// ************************************************
// Instant Developer RD3 Framework for Bootstrap
// (c) 1999-2016 Pro Gamma Srl - All rights reserved
//
// Classe Command: gestisce un comando o command set
// di tutti i tipi (menu', toolbar, popup...)
// ************************************************

function Command()
{
  // Proprieta' di questo oggetto di modello
  this.Identifier = "";         // Identificatore del comando o command set
  this.FormIndex = 0;           // Indice della form a cui appartiene il command set (0=globale)
  this.Caption = "";            // Titolo del command set o comando
  this.Tooltip = "";            // Tooltip del command set o comando
  this.Index = 0;               // Indice del comando o command set all'interno del parent
  this.FKNum = 0;               // Numero di tasto funzione per attivare questo comando
  this.Expanded = false;        // Se Vero indica che il command set e' espanso
  this.Level = 1;               // Livello del comando o command set
  this.IsMenu = true;           // Indica che questo command set e' parte del menu' laterale
  this.IsToolbar = false;       // Indica che questo command set genera una toolbar
  this.ShowNames = false;       // Se vero indica che nella toolbar occorre mostrare il nome dei comandi e non solo l'icona
  this.RequireConf = false;     // Se vero indica che prima di inviare il comando al server occorre chiedere conferma all'utente
  this.ConfText = "";           // Testo della richiesta di conferma (se RequireConf = true)
  this.ToolName = "";           // Nome da mostrare nella toolbar quando ShowNames = true
  this.Image = "";              // Icona del command o command set
  this.CommandCode = "";        // Codice del comando
  this.Enabled = true;          // Comando / Command set abilitato
  this.Visible = true;          // Comando / Command set visibile
  this.ToolCont = -1;           // -1 = toolbar globale, 0 = toolbar di form, > 0 = toolbar di frame
  this.UseHL = false;           // Hilight della toolbar?
  this.FormList = false;        // Form List?
  this.Accell = "";             // Tasto Acceleratore?
  this.CommandCanDrag = false;  // Il comando puo' essere draggato?
  this.CommandCanDrop = false;  // Il comando puo' accettare Drag?
  this.Badge = "";              // Badge del comando
  this.ClassName = "";          // Classe del comando
  //this.Width = 0;             // Larghezza calcolata da IN.DE solo per button bar
  //
  // Oggetti secondari gestiti da questo oggetto di modello
  this.Commands = new Array();  // Elenco dei comandi o command set di livello successivo
  //
  // Variabili per la gestione di questo oggetto di modello
  this.BBParent = null;               // Oggetto Button Bar padre di questo command set durante la visualizzazione in Button Bar
  this.ToolTimer = null;              //
  this.IsClosing = false;             // true se c'e' una animazione di chiusura su questo menu popup
  this.Gfx = null;                    // Animazione attualmente attiva sul menu
  this.ParentCmdSet = null;           // CmdSet Padre
  this.RecalcSeparatorLayout = false; // True se devo ricalcolare la visibilita' dei separatori figli perche' la visibilita' di uno dei comandi e' cambiata
  //
  // Struttura per la definizione delle caratteristiche degli eventi di questo comando o command set
  this.ClickEventDef = RD3_Glb.EVENT_URGENT;
  this.ExpandEventDef = RD3_Glb.EVENT_CLIENTSIDE;
  //
  // Variabili di collegamento con il DOM
  this.Realized = false; // Se vero, gli oggetti del DOM sono gia' stati creati
  //
  // Oggetti visuali relativi alla visualizzazione in Menu
  this.MyBox = null;          // Div che contiene tutto il command set / command
  this.CmdBox = null;         // Div che contiene la riga singola (testata del command set oppure command)
  this.MenuBox = null;        // Div che contiene tutti gli altri comandi
  this.BranchImg = null;      // Immagine che rappresenta il ramo dell'albero dei comandi
  this.CommandLink = null;    // Link del comando
  this.CommandImg = null;     // immagine relativa al comando / command set
  this.CommandImgDX = null;   // immagine relativa al comando / command set e posizionata a destra
  this.CommandText = null;    // nodo testuale del comando
  this.MenuSep = null;        // Separatore dal menu' precedente (solo menu' primo livello)
  this.MenuSepImg = null;     // Immagine nel separatore (solo menu' primo livello)
  this.BadgeObj = null;       // Badge del comando
  //
  // Oggetti visuali relativi alla visualizzazione in Toolbar
  this.MyToolBox = null;  // Span contenitore del comando o del command set
  this.ToolImg = null;    // Immagine del comando
  this.ToolText = null;   // Span che coniene il nome di un comando in toolbar
  //
  // Oggetti visuali relativi alla visualizzazione in Button Bar
  this.ButtonBox = null;  // Box contenitore del comando o dei comandi figli nella renderizzazione in ButtonBar
  this.Button = null;     // Pulsante realizzato in Button bar
  //
  // Oggetti visuali relativi alla visualizzazione in popup
  this.PopupContainerBox = null; // Box che contiene l'intero menu' popup
  this.PopupItemBox = null;      // Box (tr) che contiene l'item della singola linea in popup
  this.PopupIconCell = null;     // td che contiene l'icona della riga del menu'
  this.PopupTextCell = null;     // td che contiene l'icona della caption del menu'
  this.PopupText = null;         // Span che contiene il testo
  this.PopupImageBox = null;     // Icona del menu' popup
  //
  this.PopupTimer = 0;           // Timer per l'apertura dei popup di secondo livello
  //this.DragMenu = true;        // E' stato fatto un drag sul menu o sulla toolbar?
  //this.CallBackFunction = null;// Eventuale funzione callback da chiamare se il comando e' lato client
}


// *******************************************************************
// Inizializza questo CommandList leggendo i dati da un nodo <cmh> XML
// *******************************************************************
Command.prototype.LoadFromXml = function (node)
{
  // Inizializzo le proprieta' locali
  this.LoadProperties(node);
  //
  var objlist = node.childNodes;
  var n = objlist.length;
  //
  // Ciclo su tutti i nodi che rappresentano oggetti figli
  for (var i = 0; i < n; i++)
  {
    var objnode = objlist.item(i);
    var nome = objnode.nodeName;
    //
    // In base al tipo di oggetto, invio il messaggio di caricamento
    switch (nome)
    {
      case "cmd":
        {
          var cmd = new Command();
          cmd.LoadFromXml(objnode);
          //
          // Memorizzo il comando in cui e' contenuto
          cmd.ParentCmdSet = this;
          //
          // "spengo" IsMenu o IsToolbar se io non li ho
          cmd.IsToolbar = (cmd.IsToolbar && this.IsToolbar);
          cmd.IsMenu = (cmd.IsMenu && this.IsMenu);
          //
          // Informo il nuovo comando figlio della form per cui valgo io
          cmd.FormIndex = this.FormIndex;
          //
          this.Commands.push(cmd);
          //
          // Se il padre (io) e' gia' stato realizzato realizzo subito il nuovo figlio
          if (this.Realized) {
            cmd.Realize(this.MenuBox, this.MyToolBox);
            //
            // Confermo che sono collassabile, potevo non esserlo
            if (this.CmdBox)
              this.CmdBox.setAttribute("data-toggle", "collapse");
          }
        }
        break;
    }
  }
}


// **********************************************************************
// Esegue un evento di change che riguarda le proprieta' di questo oggetto
// **********************************************************************
Command.prototype.ChangeProperties = function (node)
{
  // Normale cambio di proprieta'
  this.LoadFromXml(node);
}


// **************************************************************
// Inizializza le proprieta' di questo oggetto leggendole dal
// nodo xml arrivato.
// **************************************************************
Command.prototype.LoadProperties = function (node)
{
  // Ciclo su tutti gli attributi del nodo
  var attrlist = node.attributes;
  var n = attrlist.length;
  //
  for (var i = 0; i < n; i++)
  {
    var attrnode = attrlist.item(i);
    var nome = attrnode.nodeName;
    var valore = attrnode.nodeValue;
    //
    switch (nome)
    {
      case "frm":
        this.SetFormIndex(parseInt(attrnode.nodeValue));
        break;
      case "cap":
        this.SetCaption(attrnode.nodeValue);
        break;
      case "tip":
        this.SetTooltip(attrnode.nodeValue);
        break;
      case "idx":
        this.SetIndex(attrnode.nodeValue);
        break;
      case "exp":
        this.SetExpanded(attrnode.nodeValue == "1");
        break;
      case "lev":
        this.SetLevel(parseInt(attrnode.nodeValue));
        break;
      case "ism":
        this.SetMenu(attrnode.nodeValue == "1");
        break;
      case "ist":
        this.SetToolbar(attrnode.nodeValue == "1");
        break;
      case "shn":
        this.SetShowNames(attrnode.nodeValue == "1");
        break;
      case "rqc":
        this.SetRequireConfirmation(attrnode.nodeValue == "1");
        break;
      case "rqt":
        this.SetConfirmationText(attrnode.nodeValue);
        break;
      case "tnm":
        this.SetToolName(attrnode.nodeValue);
        break;
      case "img":
        this.SetImage(attrnode.nodeValue);
        break;
      case "ena":
        this.SetEnabled(attrnode.nodeValue == "1");
        break;
      case "vis":
        this.SetVisible(attrnode.nodeValue == "1");
        break;
      case "cnt":
        this.SetToolCont(parseInt(attrnode.nodeValue));
        break;
      case "fkn":
        this.SetFKNum(parseInt(attrnode.nodeValue));
        break;
      case "cco":
        this.SetCommandCode(attrnode.nodeValue);
        break;
      case "wid":
        this.SetWidth(parseInt(attrnode.nodeValue));
        break;
      case "thl":
        this.SetUseHL(attrnode.nodeValue == "1");
        break;
      case "fli":
        this.SetFormList(attrnode.nodeValue == "1");
        break;
      case "acc":
        this.SetAccell(attrnode.nodeValue);
        break;
      case "cdg":
        this.SetCanDrag(attrnode.nodeValue == "1");
        break;
      case "cdp":
        this.SetCanDrop(attrnode.nodeValue == "1");
        break;
      case "bdg":
        this.SetBadge(attrnode.nodeValue);
        break;

      case "cln": 
        this.SetClassName(attrnode.nodeValue); 
        break;
      
      case "del": 
        this.DeleteCommand(); 
        break;

      case "clk":
        this.ClickEventDef = parseInt(attrnode.nodeValue);
        break;
      case "exe":
        this.ExpandEventDef = parseInt(attrnode.nodeValue);
        break;

      case "exa":
        this.ExpandAnimDef = valore;
        break;
      case "poa":
        this.PopupAnimDef = valore;
        break;
      case "cmds":
        this.IsCmdSet = true; 
        break;

      case "id":
        this.Identifier = valore;
        RD3_DesktopManager.ObjectMap.add(valore, this);
        break;
    }
  }
}


// *******************************************************************
// Setter delle proprieta'
// *******************************************************************
Command.prototype.SetFormIndex = function (value)
{
  // Devo solo impostare il valore in quanto questa proprieta'
  // non puo' cambiare dopo che l'oggetto e' stato realizzato
  this.FormIndex = value;
}

Command.prototype.SetIndex = function (value)
{
  // Devo solo impostare il valore in quanto questa proprieta'
  // non puo' cambiare dopo che l'oggetto e' stato realizzato
  this.Index = value;
}

Command.prototype.SetFKNum = function (value)
{
  // Devo solo impostare il valore in quanto questa proprieta'
  // non puo' cambiare dopo che l'oggetto e' stato realizzato
  this.FKNum = value;
}

Command.prototype.SetAccell = function (value)
{
  // Devo solo impostare il valore in quanto questa proprieta'
  // non puo' cambiare dopo che l'oggetto e' stato realizzato
  this.Accell = value;
}

Command.prototype.SetFormList = function (value)
{
  // Devo solo impostare il valore in quanto questa proprieta'
  // non puo' cambiare dopo che l'oggetto e' stato realizzato
  this.FormList = value;
}

Command.prototype.SetWidth = function (value)
{
  // Devo solo impostare il valore in quanto questa proprieta'
  // non puo' cambiare dopo che l'oggetto e' stato realizzato
  this.Width = value;
  if (RD3_Glb.IsTouch() && !RD3_Glb.IsMobile())
    this.Width = value * 1.3;
}

Command.prototype.SetUseHL = function (value)
{
  // Devo solo impostare il valore in quanto questa proprieta'
  // non puo' cambiare dopo che l'oggetto e' stato realizzato
  this.UseHL = value;
}

Command.prototype.SetFKNum = function (value)
{
  // Devo solo impostare il valore in quanto questa proprieta'
  // non puo' cambiare dopo che l'oggetto e' stato realizzato
  this.FKNum = value;
}

Command.prototype.SetLevel = function (value)
{
  // Devo solo impostare il valore in quanto questa proprieta'
  // non puo' cambiare dopo che l'oggetto e' stato realizzato
  this.Level = value;
}

Command.prototype.SetMenu = function (value)
{
  // Devo solo impostare il valore in quanto questa proprieta'
  // non puo' cambiare dopo che l'oggetto e' stato realizzato
  this.IsMenu = value;
}

Command.prototype.SetToolbar = function (value)
{
  // Devo solo impostare il valore in quanto questa proprieta'
  // non puo' cambiare dopo che l'oggetto e' stato realizzato
  this.IsToolbar = value;
}

Command.prototype.SetShowNames = function (value)
{
  // Devo solo impostare il valore in quanto questa proprieta'
  // non puo' cambiare dopo che l'oggetto e' stato realizzato
  this.ShowNames = value;
}

Command.prototype.SetRequireConfirmation = function (value)
{
  // Devo solo impostare il valore in quanto questa proprieta'
  // non puo' cambiare dopo che l'oggetto e' stato realizzato
  this.RequireConf = value;
}

Command.prototype.SetConfirmationText = function (value)
{
  // Devo solo impostare il valore in quanto questa proprieta'
  // non puo' cambiare dopo che l'oggetto e' stato realizzato
  this.ConfText = value;
}

Command.prototype.SetToolCont = function (value)
{
  if (value != undefined)
    this.ToolCont = value;
  //
  // Questa proprieta' puo' cambiare anche dopo, in quanto e' quando il pannello
  // si apre che il command set viene a sapere che gli appartiene
  if (this.Realized)
  {
    this.ActiveFormChanged();
  }
}

Command.prototype.SetToolName = function (value)
{
  this.ToolName = value;
  //
  if (this.Realized && !this.IsSeparator())
  {
    // Se sono in una toolbar e devo mostrare la caption
    if (this.ToolText != null) {
      if (RD3_Glb.IsIconString(this.Caption))
        this.ToolText.innerHTML = RD3_Glb.HandleIconString(this.Caption);
      else
        this.ToolText.textContent = this.Caption;
    }
  }
}

Command.prototype.SetCommandCode = function (value)
{
  // Devo solo impostare il valore in quanto questa proprieta'
  // non puo' cambiare dopo che l'oggetto e' stato realizzato
  this.CommandCode = value;
}


Command.prototype.SetCaption = function (value)
{
  var old = this.Caption;
  if (value != undefined)
    this.Caption = value;
  //
  if (this.Realized && !this.IsSeparator()) {
    if (this.CommandText)
      this.CommandText.innerHTML = RD3_Glb.HandleIconString(this.Caption);
    //
    if (this.Button)
      this.Button.innerHTML = RD3_Glb.HandleIconString(this.Caption);
    //
    if (this.PopupText)
      this.PopupText.innerHTML = RD3_Glb.HandleIconString(this.Caption);
    //
    // Se sono in una toolbar e devo mostrare la caption
    if (this.ToolText != null)
      this.ToolText.innerHTML = RD3_Glb.HandleIconString(this.Caption);
  }
}

Command.prototype.SetTooltip = function (value)
{
  if (value != undefined)
    this.Tooltip = value;
  //
  if (this.Realized && !this.IsSeparator()) {
    var s = RD3_KBManager.GetFKTip(this.FKNum);
    if (s == "" && this.CommandCode != "")
      s = " (" + this.CommandCode + ")";
    //
    if (this.CommandLink)
      RD3_TooltipManager.SetObjTitle(this.CommandLink, this.Tooltip + s);
    //
    if (this.MyToolBox && !this.IsCmdSet)
      RD3_TooltipManager.SetObjTitle(this.MyToolBox, this.Tooltip + s);
    //
    if (this.Button)
      RD3_TooltipManager.SetObjTitle(this.Button, this.Tooltip + s);
    //
    if (this.PopupText)
      RD3_TooltipManager.SetObjTitle(this.PopupText, this.Tooltip + s);
    //
    // Gestisco i bottoni senza immagine
    if (this.Image == "" && this.ToolImg)
      RD3_TooltipManager.SetObjTitle(this.ToolImg, this.Tooltip + s);
  }
}

Command.prototype.SetBadge = function (value)
{
  if (value != undefined)
    this.Badge = value;
  //
  if (this.Realized && !this.IsSeparator() && this.ShowBadge())
  {
    if (this.Badge == "")
    {
      if (this.BadgeObj != null && this.BadgeObj.parentNode)
        this.BadgeObj.parentNode.removeChild(this.BadgeObj);
      //
      this.BadgeObj = null;
    }
    else
    {
      if (this.BadgeObj == null)
      {
        this.BadgeObj = document.createElement("span");
        this.BadgeObj.className = "badge";
        //
        if (this.CommandLink)
          this.CommandLink.appendChild(this.BadgeObj);
        else if (this.MyToolBox)
          this.MyToolBox.appendChild(this.BadgeObj);
      }
      //
      this.BadgeObj.innerHTML = this.Badge;
    }
  }
}

Command.prototype.ShowBadge = function (value)
{
  return this.IsMenu || (this.IsToolbar && !this.IsCmdSet);
}

Command.prototype.SetClassName = function (value) 
{
  var old = this.ClassName;
  if (value != undefined)
    this.ClassName = value;
  //
  if (this.Realized && (old != this.ClassName || !value)) 
  {
    // nel caso di SetClassName() non devo rimuovere il vecchio valore (che e' uguale all'attuale)
    if (old && this.ClassName != old) 
    {
      RD3_Glb.RemoveClass2(this.MyBox, old);
      RD3_Glb.RemoveClass2(this.MyToolBox, old);
      RD3_Glb.RemoveClass2(this.PopupContainerBox, old);
      RD3_Glb.RemoveClass2(this.PopupItemBox, old);
    }
    if (this.ClassName) 
    {
      RD3_Glb.AddClass(this.MyBox, this.ClassName);
      RD3_Glb.AddClass(this.MyToolBox, this.ClassName);
      RD3_Glb.AddClass(this.PopupContainerBox, this.ClassName);
      RD3_Glb.AddClass(this.PopupItemBox, this.ClassName);
    }
  }
}

Command.prototype.SetExpanded = function (value)
{
  var old = this.Expanded;
  if (value != undefined)
    this.Expanded = value;
  //
  if (this.Realized && this.IsCmdSet && this.IsMenu)
  {
    if (this.ChildrenBox) {
      if (this.Expanded)
        RD3_Glb.AddClass(this.ChildrenBox, "in");
      else
        RD3_Glb.RemoveClass(this.ChildrenBox, "in");
    }
  }
}


// ********************************************************************************
// Resituisce il popupframe in cui questo menu' e' incluso
// ********************************************************************************
Command.prototype.GetPopover = function ()
{
  var obj = this;
  while (obj)
  {
    if (obj.Popover)
      return obj.Popover;
    obj = obj.ParentCmdSet;
  }
}


// ********************************************************************************
// Carica la lista successiva in caso di menu' mobile
// ********************************************************************************
Command.prototype.LoadNestedMenu = function (flInner, flpopover)
{
 
}


Command.prototype.SetImage = function (value, force)
{
  var old = this.Image;
  if (value != undefined)
    this.Image = value;
  //
  if (this.Image == "") {
    if (this.CommandImg)
      this.CommandImg.style.display = "none";
    if (this.ToolImg)
      this.ToolImg.style.display = "none";
    if (this.PopupImageBox)
      this.PopupImageBox.style.display = "none";
  }
  else {
    if (this.CommandImg)
      this.CommandImg.src = RD3_Glb.GetImgSrc("images/" + this.Image);
    //
    if (!this.ToolImg && this.IsToolbar && this.MyToolBox) {
      this.ToolImg = document.createElement("img");
      this.ToolImg.setAttribute("id", this.Identifier + ":img");
      this.ToolImg.className = "toolbar-image";
      this.MyToolBox.appendChild(this.ToolImg);
    }
    if (this.ToolImg)
      this.ToolImg.src = RD3_Glb.GetImgSrc("images/" + this.Image);
    if (this.PopupImageBox)
      this.PopupImageBox.src = RD3_Glb.GetImgSrc("images/" + this.Image);
  }
}

Command.prototype.SetEnabled = function (value)
{
  var old = this.Enabled;
  if (value != undefined)
    this.Enabled = value;
  //
  if (this.Realized && !this.IsSeparator()) {
    if (this.Enabled) {
      if (this.IsMenu && this.CommandLink)
        RD3_Glb.RemoveClass(this.CommandLink, "menu-command-disabled");
      if (this.IsToolbar && this.MyToolBox)
        this.MyToolBox.disabled = "";
      if (this.Button)
       this.Button.disabled = "";
      if (this.PopupItemBox)
        RD3_Glb.RemoveClass(this.PopupItemBox, "disabled");
    }
    else {
      if (this.IsMenu && this.CommandLink)
        RD3_Glb.AddClass(this.CommandLink, "menu-command-disabled");
      if (this.IsToolbar && this.MyToolBox)
        this.MyToolBox.disabled = "disabled";
      if (this.Button)
       this.Button.disabled = "disabled";
      if (this.PopupItemBox)
        RD3_Glb.AddClass(this.PopupItemBox, "disabled");
    }
  }
}

Command.prototype.SetVisible = function (value)
{
  var old = this.Visible;
  //
  if (value != undefined)
    this.Visible = value;
  //
  if (this.Realized && (old != this.Visible || value == undefined)) {
    var vis = this.IsVisible();
    var dis = vis ? "" : "none";
    if (this.MyBox && this.MyBox.style.display != dis)
      this.MyBox.style.display = dis;
    if (this.MyToolBox != RD3_DesktopManager.WebEntryPoint.RightHeaderCommandsList && this.MyToolBox && this.MyToolBox.style.display != dis)
      this.MyToolBox.style.display = dis;
    if (this.Button && this.Button.style.display != dis)
      this.Button.style.display = dis;
    if (this.PopupItemBox && this.PopupItemBox.style.display != dis)
      this.PopupItemBox.style.display = dis;
    //
    // Quando cambia la visibilita' a run-time di un commandset di primo livello dell'applicazione 
    // devo far valutare la struttura, magari va nascosto/mostrato qualcosa
    // (mi serve solo per il menu' superiore, per quello laterale non serve - deve rimanere sempre visibile a causa della lista delle videate)
    if (this.Level === 1 && this.IsCmdSet && this.ToolCont < 0 && value != undefined && !RD3_DesktopManager.WebEntryPoint.HasSideMenu())
      RD3_DesktopManager.WebEntryPoint.AdjustMenuVisibility();
  }
}


Command.prototype.SetCanDrag = function (value)
{
  if (value != undefined)
    this.CommandCanDrag = value;
  //
  // Non puo' cambiare dopo l'inizializzazione
}


Command.prototype.SetCanDrop = function (value)
{
  if (value != undefined)
    this.CommandCanDrop = value;
  //
  // Non puo' cambiare dopo l'inizializzazione
}


// ***************************************************************
// Crea gli oggetti DOM utili a questo oggetto
// L'oggetto parent indica all'oggetto dove devono essere contenuti
// i suoi oggetti figli nel DOM
// ***************************************************************
Command.prototype.Realize = function (parentm, parentt)
{
  // realizzo i miei oggetti visuali
  if (this.IsMenu)
  {
    if (RD3_DesktopManager.WebEntryPoint.HasSideMenu())
      this.RealizeSideMenu(parentm);
    else
      this.RealizeMenuBar(parentm);
  }
  if (this.IsToolbar)
    this.RealizeToolbar(parentt);
  //
  // Poi chiedo ai miei figli di realizzarsi
  var toolIdx = 0;
  var toolbCont = this.BtGroups ? this.BtGroups[toolIdx] : this.MyToolBox;
  var n = this.Commands.length;
  for (var i = 0; i < n; i++)
  {
    // Se sono un commandset di secondo livello contenuto in un cmdset di primo livello il mio
    // flag IsToolbar e' sempre false, non creo un mio oggetto visuale e passo ai miei figli il parentt
    if (this.MyToolBox == null && !toolbCont)
      toolbCont = parentt;
    //
    //
    if (this.Commands[i].IsSeparator() && this.BtGroups && this.BtGroups.length > toolIdx) {
      toolIdx ++;
      toolbCont = this.BtGroups[toolIdx];
    } 
    //
    this.Commands[i].Realize(this.MenuBox, toolbCont);
  }
  
  
  //
  // Eseguo l'impostazione iniziale delle mie proprieta' (quelle che cambiano l'aspetto visuale)
  this.Realized = true;
  this.SetCaption();
  this.SetTooltip();
  this.SetToolName();
  this.SetExpanded();
  this.SetImage();
  this.SetEnabled();
  this.SetVisible();
  this.SetBadge();
  this.SetClassName();
}


// ***************************************************************
// Menu
// ***************************************************************
Command.prototype.RealizeMenuBar = function (parent)
{
  if (!parent)
    return;
  //
  // Se sono un commandSet di primo livello creo il dropdown
  if (this.Level === 1 && this.IsCmdSet) {
    this.MyBox = document.createElement("li");
    this.MyBox.setAttribute("id", this.Identifier);
    this.MyBox.className = "dropdown";
    parent.appendChild(this.MyBox);
    //
    this.CommandLink = document.createElement("a");
    this.CommandLink.className = "dropdown-toggle";
    this.CommandLink.setAttribute("id", this.Identifier + ":link");
    this.CommandLink.setAttribute("data-toggle", "dropdown");
    this.CommandLink.setAttribute("role", "button");
    this.CommandLink.setAttribute("aria-haspopup", "true");
    this.CommandLink.setAttribute("aria-expanded", "false");
    this.MyBox.appendChild(this.CommandLink);
    //
    this.CommandText = document.createElement("span");
    this.CommandLink.appendChild(this.CommandText);
    //
    var caret = document.createElement("span");
    caret.className = "caret";
    this.CommandLink.appendChild(caret);
    //
    // Creo il box che conterra' i sottocomandi
    this.MenuBox = document.createElement("ul");
    this.MenuBox.setAttribute("id", this.Identifier + ":sub");
    this.MenuBox.className = "dropdown-menu";
    this.MyBox.appendChild(this.MenuBox);
    //
    if (this.FormList)
      RD3_DesktopManager.WebEntryPoint.FormListBarMenuBox = this.MenuBox;
  }
  else {  // Se sono un comando mi creo nel padre
    this.MyBox = document.createElement("li");
    this.MyBox.setAttribute("id", this.Identifier);
    parent.appendChild(this.MyBox);
    //
    if (this.IsSeparator()) {
      this.MyBox.className = "divider";
      this.MyBox.setAttribute("role", "separator");
    }
    else {
      this.CommandLink = document.createElement("a");
      this.CommandLink.className = "has-pointer";
      this.MyBox.appendChild(this.CommandLink);
      //
      // Creo l'immagine del commandset
      this.CommandImg = document.createElement("img");
      this.CommandImg.setAttribute("id", this.Identifier + ":image");
      this.CommandLink.appendChild(this.CommandImg);
      //
      this.CommandText = document.createElement("span");
      this.CommandLink.appendChild(this.CommandText);
      //
      // Gestisco il click
      var _this = this;
      this.CommandLink.onclick = function (ev) {
        _this.OnClick(ev);
      };
    }
  }
}


// ***************************************************************
// Nel caso mobile crea un menu'
// ***************************************************************
Command.prototype.RealizeGroupedMenu = function (parent)
{

}


// ***************************************************************
// Crea gli oggetti DOM per il menu
// L'oggetto parent indica all'oggetto dove devono essere contenuti
// i suoi oggetti figli nel DOM
// ***************************************************************
Command.prototype.RealizeSideMenu = function (parent)
{
  if (!parent)
    return;
  //
  // CommandSet
  if (this.IsCmdSet) {
    this.MyBox = document.createElement("div");
    this.MyBox.setAttribute("id", this.Identifier);
    this.MyBox.className = "panel panel-default" + (this.Level > 1 ? " sub-cms" : "");
    parent.appendChild(this.MyBox);
    //
    // Creo il div per la testata del comando
    this.CmdBox = document.createElement("div");
    this.CmdBox.setAttribute("id", this.Identifier + ":header");
    this.CmdBox.className = "panel-heading has-pointer";
    if (this.Commands && this.Commands.length > 0)
      this.CmdBox.setAttribute("data-toggle", "collapse");
    if (this.Level == 1)
      this.CmdBox.setAttribute("data-parent", "#menu-accordion");
    this.CmdBox.setAttribute("data-target", RD3_Glb.escapeID("#" + this.Identifier + ":sub"));
    this.MyBox.appendChild(this.CmdBox);
    //
    var title = document.createElement("h4");
    title.className = "panel-title";
    this.CmdBox.appendChild(title);
    //
    this.CommandLink = document.createElement("div");
    title.appendChild(this.CommandLink);
    this.CommandLink.setAttribute("id", this.Identifier + ":link");
    //
    // Creo l'immagine del commandset
    this.CommandImg = document.createElement("img");
    this.CommandImg.setAttribute("id", this.Identifier + ":image");
    this.CommandLink.appendChild(this.CommandImg);
    //
    this.CommandText = document.createElement("span");
    this.CommandLink.appendChild(this.CommandText);
    //
    // Creo il box che conterra' i sottocomandi
    this.ChildrenBox = document.createElement("div");
    this.ChildrenBox.setAttribute("id", this.Identifier + ":sub");
    this.ChildrenBox.className = "panel-collapse collapse";
    this.MyBox.appendChild(this.ChildrenBox);
    //
    var pbody = document.createElement("div");
    pbody.className = "panel-body";
    this.ChildrenBox.appendChild(pbody);
    //
    var ptable = document.createElement("table");
    ptable.className = "table";
    pbody.appendChild(ptable);
    //
    this.MenuBox = document.createElement("tbody");
    ptable.appendChild(this.MenuBox);
    //
    // Gestisco gli eventi di collassamento/espansione
    var cmds = this;
    $(this.ChildrenBox).on('hidden.bs.collapse', function () {
      var ev = new IDEvent("clk", cmds.Identifier, null, cmds.ExpandEventDef);
    })
    $(this.ChildrenBox).on('shown.bs.collapse', function () {
      var ev = new IDEvent("clk", cmds.Identifier, null, cmds.ExpandEventDef);
    })
  }
  else { // Comando
    if (this.IsSeparator())
    {

    }
    else {
      this.MyBox = document.createElement("tr");
      this.MyBox.setAttribute("id", this.Identifier);
      parent.appendChild(this.MyBox);
      //
      var td = document.createElement("td");
      this.MyBox.appendChild(td);
      //
      this.CommandLink = document.createElement("a");
      this.CommandLink.className = "has-pointer";
      td.appendChild(this.CommandLink);
      //
      // Creo l'immagine del commandset
      this.CommandImg = document.createElement("img");
      this.CommandImg.setAttribute("id", this.Identifier + ":image");
      this.CommandLink.appendChild(this.CommandImg);
      //
      this.CommandText = document.createElement("span");
      this.CommandLink.appendChild(this.CommandText);
      //
      // Gestisco il click
      var _this = this;
      this.CommandLink.onclick = function (ev) {
        _this.OnClick(ev);
      };
    }
  }
}


// ***************************************************************
// Crea gli oggetti DOM per la toolbar
// L'oggetto parent indica all'oggetto dove devono essere contenuti
// i suoi oggetti figli nel DOM
// ***************************************************************
Command.prototype.RealizeToolbar = function (parent)
{
  if (!parent)
    return;
  //
  // Sono un command set
  if (this.IsCmdSet)
  {
    // Sono un CommandSet: devo verificare se sono un CommandSet di primo livello
    // Se non sono un CmdSet di primo livello non devo creare nessun elemento nel DOM
    if (this.Level == 1)
    {
      if (this.FormIndex <=0 && this.ToolCont<0) {
        // Sono una toolbar globale, devo disegnarmi in modo specifico
        this.MyToolBox = document.createElement("li");
        this.MyToolBox.setAttribute("id", this.Identifier);
        this.MyToolBox.className = "pull-right navbar-toolbar";
        parent.appendChild(this.MyToolBox);
      }
      else {
        // Creo l'unico oggetto visuale necessario
        this.MyToolBox = document.createElement("div");
        this.MyToolBox.setAttribute("id", this.Identifier);
        this.MyToolBox.className = "btn-toolbar";
        parent.appendChild(this.MyToolBox);
        //
        // Adesso devo verificare se tra i miei figli ci sono separatori, in quel caso devo crear  tanti btn-group, altrimenti ne creao uno solo
        this.BtGroups = [];
        //
        var b = document.createElement("div");
        b.className = "btn-group";
        this.MyToolBox.appendChild(b);
        this.BtGroups.push(b);
        for (var i = 0; i < this.Commands.length; i++) {
           if (this.Commands[i].IsSeparator()) {
              b = document.createElement("div");
              b.className = "btn-group";
              this.MyToolBox.appendChild(b);
              this.BtGroups.push(b);
           }
        }
      }
    }
  }
  else // sono un comando
  {
    if (this.IsSeparator())
    {
      
    }
    else
    {
      // Sono un comando: creo gli oggetti DOM comuni
      this.MyToolBox = document.createElement("button");
      this.MyToolBox.setAttribute("id", this.Identifier);
      this.MyToolBox.className = "btn btn-default";
      //
      if (this.ParentCmdSet && this.ParentCmdSet.FormIndex <= 0 && this.ParentCmdSet.ToolCont < 0)
        this.MyToolBox.className += " navbar-btn";
      //
      // Gestisco il click
      var _this = this;
      this.MyToolBox.onclick = function(ev) {
        _this.OnClick(ev);
      };
      //
      // Inserisco gli oggetti nel DOM
      parent.appendChild(this.MyToolBox);
      //
      // Gestisco le impostazioni relative al nome del comando se deve essere mostrata
      if (this.ShowNames || (this.Caption !== "")) {
        this.ToolText = document.createElement("span");
        this.MyToolBox.appendChild(this.ToolText);
      }
    }
  }
}


// ***************************************************************
// Crea gli oggetti DOM per la button bar (crea dei bottoni)
// Parent e' l'oggetto ButtonBar che contiene questo CommandSet
// ***************************************************************
Command.prototype.RealizeButtonBar = function (parent)
{
  if (this.IsCmdSet)
  {
    // Creo gli elementi del DOM
    parent.ButtonBarContainer = document.createElement("div");
    if (parent.VerticalLayout)
      parent.ButtonBarContainer.className = "btn-group-vertical button-bar-group";
    else
      parent.ButtonBarContainer.className = "btn-group button-bar-group";
    //
    // Sono un commandSet: devo fare realizzare tutti i miei figli come bottoni
    var n = this.Commands.length;
    for (var i = 0; i < n; i++)
      this.Commands[i].RealizeButtonBar(parent);
    //
    // Inserisco gli elementi nel DOM e mi memorizzo i riferimenti
    parent.ContentBox.appendChild(parent.ButtonBarContainer);
    this.ButtonBox = parent.ButtonBarContainer;
    //
    this.SetVisible(this.Visible);
    this.SetEnabled(this.Enabled);
  }
  else
  {
    if (this.IsSeparator())
    {
     
    }
    else
    {
      // Sono un Comando: mi devo realizzare come Bottone
      this.Button = document.createElement("button");
      this.Button.setAttribute("id", this.Identifier);
      //
      // Assegno le classi ed i tipi corretti
      this.Button.className = "btn btn-default";
      //
      //if (this.Width)
        //this.Button.style.width = this.Width + "px";
      //
      // Assegno gli eventi
      var _this = this;
      this.Button.onclick = function (ev) {
        _this.OnButtonBarClick(ev);
      };
      //
      // Aggiungo i componenti al DOM
      parent.ButtonBarContainer.appendChild(this.Button);
      //
      this.Realized = true;
      //
      // Impostazioni iniziali
      this.SetCaption(this.Caption);
      this.SetTooltip(this.Tooltip);
      this.SetImage(this.Image);
      this.SetEnabled(this.Enabled);
      this.SetVisible(this.Visible);
    }
  }
  //
  // Mi memorizzo in ogni caso il riferimento al padre
  this.BBParent = parent;
}


// ********************************************************************************
// Calcola le dimensioni dei div in base alla dimensione del contenuto
// Adapt Layput non vuole parametri: calcolarli
// ********************************************************************************
Command.prototype.AdaptLayout = function ()
{
  
}

// *******************************************************
// Metodo che gestisce il cambio dello stato dell'immagine
// *******************************************************
Command.prototype.OnReadyStateChange = function ()
{
  
}


// ********************************************************************************
// Toglie gli elementi visuali dal DOM perche' questo oggetto sta per essere
// distrutto
// ********************************************************************************
Command.prototype.Unrealize = function (flonlydom)
{
  // Se ero collegato a mio padremi stacco
  if (this.ParentCmdSet && !flonlydom)
    this.ParentCmdSet = null;
  //
  if (this.MyBox)
    this.MyBox.parentNode.removeChild(this.MyBox);
  //
  if (this.MyToolBox && this.MyToolBox.parentNode)
    this.MyToolBox.parentNode.removeChild(this.MyToolBox);
  //
  if (this.ButtonBox)
    this.ButtonBox.parentNode.removeChild(this.ButtonBox);
  if (this.Button)
    this.Button.parentNode.removeChild(this.Button);
  //
  if (this.FormList && RD3_DesktopManager.WebEntryPoint.FormListBarMenuBox === this.MyBox)
    RD3_DesktopManager.WebEntryPoint.FormListBarMenuBox = null;
  //
  if (!flonlydom)
  {
    // Mi tolgo dalla mappa degli oggetti
    RD3_DesktopManager.ObjectMap[this.Identifier] = null;
    this.Commands = new Array();
  }
  //
  // Passo il messaggio ai miei figli
  var n = this.Commands.length;
  for (var i = 0; i < n; i++)
    this.Commands[i].Unrealize(flonlydom);
  //
  // Elimino i riferimenti al DOM
  this.MyBox = null;
  this.CmdBox = null;
  this.MenuBox = null;
  this.CommandLink = null;
  this.CommandImg = null;
  this.CommandText = null;
  this.MenuSep = null;
  this.MenuSepImg = null;
  //
  this.MyToolBox = null;
  this.ToolImg = null;
  this.ToolText = null;
  //
  this.ButtonBox = null;
  this.Button = null;
  this.BBParent = null;
  this.Expanded = false;
  this.BadgeObj = null;
  //
  this.Realized = false;
}


// ********************************************************************************
// Toglie gli elementi visuali della Button Bar, senza eliminare questo command
// set
// ********************************************************************************
Command.prototype.UnrealizeButtonBar = function ()
{
  // Rimuovo il mio box
  if (this.ButtonBox)
    this.ButtonBox.parentNode.removeChild(this.ButtonBox);
  if (this.Button)
    this.Button.parentNode.removeChild(this.Button);
  //
  // Passo il messaggio ai miei figli
  var n = this.Commands.length;
  for (var i = 0; i < n; i++)
    this.Commands[i].UnrealizeButtonBar();
  //
  // Elimino i riferimenti al DOM
  this.ButtonBox = null;
  this.Button = null;
  this.BBParent = null;
}


// ********************************************************************************
// Ritorna vero se e' visibile
// ********************************************************************************
Command.prototype.IsVisible = function ()
{
  var vis = this.Visible;
  //
  // Se l'impostazione visible e' true devo verificare se sono globale (e quindi conta visible) oppure se sono di form
  // (Allora la visibilita' deve essere condizionata alla attivazione della form)
  if (vis)
  {
    // Verifico se sono globale o di form
    if (this.FormIndex > 0 && (this.IsMenu || this.IsToolbar))
    {
      // Dato che sono di form devo verificare se la mia form e' quella attiva
      var fnd = false;
      var actf = RD3_DesktopManager.WebEntryPoint.ActiveForm;
      if (actf && actf.IdxForm == this.FormIndex)
        fnd = true;
      //
      var nf = fnd ? 0 : RD3_DesktopManager.WebEntryPoint.StackForm.length;
      for (var t = 0; t < nf; t++)
      {
        var f = RD3_DesktopManager.WebEntryPoint.StackForm[t];
        if (f.Docked && f.IdxForm == this.FormIndex)
        {
          fnd = true;
          break;
        }
      }
      //
      if (!fnd)
        fnd = this.CheckSubForm();
      //
      if (!fnd)
        vis = false;
      //
      // Sono di form e la mia form non e' ne' quella attiva ne' quella docked...
      // Verifico se il comando e' gia' contenuto nella form... Se lo e' gia' allora sono visibile...
      // tanto se la si nasconde (es: minimizza) lei porta nel limbo anche me
      // NPQ 1185: questo e' vero solo se il comando non e' anche un menu.. perche' in quel caso e' fuori dalla form e quindi si deve nascondere
      if (!vis && !this.IsMenu && this.IsToolbar && this.MyToolBox && this.MyToolBox.parentNode && this.MyToolBox.parentNode != RD3_DesktopManager.WebEntryPoint.RightHeaderCommandsList)
        vis = true;
    }
    //
    // Se sono di frame, sono visibile se il frame e' non collassato
    if (this.ToolCont > 0 && vis)
    {
      var f = RD3_DesktopManager.ObjectMap["frm:" + this.FormIndex];
      if (f)
      {
        var fr = f.Frames[this.ToolCont - 1];
        if (fr && (fr.Collapsed || !fr.Visible || !fr.Realized))
        {
          vis = false;
        }
      }
    }
  }
  //
  return vis;
}


// ********************************************************************************
// Ritorna vero se e' abilitato
// ********************************************************************************
Command.prototype.IsEnabled = function ()
{
  var en = this.Enabled;
  //
  // Se sono abilitato verifico se sono globale (e quindi abilitato) oppure se sono di form
  // (Allora l'abilitazione deve essere condizionata alla attivazione della form)
  if (en)
  {
    // Se sono un comando di form: verifico se la mia form e' la form attiva
    if (this.FormIndex > 0 && (this.IsMenu || this.IsToolbar))
    {
      var fnd = false;
      var actf = RD3_DesktopManager.WebEntryPoint.ActiveForm;
      if (actf && actf.IdxForm == this.FormIndex)
        fnd = true;
      //
      var nf = fnd ? 0 : RD3_DesktopManager.WebEntryPoint.StackForm.length;
      for (var t = 0; t < nf; t++)
      {
        var f = RD3_DesktopManager.WebEntryPoint.StackForm[t];
        if (f.Docked && f.IdxForm == this.FormIndex)
        {
          fnd = true;
          break;
        }
      }
      //
      if (!fnd)
        fnd = this.CheckSubForm();
      //
      // La form non e' stata trovata tra quelle attive, ma io sono abilitato e sono visibile: allora faccio passare il click
      if (!fnd && !this.IsMenu && this.IsToolbar && this.MyToolBox && this.MyToolBox.parentNode && this.MyToolBox.parentNode != RD3_DesktopManager.WebEntryPoint.ToolBarBox)
        fnd = true;
      //
      if (!fnd)
        en = false;
      //
      // Bottone di form: non controllo la form attiva
      if (this.BBParent)
        en = true;
    }
  }
  //
  return en;
}


// ********************************************************************************
// Ritorna vero se e' un separatore
// ********************************************************************************
Command.prototype.IsSeparator = function ()
{
  return this.Caption == "";
}


// ********************************************************************************
// Gestore evento di click
// ********************************************************************************
Command.prototype.OnClick = function (evento, confirmed)
{
  var wep = RD3_DesktopManager.WebEntryPoint;
  //
  // Voglio evitare un doppio click sugli oggetti
  if (RD3_Glb.IsAndroid())
    RD3_DDManager.ChompClick();
  //
  wep.CmdObj.ClosePopup(this.IsCmdSet ? this : null);
  //
  // Chiudo anche il popover se ero un comando
  if (!this.IsCmdSet && this.FormIndex == 0)
    RD3_DDManager.ClosePopup(true);
  //
  if (this.IsVisible() && this.IsEnabled())
  {
    if (this.RequireConf && !confirmed)
    {
      // Chiedo conferma per il comando
      this.MsgBox = new MessageBox(this.Caption + ": " + RD3_ServerParams.ConfirmMenu, RD3_Glb.MSG_CONFIRM, false);
      this.MsgBox.CallBackFunction = new Function("ev", "return RD3_DesktopManager.CallEventHandler('" + this.Identifier + "', 'OnClick', ev, true)");
      this.MsgBox.Open();
      return;
    }
    if (this.RequireConf && confirmed)
    {
      if (this.MsgBox.UserResponse != "Y")
        return;
    }
    //
    // Notifico l'evento...
    var evt = this.IsCmdSet ? this.ExpandEventDef : this.ClickEventDef;
    var ev = new IDEvent("clk", this.Identifier, evento, evt);
    //
    // Sistemo l'elemento attivo per i comandi foglia
    if (!this.IsCmdSet && this.MyBox) {
      // Per prima cosa rimuoviamo gli altri comandi selezionati
      var vl = document.getElementsByClassName("side_selected_command");
      while (vl.length > 0)
        RD3_Glb.RemoveClass(vl[0], "side_selected_command");
      //
      // Aggiungiamo la classe necessaria
      RD3_Glb.AddClass(this.MyBox, "side_selected_command");
    }
    //
    if (ev.ClientSide)
    {
      if (this.IsCmdSet)
      {
        if (wep.MenuType != RD3_Glb.MENUTYPE_MENUBAR && !this.PopupTextCell && !RD3_Glb.IsMobile())
        {
          // L'esecuzione locale di un evento di espansione apre o chiude il command set
          // Se e' un menu laterale...
          this.SetExpanded(!this.Expanded);
          //
          // Non voglio far fluire l'evento al WEP se no chiude di nuovo i popup!!!
          if (evento && wep.MenuType == RD3_Glb.MENUTYPE_TASKBAR)
            RD3_Glb.StopEvent(evento);
        }
        else
        {
          // Devo aprire il command set in popup!!!
          // ed evidenziare la voce di menu' nella barra
          if (this.MyBox)
            RD3_Glb.AddClass(this.MyBox, "menu-bar-hover");
          if (this.PopupTextCell)
            RD3_Glb.AddClass(this.PopupTextCell, "popup-menu-hover");
          this.Popup(this);
          wep.CmdObj.MenuBarOpen = true;
          //
          // Non voglio far fluire l'evento al WEP se no chiude di nuovo i popup!!!
          if (evento)
            RD3_Glb.StopEvent(evento);
        }
      }
      else if (this.CallBackFunction)
        this.CallBackFunction();
    }
  }
}


// ********************************************************************************
// Gestore evento di BACK del BOTTONE
// ********************************************************************************
Command.prototype.OnBack = function (evento)
{
  
}


// ********************************************************************************
// Gestore evento di BACK del BOTTONE
// ********************************************************************************
Command.prototype.OnEndAnimation = function (evento)
{
  
}


// ********************************************************************************
// Elimina la classe hover da tutto il menu', escluso l'oggetto indicato
// ********************************************************************************
Command.prototype.SetMenuClass = function (mycmd)
{
  
}


// ********************************************************************************
// Gestore evento di mouse over su uno degli oggetti di questo comando
// ********************************************************************************
Command.prototype.OnMouseOverObj = function (evento, obj)
{
  
}


// ********************************************************************************
// Gestore evento di mouse out su uno degli oggetti di questo comando
// ********************************************************************************
Command.prototype.OnMouseOutObj = function (evento, obj)
{
  
}


// ********************************************************************************
// Gestore evento di mouse down su uno degli oggetti di questo comando
// ********************************************************************************
Command.prototype.OnMouseDownObj = function (evento, obj)
{
  
}


// ********************************************************************************
// Gestore evento di mouse up su uno degli oggetti di questo comando
// ********************************************************************************
Command.prototype.OnMouseUpObj = function ()
{
  
}


// ********************************************************************************
// Metodo chiamato quando cambia la form attiva: rifa' il controllo della visibilita'
// del comando
// ********************************************************************************
Command.prototype.ActiveFormChanged = function ()
{
  // Lavoriamo per CommandSet di tipo toolbar di Form o Frame, 
  // bisogna agganciare i commandset al punto giusto, poi per i figli va solo impostata la visibilità 
  if (this.ToolCont >= 0 && this.IsToolbar && this.IsCmdSet)
  {
    // Se e' di form la metto nella toolbar sopra la formcaption
    if (this.ToolCont == 0)
    {
      var f = RD3_DesktopManager.ObjectMap["frm:" + this.FormIndex];
      if (f && f.Realized)
      {
        RD3_Glb.RemoveClass(this.MyToolBox, "frame-toolbar-group");
        RD3_Glb.AddClass(this.MyToolBox, "form-toolbar-group");
        f.AttachToolbar(this);
      }
    }
    //
    // Se e' legata a un pannello la metto nella toolbar del pannello stesso
    if (this.ToolCont > 0)
    {
      var f = RD3_DesktopManager.ObjectMap["frm:" + this.FormIndex];
      if (f)
      {
        var fr = f.Frames[this.ToolCont - 1];
        //
        if (fr && fr.ToolbarBox && this.MyToolBox && this.MyToolBox.parentNode != fr.ToolbarBox)
        {
          if (this.MyToolBox.parentNode)
            this.MyToolBox.parentNode.removeChild(this.MyToolBox);
          //
          // Nel caso di pannello devo tenere conto delle zone
          if (fr instanceof IDPanel)
            fr.AppendCmsToToolbar(this.MyToolBox);
          else 
            fr.ToolbarBox.appendChild(this.MyToolBox);
          //
          RD3_Glb.RemoveClass(this.MyToolBox, "form-toolbar-group");
          RD3_Glb.AddClass(this.MyToolBox, "frame-toolbar-group");
          RD3_Glb.RemoveClass(this.MyToolBox, "pull-left");
          RD3_Glb.AddClass(this.MyToolBox, "pull-right");
          //
          if (fr.SmallIcons) {
            RD3_Glb.RemoveClass(this.MyToolBox, "btn-group-md");
            RD3_Glb.AddClass(this.MyToolBox, "btn-group-xs");
          }
          else {
            RD3_Glb.RemoveClass(this.MyToolBox, "btn-group-xs");
            RD3_Glb.AddClass(this.MyToolBox, "btn-group-md");
          }
        }
      }
    }
  }
  //
  this.SetVisible();
  this.SetEnabled();
  //
  var cmdl = this.Commands.length;
  for (var i = 0; i < cmdl; i++)
    this.Commands[i].ActiveFormChanged();
}


// ********************************************************************************
// Gestore evento di mouse over sul pulsante Collapse della Toolbar
// ********************************************************************************
Command.prototype.OnBBMouseOver = function (evento)
{

}


// ********************************************************************************
// Gestore evento di mouse out sul pulsante Collapse della Toolbar
// ********************************************************************************
Command.prototype.OnBBMouseOut = function (evento)
{

}


// ********************************************************************************
// Gestore evento di click sul pulsante Collapse della Toolbar
// ********************************************************************************
Command.prototype.OnButtonBarClick = function (evento)
{
  if (this.IsVisible() && this.IsEnabled())
    var ev = new IDEvent("btb", this.BBParent.Identifier, evento, this.ClickEventDef, this.Identifier);
}

// ********************************************************************************
// Apro il command set di tipo popup
// ********************************************************************************
Command.prototype.Popup = function (node)
{
  // Se il popup era gia' aperto non posso aprirlo ancora
  if (this.PopupContainerBox != null)
    this.ClosePopup();
  //
  if (this.PopupTimer)
  {
    window.clearTimeout(this.PopupTimer);
    this.PopupTimer = 0;
  }
  //
  this.RealizePopup(node);
}


// ***************************************************************
// Crea gli oggetti DOM per il popup
// ***************************************************************
Command.prototype.RealizePopupItem = function (parent)
{
  if (this.IsSeparator()) {
    this.PopupItemBox = document.createElement("li");
    this.PopupItemBox.setAttribute("id", this.Identifier + ":li");
    this.PopupItemBox.className = "divider";
  }
  else {
    // Creo il div della linea del menu'
    this.PopupItemBox = document.createElement("li");
    this.PopupItemBox.setAttribute("id", this.Identifier + ":li");
    //this.PopupItemBox.className = "popup-menu-item";
    var _this = this;
    this.PopupItemBox.onclick = function(ev) { _this.OnClick(ev); };
    //
    this.PopupTextCell = document.createElement("a");
    this.PopupTextCell.setAttribute("id", this.Identifier + ":link");
    //this.PopupTextCell.className = "popup-cell-text" + (this.IsSeparator() ? "-sep" : "");
    //
    // Creo l'img dell'icona
    this.PopupImageBox = document.createElement("img");
    this.PopupImageBox.setAttribute("id", this.Identifier + ":pm");
    this.PopupImageBox.className = "popup-menu-image";
    //
    // Creo la linea di testo
    this.PopupText = document.createElement("span");
    //
    // Creo la struttura
    this.PopupTextCell.appendChild(this.PopupImageBox);
    this.PopupTextCell.appendChild(this.PopupText);
    this.PopupItemBox.appendChild(this.PopupTextCell);
    parent.appendChild(this.PopupItemBox);
  }
  //
  // Imposto le varie proprieta' grafiche...
  this.SetVisible();
  this.SetCaption();
  this.SetTooltip();
  this.SetImage(undefined, true);
  this.SetEnabled();
  this.SetClassName();
}


// ********************************************************************************
// Vediamo se il popup e' aperto
// ********************************************************************************
Command.prototype.ClosePopup = function (filtobj)
{
  // Vediamo se lo devo chiudere veramente
  var ok = true;
  //
  var v = filtobj;
  while (v && ok)
  {
    if (v == this)
      ok = false
    v = v.ParentCmdSet;
  }
  //
  if (!ok)
    return;
  //
  // Cancello un eventuale timer di apertura...
  if (this.PopupTimer)
  {
    window.clearTimeout(this.PopupTimer);
    this.PopupTimer = 0;
  }
  //
  // Se ho un popup aperto lo chiudo con l'animazione
  if (this.PopupContainerBox != null && !this.IsClosing)
  {
    this.IsClosing = true;
    //
    // Tolgo il box dal DOM
    if (this.PopupContainerBackDrop) {
      if (this.PopupContainerBackDrop.parentNode)
        this.PopupContainerBackDrop.parentNode.removeChild(this.PopupContainerBackDrop);
      this.PopupContainerBackDrop = null;
      this.PopupContainerBox = null;
    }
    //
    // Passo il messaggio ai comandi figli
    var n = this.Commands.length;
    for (var i = 0; i < n; i++)
      this.Commands[i].ClosePopup(filtobj);
  }
}


// ********************************************************************************
// Ritorna l'oggetto principale del DOM
// ********************************************************************************
Command.prototype.GetDOMObj = function (type)
{
  if (type == "toolbar")
    return this.MyToolBox;
  //
  // default=menu
  if (this.CommandLink)
    return this.CommandLink;
  //
  if (this.Level > 1 && this.PopupItemBox)
    return this.PopupItemBox;
  //
  return this.MyBox ? this.MyBox : this.MyToolBox;
}


// **********************************************************************
// Gestisco la pressione dei tasti FK
// formidx: indice della form, se -1 allora tutte le form
// frameidx: indice del frame, se -1 allora tutti i frame
// **********************************************************************
Command.prototype.HandleFunctionKeys = function (eve, formidx, frameidx)
{
  var code = (eve.charCode) ? eve.charCode : eve.keyCode;
  var ok = false;
  //
  // Calcolo il numero di FK da 1 a 48
  var fkn = (code - 111) + (eve.shiftKey ? 12 : 0) + (eve.ctrlKey ? 24 : 0);
  //
  // Sono io?
  ok = fkn == this.FKNum; // deve essere uguale al mio FK
  ok = ok && (this.IsMenu || this.IsToolbar); // deve essere o un menu' o una toolbar
  if (formidx > -1 && frameidx == -1) // Toolbar di form?
    ok = ok && (this.ToolCont == -1 && formidx == this.FormIndex)
  if (formidx > -1 && frameidx > -1) // Toolbar di frame?
    ok = ok && (this.ToolCont == frameidx && formidx == this.FormIndex)
  //
  if (ok)
    this.OnClick(eve);
  //
  // Passo il messaggio a tutti i command figli
  var cmdl = this.Commands.length;
  for (var i = 0; i < cmdl && !ok; i++)
  {
    ok = this.Commands[i].HandleFunctionKeys(eve, formidx, frameidx);
  }
  //
  return ok;
}


// ********************************************************************************
//  Metodo che verifica se le immagini della toolbar sono state caricate
// ********************************************************************************
Command.prototype.IsToolBarLoaded = function ()
{
  var n = this.Commands.length;
  for (var i = 0; i < n; i++)
  {
    var cmd = this.Commands[i];
    if (cmd.ToolImg && cmd.Image != "" && !cmd.ToolImg.complete)
      return false;
  }
  //
  return true;
}


// ********************************************************************************
// Metodo chiamato periodicamente: se le immagini della toolbar sono state caricate
// rifa' adattare la form, altrimenti si reimposta per scattare dopo 500 milli
// ********************************************************************************
Command.prototype.ToolbarLoadedUpdate = function ()
{
  // Verifico se la form c'e' ancora: se non c'e' non faccio nulla e non mi reimposto
  var f = RD3_DesktopManager.ObjectMap["frm:" + this.FormIndex];
  if (f)
  {
    if (this.IsToolBarLoaded())
    {
      // Potrebbe essere stata chiamata da Command::OnAdaptRetina ed esserci un timer pendente, se le immagini ci sono tutte lo annullo
      if (this.ToolTimer)
        window.clearTimeout(this.ToolTimer);
      this.ToolTimer = null;
      //
      f.AdaptToolbarLayout();
      //
      // E' una toolbar di Frame (lo faccio solo per Q e Mob7 dove ho visto dei problemi causati dal retina)
      if (this.ToolCont > 0 && (RD3_Glb.IsQuadro() || RD3_Glb.IsMobile7()))
      {
        var fr = f.Frames[this.ToolCont - 1];
        if (fr && fr.UpdateToolbar)
          fr.UpdateToolbar();
      }
    }
    else
    {
      this.ToolTimer = window.setTimeout(new Function("ev", "return RD3_DesktopManager.CallEventHandler('" + this.Identifier + "', 'ToolbarLoadedUpdate', ev, true)"), 100);
    }
  }
}


// ****************************************************************
// Ritorna TRUE se il comando e' un bottone di una toolbar collegata
// ad un frame che utilizza le icone piccole
// ****************************************************************
Command.prototype.UseSmallIcon = function ()
{
  var cmd = this;
  if (this.ParentCmdSet)
    cmd = this.ParentCmdSet;
  //
  if (cmd.ToolCont > 0)
  {
    var f = RD3_DesktopManager.ObjectMap["frm:" + cmd.FormIndex];
    if (f)
    {
      var fr = f.Frames[cmd.ToolCont - 1];
      if (fr)
        return fr.SmallIcons;
    }
  }
  //
  return false;
}


// ****************************************************************
// Ricalcola la visibilita' dei separatori in base a quella dei menu
// ****************************************************************
Command.prototype.RecalcSeparator = function ()
{
  // Se non sono realizzato non faccio nulla
  if (!this.Realized)
    return false;
  //
  // se non sono un CmdSet non faccio nulla
  if (!this.IsCmdSet)
    return;
  //
  if (this.RecalcSeparatorLayout)
  {
    var vis = false;
    //
    // Ciclo su tutti i miei figli: se trovo un menu visibile metto vis a true;
    // se trovo un separatore imposto la sua visibilia' a vis e rimetto vis a false
    var n = this.Commands.length;
    for (var i = 0; i < n; i++)
    {
      var cmd = this.Commands[i];
      //
      if (cmd && !cmd.IsSeparator() && cmd.IsVisible())
        vis = true;
      //
      if (cmd && cmd.IsSeparator())
      {
        // Verifico se dopo questo separatore c'e' un comando visibile: se si allora posso farlo diventare visibile, altrimenti lo nascondo..
        var nextVis = false;
        for (var i2 = i + 1; i2 < n; i2++)
        {
          var cmdNext = this.Commands[i2];
          if (cmdNext && !cmdNext.IsSeparator() && cmdNext.IsVisible())
          {
            nextVis = true;
            break;
          }
        }
        //
        if (vis && !nextVis)
          vis = false;
        //
        // Rendo visibile o invisibile il separatore
        cmd.SetVisible(vis);
        //
        vis = false;
      }
    }
  }
  //
  // Passo la palla ai miei cmdset figli
  var n = this.Commands.length;
  for (var i = 0; i < n; i++)
  {
    this.Commands[i].RecalcSeparator();
  }
}


// ****************************************************************
// Torna true se il comando deve mostrare l'acceleratore
// ****************************************************************
Command.prototype.ShowAccell = function ()
{
  return (this.IsMenu && !RD3_DesktopManager.WebEntryPoint.HasSideMenu() && this.Accell != "" && !RD3_Glb.IsTouch());
}


// **********************************************************************
// Gestisco la pressione dell'acceleratore
// **********************************************************************
Command.prototype.HandleAccell = function (eve, code, force)
{
  var ok = false;
  //
  // Solo CMS primo livello
  if ((this.Level > 1 || !this.IsMenu || !this.IsCmdSet) && !force)
    return false;
  //
  // Se ho il menu' aperto, provo con loro...
  if (this.PopupContainerBox)
  {
    var n = this.Commands.length;
    for (var i = 0; i < n && !ok; i++)
    {
      ok = this.Commands[i].HandleAccell(eve, code, true);
    }
    //
    if (!ok && this.FormList && code > 48 && code <= 57)
    {
      // Vediamo se e' una formlist...
      var sf = RD3_DesktopManager.WebEntryPoint.StackForm;
      var nn = sf.length;
      var c = 0;
      for (var ii = 0; ii < nn && !ok; ii++)
      {
        if (sf[ii].HasFormList())
          c++;
        if (c == (code - 48))
        {
          sf[ii].OnFLClick(eve);
          RD3_DesktopManager.WebEntryPoint.CmdObj.ClosePopup();
          ok = true;
        }
      }
    }
  }
  //
  // Gia' gestito...
  if (ok)
    return ok;
  //
  // Vediamo se sono io...
  if (this.Accell != "" && this.Accell.charCodeAt(0) == code)
    ok = true;
  //
  if (ok)
    this.OnClick(eve);
  //
  return ok;
}


// **********************************************************************
// Realizza gli oggetti dell'intero popup menu'
// Ritorno il TBODY per poter aggiungere ulteriori comandi facilmente
// **********************************************************************
Command.prototype.RealizePopupMenu = function ()
{
  // Il backdrop impedisce di cliccare/scrollare oggetti sottostanti
  this.PopupContainerBackDrop = document.createElement("div");
  this.PopupContainerBackDrop.className =  "dropdown-backdrop";
  //
  this.PopupContainerBox = document.createElement("ul");
  this.PopupContainerBox.setAttribute("id", this.Identifier + ":pp");
  this.PopupContainerBox.className =  "dropdown-menu popup-menu-container";
  this.PopupContainerBackDrop.appendChild(this.PopupContainerBox);
  //
  // Renderizzo tutti i comandi
  var n = this.Commands.length;
  for (var i = 0; i < n; i++)
    this.Commands[i].RealizePopupItem(this.PopupContainerBox);
  //
  this.SetClassName();
  //
  return this.PopupContainerBox;
}


// ********************************************************************************
// Chiude gli altri popup del padre a parte questo
// ********************************************************************************
Command.prototype.CloseOtherPopup = function ()
{
  var n = this.ParentCmdSet.Commands.length;
  for (var i = 0; i < n; i++)
  {
    var c = this.ParentCmdSet.Commands[i];
    //
    if (c != this)
    {
      c.ClosePopup();
      RD3_Glb.RemoveClass(c.PopupTextCell, "popup-menu-hover");
    }
  }
}

// *********************************************************
// Imposta il tooltip
// *********************************************************
Command.prototype.GetTooltip = function (tip, obj)
{
  if (this.Tooltip == "")
    return false;
  //
  var s = RD3_KBManager.GetFKTip(this.FKNum);
  if (s == "" && this.CommandCode != "")
    s = " (" + this.CommandCode + ")";
  //
  tip.SetTitle(this.Caption);
  tip.SetText(this.Tooltip + s);
  //
  // E' stato chiesto il tooltip per la toolbar
  if (obj == this.MyToolBox || obj == this.ToolImg || obj == this.ToolText)
  {
    tip.SetObj(this.MyToolBox);
    obj = this.MyToolBox;
    //
    tip.SetAnchor(RD3_Glb.GetScreenLeft(obj) + ((obj.offsetWidth - (this.ShowNames ? 0 : 4)) / 2), RD3_Glb.GetScreenTop(obj));
    tip.SetPosition(0);
    //
    return true;
  }
  //
  // E' stato chiesto il tooltip per il menu
  if (obj == this.CmdBox || obj == this.CommandLink || obj == this.CommandImg || obj == this.CommandImgDX || obj == this.BranchImg)
  {
    tip.SetObj(this.CmdBox);
    obj = this.CmdBox;
    //
    tip.SetAnchor(RD3_Glb.GetScreenLeft(obj) + obj.offsetWidth, RD3_Glb.GetScreenTop(obj) + ((obj.offsetHeight) / 2));
    tip.SetPosition(1);
    return true;
  }
  //
  // E' stato chiesto il tooltip per il popup
  if (obj == this.PopupItemBox || obj == this.PopupImageBox || obj == this.PopupTextCell || obj == this.PopupText)
  {
    tip.SetObj(this.PopupItemBox);
    obj = this.PopupItemBox;
    //
    tip.SetAnchor(RD3_Glb.GetScreenLeft(obj) + obj.offsetWidth, RD3_Glb.GetScreenTop(obj) + (obj.offsetHeight / 2));
    tip.SetPosition(1);
    return true;
  }
  //
  // E' stato chiesto il tooltip per la button bar
  if (this.Button)
  {
    tip.SetAnchor(RD3_Glb.GetScreenLeft(obj) + (obj.clientWidth / 2), RD3_Glb.GetScreenTop(obj) + obj.clientHeight);
    tip.SetPosition(2);
    return true;
  }
}


// ***********************************************
// Trova l'oggetto a cui attaccare il popup menu
// ***********************************************
Command.prototype.GetObjToAttach = function (node)
{
  var id = ""
  if (node instanceof Command)
    id = node.Identifier;
  else
    id = node.getAttribute("objid");
  //
  // Dovrei identificare l'oggetto per sapere dove aprirlo
  // Vediamo a quale oggetto devo essere attaccato
  var obj = RD3_DesktopManager.ObjectMap[id];
  var dobj = (obj && obj.GetDOMObj) ? obj.GetDOMObj((obj.IsToolbar) ? "toolbar" : "") : null;
  //
  // Se non ho trovato l'oggetto nella mappa provo a cercarlo nel DOM
  if (!obj)
    dobj = document.getElementById(id);
  //
  return dobj;
}


// ***********************************************
// Realizza un Popup Menu
// Chiamato dal GFX
// ***********************************************
Command.prototype.RealizePopup = function (node)
{
  var dir = 0;
  var form = "";
  var id = ""
  var x = 0;
  var y = 0;
  //
  if (!node)
    return;
  //
  if (node instanceof Command)
  {
    id = node.Identifier;
    dir = (node.Level == 1) ? 1 : 0;
  }
  else
  {
    dir = parseInt(node.getAttribute("dir"));
    form = node.getAttribute("form");
    id = node.getAttribute("objid");
    x = parseInt(node.getAttribute("xpos"));
    y = parseInt(node.getAttribute("ypos"));
  }
  //
  // Dovrei identificare l'oggetto per sapere dove aprirlo
  // Vediamo a quale oggetto devo essere attaccato
  var obj = RD3_DesktopManager.ObjectMap[id];
  var dobj = (obj && obj.GetDOMObj) ? obj.GetDOMObj((obj.IsToolbar) ? "toolbar" : "") : null;
  //
  // Se non ho trovato l'oggetto nella mappa provo a cercarlo nel DOM
  if (!obj)
  {
    dobj = document.getElementById(id);
    if (dobj)
      obj = RD3_KBManager.GetObject(dobj);
  }
  //
  // Adesso apro il popup, creandolo
  this.RealizePopupMenu();
  //
  // calcolo in base all'oggetto
  var mx = RD3_DesktopManager.WebEntryPoint.WepBox.offsetWidth;
  var my = RD3_DesktopManager.WebEntryPoint.WepBox.offsetHeight;
  //
  // Visualizzo il popup all'interno del WEPBOX, altrimenti non calcola le dimensioni
  this.PopupContainerBox.style.visibility = "hidden";
  document.body.appendChild(this.PopupContainerBackDrop);
  //
  if (dobj)
  {
    var ancora = true;
    var giro = 0;
    while (ancora)
    {
      ancora = false;
      giro++;
      //
      // Tengo conto anche di eventuali offset dovuti alle scrollbar, chiedendoli all'oggetto di modello
      // (per ora AccountOverFlowX e AccountOverFlowY ce li ha solo il TreeNode che li usava per il D&D)
      var ovx = obj && obj.AccountOverFlowX ? obj.AccountOverFlowX() : 0;
      var ovy = obj && obj.AccountOverFlowY ? obj.AccountOverFlowY() : 0;
      //
      x = RD3_Glb.GetScreenLeft(dobj) - ovx;
      y = RD3_Glb.GetScreenTop(dobj) - ovy;
      //
      // Interpreto la direzione
      switch (dir)
      {
        case 0: // Right
          x += dobj.offsetWidth;
          break;

        case 1: // Bottom
          y += dobj.offsetHeight;
          break;

        case 2: // Left
          x -= this.PopupContainerBox.offsetWidth;
          break;

        case 3: // Top
          y -= this.PopupContainerBox.offsetHeight;
          break;
      }
      //
      // Controllo di non essere uscito dallo schermo
      // faccio al max 2 giri per far prevalere quello che ci sta meglio
      // oppure l'impostazione dell'utente
      if (giro < 3)
      {
        if (dir == 1 && y + this.PopupContainerBox.offsetHeight > my)
        {
          dir = 3;
          ancora = true;
        }
        if (dir == 3 && y < 0)
        {
          dir = 1;
          ancora = true;
        }
        if (dir == 0 && x + this.PopupContainerBox.offsetWidth > mx)
        {
          dir = 2;
          ancora = true;
        }
        if (dir == 2 && x < 0)
        {
          dir = 0;
          ancora = true;
        }
      }
    }
  }
  //
  // Mi memorizzo la direzione: verra' utilizzata per l'animazione
  this.Direction = dir;
  //
  this.PopupContainerBox.style.left = x + "px";
  this.PopupContainerBox.style.top = y + "px";
  //
  this.PopupContainerBox.style.visibility = "visible";
  //
  // Gestisco la visibilita' dei separatori
  //this.RecalcSeparator();
  //
  this.AdaptLayout();
  //
  // Riverifico la posizione dopo aver dimensionato correttamente il menu popup
  if (dir == 1 || dir == 3) // Top o Bottom
  {
    // L'impostazione bottom o top verifica se usciamo dallo schermo in basso o in alto.. ma dobbiamo considerare anche se
    // usciamo a destra o sinistra..
    if (x + this.PopupContainerBox.offsetWidth > mx)
    {
      // Sforiamo sulla destra.. dobbiamo spostare la X per recuperare spazio
      x -= x + this.PopupContainerBox.offsetWidth - mx;
      this.PopupContainerBox.style.left = x + "px";
    }
    if (x < 0)
    {
      // Sforiamo sulla sinistra.. dobbiamo spostare la X
      x = 0;
      this.PopupContainerBox.style.left = x + "px";
    }
  }
  if (dir == 0 || dir == 2) // Rigth o Left
  {
    // L'impostazione right o left verifica se usciamo dallo schermo a destra o sinistra, dobbiamo verificare se
    // sforiamo anche in basso o in alto
    if (y + this.PopupContainerBox.offsetHeight > my)
    {
      // Sforiamo in basso.. dobbiamo spostare la Y per recuperare spazio
      y -= y + this.PopupContainerBox.offsetHeight - my;
      this.PopupContainerBox.style.top = y + "px";
    }
    if (y < 0)
    {
      // Sforiamo in alto.. dobbiamo spostare la Y
      y = 0;
      this.PopupContainerBox.style.top = y + "px";
    }
  }
}

// *******************************************************************************
// Metodo chiamato dal DDManager quando viene iniziato un D&D
// ident: id dell'oggetto su cui l'utente ha cliccato per iniziare il D&D
// *******************************************************************************
Command.prototype.IsDraggable = function (ident)
{
  // il comando e' draggabile se:
  // - candrag del commandset padre e' attivo
  // - non e' un commandset
  var ok = !this.IsCmdSet && this.ParentCmdSet && this.ParentCmdSet.CommandCanDrag;
  //
  // Mi devo memorizzare se vengo da Toolbar o Comando..
  if (ok)
  {
    if (ident.indexOf(":header") != -1 || ident.indexOf(":image") != -1 || ident.indexOf(":link") != -1 || (RD3_Glb.IsMobile() && ident == this.Identifier))
      this.DragMenu = true;
    else
      this.DragMenu = false;
  }
  return ok;
}

// *************************************************
// Restituisce l'oggetto da trascinare
// *************************************************
Command.prototype.DropElement = function ()
{
  if (this.DragMenu)
    return this.CmdBox ? this.CmdBox : this.MyBox;
  else if (this.MyToolBox)
    return this.MyToolBox;
  //
  return this.CmdBox;
}


// *************************************************
// Verifica se il comando puo' essere draggato
// *************************************************
Command.prototype.ComputeDropList = function (list, dragobj)
{
  // Un CommandSet deve solo passare il messaggio ai suoi figli
  if (this.IsCmdSet)
  {
    var n = this.Commands.length;
    for (var i = 0; i < n; i++)
    {
      this.Commands[i].ComputeDropList(list, dragobj);
    }
  }
  else
  {
    // Se non sono visibile o mio padre non accetta il drag ho finito
    if (!this.ParentCmdSet.CommandCanDrop || !this.IsVisible())
      return;
    //
    list.push(this);
    //
    // Aggiungo le mie coordinate (sia come menu che come toolbar)
    if (this.IsMenu)
    {
      var o = this.CmdBox ? this.CmdBox : this.MyBox;
      this.AbsLeft = RD3_Glb.GetScreenLeft(o, true);
      this.AbsTop = RD3_Glb.GetScreenTop(o, true);
      this.AbsRight = this.AbsLeft + o.offsetWidth - 1;
      this.AbsBottom = this.AbsTop + o.offsetHeight - 1;
    }
    //
    if (this.IsToolbar)
    {
      var o = this.MyToolBox;
      this.AbsLeft2 = RD3_Glb.GetScreenLeft(o);
      this.AbsTop2 = RD3_Glb.GetScreenTop(o);
      this.AbsRight2 = this.AbsLeft2 + o.offsetWidth - 1;
      this.AbsBottom2 = this.AbsTop2 + o.offsetHeight - 1;
    }
  }
}


// ********************************************************************************
// Torna true se questo elemento e' quello attivo nel menu' principale mobile
// ********************************************************************************
Command.prototype.IsActiveMenu = function ()
{
  return RD3_DesktopManager.WebEntryPoint.CmdObj.ActiveCommand == this;
}


// ********************************************************************************
// Il comando e' stato toccato dall'utente
// ********************************************************************************
Command.prototype.OnTouchDown = function (evento)
{
  
}

// ********************************************************************************
// Il comando e' stato smesso di toccare dall'utente
// ********************************************************************************
Command.prototype.OnTouchUp = function (evento, click)
{
  
}

// ********************************************************************************
// Se il popup si e' chiuso, mi dimentico della mia referenza
// ********************************************************************************
Command.prototype.OnClosePopup = function (popup)
{
  if (popup == this.Popover)
  {
    this.Popover = null;
    //
    // Derealizzo tutti i comandi contenuti
    // per ripartire da una situazione "pulita"
    for (var i = 0; i < this.Commands.length; i++)
      this.Commands[i].Unrealize(true);
    //
    // Distruggo il menu box
    if (this.MenuBox)
    {
      this.MenuBox.parentNode.removeChild(this.MenuBox);
      this.MenuBox = null;
    }
    if (this.PopupMobileBox)
    {
      if (this.PopupMobileBox.parentNode)
        this.PopupMobileBox.parentNode.removeChild(this.PopupMobileBox);
      this.PopupMobileBox = null;
    }
  }
}


// ********************************************************************************
// Torna vero se questo command set deve mostrare i comandi interni
// come menu' raggruppato
// ********************************************************************************
Command.prototype.ShowGroupedMenu = function ()
{
  // Mostro il menu raggruppato solo se sono di primo livello, l'applicazione ha il menu raggruppato e sono un menu (no toolbar e no menu popup)
  return this.ParentCmdSet == null && RD3_DesktopManager.WebEntryPoint.CmdObj.ShowGroups && this.IsMenu;
}

// ********************************************************************************
// Rispondo true se sono al primo livello
// ********************************************************************************
Command.prototype.IsFirstLevel = function ()
{
  return this.ParentCmdSet == null || this.ParentCmdSet.ShowGroupedMenu();
}


// ********************************************************************************
// Ritorna True se il comando appartiene ad una SubForm e questa e' visibile
// ********************************************************************************
Command.prototype.CheckSubForm = function ()
{
  if (this.FormIndex < 65536)
    return false;
  //
  var frmId = "frm:" + this.FormIndex;
  var frm = RD3_DesktopManager.ObjectMap[frmId];
  //
  if (frm == null)
    return false;
  //
  var masterFrm = frm.GetMasterForm();
  var fnd = false;
  var actf = RD3_DesktopManager.WebEntryPoint.ActiveForm;
  if (actf && actf.IdxForm == masterFrm.IdxForm)
    fnd = true;
  //
  var nf = fnd ? 0 : RD3_DesktopManager.WebEntryPoint.StackForm.length;
  for (var t = 0; t < nf; t++)
  {
    var f = RD3_DesktopManager.WebEntryPoint.StackForm[t];
    if (f.Docked && f.IdxForm == masterFrm.IdxForm)
    {
      fnd = true;
      break;
    }
  }
  //
  return fnd;
}

// *******************************************************************
// Gestisce il comando back girandolo eventualmente ai figli;
// ritorna True se e' stato gestito
// *******************************************************************
Command.prototype.HandleBackButton = function ()
{
  
}


// *******************************************************************
// Chiamato quando cambia il colore di accento
// *******************************************************************
Command.prototype.AccentColorChanged = function (reg, newc)
{
  
}

// *******************************************************************
// Chiamato quando ho l'immagine da retinare
// *******************************************************************
Command.prototype.OnAdaptRetina = function (w, h, par)
{
  
}

// *******************************************************************
// Chiamato quando il server chiede di rimuovere il comando/commandset
// *******************************************************************
Command.prototype.DeleteCommand = function()
{
  if (this.ParentCmdSet)
  {
    // Mi rimuovo da mio padre, lo devo fare prima dell'urealize perche' altrimenti perdo il puntatore
    var parentIdx = this.ParentCmdSet.Commands.indexOf(this);
    if (parentIdx >= 0)
      this.ParentCmdSet.Commands.splice(parentIdx, 1);
  }
  else 
  {
    // Provo a cercarmi nell'array globale e a togliermi da li'
    var cmsIdx = RD3_DesktopManager.WebEntryPoint.CmdObj.CommandSets.indexOf(this);
    if (cmsIdx >= 0)
      RD3_DesktopManager.WebEntryPoint.CmdObj.CommandSets.splice(cmsIdx, 1);
  }
  //
  // Rimuovo i miei figli se presenti (la Unrealize in questo caso particolare non passerebbe il messaggio)
  for(var i=0; i<this.Commands.length; i++)
    this.Commands[i].Unrealize();
  //
  // Mi rimuovo
  this.Unrealize();
}
