/*
 * Instant Developer Cloud
 * Copyright Pro Gamma Spa 2000-2021
 * All rights reserved
 */

var Client = Client || {};
/**
 * @class A frame object of type panel
 * @param {Object} widget
 * @param {View|Element} parent - the parent element
 * @param {View} view
 */
Client.IdfPanel = function (widget, parent, view)
{
  this.fields = [];
  this.groups = [];
  this.pages = [];
  this.multiSelStatus = [];
  this.listGridRows = [];
  this.formGridRows = [];
  this.rows = [];
  this.detachedRows = {};
  this.commandsZones = [4, 4, 4, 6, 5, 4, 4, 6, 6, 0, 0, 0, 0, 0, 7, 7, 7, 7, 8, 8, 8, 8, 8, 8, 8, 8, 0, 0, 0, 0, 3, 0, 1, 1, 2, 9, 2, 8, 8, 8, 8, 8, 8, 8, 8, 0, 0, 0];
  this.customCommands = Client.mainFrame.wep?.customCommands || [];
  this.showFieldImageInValue = Client.mainFrame.wep?.showFieldImageInValue !== undefined ? Client.mainFrame.wep.showFieldImageInValue : true;
  //
  // Set default values
  widget = Object.assign({
    showStatusbar: true,
    gridHeight: parent.height,
    gridTop: 0,
    gridLeft: 0,
    headerHeight: Client.IdfPanel.defaultHeaderHeight,
    layout: Client.IdfPanel.layouts.list,
    status: Client.IdfPanel.statuses.qbe,
    showRowSelector: true,
    showMultiSelection: false,
    enableMultiSelection: true,
    selectOnlyVisibleRows: false,
    hasList: true,
    hasForm: true,
    rowHeightResize: false,
    canUpdate: true,
    canDelete: true,
    canInsert: true,
    canSearch: true,
    canSort: true,
    cangGroup: false,
    showGroups: false,
    confirmDelete: true,
    highlightDelete: true,
    actualPosition: 1,
    actualRow: 0,
    resizeWidth: Client.IdfPanel.resizeModes.stretch,
    resizeHeight: Client.IdfPanel.resizeModes.stretch,
    fixedColumns: 0,
    blockingCommands: 0,
    enabledCommands: -1,
    extEnabledCommands: -1,
    automaticLayout: false,
    searchMode: Client.IdfPanel.searchModes.toolbar,
    allowNavigationWhenModified: true,
    enableInsertWhenLocked: false,
    hasBook: false,
    isDO: false,
    hasDocTemplate: false,
    DOMaster: false,
    DOModified: false,
    DOSingleDoc: false,
    DOCanSave: true,
    activateOnRightClick: false,
    pullToRefresh: true,
    activePage: 0,
    advancedTabOrder: false,
    canReorderColumn: false,
    canResizeColumn: false,
    toolbarEventDef: Client.IdfMessagesPump?.eventTypes.ACTIVE,
    scrollEventDef: Client.IdfMessagesPump?.eventTypes.ACTIVE,
    rowSelectEventDef: Client.IdfMessagesPump?.eventTypes.ACTIVE,
    pageClickEventDef: Client.IdfMessagesPump?.eventTypes.ACTIVE,
    multiSelEventDef: Client.IdfMessagesPump?.eventTypes.DEFERRED,
    focusEventDef: Client.IdfMessagesPump?.eventTypes.CLIENTSIDE,
    selectionChangeEventDef: Client.IdfMessagesPump?.eventTypes.DEFERRED,
    changeLayoutAnimationDef: Client.IdfWebEntryPoint.getAnimationDefault("list"),
    qbeTipAnimationDef: Client.IdfWebEntryPoint.getAnimationDefault("qbeTip")
  }, widget);
  //
  // Set original dimensions and position
  this.orgGridHeight = widget.gridHeight;
  this.orgGridTop = widget.gridTop;
  this.orgGridLeft = widget.gridLeft;
  //
  Client.IdfFrame.call(this, widget, parent, view);
};


// Make Client.IdfPanel extend Client.IdfFrame
Client.IdfPanel.prototype = new Client.IdfFrame();

Client.IdfPanel.getRequirements = Client.IdfFrame.getRequirements;

Client.IdfPanel.transPropMap = Object.assign({}, Client.IdfFrame.transPropMap, {
  mod: "layout",
  sta: "status",
  num: "numRows",
  mhr: "maxRowHeight",
  hds: "headerHeight",
  srs: "showRowSelector",
  sms: "showMultiSelection",
  ems: "enableMultiSelection",
  sov: "selectOnlyVisibleRows",
  hli: "hasList",
  hfo: "hasForm",
  upd: "canUpdate",
  del: "canDelete",
  ins: "canInsert",
  sea: "canSearch",
  sor: "canSort",
  grn: "canGroup",
  sgr: "showGroups",
  cde: "confirmDelete",
  hde: "highlightDelete",
  lle: "gridLeft",
  lto: "gridTop",
  lwi: "gridWidth",
  lhe: "gridHeight",
  atr: "actualRow",
  vre: "resizeHeight",
  hre: "resizeWidth",
  fix: "fixedColumns",
  qtp: "qbeTip",
  acp: "actualPosition",
  tot: "totalRows",
  cbk: "blockingCommands",
  enc: "enabledCommands",
  eec: "extEnabledCommands",
  qbf: "automaticLayout",
  lqb: "searchMode",
  rhr: "rowHeightResize",
  amp: "allowNavigationWhenModified",
  eil: "enableInsertWhenLocked",
  bok: "hasBook",
  ido: "isDO",
  hdt: "hasDocTemplate",
  dom: "DOModified",
  mst: "DOMaster",
  csa: "DOCanSave",
  sdo: "DOSingleDoc",
  arc: "activateOnRightClick",
  pag: "activePage",
  tck: "toolbarEventDef",
  sck: "scrollEventDef",
  rck: "rowSelectEventDef",
  pck: "pageClickEventDef",
  mse: "multiSelEventDef",
  fed: "focusEventDef",
  tsk: "selectionChangeEventDef",
  cla: "changeLayoutAnimationDef",
  qta: "qbeTipAnimationDef",
  rsc: "canResizeColumn",
  rcl: "canReorderColumn",
  pre: "pullToRefresh",
  ata: "advancedTabOrder",
  vfl: "visualFlags"
});


Client.IdfPanel.defaultHeaderHeight = 32;
Client.IdfPanel.defaultListRowHeight = 32;
Client.IdfPanel.defaultRowSelectorWidth = 40;
Client.IdfPanel.rowSelectorOffset = 20;
Client.IdfPanel.maxToolbarZones = 11;


Client.IdfPanel.commands = {
  // Commands indexes to handle commands zones
  CZ_FORMLIST: 0, CZ_SEARCH: 1, CZ_FIND: 2, CZ_INSERT: 3, CZ_DELETE: 4, CZ_CANCEL: 5, CZ_REQUERY: 6, CZ_UPDATE: 7, CZ_DUPLICATE: 8, CZ_LOOKUP: 9,
  CZ_BLOBEDIT: 10, CZ_BLOBDELETE: 11, CZ_BLOBNEW: 12, CZ_BLOBSAVEAS: 13, CZ_PRINT: 14, CZ_GROUP: 15, CZ_ATTACH: 16, CZ_CSV: 17, CZ_CUSTOM1: 18,
  CZ_CUSTOM2: 19, CZ_CUSTOM3: 20, CZ_CUSTOM4: 21, CZ_CUSTOM5: 22, CZ_CUSTOM6: 23, CZ_CUSTOM7: 24, CZ_CUSTOM8: 25, CZ_NAVIGATE: 30, CZ_COLLAPSE: 32,
  CZ_LOCK: 33, CZ_STATUSBAR: 34, CZ_CMDSET: 35, CZ_QBETIP: 36,
  //
  // Commands values to handle enable commands
  CMD_FORMLIST: 0x1, CMD_SEARCH: 0x2, CMD_FIND: 0x4, CMD_INSERT: 0x8, CMD_DELETE: 0x10, CMD_CANCEL: 0x20, CMD_REFRESH: 0x40, CMD_SAVE: 0x80,
  CMD_DUPLICATE: 0x100, CMD_LOOKUP: 0x200, CMD_BLOBEDIT: 0x400, CMD_BLOBDELETE: 0x800, CMD_BLOBNEW: 0x1000, CMD_BLOBSAVEAS: 0x2000, CMD_PRINT: 0x4000,
  CMD_ATTACH: 0x10000, CMD_CSV: 0x20000, CMD_GROUP: 0x8000, CMD_CUSTOM1: -0x1, CMD_CUSTOM2: -0x2, CMD_CUSTOM3: -0x4, CMD_CUSTOM4: -0x8,
  CMD_CUSTOM5: -0x16, CMD_CUSTOM6: -0x32, CMD_CUSTOM7: -0x64, CMD_CUSTOM8: -0x128, CMD_NAVIGATION: 0x40000000
};


Client.IdfPanel.resizeModes = {
  none: 1,
  move: 2,
  stretch: 3
};


Client.IdfPanel.statuses = {
  qbe: 1,
  data: 2,
  updated: 3
};


Client.IdfPanel.layouts = {
  list: 0,
  form: 1
};


Client.IdfPanel.searchModes = {
  toolbar: 0,
  header: -1,
  row: -2
};


/**
 * Create element configuration from xml
 * @param {XmlNode} xml
 */
Client.IdfPanel.createConfigFromXml = function (xml)
{
  let config = {};
  //
  let dataChange = false;
  let dataBlockStart;
  let dataBlockEnd;
  //
  // Look into panel children (i.e. fields) in order to find start and end index of given block of data
  for (let i = 0; i < xml.childNodes.length; i++) {
    let child = xml.childNodes[i];
    //
    if (child.nodeName === "lsg") {
      config.groupedRows = true;
      continue;
    }
    //
    let fieldConfig = Client.Widget.createConfigFromXml(child);
    if (!fieldConfig || !Client.Widget.isFieldClass(fieldConfig.c))
      continue;
    //
    if (!fieldConfig.valuesConfig.length)
      continue;
    //
    dataChange = true;
    //
    if (dataBlockStart === undefined)
      dataBlockStart = parseInt(fieldConfig.valuesConfig[0].idx);
    //
    if (dataBlockEnd === undefined)
      dataBlockEnd = parseInt(fieldConfig.valuesConfig[fieldConfig.valuesConfig.length - 1].idx);
  }
  //
  if (dataChange) {
    config.dataBlockStart = dataBlockStart || 1;
    config.dataBlockEnd = dataBlockEnd || 1;
    //
    // An IDC panel receives data into "data" property, while an IDF panel gets its data at creation time.
    // Since I populate panel grid on "data" property change (see IdfPanel.updateElement method), set dummy "data" property on IDF panel too.
    config.data = {};
  }
  //
  return config;
};


/**
 * Convert properties values
 * @param {Object} props
 */
Client.IdfPanel.convertPropValues = function (props)
{
  props = props || {};
  //
  Client.IdfFrame.convertPropValues(props);
  //
  for (let p in props) {
    switch (p) {
      case Client.IdfPanel.transPropMap.mod:
      case Client.IdfPanel.transPropMap.sta:
      case Client.IdfPanel.transPropMap.num:
      case Client.IdfPanel.transPropMap.hds:
      case Client.IdfPanel.transPropMap.tot:
      case Client.IdfPanel.transPropMap.lwi:
      case Client.IdfPanel.transPropMap.lhe:
      case Client.IdfPanel.transPropMap.lle:
      case Client.IdfPanel.transPropMap.lto:
      case Client.IdfPanel.transPropMap.hre:
      case Client.IdfPanel.transPropMap.vre:
      case Client.IdfPanel.transPropMap.atr:
      case Client.IdfPanel.transPropMap.acp:
      case Client.IdfPanel.transPropMap.cbk:
      case Client.IdfPanel.transPropMap.enc:
      case Client.IdfPanel.transPropMap.eec:
      case Client.IdfPanel.transPropMap.lqb:
      case Client.IdfPanel.transPropMap.pag:
      case Client.IdfPanel.transPropMap.tck:
      case Client.IdfPanel.transPropMap.sck:
      case Client.IdfPanel.transPropMap.rck:
      case Client.IdfPanel.transPropMap.pck:
      case Client.IdfPanel.transPropMap.mse:
      case Client.IdfPanel.transPropMap.rsc:
      case Client.IdfPanel.transPropMap.rcl:
      case Client.IdfPanel.transPropMap.fix:
      case Client.IdfPanel.transPropMap.vfl:
        props[p] = parseInt(props[p]);
        break;

      case Client.IdfPanel.transPropMap.srs:
      case Client.IdfPanel.transPropMap.sms:
      case Client.IdfPanel.transPropMap.ems:
      case Client.IdfPanel.transPropMap.sov:
      case Client.IdfPanel.transPropMap.hli:
      case Client.IdfPanel.transPropMap.hfo:
      case Client.IdfPanel.transPropMap.upd:
      case Client.IdfPanel.transPropMap.del:
      case Client.IdfPanel.transPropMap.ins:
      case Client.IdfPanel.transPropMap.sea:
      case Client.IdfPanel.transPropMap.sor:
      case Client.IdfPanel.transPropMap.grn:
      case Client.IdfPanel.transPropMap.sgr:
      case Client.IdfPanel.transPropMap.cde:
      case Client.IdfPanel.transPropMap.hde:
      case Client.IdfPanel.transPropMap.pfp:
      case Client.IdfPanel.transPropMap.amp:
      case Client.IdfPanel.transPropMap.eil:
      case Client.IdfPanel.transPropMap.qbf:
      case Client.IdfPanel.transPropMap.bok:
      case Client.IdfPanel.transPropMap.ido:
      case Client.IdfPanel.transPropMap.dom:
      case Client.IdfPanel.transPropMap.mst:
      case Client.IdfPanel.transPropMap.csa:
      case Client.IdfPanel.transPropMap.sdo:
      case Client.IdfPanel.transPropMap.arc:
      case Client.IdfPanel.transPropMap.pre:
      case Client.IdfPanel.transPropMap.ata:
        props[p] = props[p] === "1";
        break;
    }
  }
};


/**
 * Get root object. Root object is the object where children will be inserted
 * @param {Boolean} el - if true, get the element itself istead of its domObj
 */
Client.IdfPanel.prototype.getRootObject = function (el)
{
  let rootObject = Client.eleMap[this.panelContainerConf.id];
  return el ? rootObject : rootObject.domObj;
};


/**
 * Create elements configuration
 * @param {Object} widget
 */
Client.IdfPanel.prototype.createElementsConfig = function (widget)
{
  Client.IdfFrame.prototype.createElementsConfig.call(this, widget);
  //
  // Create pages container configuration and put it between toolbar and content
  this.pagesContainerConf = this.createElementConfig({c: "Container", className: "panel-pages-container", visible: false});
  this.mainContainerConf.children.splice(1, 0, this.pagesContainerConf);
  //
  // Create an alternate container configuration that will contain two pages: one for list mode and another one for form mode
  this.panelContainerConf = this.createElementConfig({c: "AltContainer", selectedPage: 0, className: "panel-container"});
  let a = Client.IdfWebEntryPoint.getAnimationByDef(this.changeLayoutAnimationDef);
  if (a)
    this.panelContainerConf.animations = [{trigger: "change", ...a}];
  this.contentContainerConf.children.push(this.panelContainerConf);
  //
  // Create list page configuration (first page)
  if (widget.hasList) {
    let page1 = this.createElementConfig({c: "Container", style: {width: "100%", height: "100%"}});
    //
    // Create list container configuration
    this.listContainerConf = this.createElementConfig({c: "IonGrid", className: "panel-list-container", noPadding: true});
    //
    // Create grid configuration
    this.createGridConfig();
    //
    page1.children.push(this.listContainerConf);
    page1.children.push(this.extGridConf);
    this.panelContainerConf.children.push(page1);
  }
  //
  // Create form page configuration (second page)
  if (widget.hasForm) {
    let page2 = this.createElementConfig({c: "Container", style: {width: "100%", height: "100%"}});
    //
    // Create form container configuration
    this.formContainerConf = this.createElementConfig({c: "IonGrid", className: "panel-form-container", noPadding: true});
    //
    page2.children.push(this.formContainerConf);
    this.panelContainerConf.children.push(page2);
  }
};


/**
 * Create grid configuration
 */
Client.IdfPanel.prototype.createGridConfig = function ()
{
  // Create the external ion-list configuration
  this.extGridConf = this.createElementConfig({c: "IonContent", className: "panel-list-ext-grid"});
  //
  this.extGridListConf = this.createElementConfig({c: "IonList", className: "panel-list-ext-grid", events: ["onRefresh"], pullText: " ", refreshText: " "});
  this.extGridConf.children.push(this.extGridListConf);
  //
  // Create grid configuration
  this.gridConf = this.createElementConfig({c: "IonGrid", className: "panel-list-grid", noPadding: true, events: ["onScroll"]});
  this.extGridListConf.children.push(this.gridConf);
  //
  // Create grid header configuration
  this.gridHeaderConf = this.createElementConfig({c: "IonRow", className: "panel-list-row panel-list-header-row", noWrap: true});
  this.gridConf.children.push(this.gridHeaderConf);
  //
  // Create row selector column configuration
  let offsetCol = this.getHeaderOffset() ? " offset-col" : "";
  this.rowSelectorColumnConf = this.createElementConfig({c: "IonCol", className: "panel-list-col row-selector-col" + offsetCol, xs: "auto"});
  this.gridHeaderConf.children.push(this.rowSelectorColumnConf);
  //
  // Create multiselection button configuration
  this.multiSelButtonConf = this.createElementConfig({c: "IonButton", icon: "checkbox", className: "generic-btn panel-multisel-button", events: ["onClick"]});
  this.rowSelectorColumnConf.children.push(this.multiSelButtonConf);
  //
  // Create aggregate row configuration
  this.aggregateRowConf = this.createElementConfig({c: "IonRow", className: "panel-list-row panel-list-aggregate-row", noWrap: true});
  this.gridConf.children.push(this.aggregateRowConf);
  //
  // Create row selector configuration for aggregate row
  this.aggregateRowSelectorConf = this.createElementConfig({c: "IonCol", className: "row-selector-col", xs: "auto"});
  this.aggregateRowConf.children.push(this.aggregateRowSelectorConf);
};


/**
 * Create toolbar configuration
 */
Client.IdfPanel.prototype.createToolbarConfig = function ()
{
  Client.IdfFrame.prototype.createToolbarConfig.call(this);
  //
  // Create toolbar zones configuration
  this.toolbarZonesConfig = [];
  for (let i = 0; i < Client.IdfPanel.maxToolbarZones; i++) {
    if (Client.mainFrame.idfMobile)
      this.toolbarZonesConfig[i] = this.toolbarConf;
    else {
      // Create zone configuration (the first one is not visible)
      this.toolbarZonesConfig[i] = this.createElementConfig({c: "Container", className: "panel-toolbar-zone"});
      this.toolbarConf.children.push(this.toolbarZonesConfig[i]);
    }
  }
  //
  let zoneIdx, commandIdx;
  //
  // Move collapse button to collapse command zone
  commandIdx = this.toolbarConf.children.indexOf(this.collapseButtonConf);
  this.toolbarConf.children.splice(commandIdx, 1);
  zoneIdx = this.getCommandZone(Client.IdfPanel.commands.CZ_COLLAPSE);
  this.toolbarZonesConfig[zoneIdx].children.push(this.collapseButtonConf);
  this.collapseButtonConf.className += " panel-toolbar-btn";
  //
  // Move menu button to collapse command zone
  commandIdx = this.toolbarConf.children.indexOf(this.menuButtonConf);
  this.toolbarConf.children.splice(commandIdx, 1);
  zoneIdx = this.getCommandZone(Client.IdfPanel.commands.CZ_COLLAPSE);
  this.toolbarZonesConfig[zoneIdx].children.push(this.menuButtonConf);
  this.menuButtonConf.className += " panel-toolbar-btn";
  //
  // Move lock button to lock command zone
  commandIdx = this.toolbarConf.children.indexOf(this.lockButtonConf);
  this.toolbarConf.children.splice(commandIdx, 1);
  zoneIdx = this.getCommandZone(Client.IdfPanel.commands.CZ_LOCK);
  this.toolbarZonesConfig[zoneIdx].children.push(this.lockButtonConf);
  this.lockButtonConf.className += " panel-toolbar-btn";
  //
  // Move icon to status bar command zone
  commandIdx = this.toolbarConf.children.indexOf(this.iconButtonConf);
  this.toolbarConf.children.splice(commandIdx, 1);
  zoneIdx = this.getCommandZone(Client.IdfPanel.commands.CZ_STATUSBAR);
  this.toolbarZonesConfig[zoneIdx].children.push(this.iconButtonConf);
  this.iconButtonConf.className += " panel-toolbar-btn";
  //
  // Move title to status bar command zone
  commandIdx = this.toolbarConf.children.indexOf(this.titleConf);
  this.toolbarConf.children.splice(commandIdx, 1);
  zoneIdx = this.getCommandZone(Client.IdfPanel.commands.CZ_STATUSBAR);
  this.toolbarZonesConfig[zoneIdx].children.push(this.titleConf);
  //
  // Create status bar configuration
  this.statusbarConf = this.createElementConfig({c: "Span", className: "panel-statusbar"});
  this.titleConf.children.push(this.statusbarConf);
  //
  // Create qbe button configuration
  this.qbeButtonConf = this.createElementConfig({c: "IonButton", icon: "information-circle-outline", className: "generic-btn panel-toolbar-btn qbe-tip-btn"});
  zoneIdx = this.getCommandZone(Client.IdfPanel.commands.CZ_STATUSBAR);
  this.toolbarZonesConfig[zoneIdx].children.push(this.qbeButtonConf);
  //
  // Create navigation buttons
  //
  // Create top button configuration
  this.topButtonConf = this.createElementConfig({c: "IonButton", icon: "rewind", className: "generic-btn panel-toolbar-btn top-btn", events: ["onClick"], customid: this.id.replace(/:/g, "_") + "_top"});
  zoneIdx = this.getCommandZone(Client.IdfPanel.commands.CZ_NAVIGATE);
  this.toolbarZonesConfig[zoneIdx].children.push(this.topButtonConf);
  //
  // Create previous button configuration
  this.prevButtonConf = this.createElementConfig({c: "IonButton", icon: "play", className: "generic-btn panel-toolbar-btn prev-btn", events: ["onClick"]});
  zoneIdx = this.getCommandZone(Client.IdfPanel.commands.CZ_NAVIGATE);
  this.toolbarZonesConfig[zoneIdx].children.push(this.prevButtonConf);
  //
  // Create next button configuration
  this.nextButtonConf = this.createElementConfig({c: "IonButton", icon: "play", className: "generic-btn panel-toolbar-btn next-btn", events: ["onClick"]});
  zoneIdx = this.getCommandZone(Client.IdfPanel.commands.CZ_NAVIGATE);
  this.toolbarZonesConfig[zoneIdx].children.push(this.nextButtonConf);
  //
  // Create bottom button configuration
  this.bottomButtonConf = this.createElementConfig({c: "IonButton", icon: "fastforward", className: "generic-btn panel-toolbar-btn bottom-btn", events: ["onClick"]});
  zoneIdx = this.getCommandZone(Client.IdfPanel.commands.CZ_NAVIGATE);
  this.toolbarZonesConfig[zoneIdx].children.push(this.bottomButtonConf);
  //
  // Create search button configuration
  this.searchButtonConf = this.createElementConfig({c: "IonButton", icon: "search", className: "generic-btn panel-toolbar-btn search-btn", events: ["onClick"], customid: this.id.replace(/:/g, "_") + "_search"});
  zoneIdx = this.getCommandZone(Client.IdfPanel.commands.CZ_SEARCH);
  this.toolbarZonesConfig[zoneIdx].children.push(this.searchButtonConf);
  //
  // Create find button configuration
  this.findButtonConf = this.createElementConfig({c: "IonButton", icon: "flash", className: "generic-btn panel-toolbar-btn find-btn", events: ["onClick"], customid: this.id.replace(/:/g, "_") + "_find"});
  zoneIdx = this.getCommandZone(Client.IdfPanel.commands.CZ_FIND);
  this.toolbarZonesConfig[zoneIdx].children.push(this.findButtonConf);
  //
  // Create form/list button configuration
  this.formListButtonConf = this.createElementConfig({c: "IonButton", icon: Client.mainFrame.idfMobile ? "arrow-back" : "list", className: "generic-btn panel-toolbar-btn formlist-btn", events: ["onClick"], customid: this.id.replace(/:/g, "_") + "_formlist"});
  zoneIdx = this.getCommandZone(Client.IdfPanel.commands.CZ_FORMLIST);
  this.toolbarZonesConfig[zoneIdx].children.push(this.formListButtonConf);
  //
  // Create cancel button configuration
  this.cancelButtonConf = this.createElementConfig({c: "IonButton", icon: "undo", className: "generic-btn panel-toolbar-btn undo-btn", events: ["onClick"], customid: this.id.replace(/:/g, "_") + "_cancel"});
  zoneIdx = this.getCommandZone(Client.IdfPanel.commands.CZ_CANCEL);
  this.toolbarZonesConfig[zoneIdx].children.push(this.cancelButtonConf);
  //
  // Create refresh button configuration
  this.refreshButtonConf = this.createElementConfig({c: "IonButton", icon: "refresh", className: "generic-btn panel-toolbar-btn refresh-btn", events: ["onClick"], customid: this.id.replace(/:/g, "_") + "_refresh"});
  zoneIdx = this.getCommandZone(Client.IdfPanel.commands.CZ_REQUERY);
  this.toolbarZonesConfig[zoneIdx].children.push(this.refreshButtonConf);
  //
  // Create delete button configuration
  this.deleteButtonConf = this.createElementConfig({c: "IonButton", icon: "trash", className: "generic-btn panel-toolbar-btn delete-btn", events: ["onClick"], customid: this.id.replace(/:/g, "_") + "_del"});
  zoneIdx = this.getCommandZone(Client.IdfPanel.commands.CZ_DELETE);
  this.toolbarZonesConfig[zoneIdx].children.push(this.deleteButtonConf);
  //
  // Create insert button configuration
  this.insertButtonConf = this.createElementConfig({c: "IonButton", icon: "add", className: "generic-btn panel-toolbar-btn insert-btn", events: ["onClick"], customid: this.id.replace(/:/g, "_") + "_new"});
  zoneIdx = this.getCommandZone(Client.IdfPanel.commands.CZ_INSERT);
  this.toolbarZonesConfig[zoneIdx].children.push(this.insertButtonConf);
  //
  // Create duplicate button configuration
  this.duplicateButtonConf = this.createElementConfig({c: "IonButton", icon: "copy", className: "generic-btn panel-toolbar-btn duplicate-btn", events: ["onClick"], customid: this.id.replace(/:/g, "_") + "_dupl"});
  zoneIdx = this.getCommandZone(Client.IdfPanel.commands.CZ_DUPLICATE);
  this.toolbarZonesConfig[zoneIdx].children.push(this.duplicateButtonConf);
  //
  // Create save button configuration
  this.saveButtonConf = this.createElementConfig({c: "IonButton", icon: "save", className: "generic-btn panel-toolbar-btn save-btn", events: ["onClick"], customid: this.id.replace(/:/g, "_") + "_save"});
  zoneIdx = this.getCommandZone(Client.IdfPanel.commands.CZ_UPDATE);
  this.toolbarZonesConfig[zoneIdx].children.push(this.saveButtonConf);
  //
  // Create print button configuration
  this.printButtonConf = this.createElementConfig({c: "IonButton", icon: "print", className: "generic-btn panel-toolbar-btn print-btn", events: ["onClick"], customid: this.id.replace(/:/g, "_") + "_print"});
  zoneIdx = this.getCommandZone(Client.IdfPanel.commands.CZ_PRINT);
  this.toolbarZonesConfig[zoneIdx].children.push(this.printButtonConf);
  //
  // Create csv button configuration
  this.csvButtonConf = this.createElementConfig({c: "IonButton", icon: "open", className: "generic-btn panel-toolbar-btn csv-btn", events: ["onClick"], customid: this.id.replace(/:/g, "_") + "_csv"});
  zoneIdx = this.getCommandZone(Client.IdfPanel.commands.CZ_CSV);
  this.toolbarZonesConfig[zoneIdx].children.push(this.csvButtonConf);
  //
  // Create attach button configuration
  this.attachButtonConf = this.createElementConfig({c: "IonButton", icon: "attach", className: "generic-btn panel-toolbar-btn attach-btn", events: ["onClick"], customid: this.id.replace(/:/g, "_") + "_attach"});
  zoneIdx = this.getCommandZone(Client.IdfPanel.commands.CZ_ATTACH);
  this.toolbarZonesConfig[zoneIdx].children.push(this.attachButtonConf);
  //
  // Create group button configuration
  this.groupButtonConf = this.createElementConfig({c: "IonButton", icon: "grid", className: "generic-btn panel-toolbar-btn group-btn", events: ["onClick"], customid: this.id.replace(/:/g, "_") + "_group"});
  zoneIdx = this.getCommandZone(Client.IdfPanel.commands.CZ_GROUP);
  this.toolbarZonesConfig[zoneIdx].children.push(this.groupButtonConf);
  //
  // Create custom buttons configuration
  this.customButtonsConf = [];
  for (let i = 0; i < this.customCommands.length; i++) {
    // Positions for "1-8" custom commands are "18-25"
    let ofs = 18 + i;
    //
    // Positions for "9-16" custom commands are "37-45"
    if (i >= 8)
      ofs = 37 + i - 8;
    //
    this.customButtonsConf[i] = this.createElementConfig({c: "IonButton", className: "generic-btn panel-toolbar-btn custom-btn" + (i + 1), events: ["onClick"], customid: (this.id.replace(/:/g, "_") + "_custom" + i)});
    zoneIdx = this.getCommandZone(ofs);
    this.toolbarZonesConfig[zoneIdx].children.push(this.customButtonsConf[i]);
  }
  //
  // Remove empty zones
  // (not the commandsets, they can be created after this phase)
  zoneIdx = this.getCommandZone(Client.IdfPanel.commands.CZ_CMDSET);
  for (let i = 0; i < this.toolbarZonesConfig.length; i++) {
    if (!this.toolbarZonesConfig[i].children.length && i !== zoneIdx) {
      let idx = this.toolbarConf.children.indexOf(this.toolbarZonesConfig[i]);
      this.toolbarConf.children.splice(idx, 1);
    }
  }
};


/**
 * Create index-th data row configuration
 * @param {Integer} index
 */
Client.IdfPanel.prototype.createDataRowConfig = function (index)
{
  let dataRow = Client.eleMap[this.rows[index]?.id];
  //
  if (this.hasGroupedRows())
    index = this.groupedRowsRoot.groupedIndexToRealIndex(index);
  //
  // If index-th data row does not exists, create new row configuration
  let rowConf;
  if (!dataRow)
    rowConf = this.createElementConfig({c: "IonRow", className: "panel-list-row", noWrap: true});
  //
  let firstInListFieldIndex;
  for (let i = 0; i < this.fields.length; i++) {
    let field = this.fields[i];
    //
    // If current field is not shown in list or is not inside the list, continue.
    // Otherwise create a value configuration (i.e. a col) and add it to current row configuration
    if (!field.isShown() || !field.isInList())
      continue;
    //
    if (firstInListFieldIndex === undefined)
      firstInListFieldIndex = i;
    //
    let fieldValue = field.getValueByIndex(index);
    //
    // If I have no value yet, continue.
    // This happens when a field is not shown in form mode and panel starts in form mode.
    // In this case server sends just the field without any value (that will be sent as soon as panel switches mode)
    // Also continue if field value exists and it's already rendered
    if (!fieldValue || fieldValue.listContainerId)
      continue;
    //
    let nextSiblingId;
    //
    // If index-th row exists, I have to fill its missing columns (if any). So I have to find the next sibling for column I'm going to create
    if (dataRow) {
      for (let j = i + 1; j < this.fields.length; j++) {
        let nextField = this.fields[j];
        // If current field is not shown in list or it's not in list, continue
        if (!nextField.isShown() || !nextField.isInList())
          continue;
        //
        // Get index-th value
        let nextValue = nextField.getValueByIndex(index);
        //
        // If value does not exists or it's not rendered yet, continue
        if (!nextValue || !nextValue.listContainerId)
          continue;
        //
        nextSiblingId = nextValue.listContainerId;
        break;
      }
    }
    //
    // The first in list field values have to handle row selectors too
    if (firstInListFieldIndex === i && !fieldValue.rowSelectorId) {
      let rowSelectorConf = fieldValue.createRowSelectorConfig();
      if (!dataRow)
        rowConf.children.push(rowSelectorConf);
      else
        dataRow.insertBefore({child: rowSelectorConf, sib: nextSiblingId});
    }
    //
    // Create value configuration
    let valueConf = fieldValue.createListConfig();
    //
    // If field belongs to a group, I have to create value container into group container
    if (field.group) {
      let groupConf = field.group.listContainersConf[index];
      if (!groupConf) {
        groupConf = field.group.createListConfig(index);
        rowConf.children.push(groupConf);
      }
      //
      if (!dataRow)
        groupConf.children[0].children.push(valueConf);
      else {
        let groupEl = Client.eleMap[groupConf.id];
        if (!groupEl) {
          groupConf.children[0].children.push(valueConf);
          dataRow.insertBefore({child: groupConf, sib: nextSiblingId});
        }
        else
          groupEl.elements[0].insertBefore({child: valueConf, sib: nextSiblingId});
      }
    }
    else {
      if (!dataRow)
        rowConf.children.push(valueConf);
      else
        dataRow.insertBefore({child: valueConf, sib: nextSiblingId});
    }
  }
  //
  return rowConf;
};


/**
 * Clone given data row
 * @param {Object} options
 *                - el: dataRow to clone
 *                - index
 *                - level
 */
Client.IdfPanel.prototype.cloneDataRow = function (options)
{
  let index = options.index;
  let level = options.level;
  let el = options.el;
  //
  let config = {children: [], c: level === 1 ? "IonRow" : "IonCol"};
  //
  let props = Object.keys(el);
  for (let i = 0; i < props.length; i++) {
    let p = props[i];
    //
    // Skip some properties
    if (["parent", "view", "elements", "children", "parentWidget", "domObj"].includes(p))
      continue;
    //
    // Copy current property
    config[p] = el[p];
  }
  //
  // If I reach level 0 (i.e. column level), cloning is done
  if (level === 0) {
    let rootObject = el.getRootObject();
    //
    // Add reference to placeholder row to id
    config.id += "phr" + index;
    //
    // Set column style
    config.style = config.style || {};
    let styleProps = rootObject.style.cssText.split(";");
    for (let i = 0; i < styleProps.length; i++) {
      let prop = styleProps[i].split(":");
      if (prop[0])
        config.style[prop[0].trim()] = prop[1].trim();
    }
    //
    config.className = rootObject.className;
    //
    return config;
  }
  else // In case of IonRow, overwrite id
    config.id = "phr" + index;
  //
  // Recursively clone children elements
  for (let i = 0; i < el.elements.length; i++)
    config.children.push(this.cloneDataRow({el: el.elements[i], index, level: level - 1}));
  //
  return config;
};


/**
 * Create children elements
 * @param {Object} el
 */
Client.IdfPanel.prototype.createChildren = function (el)
{
  if (!el.children)
    return;
  //
  // On IDC, fields belonging to a group are children of that group.
  // On IDF, the group and the fields belonging to it are siblings.
  // I want IDC behaves like IDF
  if (!Client.mainFrame.isIDF) {
    // Get IdfGroup children
    let groups = el.children.filter(obj => obj.c === Client.Widget.transXmlNodeMap.grp);
    //
    for (let i = 0; i < groups.length; i++) {
      let group = groups[i];
      let index = el.children.findIndex(obj => obj.id === group.id);
      //
      // Get IdfField children
      let fields = group.children.filter(obj => Client.Widget.isFieldClass(obj.c));
      //
      // Remove fields from groups and push them into panel children array, at group's old position.
      for (let j = 0; j < fields.length; j++) {
        let field = group.children.splice(j, 1)[0];
        el.children.splice(index, 0, field);
        index++;
      }
      //
      // Recalculate group index, because it may has changed due to fields splice
      index = el.children.findIndex(obj => obj.id === group.id);
      //
      // Since I want groups after fields, remove groups from panel children array and push them again at the end
      group = el.children.splice(index, 1)[0];
      el.children.push(group);
    }
  }
  //
  // Get IdfPanelPage children
  let pages = el.children.filter(obj => obj.c === Client.Widget.transXmlNodeMap.ppg);
  //
  // Since I want pages to be the last children, remove them from children array and push them again at the end
  for (let i = 0; i < pages.length; i++) {
    let index = el.children.findIndex(obj => obj.id === pages[i].id);
    let page = el.children.splice(index, 1)[0];
    el.children.push(page);
  }
  //
  el.hasPages = !!pages.length;
  //
  Client.Widget.prototype.createChildren.call(this, el);
};


/**
 * Realize widget UI
 * @param {Object} widget
 * @param {View|Element|Widget} parent
 * @param {View} view
 */
Client.IdfPanel.prototype.realize = function (widget, parent, view)
{
  Client.IdfFrame.prototype.realize.call(this, widget, parent, view);
  //
  this.updateStructure();
  this.initializeListFilters();
};


/**
 * Add given field to fields array
 * @param {IdfField} field
 */
Client.IdfPanel.prototype.addField = function (field)
{
  this.fields.push(field);
};


/**
 * Add given group to groups array
 * @param {IdfGroup} group
 */
Client.IdfPanel.prototype.addGroup = function (group)
{
  this.groups.push(group);
};


/**
 * Add given page to pages array
 * @param {IdfPanelPage} page
 */
Client.IdfPanel.prototype.addPage = function (page)
{
  this.pages.push(page);
};


/**
 * Detach grid, fields and groups dom objects from their old columns.
 * Then destroy and recreate grid structure and finally place grid, fields and groups dom objects to their new columns
 */
Client.IdfPanel.prototype.updateStructure = function ()
{
  // Since I'm going to destroy main grid structure, detach grid, groups and out of list fields.
  // They will be placed in the new structure
  this.unplace();
  //
  if (this.hasList) {
    let listContainer = Client.eleMap[this.listContainerConf.id];
    //
    // Remove old list rows
    this.removeGridRows();
    //
    // Create new list structure
    this.listGridRows = this.createStructure();
    for (let i = 0; i < this.listGridRows.length; i++) {
      let listRow = this.view.createElement(this.listGridRows[i].conf, listContainer, this.view);
      listContainer.elements.push(listRow);
    }
  }
  //
  if (this.hasForm) {
    let formContainer = Client.eleMap[this.formContainerConf.id];
    //
    // Remove old form rows
    this.removeGridRows(true);
    //
    // Create new form structure
    this.formGridRows = this.createStructure(true);
    for (let i = 0; i < this.formGridRows.length; i++) {
      let formRow = this.view.createElement(this.formGridRows[i].conf, formContainer, this.view);
      formContainer.elements.push(formRow);
    }
  }
  //
  // Place grid, groups and out of list fields in the new structure
  this.place();
  //
  // If this panel uses ROW as qbe mode and qbe row does not exists yet, create it
  if (this.canUseRowQbe() && !this.qbeRowConf)
    this.createQbeRow();
};


/**
 * Remove form/list grid rows
 * @param {Boolean} form
 */
Client.IdfPanel.prototype.removeGridRows = function (form)
{
  let rows = form ? this.formGridRows : this.listGridRows;
  let rowsContainer = form ? Client.eleMap[this.formContainerConf.id] : Client.eleMap[this.listContainerConf.id];
  //
  for (let i = 0; i < rows.length; i++) {
    rowsContainer.removeChild(rows[i].conf);
    rows.splice(i--, 1);
  }
};


/**
 * Append grid, fields and groups dom objects to their own columns
 */
Client.IdfPanel.prototype.place = function ()
{
  if (this.hasList) {
    // Get grid parent column
    let parentColumn = this.getListFieldColumn(this.id);
    this.gridColConf = parentColumn.conf;
    //
    // Set its className
    this.gridColConf.className = "panel-list-grid-col";
    //
    let gridCol = Client.eleMap[this.gridColConf.id];
    let grid = Client.eleMap[this.extGridConf.id];
    //
    gridCol.appendChildObject(undefined, grid.getRootObject());
    gridCol.elements.push(grid);
    grid.parent = gridCol;
  }
  //
  for (let i = 0; i < this.fields.length; i++)
    this.fields[i].place();
  //
  for (let i = 0; i < this.groups.length; i++)
    this.groups[i].place();
};


/**
 * Remove grid, fields and groups dom objects from their parent columns
 */
Client.IdfPanel.prototype.unplace = function ()
{
  if (this.hasList) {
    let grid = Client.eleMap[this.extGridConf.id];
    for (let i = 0; i < grid.parent.elements.length; i++) {
      if (grid.parent.elements[i].id === grid.id) {
        grid.parent.elements.splice(i, 1);
        break;
      }
    }
  }
  //
  for (let i = 0; i < this.groups.length; i++)
    this.groups[i].unplace();
  //
  for (let i = 0; i < this.fields.length; i++)
    this.fields[i].unplace();
};


/**
 * Update element properties
 * @param {Object} props
 */
Client.IdfPanel.prototype.updateElement = function (props)
{
  this.lastActiveRowIndex = this.getActiveRowIndex();
  //
  let objectsToUpdate = {};
  //
  props = props || {};
  //
  Client.IdfFrame.prototype.updateElement.call(this, props);
  //
  if (props.applyVisualStyle) {
    delete props.applyVisualStyle;
    objectsToUpdate.visualStyle = true;
  }
  //
  if (props.calcLayout) {
    delete props.calcLayout;
    objectsToUpdate.calcLayout = true;
  }
  //
  if (props.updateToolbar) {
    delete props.updateToolbar;
    objectsToUpdate.toolbar = true;
  }
  //
  if (props.totalRows !== undefined) {
    this.totalRows = props.totalRows;
    objectsToUpdate.toolbar = true;
    objectsToUpdate.multiSel = true;
  }
  //
  if (props.numRows !== undefined)
    this.numRows = props.numRows;
  //
  if (props.activePage !== undefined) {
    this.activePage = props.activePage;
    objectsToUpdate.activePage = true;
    objectsToUpdate.calcLayout = true;
  }
  //
  if (props.headerHeight !== undefined) {
    this.headerHeight = props.headerHeight;
    objectsToUpdate.calcLayout = true;
  }
  //
  if (props.rowHeightResize !== undefined)
    this.rowHeightResize = props.rowHeightResize;
  //
  if (props.dataBlockStart !== undefined)
    this.dataBlockStart = props.dataBlockStart;
  //
  if (props.dataBlockEnd !== undefined)
    this.dataBlockEnd = props.dataBlockEnd;
  //
  if (props.canResizeColumn !== undefined)
    this.canResizeColumn = props.canResizeColumn;
  if (props.canReorderColumn !== undefined)
    this.canReorderColumn = props.canReorderColumn;
  //
  if (props.groupedRows !== undefined) {
    objectsToUpdate.groupedRows = true;
    objectsToUpdate.bufferVideo = true;
  }
  //
  if (props.data !== undefined) {
    this.data = props.data;
    objectsToUpdate.data = true;
    objectsToUpdate.bufferVideo = true;
    objectsToUpdate.calcLayout = true;
    objectsToUpdate.rowSelectors = true;
    objectsToUpdate.multiSel = true;
    objectsToUpdate.toolbar = true;
    objectsToUpdate.visualStyle = true;
    objectsToUpdate.scrollbar = true;
    objectsToUpdate.skipScroll = true;
  }
  //
  if (props.actualRow !== undefined && (props.actualRow !== this.actualRow || this.realizing)) {
    this.actualRow = props.actualRow;
    objectsToUpdate.calcLayout = true;
    objectsToUpdate.activeRow = true;
    objectsToUpdate.toolbar = true;
    objectsToUpdate.bufferVideo = true;
    objectsToUpdate.skipScroll = false;
  }
  //
  if (props.actualPosition !== undefined && (props.actualPosition !== this.actualPosition || this.realizing)) {
    this.actualPosition = props.actualPosition;
    objectsToUpdate.calcLayout = true;
    objectsToUpdate.activeRow = true;
    objectsToUpdate.toolbar = true;
    objectsToUpdate.bufferVideo = true;
    objectsToUpdate.skipScroll = false;
  }
  //
  if (props.skipScroll)
    objectsToUpdate.skipScroll = true;
  //
  if (props.showRowSelector !== undefined) {
    this.showRowSelector = props.showRowSelector;
    objectsToUpdate.rowSelectors = true;
    objectsToUpdate.calcLayout = true;
  }
  //
  if (props.showMultiSelection !== undefined) {
    this.showMultiSelection = props.showMultiSelection;
    objectsToUpdate.multiSel = true;
    objectsToUpdate.statusbar = true;
  }
  //
  if (props.enableMultiSelection !== undefined) {
    this.enableMultiSelection = props.enableMultiSelection;
    //
    if (this.multiSelButtonConf)
      Client.eleMap[this.multiSelButtonConf.id].updateElement({visible: this.enableMultiSelection});
  }
  //
  if (props.selectOnlyVisibleRows !== undefined)
    this.selectOnlyVisibleRows = props.selectOnlyVisibleRows;
  //
  if (props.gridHeight !== undefined) {
    this.gridHeight = isNaN(props.gridHeight) ? undefined : props.gridHeight;
    this.orgGridHeight = this.gridHeight;
    objectsToUpdate.structure = true;
    objectsToUpdate.calcLayout = true;
  }
  //
  if (props.gridLeft !== undefined) {
    this.gridLeft = isNaN(props.gridLeft) ? undefined : props.gridLeft;
    this.orgGridLeft = this.gridLeft;
    objectsToUpdate.structure = true;
    objectsToUpdate.calcLayout = true;
  }
  //
  if (props.gridTop !== undefined) {
    this.gridTop = isNaN(props.gridTop) ? undefined : props.gridTop;
    this.orgGridTop = this.gridTop;
    objectsToUpdate.structure = true;
    objectsToUpdate.calcLayout = true;
  }
  //
  if (props.resizeWidth !== undefined) {
    this.resizeWidth = props.resizeWidth;
    objectsToUpdate.calcLayout = true;
  }
  //
  if (props.resizeHeight !== undefined) {
    this.resizeHeight = props.resizeHeight;
    objectsToUpdate.calcLayout = true;
  }
  //
  if (props.fixedColumns !== undefined) {
    this.fixedColumns = props.fixedColumns;
    objectsToUpdate.fixedColumns = true;
  }
  //
  if (props.layout !== undefined && (props.layout !== this.layout || this.realizing)) {
    this.layout = props.layout;
    //
    objectsToUpdate.layout = true;
    objectsToUpdate.toolbar = true;
    objectsToUpdate.className = true;
    objectsToUpdate.visualStyle = true;
  }
  //
  if (props.status !== undefined) {
    this.status = props.status;
    objectsToUpdate.toolbar = true;
    objectsToUpdate.visualStyle = true;
    //
    // Tell my children that status is changed
    for (let i = 0; i < this.fields.length; i++)
      this.fields[i].onPanelStatusChange();
  }
  //
  if (props.hasList !== undefined)
    this.hasList = props.hasList;
  //
  if (props.hasForm !== undefined)
    this.hasForm = props.hasForm;
  //
  if (props.canUpdate !== undefined) {
    this.canUpdate = props.canUpdate;
    objectsToUpdate.toolbar = true;
  }
  //
  if (props.canDelete !== undefined) {
    this.canDelete = props.canDelete;
    objectsToUpdate.toolbar = true;
  }
  //
  if (props.canInsert !== undefined) {
    this.canInsert = props.canInsert;
    objectsToUpdate.toolbar = true;
  }
  //
  if (props.canSearch !== undefined) {
    this.canSearch = props.canSearch;
    objectsToUpdate.toolbar = true;
  }
  //
  if (props.canSort !== undefined) {
    this.canSort = props.canSort;
    objectsToUpdate.toolbar = true;
  }
  //
  if (props.canGroup !== undefined) {
    this.canGroup = props.canGroup;
    objectsToUpdate.toolbar = true;
  }
  //
  if (props.showGroups !== undefined) {
    this.showGroups = props.showGroups;
    objectsToUpdate.toolbar = true;
  }
  //
  if (props.enableInsertWhenLocked !== undefined) {
    this.enableInsertWhenLocked = props.enableInsertWhenLocked;
    objectsToUpdate.toolbar = true;
  }
  //
  if (props.qbeTip !== undefined) {
    this.qbeTip = props.qbeTip;
    //
    this.qbeTip = this.qbeTip.replace(/QBEF1/g, "qbe-field");
    this.qbeTip = this.qbeTip.replace(/QBEF2/g, "qbe-value");
    //
    objectsToUpdate.toolbar = true;
  }
  //
  if (props.searchMode !== undefined) {
    this.searchMode = props.searchMode;
    objectsToUpdate.toolbar = true;
  }
  //
  if (props.isDO !== undefined) {
    this.isDO = props.isDO;
    objectsToUpdate.toolbar = true;
  }
  //
  if (props.hasDocTemplate !== undefined) {
    this.hasDocTemplate = props.hasDocTemplate;
    objectsToUpdate.toolbar = true;
  }
  //
  if (props.DOModified !== undefined) {
    this.DOModified = props.DOModified;
    objectsToUpdate.toolbar = true;
  }
  //
  if (props.DOMaster !== undefined) {
    this.DOMaster = props.DOMaster;
    objectsToUpdate.toolbar = true;
  }
  //
  if (props.DOCanSave !== undefined) {
    this.DOCanSave = props.DOCanSave;
    objectsToUpdate.toolbar = true;
  }
  //
  if (props.DOSingleDoc !== undefined) {
    this.DOSingleDoc = props.DOSingleDoc;
    objectsToUpdate.toolbar = true;
  }
  //
  if (props.pullToRefresh !== undefined) {
    this.pullToRefresh = props.pullToRefresh;
    //
    if (this.hasList) {
      let el = Client.eleMap[this.extGridListConf.id];
      //
      // Nasty bug of the refresher: is activated when set !== undefined, so true or false will activate it...
      if (this.pullToRefresh && (Client.mainFrame.device.isMobile || Client.mainFrame.idfMobile))
        el.updateElement({refresher: true});
    }
  }
  //
  if (props.advancedTabOrder !== undefined)
    this.advancedTabOrder = props.advancedTabOrder;
  //
  if (props.hasPages !== undefined)
    objectsToUpdate.pagesContainer = true;
  //
  // "visualFlags" property exists just on IDF. Thus, transform variation on visualFlags into variations on each single property
  if (props.visualFlags !== undefined) {
    this.visualFlags = props.visualFlags;
    //
    props.hidePages = (this.visualFlags & 0x4000) !== 0;
  }
  //
  if (props.hidePages !== undefined) {
    this.hidePages = props.hidePages;
    objectsToUpdate.pagesContainer = true;
  }
  //
  if (props.toolbarEventDef !== undefined)
    this.toolbarEventDef = props.toolbarEventDef;
  //
  if (props.scrollEventDef !== undefined)
    this.scrollEventDef = props.scrollEventDef;
  //
  if (props.rowSelectEventDef !== undefined)
    this.rowSelectEventDef = props.rowSelectEventDef;
  //
  if (props.pageClickEventDef !== undefined)
    this.pageClickEventDef = props.pageClickEventDef;
  //
  if (props.multiSelEventDef !== undefined)
    this.multiSelEventDef = props.multiSelEventDef;
  //
  if (props.focusEventDef !== undefined)
    this.focusEventDef = props.focusEventDef;
  //
  if (props.selectionChangeEventDef !== undefined)
    this.selectionChangeEventDef = props.selectionChangeEventDef;
  //
  // If there's something to update, do it now
  if (Object.keys(objectsToUpdate).length)
    this.updateObjects(objectsToUpdate);
};


/**
 * Handle an event
 * @param {Object} event
 */
Client.IdfPanel.prototype.onEvent = function (event)
{
  let events = Client.IdfFrame.prototype.onEvent.call(this, event);
  //
  switch (event.id) {
    case "onClick":
      switch (event.obj) {
        case this.multiSelButtonConf?.id: // If user clicks on multiple selection button
          events.push(...this.handleMultiSelClick(event));
          break;

        case this.lockButtonConf.id: // If user clicks on lock button
        case this.formListButtonConf.id: // If user clicks on form/list button
        case this.topButtonConf.id:
        case this.prevButtonConf.id:
        case this.nextButtonConf.id:
        case this.bottomButtonConf.id: // If user clicks on a navigation button
        case this.saveButtonConf.id: // If user clicks on save button
        case this.cancelButtonConf.id: // If user clicks on cancel button
        case this.refreshButtonConf.id: // If user clicks on refresh button
        case this.insertButtonConf.id: // If user clicks on insert button
        case this.deleteButtonConf.id: // If user clicks on delete button
        case this.searchButtonConf.id: // If user clicks on search button
        case this.findButtonConf.id: // If user clicks on find button
        case this.duplicateButtonConf.id: // If user clicks on duplicate button
        case this.csvButtonConf.id: // If user clicks on export button
        case this.printButtonConf.id: // If user clicks on print button
        case this.attachButtonConf.id: // If user clicks on attach button
        case this.groupButtonConf.id: // If user clicks on group button
          events.push(...this.handleToolbarClick(event));
          break;

        default:
          // Otherwise check if user clicks on a custom button
          for (let i = 0; i < this.customButtonsConf.length; i++) {
            let customButtonId = this.customButtonsConf[i].id;
            if (event.obj === customButtonId) {
              events.push(...this.handleToolbarClick(event));
              break;
            }
          }
          break;
      }
      break;

    case "onScroll":
      // I don't have to return any event. There is a scroll timeout that directly sends "panscr" event if needed
      if (event.obj === this.gridConf.id)
        this.handleScroll();
      break;

    case "onRefresh":
      let el = Client.eleMap[this.extGridListConf.id];
      el.refreshCompleted();
      //
      event.obj = this.refreshButtonConf.id;
      events.push(...this.handleToolbarClick(event));
      break;

    case "onKey":
      events.push(...this.handleFunctionKeys(event));
      break;
  }
  //
  return events;
};


/**
 * Update objects
 * @param {Object} objectsToUpdate
 */
Client.IdfPanel.prototype.updateObjects = function (objectsToUpdate)
{
  let activeRowIndex = this.getActiveRowIndex();
  //
  // In order to update panel structure I have to:
  // 1) detach grid, fields and groups dom objects from their old columns
  // 2) destroy and recreate structure (rows and columns)
  // 3) place grid, fields and groups dom objects to their new columns
  // 4) calculate layout
  if (objectsToUpdate.structure && !this.realizing)
    this.updateStructure();
  //
  if (objectsToUpdate.activePage)
    this.updateActivePage();
  //
  if (objectsToUpdate.groupedRows)
    this.updateGroupedRows();
  //
  if (objectsToUpdate.data) {
    // Update data rows
    this.updateDataRows();
    //
    // Assign subFrame
    this.fields.forEach(f => f.assignSubFrame());
    //
    // I have to assign shared controls to active row
    this.setActiveRow(true);
  }
  //
  // If I have to update row selectors visibility, do it now
  if (objectsToUpdate.rowSelectors)
    this.updateRowSelectorsVisibility();
  //
  if (objectsToUpdate.layout)
    this.updateLayout();
  //
  // If I have to calculate layout, do it now
  if (objectsToUpdate.calcLayout || objectsToUpdate.fixedColumns) {
    let index = objectsToUpdate.activeRow && !objectsToUpdate.data && !objectsToUpdate.fixedColumns ? activeRowIndex : undefined;
    this.calcLayout(index);
  }
  //
  // If I have to set active row, do it now
  if (objectsToUpdate.activeRow)
    this.setActiveRow(objectsToUpdate.skipScroll);
  //
  // If I have to update toolbar, do it now
  if (objectsToUpdate.toolbar)
    this.updateToolbar();
  //
  // If I have to update className, do it now
  if (objectsToUpdate.className)
    this.updateClassName();
  //
  // If I have to apply visual style, do it now
  if (objectsToUpdate.visualStyle)
    this.applyVisualStyle();
  //
  // If I have to fill buffer video, do it now
  if (objectsToUpdate.bufferVideo)
    this.fillBufferVideo();
  //
  // If I have to adjust scrollbar, do it now
  // I have to adjust scrollbar just in case I'm not going to fill buffer video.
  // In that case scrollbar will be adjusted after that operation.
  if (objectsToUpdate.scrollbar && !objectsToUpdate.bufferVideo)
    this.adjustScrollbar();
  //
  // If I have to update multiple selection checkboxes visibility, do it now
  if (objectsToUpdate.multiSel) {
    this.fields.forEach(f => f.updateMultiSelVisibility(this.showMultiSelection));
    //
    if (this.showMultiSelection) {
      for (let i = 1; i <= this.totalRows; i++)
        this.fields.forEach(f => f.selectRow(this.multiSelStatus[i], i));
    }
  }
  //
  // If I have to update statusbar, do it now
  if (objectsToUpdate.statusbar)
    this.updateStatusbar();
  //
  if (objectsToUpdate.data) {
    delete this.data;
    delete this.dataBlockStart;
    delete this.dataBlockEnd;
  }
  //
  // If I have to update pages container, do it now
  if (objectsToUpdate.pagesContainer)
    Client.eleMap[this.pagesContainerConf.id].updateElement({visible: !!this.pages.length && !this.hidePages});
};


/**
 * Create fake fields value to fill panel in the view editor
 */
Client.IdfPanel.prototype.createFakeFieldsValues = function ()
{
  for (let i = 0; i < this.fields.length; i++) {
    let field = this.fields[i];
    //
    let fieldsValues = [];
    //
    // Create values
    for (let j = 1; j <= 10; j++) {
      let valueConfig = this.createElementConfig({c: "IdfFieldValue", index: j, text: field.listHeader + j});
      fieldsValues.push(valueConfig);
    }
    //
    field.createChildren({children: fieldsValues});
  }
};


/**
 * Set active row
 * @param {Boolean} skipScroll
 */
Client.IdfPanel.prototype.setActiveRow = function (skipScroll)
{
  let index = this.getActiveRowIndex();
  //
  // Assign shared out list value container and form value container to index-th value. Now it is the temporary owner
  for (let i = 0; i < this.fields.length; i++) {
    let field = this.fields[i];
    field.assignControls(index);
    field.writeValue(index, true);
    //
    field.applyVisualStyle(this.lastActiveRowIndex);
    field.applyVisualStyle(index);
  }
  //
  // When user selects a row by clicking on it, skip scroll because that row is already visibile.
  // Instead, if this is not a row click, show row scrolling grid container (if I'm in list mode)
  if (!skipScroll && this.layout === Client.IdfPanel.layouts.list)
    this.scrollToDataRow(index);
};


/**
 * Handle click on data row (called by IdfFieldValue)
 * @param {Object} event
 * @param {IdfFieldValue} fieldValue
 */
Client.IdfPanel.prototype.handleDataRowClick = function (event, fieldValue)
{
  let events = [];
  //
  if (this.status === Client.IdfPanel.statuses.qbe)
    return events;
  //
  if (!fieldValue.parentField.isInList() || this.layout === Client.IdfPanel.layouts.form)
    return events;
  //
  let rowSelector = event && event.obj === fieldValue.rowSelector?.id;
  //
  // Don't click current row if click occurred on row selector column when multiple selection is enabled.
  // In this case user just wants to select row
  if (rowSelector && this.showMultiSelection)
    return events;
  //
  if (fieldValue.isRowQbe) {
    if (rowSelector)
      events.push({
        id: "qbeclall",
        def: Client.IdfMessagesPump.eventTypes.URGENT,
        content: {
          oid: this.id
        }
      });
  }
  else {
    // Tell server that selected row is changed
    let isRowSelectorClick = rowSelector || event?.id === "onDblclick";
    events.push(...this.handleRowChange(fieldValue.index, isRowSelectorClick));
    //
    // In case of mobile, if panel has form and automatic layout, change mode on data row click
    if (Client.mainFrame.idfMobile && this.hasForm && this.automaticLayout)
      events.push(...this.onEvent({
        id: "onClick",
        obj: this.formListButtonConf.id,
        content: {
          offsetX: 0,
          offsetY: 0
        }
      }));
  }
  //
  return events;
};


/**
 * Handle click on multiple selection button
 * @param {Object} event
 */
Client.IdfPanel.prototype.handleMultiSelClick = function (event)
{
  let events = [];
  //
  let obn;
  //
  // If I'm not showing multiple selection, tell server to toggle it
  if (!this.showMultiSelection)
    obn = "seltog";
  else {
    // Get eventual popup box result
    obn = event.content.res;
    //
    // If popup box returned an unexpected value, return no events
    if (event.id === "popupCallback" && !["selall", "selnone", "seltog"].includes(obn))
      return events;
  }
  //
  // If I have an obn to send to server, send it
  if (obn) {
    // If I'm on IDF, do operation just if toolbar events have to be handled client side too
    if (!Client.mainFrame.isIDF || Client.IdfMessagesPump.isClientSideEvent(this.toolbarEventDef)) {
      switch (obn) {
        case "selall":
          this.updateMultiSel({value: true});
          break;

        case "selnone":
          this.updateMultiSel();
          break;

        case "seltog":
          this.updateElement({showMultiSelection: !this.showMultiSelection});
          break;
      }
    }
    //
    if (Client.mainFrame.isIDF) {
      events.push({
        id: "pantb",
        def: this.toolbarEventDef,
        content: {
          oid: this.id,
          obn
        }
      });
    }
    else {
      // TODO
    }
  }
  else { // Otherwise open popup with three options: select all, unselect all, show row selectors
    let items = [];
    if (this.status !== Client.IdfPanel.statuses.qbe) {
      items.push({id: "selall", title: Client.IdfResources.t("TIP_TITLE_TooltipSelectAll"), icon: "checkbox"});
      items.push({id: "selnone", title: Client.IdfResources.t("TIP_TITLE_TooltipDeseleziona"), icon: "square-outline"});
    }
    //
    items.push({id: "seltog", title: Client.IdfResources.t("TIP_TITLE_SelectPopupCmd"), icon: "arrow-round-forward"});
    //
    let buttonRects = Client.eleMap[this.multiSelButtonConf.id].getRootObject().getBoundingClientRect();
    let rect = {left: buttonRects.left + "px", top: buttonRects.bottom + "px"};
    //
    Client.mainFrame.popup({
      options: {
        type: "menu",
        style: "menu-popup multisel-popup",
        rect,
        items,
        callback: res => {
          Client.mainFrame.sendEvents(this.handleMultiSelClick({id: "popupCallback", content: {res}}));
        }
      }
    });
  }
  //
  return events;
};


/**
 * Handle click on toolbar button
 * @param {Object} event
 */
Client.IdfPanel.prototype.handleToolbarClick = function (event)
{
  let updateProps = {};
  let updateMultiSel;
  //
  // Count selected row if multiple selection is active
  let selectedRows = this.showMultiSelection ? this.getSelectedDataRows() : 0;
  //
  let type = Client.Widget.msgTypes.CONFIRM;
  let commandName, commandValue, msg;
  switch (event.obj) {
    case this.lockButtonConf.id:
      commandName = this.locked ? "unlock" : "lock";
      updateProps.locked = !this.locked;
      break;

    case this.formListButtonConf.id:
      commandName = "list";
      commandValue = Client.IdfPanel.commands.CMD_FORMLIST;
      //
      // Change mode just if toolbar events have to be handled client side too
      updateProps.layout = this.layout === Client.IdfPanel.layouts.list ? Client.IdfPanel.layouts.form : Client.IdfPanel.layouts.list;
      if (updateProps.layout === Client.IdfPanel.layouts.list)
        updateProps.actualPosition = this.actualPosition;
      break;

    case this.saveButtonConf.id:
      commandName = "save";
      commandValue = Client.IdfPanel.commands.CMD_SAVE;
      break;

    case this.cancelButtonConf.id:
      commandName = "cancel";
      commandValue = Client.IdfPanel.commands.CMD_CANCEL;
      break;

    case this.refreshButtonConf.id:
      commandName = "refresh";
      commandValue = Client.IdfPanel.commands.CMD_REFRESH;
      break;

    case this.insertButtonConf.id:
      commandName = "insert";
      commandValue = Client.IdfPanel.commands.CMD_INSERT;
      //
      if (this.locked)
        updateProps.locked = false;
      break;

    case this.deleteButtonConf.id:
      commandName = "delete";
      commandValue = Client.IdfPanel.commands.CMD_DELETE;
      //
      // If multiple selection is enabled, remember I have to update it after delete
      updateMultiSel = this.showMultiSelection;
      //
      if (this.confirmDelete) {
        // TODO: this.DoHighlightDelete(true);
        //
        // Calculate message to show
        if (!this.showMultiSelection)
          msg = Client.IdfResources.t("PAN_MSG_ConfirmDeleteRS", [this.caption]);
        else {
          if (selectedRows === 0) {
            type = Client.Widget.msgTypes.ALERT;
            msg = Client.IdfResources.t("PAN_MSG_ConfirmDeleteNR", [this.caption]);
          }
          else if (selectedRows === 1)
            msg = Client.IdfResources.t("PAN_MSG_ConfirmDeleteRS", [this.caption]);
          else if (selectedRows < this.totalRows)
            msg = Client.IdfResources.t("PAN_MSG_ConfirmDeleteRR", [this.caption, selectedRows]);
          else
            msg = Client.IdfResources.t("PAN_MSG_ConfirmDeleteAR", [this.caption]);
        }
      }
      break;

    case this.searchButtonConf.id:
      commandName = "search";
      commandValue = Client.IdfPanel.commands.CMD_SEARCH;
      break;

    case this.findButtonConf.id:
      commandName = "find";
      commandValue = Client.IdfPanel.commands.CMD_FIND;
      break;

    case this.duplicateButtonConf.id:
      commandName = "dupl";
      commandValue = Client.IdfPanel.commands.CMD_DUPLICATE;
      //
      // If multiple selection is active but there are no selected rows, ask user for confirmation about duplicate
      if (this.showMultiSelection && !selectedRows)
        msg = Client.IdfResources.t("PAN_MSG_ConfirmDuplicateNR", [this.caption]);
      break;

    case this.csvButtonConf.id:
      commandName = "csv";
      commandValue = Client.IdfPanel.commands.CMD_CSV;
      //
      // If multiple selection is active but there are no selected rows, ask user for confirmation about duplicate/export
      if (this.showMultiSelection && !selectedRows)
        msg = Client.IdfResources.t("PAN_MSG_ConfirmExportNR", [this.caption]);
      break;

    case this.printButtonConf.id:
      commandName = "print";
      commandValue = Client.IdfPanel.commands.CMD_PRINT;
      break;

    case this.attachButtonConf.id:
      commandName = "attach";
      commandValue = Client.IdfPanel.commands.CMD_ATTACH;
      break;

    case this.groupButtonConf.id:
      commandName = "group";
      commandValue = Client.IdfPanel.commands.CMD_GROUP;
      break;

    case this.topButtonConf.id:
      commandName = "top";
      commandValue = Client.IdfPanel.commands.CMD_NAVIGATION;
      updateProps.actualRow = 0;
      updateProps.actualPosition = 1;
      //
      this.dataBlockStart = 1;
      this.dataBlockEnd = this.getNumRows();
      break;

    case this.prevButtonConf.id:
    {
      commandName = "prev";
      commandValue = Client.IdfPanel.commands.CMD_NAVIGATION;
      updateProps.actualRow = 0;
      //
      let numRows = (this.layout === Client.IdfPanel.layouts.list) ? this.getNumRows() : 1;
      updateProps.actualPosition = this.actualPosition - numRows + 1;
      if (updateProps.actualPosition < 1)
        updateProps.actualPosition = 1;
      //
      this.dataBlockStart = updateProps.actualPosition - this.getNumRows();
      this.dataBlockEnd = updateProps.actualPosition;
      break;
    }

    case this.nextButtonConf.id:
      commandName = "next";
      commandValue = Client.IdfPanel.commands.CMD_NAVIGATION;
      updateProps.actualRow = 0;
      //
      let numRows = (this.layout === Client.IdfPanel.layouts.list) ? this.getNumRows() : 1;
      updateProps.actualPosition = this.actualPosition + numRows - 1;
      if (updateProps.actualPosition >= this.totalRows)
        updateProps.actualPosition = this.totalRows;
      else if (updateProps.actualPosition > this.totalRows - numRows + 1)
        updateProps.actualPosition = this.totalRows - numRows + 1;
      //
      this.dataBlockStart = updateProps.actualPosition;
      this.dataBlockEnd = updateProps.actualPosition + this.getNumRows();
      break;

    case this.bottomButtonConf.id:
      commandName = "bottom";
      commandValue = Client.IdfPanel.commands.CMD_NAVIGATION;
      updateProps.actualRow = 0;
      updateProps.actualPosition = this.totalRows;
      //
      this.dataBlockStart = this.totalRows - this.getNumRows();
      this.dataBlockEnd = this.totalRows;
      break;

    default:
      for (let i = 0; i < this.customButtonsConf.length; i++) {
        if (event.obj === this.customButtonsConf[i].id) {
          commandName = "cb" + i;
          commandValue = Client.IdfPanel.commands["CMD_CUSTOM" + (i + 1)];
          break;
        }
      }
      break;
  }
  //
  let events = [];
  let isBlocking;
  let eventDef;
  if (Client.mainFrame.isIDF) {
    isBlocking = (this.blockingCommands & commandValue);
    eventDef = this.toolbarEventDef | (isBlocking ? Client.IdfMessagesPump.eventTypes.BLOCKING : 0);
    //
    events.push({
      id: "pantb",
      def: eventDef,
      content: {
        oid: this.id,
        obn: commandName
      }
    });
  }
  else {
    events.push({
      id: "fireOnCommand",
      obj: this.id,
      content: {command: commandName}
    });
  }
  //
  // If I'm on IDF, update panel's properties just if toolbar events have to be handled client side too
  if (Object.keys(updateProps).length > 0 && (!Client.mainFrame.isIDF || (Client.IdfMessagesPump.isClientSideEvent(eventDef) && !isBlocking)))
    this.updateElement(updateProps);
  //
  // If no confirm is required, return event
  if (!msg) {
    if (updateMultiSel)
      this.updateMultiSel();
    //
    return events;
  }
  //
  // Show confirm message and eventually send event on user confirm
  Client.Widget.showMessageBox({type, text: msg}, result => {
    // TODO: this.DoHighlightDelete(false);
    if (result === "Y") {
      if (updateMultiSel)
        this.updateMultiSel();
      //
      Client.mainFrame.sendEvents(events);
    }
  });
  //
  return [];
};


/**
 * Handle function keys
 * @param {Object} event
 */
Client.IdfPanel.prototype.handleFunctionKeys = function (event)
{
  let events = [];
  if (event.content.type !== "keydown")
    return events;
  //
  // I check the commands related to the fields
  for (let i = 0; i < this.elements.length && events.length === 0; i++) {
    let el = this.elements[i];
    if (el instanceof Client.IdfField)
      events.push(...el.handleFunctionKeys(event));
  }
  //
  if (events.length > 0)
    return events;
  //
  events.push(...Client.IdfFrame.prototype.handleFunctionKeys.call(this, event));
  if (events.length > 0)
    return events;
  //
  // Calculate the number of FKs from 1 to 48
  let fkn = (event.content.keyCode - 111) + (event.content.shiftKey ? 12 : 0) + (event.content.ctrlKey ? 24 : 0);
  //
  // Let's see if it matches one of my default keys
  let button, value, enabled;
  switch (fkn) {
    case Client.mainFrame.wep?.FKEnterQBE:
      button = this.searchButtonConf;
      value = Client.IdfPanel.commands.CMD_SEARCH;
      break;

    case Client.mainFrame.wep?.FKFindData:
      button = this.findButtonConf;
      value = Client.IdfPanel.commands.CMD_FIND;
      break;

    case Client.mainFrame.wep?.FKFormList:
      button = this.formListButtonConf;
      value = Client.IdfPanel.commands.CMD_FORMLIST;
      break;

    case Client.mainFrame.wep?.FKRefresh:
      button = this.refreshButtonConf;
      value = Client.IdfPanel.commands.CMD_REFRESH;
      break;

    case Client.mainFrame.wep?.FKCancel:
      button = this.cancelButtonConf;
      value = Client.IdfPanel.commands.CMD_CANCEL;
      break;

    case Client.mainFrame.wep?.FKInsert:
      button = this.insertButtonConf;
      value = Client.IdfPanel.commands.CMD_INSERT;
      break;

    case Client.mainFrame.wep?.FKDelete:
      button = this.deleteButtonConf;
      value = Client.IdfPanel.commands.CMD_DELETE;
      break;

    case Client.mainFrame.wep?.FKUpdate:
      button = this.saveButtonConf;
      value = Client.IdfPanel.commands.CMD_SAVE;
      break;

    case Client.mainFrame.wep?.FKDuplicate:
      button = this.duplicateButtonConf;
      value = Client.IdfPanel.commands.CMD_DUPLICATE;
      break;

    case Client.mainFrame.wep?.FKPrint:
      button = this.printButtonConf;
      value = Client.IdfPanel.commands.CMD_PRINT;
      break;

    case Client.mainFrame.wep?.FKLocked:
      button = this.lockButtonConf;
      enabled = this.lockable;
      break;

    default:
    for (let i = 0; i < this.customCommands.length; i++) {
      if (Client.eleMap[this.customButtonsConf[i].id].fknum === fkn) {
        button = this.customButtonsConf[i];
        value = Client.IdfPanel.commands["CMD_CUSTOM" + (i + 1)];
        break;
      }
    }
  }
  //
  if (value)
    enabled = this.isCommandEnabled(value);
  if (button)
    events.push(...this.handleToolbarClick({obj: button.id}));
  //
  if (this.enableMultiSelection) {
    let res;
    if (this.showMultiSelection) {
      if (fkn === Client.mainFrame.wep?.FKSelAll)
        res = "selall";
      else if (fkn === Client.mainFrame.wep?.FKSelNone)
        res = "selnone";
    }
    else if (fkn === Client.mainFrame.wep?.FKSelTog)
      res = "seltog";
    //
    if (res)
      events.push(...this.handleMultiSelClick({content: {res}}));
  }
  //
  if (fkn === Client.mainFrame.wep?.FKActRow)
    events.push(...this.handleDataRowClick({id: "onDblclick"}, this.fields[0].getValueByIndex(this.getActiveRowIndex())));
  //
  if (events.length > 0)
    event.content.srcEvent.preventDefault();
  //
  return events;
};


/**
 * Handle click on lock button
 * @param {Object} event
 */
Client.IdfPanel.prototype.handlePageClick = function (event)
{
  let events = [];
  let pageIndex = event.page.index;
  //
  // If click occurred on a page that is already active, do nothing
  if (pageIndex === this.activePage)
    return events;
  //
  // Give event the IDF format
  if (Client.mainFrame.isIDF) {
    events.push({
      id: "panpg",
      def: this.pageClickEventDef,
      content: {
        oid: event.page.id,
        obn: this.id,
        xck: event.content.offsetX,
        yck: event.content.offsetY
      }
    });
  }
  else { // On IDC send onChangePage event
    events.push({
      id: "chgProp",
      obj: this.id,
      content: {
        name: "page",
        value: pageIndex,
        clid: Client.id
      }
    });
    //
    events.push({
      id: "onChangePage",
      obj: this.id,
      content: {
        oldPage: this.activePage,
        newPage: pageIndex
      }
    });
  }
  //
  // If I'm on IDF, change active page just if page click events have to be handled client side too
  if (!Client.mainFrame.isIDF || Client.IdfMessagesPump.isClientSideEvent(this.pageClickEventDef)) {
    this.updateElement({activePage: pageIndex});
    //
    Client.mainFrame.messagesPump.sendEvents(true);
  }
  //
  return events;
};


/**
 * Apply visual style
 */
Client.IdfPanel.prototype.applyVisualStyle = function ()
{
  // Set visual style on panel container
  this.addVisualStyleClasses(Client.eleMap[this.contentContainerConf.id], {objType: "panel"});
  //
  if (this.hasList) {
    // Set panel visual style on aggregate row too
    this.addVisualStyleClasses(Client.eleMap[this.aggregateRowConf.id], {objType: "panel"});
    //
    // Set visual style on row selector header
    this.addVisualStyleClasses(Client.eleMap[this.rowSelectorColumnConf.id], {objType: "fieldHeader", list: true});
  }
  //
  // Apply visual style on fields
  for (let i = 0; i < this.fields.length; i++)
    this.fields[i].applyVisualStyle();
  //
  // Apply visual style on groups
  for (let i = 0; i < this.groups.length; i++)
    this.groups[i].applyVisualStyle();
};


/**
 * Get index-th row
 * @param {Integer} index
 */
Client.IdfPanel.prototype.getRow = function (index)
{
  return Client.eleMap[this.rows[index]?.id];
};


/**
 * Get id of next existing row after index-th row
 * @param {Integer} index
 */
Client.IdfPanel.prototype.getNextRowId = function (index)
{
  let nextRowId;
  //
  let rowsCount = this.getMaxRows();
  for (let i = index + 1; i <= rowsCount; i++) {
    let currentRowId = this.rows[i]?.id;
    //
    // If current row exists and it's not detached, it is the row I was finding
    if (currentRowId && !this.isRowDetached(i)) {
      nextRowId = currentRowId;
      break;
    }
  }
  //
  return nextRowId || this.aggregateRowConf.id;
};


/**
 * Check if index-th row is detached
 * @param {Integer} index
 */
Client.IdfPanel.prototype.isRowDetached = function (index)
{
  return this.detachedRows[this.rows[index]?.id];
};


/**
 * Attach index-th row
 * @param {Integer} index
 */
Client.IdfPanel.prototype.attachRow = function (index)
{
  // If in current position there is a data row or placeholder row that is not detached, I don't need to add any row. So continue
  if (this.getRow(index) && !this.isRowDetached(index))
    return;
  //
  let detachedRow = this.isRowDetached(index);
  let groupHeaderRow;
  let newRow;
  //
  // Since current position is free, try to attach a row. It can be a detached row, a group header row or a new placeholder row
  if (detachedRow)
    newRow = detachedRow;
  else {
    if (this.hasGroupedRows()) {
      let rowsGroup = this.getRowsGroupByIndex(index);
      groupHeaderRow = rowsGroup ? rowsGroup.createHeaderRowConf(index) : undefined;
    }
    //
    newRow = groupHeaderRow || this.createPlaceholderRowConf(index);
  }
  //
  // If there isn't a row to attach in current position, continue
  if (!newRow)
    return;
  //
  let grid = Client.eleMap[this.gridConf.id];
  //
  let nextRow = Client.eleMap[this.getNextRowId(index)];
  //
  // If row to attach is a detached one, I have to insert it directly on DOM
  if (detachedRow) {
    grid.getRootObject().insertBefore(newRow, nextRow.getRootObject());
    //
    // First or last row could be changed, so check if I need to move margin top and bottom
    this.moveScrollMargins(Client.eleMap[newRow.id]);
    //
    // Now it's no more detached
    delete this.detachedRows[detachedRow.id];
  }
  else { // Otherwise create new placeholder
    newRow = grid.insertBefore({child: newRow, sib: nextRow.id});
    this.rows[index] = {id: newRow.id};
    if (groupHeaderRow)
      this.rows[index].isGroupHeader = true;
    else
      this.rows[index].isPlaceholder = true;
    //
    // First or last row could be changed, so check if I need to move margin top and bottom
    this.moveScrollMargins(newRow);
  }
};


/**
 * Detach index-th row
 * @param {Integer} index
 * @param {Boolean} deleted
 */
Client.IdfPanel.prototype.detachRow = function (index, deleted)
{
  let rowEl = this.getRow(index);
  //
  // If row does not exist or it's already detached, do nothing
  if (!rowEl)
    return;
  //
  if (!this.isRowDetached(index)) {
    // First or last row may be about to change, so check if I need to move margin top and bottom
    this.moveScrollMargins(rowEl, true);
    //
    let rootObject = rowEl.getRootObject();
    //
    // Remove current row from dom
    rootObject.parentNode.removeChild(rootObject);
    //
    this.detachedRows[rowEl.id] = rootObject;
  }
  //
  // If current row is deleted, remove it from rows
  if (deleted) {
    delete this.rows[index];
    delete this.detachedRows[rowEl.id];
  }
};


/**
 * Move margin top and margin bottom to the right row, after inserting a new row or before removing a row
 * @param {Object} row
 * @param {Boolean} removing
 */
Client.IdfPanel.prototype.moveScrollMargins = function (row, removing)
{
  let rootObject = row.getRootObject();
  let nextRow = rootObject.nextSibling;
  let previousRow = rootObject.previousSibling;
  let marginTop;
  let marginBottom;
  //
  if (removing) {
    // If I'm going to remove a row having margin bottom, I have to avoid grid scrollbar from moving.
    // So assign margin bottom to previous row
    marginBottom = rootObject.style.marginBottom;
    if (marginBottom && previousRow && previousRow.id !== this.gridHeaderConf.id) {
      rootObject.style.marginBottom = "";
      previousRow.style.marginBottom = marginBottom;
    }
    //
    // If I'm going to remove a row having margin top, I have to avoid grid scrollbar from moving.
    // So assign margin top to next row
    marginTop = rootObject.style.marginTop;
    if (marginTop && nextRow && nextRow.id !== this.aggregateRowConf.id) {
      rootObject.style.marginTop = "";
      nextRow.style.marginTop = marginTop;
    }
  }
  else {
    // If I just added a row, check if previous row has margin bottom.
    // Since now the last row is the current one, remove margin bottom from previous row and assign it to the current one
    marginBottom = previousRow?.style.marginBottom;
    if (marginBottom) {
      previousRow.style.marginBottom = "";
      rootObject.style.marginBottom = marginBottom;
    }
    //
    // If I just added a row, check if next row has margin top.
    // Since now the first row is the current one, remove margin top from next row and assign it to the current one
    marginTop = nextRow?.style.marginTop;
    if (marginTop) {
      nextRow.style.marginTop = "";
      rootObject.style.marginTop = marginTop;
    }
  }
};


/**
 * Update grouped rows
 */
Client.IdfPanel.prototype.updateGroupedRows = function ()
{
  Client.IdfRowsGroup.setIndex(this.groupedRowsRoot, 0);
  //
  // Existing rows have an index that doesn't take in account groups. So convert it to grouped format
  let newRows = [];
  for (let i = 0; i < this.rows.length; i++) {
    let row = this.rows[i];
    if (!row)
      continue;
    //
    let groupedIndex = this.groupedRowsRoot.realIndexToGroupedIndex(i);
    newRows[groupedIndex] = row;
  }
  //
  // Replace rows array with its grouped version
  this.rows = newRows.slice();
  //
  this.groupedRowsRoot.handleExpansion();
};
/**
 * Update data rows
 */
Client.IdfPanel.prototype.updateDataRows = function ()
{
  if (!this.hasList)
    return;
  //
  let dataBlockStart = this.getDataBlockStart();
  let dataBlockEnd = this.getDataBlockEnd();
  //
  for (let i = dataBlockStart; i <= dataBlockEnd; i++) {
    let index = this.hasGroupedRows() ? this.groupedRowsRoot.realIndexToGroupedIndex(i) : i;
    //
    let currentRow = this.rows[index];
    //
    // I'm updating data rows, so skip groups headers
    if (currentRow?.isGroupHeader)
      continue;
    //
    // If current data row already exists, eventually fill missing columns
    if (currentRow?.isData)
      this.createDataRowConfig(index);
    else { // Otherwise create new data row
      let grid = Client.eleMap[this.gridConf.id];
      //
      // If there was a placeholder row at current position, remove it because the real data row is here
      if (currentRow?.isPlaceholder)
        this.detachRow(index, true);
      //
      // Create row element before next row or before the aggregate one if there isn't a next data row
      let dataRow = grid.insertBefore({child: this.createDataRowConfig(index), sib: this.getNextRowId(index)});
      //
      // Save new data row id
      this.rows[index] = {id: dataRow.id, isData: true};
      //
      if (this.hasGroupedRows() && !this.groupedRowsRoot.isRowVisible(index))
        this.detachRow(index);
      else // First or last row could be changed, so check if I need to move margin top and bottom
        this.moveScrollMargins(dataRow);
    }
  }
  //
  // Since I've just create new data rows, tell values to update their controls with all properties
  for (let i = 0; i < this.fields.length; i++) {
    let f = this.fields[i];
    if (!f.isInList())
      continue;
    //
    f.updateControls({all: true});
  }
};


/**
 * Since server sends me a page of data on scrolling, there may be some empty spaces between one data row and the next.
 * So I apply a margin top to first data row of each data block, making room for missing data row (if any)
 * @param {Integer} start
 * @param {Integer} end
 */
Client.IdfPanel.prototype.adjustScrollbar = function (start, end)
{
  if (!this.hasList)
    return;
  //
  // Reset rows margins
  this.resetRowsMargins();
  //
  // Apply margin top to first data row
  let firstRow = this.getFirstGridRow();
  let marginTop = this.getRowMarginTop(firstRow);
  marginTop = marginTop ? marginTop + "px" : "";
  firstRow?.updateElement({style: {marginTop}});
  //
  // Apply margin bottom to last data row
  let lastRow = this.getLastGridRow();
  let marginBottom = this.getRowMarginBottom(lastRow);
  marginBottom = marginBottom ? marginBottom + "px" : "";
  lastRow?.updateElement({style: {marginBottom}});
};


/**
 * Scroll grid to index-th row
 * @param {Integer} index
 */
Client.IdfPanel.prototype.scrollToDataRow = function (index)
{
  let rowHeight = this.getListRowHeight();
  let rowOffset = this.getListRowOffset();
  let scrollTop = 0;
  //
  for (let i = 1; i < index; i++) {
    scrollTop += (this.getRow(i)?.getRootObject().clientHeight || rowHeight);
    scrollTop += rowOffset;
  }
  //
  if (scrollTop < 0)
    scrollTop = 0;
  //
  // Apply scrollTop after a while
  setTimeout(() => {
    let grid = Client.eleMap[this.gridConf.id].getRootObject();
    if (grid.scrollTop !== scrollTop) {
      // I don't want to send "panscr" event if scroll is not performed by user
      this.skipPanScr = true;
      grid.scrollTop = scrollTop;
    }
  }, 0);
};


/**
 * Handle multiple selection of data row
 * @param {Boolean} value
 * @param {Integer} index
 */
Client.IdfPanel.prototype.handleDataRowSelection = function (value, index)
{
  let events = [];
  //
  // If I'm on IDF, select row just if multi selection event has to be handled client side too
  if (!Client.mainFrame.isIDF || Client.IdfMessagesPump.isClientSideEvent(this.multiSelEventDef))
    this.updateMultiSel({value, index});
  //
  if (Client.mainFrame.isIDF) {
    events.push({
      id: "panms",
      def: this.multiSelEventDef,
      content: {
        oid: this.id,
        obn: index - 1, // I have to send index 0 based!
        par1: value ? "-1" : "0"
      }
    });
  }
  else {
    // TODO
  }
  //
  return events;
};


/**
 * Get selected data rows count
 */
Client.IdfPanel.prototype.getSelectedDataRows = function ()
{
  let selectedRows = 0;
  this.multiSelStatus.forEach(status => selectedRows += (status ? 1 : 0));
  //
  return selectedRows;
};


/**
 * Get list field parent column configuration
 * @param {String} fieldId
 * @param {Boolean} aggregate - true if I have to get aggregate container of given field
 */
Client.IdfPanel.prototype.getListFieldColumn = function (fieldId, aggregate)
{
  // fieldId could be also the panel id. In this case it means I want to find the column for list grid
  let field = Client.eleMap[fieldId];
  let inList = field.id !== this.id ? field.isInList() : false;
  //
  if (inList) {
    let parentGroup = Client.eleMap[field.groupId];
    if (aggregate)
      return {conf: parentGroup ? parentGroup.aggregateRowConf : this.aggregateRowConf};
    else
      return {conf: parentGroup ? parentGroup.listContentConf : this.gridHeaderConf};
  }
  else if (field instanceof Client.IdfField && field.aggregateOfField !== -1) {
    let parentField = this.fields.find(f => f.index === field.aggregateOfField);
    return {conf: parentField.aggregateContainerConf};
  }
  //
  for (let i = 0; i < this.listGridRows.length; i++) {
    let row = this.listGridRows[i];
    for (let j = 0; j < row.cols.length; j++) {
      let fieldCol = this.recursivelyFindFieldColumn(field.id, row.cols[j]);
      //
      if (fieldCol)
        return fieldCol;
    }
  }
};


/**
 * Get form field parent column configuration
 * @param {String} fieldId
 */
Client.IdfPanel.prototype.getFormFieldColumn = function (fieldId)
{
  for (let i = 0; i < this.formGridRows.length; i++) {
    let row = this.formGridRows[i];
    for (let j = 0; j < row.cols.length; j++) {
      let fieldCol = this.recursivelyFindFieldColumn(fieldId, row.cols[j]);
      //
      if (fieldCol)
        return fieldCol;
    }
  }
};


/**
 * Find field column configuration
 * @param {String} fieldId
 * @param {Object} col
 */
Client.IdfPanel.prototype.recursivelyFindFieldColumn = function (fieldId, col)
{
  if (col.fields.length === 1 && col.fields[0].id === fieldId && (!col.rows.length || col.fields[0].isGroup))
    return col;
  //
  for (let i = 0; i < col.rows.length; i++) {
    let row = col.rows[i];
    for (let j = 0; j < row.cols.length; j++) {
      let fieldCol = this.recursivelyFindFieldColumn(fieldId, row.cols[j]);
      //
      if (fieldCol)
        return fieldCol;
    }
  }
};


/**
 * Reset edge column properties (isMostLeft, isMostRight, isMostTop, isMostBottom)
 * @param {Object} col
 */
Client.IdfPanel.prototype.recursivelyResetEdgeColumn = function (col)
{
  if (!col)
    return;
  //
  col.isMostLeft = false;
  col.isMostRight = false;
  col.isMostTop = false;
  col.isMostBottom = false;
  //
  for (let i = 0; i < col.rows.length; i++) {
    let row = col.rows[i];
    //
    for (let j = 0; j < row.cols.length; j++)
      this.recursivelyResetEdgeColumn(row.cols[j]);
  }
};


/**
 * Find and mark most right columns
 * @param {Object} col
 */
Client.IdfPanel.prototype.recursivelySetMostRightColumn = function (col)
{
  if (!col)
    return;
  //
  if (!col.rows.length || (col.fields[0] && col.fields[0].isGroup)) {
    col.isMostRight = true;
    return;
  }
  //
  for (let i = 0; i < col.rows.length; i++) {
    let row = col.rows[i];
    //
    // Get last visible column
    let lastVisibleCol = row.cols[row.cols.length - 1];
    for (let j = row.cols.length - 1; j >= 0; j--) {
      if (row.cols[j].visible) {
        lastVisibleCol = row.cols[j];
        break;
      }
    }
    //
    this.recursivelySetMostRightColumn(lastVisibleCol);
  }
};


/**
 * Find and mark most left columns
 * @param {Object} col
 */
Client.IdfPanel.prototype.recursivelySetMostLeftColumn = function (col)
{
  if (!col)
    return;
  //
  if (!col.rows.length || (col.fields[0] && col.fields[0].isGroup)) {
    col.isMostLeft = true;
    return;
  }
  //
  for (let i = 0; i < col.rows.length; i++) {
    let row = col.rows[i];
    this.recursivelySetMostLeftColumn(row.cols[0]);
  }
};


/**
 * Find and mark most bottom columns
 * @param {Object} col
 */
Client.IdfPanel.prototype.recursivelySetMostBottomColumn = function (col)
{
  if (!col)
    return;
  //
  if (!col.rows.length || (col.fields[0] && col.fields[0].isGroup)) {
    col.isMostBottom = true;
    return;
  }
  //
  // Get last row having at least a visible column
  let lastVisibleRow = col.rows[col.rows.length - 1];
  let found;
  for (let i = col.rows.length - 1; i >= 0; i--) {
    let colRow = col.rows[i];
    for (let j = 0; j < colRow.cols.length; j++) {
      if (colRow.cols[j].visible) {
        lastVisibleRow = colRow;
        found = true;
        break;
      }
    }
    //
    if (found)
      break;
  }
  //
  for (let i = 0; i < lastVisibleRow.cols.length; i++)
    this.recursivelySetMostBottomColumn(lastVisibleRow.cols[i]);
};


/**
 * Find and mark most top columns
 * @param {Object} col
 */
Client.IdfPanel.prototype.recursivelySetMostTopColumn = function (col)
{
  if (!col)
    return;
  //
  if (!col.rows.length || (col.fields[0] && col.fields[0].isGroup)) {
    col.isMostTop = true;
    return;
  }
  //
  let firstRow = col.rows[0];
  for (let i = 0; i < firstRow.cols.length; i++)
    this.recursivelySetMostTopColumn(firstRow.cols[i]);
};


/**
 * Recursively set flex grow on non-leaf columns
 * @param {Object} col
 */
Client.IdfPanel.prototype.recursivelySetColumnsGrow = function (col)
{
  // If col is a leaf column, return its width adaptable status
  if (!col.rows.length)
    return col.canAdaptWidth && col.visible;
  //
  // A non-leaf column has to grow if one of its children grows
  let canAdaptWidth = false;
  for (let i = 0; i < col.rows.length; i++) {
    let row = col.rows[i];
    //
    for (let j = 0; j < row.cols.length; j++) {
      canAdaptWidth = this.recursivelySetColumnsGrow(row.cols[j]);
      if (canAdaptWidth)
        break;
    }
    //
    if (canAdaptWidth)
      break;
  }
  //
  // Set flex grow on given column
  let el = Client.eleMap[col.conf.id];
  el.updateElement({style: {flexGrow: canAdaptWidth ? "1" : "0"}});
  //
  return canAdaptWidth;
};


/**
 * Recursively set flex grow on rows
 * @param {Object} row
 */
Client.IdfPanel.prototype.recursivelySetRowsGrow = function (row)
{
  let canAdaptHeight = false;
  for (let i = 0; i < row.cols.length; i++) {
    let col = row.cols[i];
    //
    // If a leaf column grows in height, parent row has to grow in height too
    if (!col.rows.length && col.canAdaptHeight && col.visible)
      canAdaptHeight = true;
    else { // Otherwise recursively check if some column's row has to grow in height
      for (let j = 0; j < col.rows.length; j++) {
        let hasToGrow = this.recursivelySetRowsGrow(col.rows[j]);
        if (hasToGrow)
          canAdaptHeight = true;
      }
    }
  }
  //
  // Set flex grow on given row
  let el = Client.eleMap[row.conf.id];
  el.updateElement({style: {flexGrow: canAdaptHeight ? "1" : "0"}});
  //
  return canAdaptHeight;
};


/**
 * Recursively set columns visible property
 * @param {Object} col
 */
Client.IdfPanel.prototype.recursivelySetColumnsVisible = function (col)
{
  // If col is a leaf column, return its visibility
  if (!col.rows.length)
    return col.visible;
  //
  // A non-leaf column is visible if one of its children is
  let visible = false;
  for (let i = 0; i < col.rows.length; i++) {
    let row = col.rows[i];
    //
    for (let j = 0; j < row.cols.length; j++) {
      visible = this.recursivelySetColumnsVisible(row.cols[j]);
      if (visible)
        break;
    }
    //
    if (visible)
      break;
  }
  //
  // Set visible property
  col.visible = visible;
};


/**
 * Calculate layout rules to handle resize mode
 * @param {Integer} index
 */
Client.IdfPanel.prototype.calcLayout = function (index)
{
  Client.IdfFrame.prototype.calcLayout.call(this);
  //
  // Tell my groups to update their visibility
  for (let i = 0; i < this.groups.length; i++)
    this.groups[i].updateVisibility(index);
  //
  // Tell my fields to update their visibility
  for (let i = 0; i < this.fields.length; i++)
    this.fields[i].updateVisibility(index);
  //
  // Set visible property on all parent columns with the same value as leaf column visible property
  this.setColumnsVisible();
  //
  // Set edge columns because they may have changed.
  // For example, a most right column that becomes not visible is no more the "most right" column.
  // The new most right column for that row will be previous column (if any)
  this.setEdgeColumns();
  this.setEdgeColumns(true);
  //
  // If I have a list, calculate grid layout
  if (this.hasList) {
    // Get row selector width
    let rowSelectorWidth = this.showRowSelector ? Client.eleMap[this.rowSelectorColumnConf.id].getRootObject().offsetWidth : 0;
    //
    // Get grid width
    this.gridWidth = this.getGridWidth();
    //
    let gridStyle = {};
    let gridColStyle = {};
    let rowSelectorStyle = {height: this.getHeaderHeight() + "px"};
    //
    gridColStyle.padding = "0px";
    gridColStyle.flexBasis = (this.gridWidth + rowSelectorWidth) + "px";
    gridColStyle.flexGrow = this.canAdaptWidth() ? "1" : "0";
    //
    let gridHeight = this.gridHeight;
    //
    // If panel hasn't dynamic height rows I don't want to see an half of a row at grid bottom.
    // So grid column height has to be a multiple of row height
    if (!this.hasDynamicHeightRows()) {
      // Get header height
      let headerHeight = this.getHeaderHeight();
      //
      // Get qbe row height
      let qbeRowHeight = this.getQbeRowHeight();
      //
      // Get row offset
      let rowOffset = this.getListRowOffset();
      //
      // Get row height
      let rowHeight = this.getListRowHeight();
      //
      // Get height of visible rows block
      let visibleRowsHeight = gridHeight - headerHeight - qbeRowHeight;
      //
      // Calculate how many rows can be contained into visible rows block
      let numRows = parseInt(visibleRowsHeight / (rowHeight + rowOffset));
      //
      // Adjust grid height in order to contain the header and the rows each with its row offset (the first row has no offset)
      gridHeight = headerHeight + qbeRowHeight + (rowHeight * numRows) + (rowOffset * (numRows - 1));
    }
    //
    gridColStyle.height = this.canAdaptHeight() ? "auto" : gridHeight + "px";
    gridStyle.height = gridHeight + "px";
    //
    // Get grid parent column
    let gridColumn = this.getListFieldColumn(this.id);
    //
    // Tell grid column if it can adapt its width and height
    gridColumn.canAdaptWidth = this.canAdaptWidth();
    gridColumn.canAdaptHeight = this.canAdaptHeight();
    //
    // Calculate margin left
    let gridColumnLeft = gridColumn.rect.left || 0;
    let deltaLeft = this.gridLeft - gridColumnLeft;
    gridColStyle.marginLeft = gridColumn.isMostLeft ? this.gridLeft + "px" : deltaLeft + "px";
    //
    // Calculate margin right
    let deltaRight = gridColumn.isMostRight ? (this.originalWidth - this.gridWidth - rowSelectorWidth - this.gridLeft) : gridColumn.rect.deltaRight;
    gridColStyle.marginRight = deltaRight + "px";
    //
    // Calculate margin top
    let gridColumnTop = gridColumn.rect.top || 0;
    let deltaTop = this.gridTop - gridColumnTop;
    gridColStyle.marginTop = gridColumn.isMostTop ? this.gridTop + "px" : deltaTop + "px";
    //
    // Get panel toolbar height
    let toolbarHeight = this.getToolbarHeight();
    //
    // Calculate margin bottom
    let deltaBottom = gridColumn.isMostBottom ? (this.originalHeight - this.gridHeight - this.gridTop - toolbarHeight) : gridColumn.rect.deltaBottom;
    gridColStyle.marginBottom = deltaBottom >= 0 ? deltaBottom + "px" : "0px";
    //
    // Update grid column style
    Client.eleMap[this.gridColConf.id].updateElement({style: gridColStyle, xs: "auto"});
    //
    // Update grid style
    Client.eleMap[this.gridConf.id].updateElement({style: gridStyle});
    //
    rowSelectorStyle.left = this.fixedColumns > 0 ? 0 : undefined;
    //
    // Update row selector style
    let el = Client.eleMap[this.rowSelectorColumnConf.id];
    el.updateElement({style: rowSelectorStyle});
    Client.Widget.updateElementClassName(el, "fixed-col", !this.fixedColumns);
    //
    // Update aggregate row selector style
    el = Client.eleMap[this.aggregateRowSelectorConf.id];
    el.updateElement({style: rowSelectorStyle});
    Client.Widget.updateElementClassName(el, "fixed-col", !this.fixedColumns);
  }
  //
  // Tell my groups to calculate their layout
  for (let i = 0; i < this.groups.length; i++)
    this.groups[i].calcLayout();
  //
  // Tell my fields to calculate their layout
  for (let i = 0; i < this.fields.length; i++)
    this.fields[i].calcLayout(index);
  //
  // Set rows min width in order to handle fixed columns properly
  if (this.hasList && this.fixedColumns)
    this.setRowsMinWidth();
  //
  // To properly handle resize, I have to set flex-grow on structure rows and columns.
  // In fact, if a leaf column has to adapt its width, all its parent columns have to grow horizontally.
  // And if a leaf column has to adapt its height, all its parent rows have to grow vertically.
  this.setFlexGrow();
  //
  // Send eventual new dimensions to server
  Client.mainFrame.sendEvents(this.handleResize());
};


/**
 * Set flex grow on rows and columns in order to properly handle resize
 */
Client.IdfPanel.prototype.setFlexGrow = function ()
{
  if (this.hasList) {
    for (let i = 0; i < this.listGridRows.length; i++) {
      let listRow = this.listGridRows[i];
      //
      // Set flex-grow on rows in order to handle vertical resize
      this.recursivelySetRowsGrow(listRow);
      //
      // Set flex-grow on columns in order to handle horizontal resize
      for (let j = 0; j < listRow.cols.length; j++)
        this.recursivelySetColumnsGrow(listRow.cols[j]);
    }
  }
  //
  if (this.hasForm) {
    for (let i = 0; i < this.formGridRows.length; i++) {
      let formRow = this.formGridRows[i];
      //
      // Set flex-grow on rows in order to handle vertical resize
      this.recursivelySetRowsGrow(formRow);
      //
      // Set flex-grow on columns in order to handle horizontal resize
      for (let j = 0; j < formRow.cols.length; j++)
        this.recursivelySetColumnsGrow(formRow.cols[j]);
    }
  }
};


/**
 * Set visible property on columns in order to properly handle resize
 */
Client.IdfPanel.prototype.setColumnsVisible = function ()
{
  if (this.hasList) {
    for (let i = 0; i < this.listGridRows.length; i++) {
      let listRow = this.listGridRows[i];
      //
      for (let j = 0; j < listRow.cols.length; j++)
        this.recursivelySetColumnsVisible(listRow.cols[j]);
    }
  }
  //
  if (this.hasForm) {
    for (let i = 0; i < this.formGridRows.length; i++) {
      let formRow = this.formGridRows[i];
      //
      for (let j = 0; j < formRow.cols.length; j++)
        this.recursivelySetColumnsVisible(formRow.cols[j]);
    }
  }
};


/**
 * Set edge columns in order to make calcLayout method to apply proper margins
 * @param {Boolean} form
 * @param {Client.IdfGroup} group
 */
Client.IdfPanel.prototype.setEdgeColumns = function (form, group)
{
  if (form && !this.hasForm)
    return;
  //
  if (!form && !this.hasList)
    return;
  //
  let rows;
  if (form)
    rows = group ? this.getFormFieldColumn(group.id).rows : this.formGridRows;
  else
    rows = group ? this.getListFieldColumn(group.id).rows : this.listGridRows;
  //
  // Get last row having at least a visible column
  let lastVisibleRowIndex = rows.length - 1;
  for (let i = rows.length - 1; i >= 0; i--) {
    let found = false;
    for (let j = 0; j < rows[i].cols.length; j++) {
      if (rows[i].cols[j].visible) {
        lastVisibleRowIndex = i;
        found = true;
        break;
      }
    }
    //
    if (found)
      break;
  }
  //
  for (let i = 0; i < rows.length; i++) {
    // First of all, reset all columns edge properties (isMostLeft, isMostRight, isMostTop, isMostBottom)
    for (let j = 0; j < rows[i].cols.length; j++)
      this.recursivelyResetEdgeColumn(rows[i].cols[j]);
    //
    // Find and mark most left column. It must be in the first col of actual row
    let firstRowCol = rows[i].cols[0];
    this.recursivelySetMostLeftColumn(firstRowCol);
    //
    // Find and mark most right column. It must be in the last visible col of actual row
    let lastVisibleRowCol = rows[i].cols[rows[i].cols.length - 1];
    for (let j = rows[i].cols.length - 1; j >= 0; j--) {
      if (rows[i].cols[j].visible) {
        lastVisibleRowCol = rows[i].cols[j];
        break;
      }
    }
    this.recursivelySetMostRightColumn(lastVisibleRowCol);
    //
    // Find and mark most top and most bottom columns.
    for (let j = 0; j < rows[i].cols.length; j++) {
      // Most top column must be in the first row
      if (i === 0)
        this.recursivelySetMostTopColumn(rows[i].cols[j]);
      //
      // Most bottom column must be in the last row having at least one visible column
      if (i === lastVisibleRowIndex)
        this.recursivelySetMostBottomColumn(rows[i].cols[j]);
    }
  }
};


/**
 * Check if panel grid can adapt its width
 */
Client.IdfPanel.prototype.canAdaptWidth = function ()
{
  let adaptableFields;
  //
  // Check if at least one of visible in list fields is adaptable
  for (let i = 0; i < this.fields.length; i++) {
    let field = this.fields[i];
    if (!field.isInList())
      continue;
    //
    if (field.canAdaptWidth() && field.isVisible()) {
      adaptableFields = true;
      break;
    }
  }
  //
  return this.parentIdfView?.resizeWidth !== Client.IdfView.resizeModes.NONE && this.resizeWidth === Client.IdfPanel.resizeModes.stretch && adaptableFields;
};


/**
 * Check if panel grid can apdapt its height
 */
Client.IdfPanel.prototype.canAdaptHeight = function ()
{
  return this.parentIdfView?.resizeHeight !== Client.IdfView.resizeModes.NONE && this.resizeHeight === Client.IdfPanel.resizeModes.stretch;
};


/**
 * Check if panel grid can move left
 */
Client.IdfPanel.prototype.canMoveLeft = function ()
{
  return this.parentIdfView?.resizeWidth !== Client.IdfView.resizeModes.NONE && this.resizeWidth === Client.IdfPanel.resizeModes.move;
};


/**
 * Check if panel grid can move top
 */
Client.IdfPanel.prototype.canMoveTop = function ()
{
  return this.parentIdfView?.resizeHeight !== Client.IdfView.resizeModes.NONE && this.resizeHeight === Client.IdfPanel.resizeModes.move;
};


/**
 * Show or hide row selectors
 */
Client.IdfPanel.prototype.updateRowSelectorsVisibility = function ()
{
  if (!this.hasList)
    return;
  //
  // Update header row selector visibility
  Client.eleMap[this.rowSelectorColumnConf.id].updateElement({visible: this.showRowSelector});
  //
  // Update aggregate row selector visibility
  Client.eleMap[this.aggregateRowSelectorConf.id].updateElement({visible: this.showRowSelector});
  //
  for (let i = 0; i < this.fields.length; i++) {
    let field = this.fields[i];
    if (!field.isInList())
      continue;
    //
    field.updateRowSelectorsVisibility(this.showRowSelector);
  }
};


/**
 * Get start position of last received data block
 */
Client.IdfPanel.prototype.getDataBlockStart = function ()
{
  return this.dataBlockStart || (this.canUseRowQbe() ? 0 : 1);
};


/**
 * Get end position of last received data block
 */
Client.IdfPanel.prototype.getDataBlockEnd = function ()
{
  return this.dataBlockEnd || this.totalRows;
};


/**
 * Calculate grid width as fields width sum
 */
Client.IdfPanel.prototype.getGridWidth = function ()
{
  let gridWidth = 0;
  //
  // Add in list fields width
  for (let i = 0; i < this.fields.length; i++) {
    let field = this.fields[i];
    if (!field.isInList())
      continue;
    //
    gridWidth += field.getRects({checkVisibility: true}).width;
  }
  //
  return gridWidth;
};


/**
 * Get fields rect
 * @param {Boolean} form
 * @param {Client.IdfGroup} group
 */
Client.IdfPanel.prototype.getFieldsRect = function (form, group)
{
  let sortChildren = function (children) {
    children.sort(function (a, b) {
      if (a.rect.top === b.rect.top) {
        return a.rect.left - b.rect.left;
      }
      //
      return a.rect.top - b.rect.top;
    });
  };
  //
  let children = [];
  let child;
  let field;
  //
  let fields = [];
  if (group)
    fields = group.fields;
  else {
    // Reset grid dimensions and position
    this.gridHeight = this.orgGridHeight;
    this.gridTop = this.orgGridTop;
    this.gridLeft = this.orgGridLeft;
    //
    // Get first level parent children (i.e. groups and fields belonging to any group)
    for (let i = 0; i < this.elements.length; i++) {
      let el = this.elements[i];
      //
      let isField = el instanceof Client.IdfField;
      let isGroup = el instanceof Client.IdfGroup;
      //
      // I just want fields and groups. So if el is neither, skip it.
      // Moreover, also skip field if:
      // 1) it belongs to a group
      // 2) it's a values aggregation of another field and I'm getting rects for list mode. In this case there's no need for a column for it in the main grid
      if ((!isField || el.groupId || (el.aggregateOfField !== -1 && !form)) && !isGroup)
        continue;
      //
      fields.push(el);
    }
  }
  //
  let gridWidth = 0;
  //
  // Add default row selector width to grid width
  gridWidth += (this.showRowSelector ? Client.IdfPanel.defaultRowSelectorWidth : 0);
  //
  // Add all out of list fields and groups as children
  for (let i = 0; i < fields.length; i++) {
    field = fields[i];
    //
    if (!field.isShown(form))
      continue;
    //
    let isGroup = field instanceof Client.IdfGroup;
    //
    // Reset list dimensions and position
    if (!form) {
      field.listWidth = field.orgListWidth;
      field.listHeight = field.orgListHeight;
      field.listLeft = field.orgListLeft;
      field.listTop = field.orgListTop;
      //
      // Just fields handle percentage dimensions
      if (!isGroup) {
        field.listWidthPerc = field.orgListWidthPerc;
        field.listHeightPerc = field.orgListHeightPerc;
        field.listLeftPerc = field.orgListLeftPerc;
        field.listTopPerc = field.orgListTopPerc;
      }
    }
    else { // Reset form dimensions and position
      field.formWidth = field.orgFormWidth;
      field.formHeight = field.orgFormHeight;
      field.formLeft = field.orgFormLeft;
      field.formTop = field.orgFormTop;
      //
      // Just fields handle percentage dimensions
      if (!isGroup) {
        field.formWidthPerc = field.orgFormWidthPerc;
        field.formHeightPerc = field.orgFormHeightPerc;
        field.formLeftPerc = field.orgFormLeftPerc;
        field.formTopPerc = field.orgFormTopPerc;
      }
    }
    //
    let width, height, left, top;
    //
    let parentWidth = field.isInList() ? this.gridWidth : this.originalWidth;
    let parentHeight = field.isInList() ? this.gridHeight : this.originalHeight;
    //
    // Use width percentage or normal width
    let widthPerc = form ? field.formWidthPerc : field.listWidthPerc;
    if (widthPerc !== undefined)
      width = (parentWidth * widthPerc / 100);
    else
      width = form ? field.formWidth : field.listWidth;
    //
    // Use height percentage or normal height
    let heightPerc = form ? field.formHeightPerc : field.listHeightPerc;
    if (heightPerc !== undefined)
      height = (parentHeight * heightPerc / 100);
    else
      height = form ? field.formHeight : field.listHeight;
    //
    // On IDF, panel grid has a left offset of 20px when row selector is visible.
    // This offset is not taken in account by out list field's left, so add it manually
    if (Client.mainFrame.isIDF && !form && !field.isInList() && this.showRowSelector) {
      // Don't add row selector offset in case of group, because group left already takes in account it
      if (!isGroup)
        field.listLeft += Client.IdfPanel.rowSelectorOffset;
      //
      if (field.listLeftPerc !== undefined) {
        field.listLeftPerc += (Client.IdfPanel.rowSelectorOffset * 100) / this.originalWidth;
        //
        if (!isGroup)
          field.listLeftPerc += (Client.IdfPanel.rowSelectorOffset * 100) / this.originalWidth;
      }
    }
    //
    // Use left percentage or normal left
    let leftPerc = form ? field.formLeftPerc : field.listLeftPerc;
    if (leftPerc !== undefined)
      left = (this.originalWidth * leftPerc / 100);
    else
      left = form ? field.formLeft : field.listLeft;
    //
    // Use top percentage or normal top
    let topPerc = form ? field.formTopPerc : field.listTopPerc;
    if (topPerc !== undefined)
      top = (this.originalHeight * topPerc / 100);
    else
      top = form ? field.formTop : field.listTop;
    //
    if (!form && field.isInList()) {
      gridWidth += width;
      continue;
    }
    //
    child = {};
    child.id = field.id;
    child.isGroup = isGroup;
    child.gridClass = field.gridClass ? field.gridClass : this.defaultGridClass;
    child.rect = {};
    child.rect.top = top;
    child.rect.bottom = child.rect.top + (height || (isGroup ? 0 : Client.IdfField.defaultHeight));
    child.rect.left = left;
    child.rect.right = child.rect.left + width;
    child.rect.width = width;
    child.rect.height = height;
    //
    children.push(child);
  }
  //
  // If I'm creating main list structure (i.e. not group structure), add also grid as child
  if (!form && !group) {
    child = {};
    child.id = this.id;
    child.rect = {};
    child.rect.top = this.gridTop;
    child.rect.bottom = child.rect.top + this.gridHeight;
    child.rect.left = this.gridLeft;
    child.rect.right = child.rect.left + gridWidth;
    child.rect.width = gridWidth;
    child.rect.height = this.gridHeight;
    //
    children.push(child);
  }
  //
  let completeIntersections = true;
  let strongConnections = true;
  //
  // Check if there are complete intersections or strong connections between children and iterate until they are not resolved
  while (completeIntersections || strongConnections) {
    // Sort children by top and left
    sortChildren(children);
    //
    completeIntersections = this.checkCompleteIntersections(children, form);
    strongConnections = this.checkStrongConnections(children, form);
  }
  //
  // Sort children by top and left
  sortChildren(children);
  //
  return children;
};


/**
 * Update child rect
 * @param {Object} options
 */
Client.IdfPanel.prototype.updateChildRect = function (options)
{
  let child = options.child;
  let newLeft = options.left;
  let newTop = options.top;
  let form = options.form;
  //
  // If current child is the panel itself I have to update its gridTop and gridLeft properties (just if I'm not creating form structure)
  if (child.id === this.id) {
    if (!form) {
      this.gridTop = newTop;
      this.gridLeft = newLeft;
    }
  }
  else { // Otherwise find the corresponding field/group and update its formTop/listTop and formLeft/listLeft properties
    let field = Client.eleMap[child.id];
    //
    if (form) {
      if (field.fromTopPerc !== undefined)
        field.formTopPerc = ((newTop * 100) / this.parent.height);
      else
        field.formTop = newTop;
      //
      if (field.formLeftPerc !== undefined)
        field.formLeftPerc = ((newLeft * 100) / this.parent.width);
      else
        field.formLeft = newLeft;
    }
    else {
      if (field.listTopPerc !== undefined)
        field.listTopPerc = ((newTop * 100) / this.parent.height);
      else
        field.listTop = newTop;
      //
      if (field.listLeftPerc !== undefined)
        field.listLeftPerc = ((newLeft * 100) / this.parent.width);
      else
        field.listLeft = newLeft;
    }
  }
};


/**
 * Look for complete intersections and resolve them if any
 * @param {Array} children
 * @param {Boolean} form
 */
Client.IdfPanel.prototype.checkCompleteIntersections = function (children, form)
{
  let found = false;
  //
  // Check if two children intersect each other. I have to avoid this behaviour
  for (let i = 0; i < children.length; i++) {
    let child1 = children[i];
    //
    for (let j = i + 1; j < children.length; j++) {
      let rightMove = false;
      let child2 = children[j];
      //
      // Check if child1 and child2 intersect each other both horizontally and vertically
      let completeIntersection = this.rectIntersection(child1.rect, child2.rect) && this.rectIntersection(child1.rect, child2.rect, true);
      //
      // If child1 and child2 don't completely intersect each other, do nothing
      if (!completeIntersection)
        continue;
      //
      found = true;
      //
      // Calculate intersection coordinates
      let intersectionLeft = child1.rect.left > child2.rect.left ? child1.rect.left : child2.rect.left;
      let intersectionRight = child1.rect.right > child2.rect.right ? child2.rect.right : child1.rect.right;
      let intersectionTop = child1.rect.top > child2.rect.top ? child1.rect.top : child2.rect.top;
      let intersectionBottom = child1.rect.bottom > child2.rect.bottom ? child2.rect.bottom : child1.rect.bottom;
      //
      // Calculate intersection width and height
      let intersectionWidth = intersectionRight - intersectionLeft;
      let intersectionHeight = intersectionBottom - intersectionTop;
      //
      // If intersection is higher than wider, move the second child at first child right (add 2 to make the two children don't touch each other)
      if (intersectionHeight > intersectionWidth) {
        child2.rect.leftGap = true;
        child2.rect.left = child1.rect.right + 2;
        child2.rect.right = child2.rect.left + child2.rect.width;
        rightMove = true;
      }
      else { // Otherwise move the second child at first child bottom (add 2 to make the two children don't touch each other)
        child2.rect.topGap = true;
        child2.rect.top = child1.rect.bottom + 2;
        child2.rect.bottom = child2.rect.top + child2.rect.height;
      }
      //
      let newLeft = child2.rect.left - (child2.rect.leftGap ? 2 : 0);
      let newTop = child2.rect.top - (child2.rect.topGap ? 2 : 0);
      delete child2.rect.leftGap;
      delete child2.rect.topGap;
      //
      this.updateChildRect({child: child2, left: newLeft, top: newTop, form});
      //
      // I just moved a field/group on right side of another field/group. But what if I moved it on another field/group? I have another intersection!
      // And if this field/group has a top value smaller than field/group I moved (i.e. it's above the moved field),
      // no one will handle this intersection because I order children by their top values and this means I already handled that field/group.
      // So in this case I have to handle children again starting from 0
      if (rightMove) {
        i = -1;
        //
        // Sort children by top and left
        children.sort(function (a, b) {
          if (a.rect.top === b.rect.top) {
            return a.rect.left - b.rect.left;
          }
          //
          return a.rect.top - b.rect.top;
        });
        //
        break;
      }
    }
  }
  //
  return found;
};


/**
 * Look for strong connections and resolve them if any
 * @param {Array} children
 * @param {Boolean} form
 */
Client.IdfPanel.prototype.checkStrongConnections = function (children, form)
{
  let intersectionsMap = this.getIntersectionsMap(children);
  //
  for (let i = 0; i < children.length; i++) {
    let child1 = children[i];
    //
    for (let j = i + 1; j < children.length; j++) {
      let child2 = children[j];
      //
      // If child1 and child2 are not part of a strong connection, do nothing
      let child4 = this.checkStrongConnection(child1, child2, intersectionsMap);
      if (!child4)
        continue;
      //
      // Strong connection: move child2 to child1 right and child4 to child1 left. This resolves the strong connection
      child2.rect.left = child1.rect.right;
      child2.rect.right = child2.rect.left + child2.rect.width;
      child4.rect.left = child1.rect.left;
      child4.rect.right = child4.rect.left + child4.rect.width;
      //
      this.updateChildRect({child: child2, left: child2.rect.left, top: child2.rect.top, form});
      this.updateChildRect({child: child4, left: child4.rect.left, top: child4.rect.top, form});
      //
      return true;
    }
  }
};


/**
 * Check if given rects intersect each other in given direction
 * @param {Object} r1 - {top, bottom, left, right}
 * @param {Object} r2 - {top, bottom, left, right}
 * @param {boolean} vertical
 */
Client.IdfPanel.prototype.rectIntersection = function (r1, r2, vertical)
{
  // Look for a vertical intersection
  if (vertical) {
    if (r1.top <= r2.bottom && r1.top >= r2.top)
      return true;
    if (r1.bottom >= r2.top && r2.top >= r1.top)
      return true;
    if (r2.top >= r1.top && r2.bottom <= r1.bottom)
      return true;
    if (r1.top >= r2.top && r1.bottom <= r2.bottom)
      return true;
  }
  else { // Otherwise look for a horizontal intersection
    if (r1.left < r2.right && r1.left >= r2.left)
      return true;
    if (r1.right > r2.left && r2.left >= r1.left)
      return true;
    if (r2.left >= r1.left && r2.right <= r1.right)
      return true;
    if (r1.left >= r2.left && r1.right <= r2.right)
      return true;
  }
  //
  return false;
};


/**
 * Get intersections map
 * @param {Object} children
 */
Client.IdfPanel.prototype.getIntersectionsMap = function (children)
{
  let intersections = {};
  //
  for (let i = 0; i < children.length; i++) {
    let child1 = children[i];
    intersections[child1.id] = intersections[child1.id] || {hor: [], ver: []};
    //
    for (let j = 0; j < children.length; j++) {
      if (j === i)
        continue;
      //
      let child2 = children[j];
      if (this.rectIntersection(child1.rect, child2.rect))
        intersections[child1.id].hor.push(child2);
      if (this.rectIntersection(child1.rect, child2.rect, true))
        intersections[child1.id].ver.push(child2);
    }
  }
  //
  return intersections;
};


/**
 * Check if child1 and child2 are part of a strongly connected group of fields
 * @param {Object} child1
 * @param {Object} child2
 * @param {Object} intersectionsMap
 */
Client.IdfPanel.prototype.checkStrongConnection = function (child1, child2, intersectionsMap)
{
  // If child2 does not belong to child1 horizontal intersections, do nothing
  if (!intersectionsMap[child1.id].hor.find(c => c.id === child2.id))
    return;
  //
  let child3, child4;
  //
  // Get child2 horizontal intersections
  let horChild2 = intersectionsMap[child2.id].hor;
  for (let j = 0; j < horChild2.length; j++) {
    let hor = horChild2[j];
    //
    // Look for a child2 horizontal intersection (different from child1) that vertically intersects child1
    if (hor.id === child1.id || !intersectionsMap[hor.id].ver.find(c => c.id === child1.id))
      continue;
    //
    // Now I have to check if current child2 horizontal intersection vertically intersects child1
    if (child2.rect.left < hor.rect.right && child1.rect.left < child2.rect.right) {
      child3 = hor;
      break;
    }
  }
  //
  // If third component of strongly connected fields is missing, there's no strong connection
  if (!child3)
    return;
  //
  // Get child2 vertical intersections
  let verChild2 = intersectionsMap[child2.id].ver;
  for (let j = 0; j < verChild2.length; j++) {
    let ver = verChild2[j];
    //
    // Look for a child2 vertical intersection (different from child1) that vertically intersects child1
    if (ver.id === child1.id)
      continue;
    //
    // If child4 candidate (ver) vertically intersects child1 and horizontally intersects child3
    if (intersectionsMap[ver.id].ver.find(c => c.id === child1.id) && intersectionsMap[ver.id].hor.find(c => c.id === child3.id) && child3.rect.left < ver.rect.left) {
      child4 = ver;
      break;
    }
    //
    // If child4 candidate (ver) horizontally intersects child1 and vertically intersects child3
    if (intersectionsMap[ver.id].hor.find(c => c.id === child1.id) && intersectionsMap[ver.id].ver.find(c => c.id === child3.id) && child3.rect.top < ver.rect.top) {
      child4 = ver;
      break;
    }
  }
  //
  // If fourth component of strongly connected fields is missing, there's no strong connection
  return child4;
};


/**
 * Create a grid structure made of rows and columns in which place each panel child based on its rect
 * @param {Boolean} form
 * @param {Object} groupConf - group for which to create structure
 */
Client.IdfPanel.prototype.createStructure = function (form, groupConf)
{
  let group = groupConf ? Client.eleMap[groupConf.id] : undefined;
  let fields = this.getFieldsRect(form, group);
  //
  let rows = [];
  //
  // Place fields/groups inside rows based on rows and fields/groups rects
  for (let i = 0; i < fields.length; i++) {
    let field = fields[i];
    //
    let found = false;
    for (let j = 0; j < rows.length; j++) {
      // If current field do not vertically intersects current row, this is not the right row for it
      if (!this.rectIntersection(rows[j].rect, field.rect, true))
        continue;
      //
      // If field is placed above the row, its top becomes row's new top
      if (rows[j].rect.top > field.rect.top)
        rows[j].rect.top = field.rect.top;
      //
      // If it's placed under the row, its bottom becomes row's new bottom
      if (rows[j].rect.bottom < field.rect.bottom)
        rows[j].rect.bottom = field.rect.bottom;
      //
      // If it's placed at row left, its left becomes row's new left
      if (rows[j].rect.left > field.rect.left)
        rows[j].rect.left = field.rect.left;
      //
      // If it's placed at row right, its right becomes row's new right
      if (rows[j].rect.right < field.rect.right)
        rows[j].rect.right = field.rect.right;
      //
      // Add current field/group to current row
      rows[j].fields.push(field);
      found = true;
      break;
    }
    //
    // If I didn't found a row that intersects current field, I have to create a new row for it
    if (!found) {
      // Give new row the same rect as field rect, then add field to row
      let newRow = {};
      newRow.rect = Object.assign({}, field.rect);
      newRow.fields = [field];
      newRow.cols = [];
      //
      // New row top has to be equal to last row bottom. In case of first row it has to be 0 if row doesn't belong to any group, otherwise it has to be equal to group top
      // New row left is 0 if row belongs to any group, otherwise it's group left
      let lastRow = rows[rows.length - 1];
      newRow.rect.top = lastRow ? lastRow.rect.bottom : (groupConf ? groupConf.rect.top : 0);
      newRow.rect.left = groupConf ? groupConf.rect.left : 0;
      //
      // Create row configuration
      newRow.conf = this.createElementConfig({c: "IonRow", noWrap: true, style: {height: "100%", "flex-basis": "content", "flex-grow": "0"}});
      //
      rows.push(newRow);
    }
  }
  //
  // Split each row into columns and assign each field to proper column
  for (let i = 0; i < rows.length; i++)
    this.assignFieldsToColumns(form, rows[i]);
  //
  return rows;
};


/**
 * Split given column into rows and assign each field to proper row
 * @param {Boolean} form
 * @param {Object} col
 */
Client.IdfPanel.prototype.assignFieldsToRows = function (form, col)
{
  let rows = [];
  //
  // Order column fields by their top
  col.fields.sort(function (a, b) {
    return a.rect.top - b.rect.top;
  });
  //
  for (let i = 0; i < col.fields.length; i++) {
    let field = col.fields[i];
    //
    let found = false;
    for (let j = 0; j < rows.length; j++) {
      // If current field do not vertically intersects current row, this is not the right row for it
      if (!this.rectIntersection(rows[j].rect, field.rect, true))
        continue;
      //
      // If field is placed above the row, its top becomes row's new top
      if (rows[j].rect.top > field.rect.top)
        rows[j].rect.top = field.rect.top;
      //
      // If it's placed under the row, its bottom becomes row's new bottom
      if (rows[j].rect.bottom < field.rect.bottom)
        rows[j].rect.bottom = field.rect.bottom;
      //
      // If it's placed above at row left, its left becomes row's new left
      if (rows[j].rect.left > field.rect.left)
        rows[j].rect.left = field.rect.left;
      //
      // If it's placed at row right, its right becomes row's new bottom
      if (rows[j].rect.right < field.rect.right)
        rows[j].rect.right = field.rect.right;
      //
      // Add current field to current row
      rows[j].fields.push(field);
      found = true;
      break;
    }
    //
    // If I didn't found a row that intersects current field, I have to create a new row for it
    if (!found) {
      // Give new row the same rect as field rect, then add field to row
      let newRow = {};
      newRow.rect = Object.assign({}, field.rect);
      newRow.fields = [field];
      newRow.cols = [];
      //
      let lastRow = rows[rows.length - 1];
      newRow.rect.top = lastRow ? lastRow.rect.bottom : col.rect.top;
      newRow.rect.bottom = field.rect.bottom;
      newRow.rect.left = col.rect.left;
      newRow.rect.right = col.rect.right;
      //
      // Create row configuration and add it to column configuration
      newRow.conf = this.createElementConfig({c: "IonRow", noWrap: true, style: {height: "100%", "flex-basis": "content", "flex-grow": "0"}});
      col.conf.children.push(newRow.conf);
      //
      // Add new row to column rows
      rows.push(newRow);
    }
  }
  //
  // Sort rows by theri top
  rows.sort(function (a, b) {
    return (a.rect && b.rect) ? a.rect.top - b.rect.top : 0;
  });
  //
  // Split each row into columns and assign each field to proper column
  col.rows = rows;
  for (let j = 0; j < rows.length; j++)
    this.assignFieldsToColumns(form, rows[j]);
};


/**
 * Split given row into columns and assign each field to proper column
 * @param {Boolean} form
 * @param {Object} row
 */
Client.IdfPanel.prototype.assignFieldsToColumns = function (form, row)
{
  let cols = [];
  //
  row.fields.sort(function (a, b) {
    return a.rect.left - b.rect.left;
  });
  //
  for (let i = 0; i < row.fields.length; i++) {
    let field = row.fields[i];
    //
    let found = false;
    for (let j = 0; j < cols.length; j++) {
      // If current field do not horizontally intersects current column, this is not the right column for it
      if (!this.rectIntersection(cols[j].rect, field.rect, false))
        continue;
      //
      // If field is placed above the column, its top becomes column's new top
      if (cols[j].rect.top > field.rect.top)
        cols[j].rect.top = field.rect.top;
      //
      // If it's placed under the column, its bottom becomes column's new bottom
      if (cols[j].rect.bottom < field.rect.bottom)
        cols[j].rect.bottom = field.rect.bottom;
      //
      // If it's placed at column left, its left becomes column's new left
      if (cols[j].rect.left > field.rect.left)
        cols[j].rect.left = field.rect.left;
      //
      // If it's placed at column right, its right becomes column's new right
      if (cols[j].rect.right < field.rect.right)
        cols[j].rect.right = field.rect.right;
      //
      // Add current field to current column
      cols[j].fields.push(field);
      found = true;
      break;
    }
    //
    // If I didn't found a column that intersects current field, I have to create a new column for it
    if (!found) {
      // Give new column the same rect as field rect, then add field to column
      let newCol = {};
      newCol.rect = Object.assign({}, field.rect);
      newCol.fields = [field];
      newCol.rows = [];
      newCol.visible = true;
      //
      let lastCol = cols[cols.length - 1];
      newCol.rect.left = lastCol ? lastCol.rect.right : row.rect.left;
      newCol.rect.right = field.rect.right;
      newCol.rect.top = row.rect.top;
      newCol.rect.bottom = field.rect.bottom;
      //
      // Create new column configuration and add it to row configuration
      newCol.conf = this.createElementConfig({c: "IonCol", style: {"flex-basis": "content", "flex-direction": "column", "display": "flex", padding: 0}});
      newCol.conf.parentRowId = row.conf.id;
      row.conf.children.push(newCol.conf);
      //
      // Add new column to row columns
      cols.push(newCol);
      //
      // For each column calculate delta right and delta bottom. I will use these values in "calcLayout" method of both IdfPanel and IdfField
      for (let k = 0; k < cols.length; k++) {
        // If parent row has just a column, calculate distance from parent row right
        cols[k].rect.deltaRight = cols.length === 1 ? row.rect.right - cols[k].rect.right : 0;
        //
        // If current column has no rows (that means it is a leaf column) calculate distance from parent row bottom
        cols[k].rect.deltaBottom = !cols[k].rows.length ? row.rect.bottom - cols[k].rect.bottom : 0;
      }
    }
  }
  //
  // Order columns by their left
  cols.sort(function (a, b) {
    if (a.rect.left === b.rect.left)
      return 0;
    else if (b.rect.left > a.rect.left)
      return -1;
    else
      return 1;
  });
  //
  row.cols = cols;
  for (let i = 0; i < cols.length; i++) {
    // If current column contains more than one field, split it into rows and assign each field to proper row
    if (cols[i].fields.length > 1)
      this.assignFieldsToRows(form, cols[i]);
    else {
      let obj = cols[i].fields[0];
      //
      // Check if the Field has a responsive set-up and update it
      if (obj.gridClass)
        cols[i].conf.className = obj.gridClass;
      //
      // If current column contains a group, create structure for that group
      if (obj.isGroup) {
        cols[i].rows = this.createStructure(form, obj);
        for (let j = 0; j < cols[i].rows.length; j++)
          cols[i].conf.children.push(cols[i].rows[j].conf);
      }
    }
  }
};


/**
 * Return true if panel has dynamic height rows
 */
Client.IdfPanel.prototype.hasDynamicHeightRows = function ()
{
  return !!this.rowHeightResize;
};


/**
 * Get active row index
 */
Client.IdfPanel.prototype.getActiveRowIndex = function ()
{
  return this.actualPosition + this.actualRow;
};


/**
 * Handle scroll
 */
Client.IdfPanel.prototype.handleScroll = function ()
{
  if (!Client.mainFrame.isIDF)
    return;
  //
  // Clear actual scroll timeout
  clearTimeout(this.scrollTimeout);
  //
  // Get grid element
  let gridEl = Client.eleMap[this.gridConf.id].getRootObject();
  //
  let oldScrollTop = this.lastScrollTop || 0;
  this.lastScrollTop = gridEl.scrollTop;
  //
  // Check if I'm scrolling up
  this.scrollUp = oldScrollTop > this.lastScrollTop;
  //
  // If this scroll is not performed by user, skip it
  if (this.skipPanScr) {
    delete this.skipPanScr;
    return;
  }
  //
  // Set scroll timeout
  this.scrollTimeout = setTimeout(() => {
    let rowsTotHeight = 0;
    let rowHeight;
    let defaultRowHeight = this.getListRowHeight();
    let dataWindowHeight = 0;
    let dataWindowStart = -1;
    //
    let visibleRowsHeight = gridEl.clientHeight - this.getHeaderHeight() + this.getQbeRowHeight();
    //
    let maxRows = this.getMaxRows();
    //
    // Calculate row index to request to server
    for (let i = 1; i <= maxRows; i++) {
      // In case of grouped rows, skip rows belonging to collapsed groups
      if (this.hasGroupedRows() && !this.groupedRowsRoot.isRowVisible(i))
        continue;
      //
      // If current data row exists get its height.
      // Otherwise in this position there is an empty space (or a placeholder row) waiting for a data row. So get default row height
      rowHeight = this.getRow(i)?.getRootObject().clientHeight || defaultRowHeight;
      //
      // Add row height to total height
      rowsTotHeight += rowHeight;
      //
      // If rows height is greater than grid scroll top, I found data window start
      if (dataWindowStart === -1 && rowsTotHeight > gridEl.scrollTop)
        dataWindowStart = i;
      //
      // If I found data window start
      if (dataWindowStart !== -1) {
        dataWindowHeight += rowHeight;
        //
        // If I reach data window end, exit
        if (dataWindowHeight >= visibleRowsHeight) {
          this.fillBufferVideo(dataWindowStart, i);
          break;
        }
      }
    }
    //
    Client.mainFrame.sendEvents(this.handlePanelScroll(dataWindowStart, this.scrollUp));
  }, 50);
};


/**
 * Update toolbar
 */
Client.IdfPanel.prototype.updateToolbar = function ()
{
  Client.IdfFrame.prototype.updateToolbar.call(this);
  //
  // Update status bar
  this.updateStatusbar();
  //
  // Update qbe button
  let qbeButton = Client.eleMap[this.qbeButtonConf.id];
  let showQbeButton = this.qbeTip && this.status === Client.IdfPanel.statuses.data && this.showStatusbar && !this.collapsed;
  qbeButton.updateElement({visible: !!showQbeButton, tooltip: this.getTooltip(this.qbeButtonConf.id)});
  //
  let numRows = this.layout === Client.IdfPanel.layouts.list ? this.getNumRows() : 1;
  //
  // Update navigation buttons
  let showNavButtons = this.isCommandEnabled(Client.IdfPanel.commands.CMD_NAVIGATION) && this.canNavigate();
  showNavButtons = showNavButtons && (this.totalRows > numRows || this.actualPosition > 1);
  //
  let topButton = Client.eleMap[this.topButtonConf.id];
  let prevButton = Client.eleMap[this.prevButtonConf.id];
  let nextButton = Client.eleMap[this.nextButtonConf.id];
  let bottomButton = Client.eleMap[this.bottomButtonConf.id];
  topButton.updateElement({visible: !!showNavButtons, tooltip: this.getTooltip(this.topButtonConf.id)});
  prevButton.updateElement({visible: !!showNavButtons, tooltip: this.getTooltip(this.prevButtonConf.id)});
  nextButton.updateElement({visible: !!showNavButtons, tooltip: this.getTooltip(this.nextButtonConf.id)});
  bottomButton.updateElement({visible: !!showNavButtons, tooltip: this.getTooltip(this.bottomButtonConf.id)});
  //
  // Update search button
  let showSearchButton = this.canNavigate() && this.isCommandEnabled(Client.IdfPanel.commands.CMD_SEARCH) && this.canSearch && this.searchMode === Client.IdfPanel.searchModes.toolbar;
  let searchButton = Client.eleMap[this.searchButtonConf.id];
  searchButton.updateElement({visible: !!showSearchButton, tooltip: this.getTooltip(this.searchButtonConf.id)});
  //
  // Update find button
  let showFindButton = this.isCommandEnabled(Client.IdfPanel.commands.CMD_FIND) && this.canSearch && this.status === Client.IdfPanel.statuses.qbe;
  let findButton = Client.eleMap[this.findButtonConf.id];
  findButton.updateElement({visible: !!showFindButton, tooltip: this.getTooltip(this.findButtonConf.id)});
  //
  // Update formList button
  let showFormListButton = this.isCommandEnabled(Client.IdfPanel.commands.CMD_FORMLIST) && this.hasList && this.hasForm;
  showFormListButton = showFormListButton && (this.status !== Client.IdfPanel.statuses.qbe || !this.automaticLayout);
  showFormListButton = showFormListButton && (this.status === Client.IdfPanel.statuses.qbe || this.canNavigate());
  let formListButton = Client.eleMap[this.formListButtonConf.id];
  formListButton.updateElement({visible: !!showFormListButton, tooltip: this.getTooltip(this.formListButtonConf.id)});
  //
  // Update cancel button
  let showCancelButton = this.isCommandEnabled(Client.IdfPanel.commands.CMD_CANCEL);
  showCancelButton = showCancelButton && ((this.status === Client.IdfPanel.statuses.updated || (Client.mainFrame.idfMobile && !this.locked && this.lockable)) && (this.canUpdate || this.canInsert) || this.status === Client.IdfPanel.statuses.qbe || this.DOModified);
  let cancelButton = Client.eleMap[this.cancelButtonConf.id];
  cancelButton.updateElement({visible: !!showCancelButton, tooltip: this.getTooltip(this.cancelButtonConf.id)});
  //
  // Update refresh button
  let showRefreshButton = this.isCommandEnabled(Client.IdfPanel.commands.CMD_REFRESH) && this.status !== Client.IdfPanel.statuses.qbe;
  showRefreshButton = showRefreshButton && (!this.isDO || this.hasDocTemplate);
  let refreshButton = Client.eleMap[this.refreshButtonConf.id];
  refreshButton.updateElement({visible: !!showRefreshButton, tooltip: this.getTooltip(this.refreshButtonConf.id)});
  //
  // Update delete button
  let showDeleteButton = this.isCommandEnabled(Client.IdfPanel.commands.CMD_DELETE) && this.canDelete && (!this.locked || Client.mainFrame.idfMobile) && this.status === Client.IdfPanel.statuses.data;
  let deleteButton = Client.eleMap[this.deleteButtonConf.id];
  deleteButton.updateElement({visible: !!showDeleteButton, tooltip: this.getTooltip(this.deleteButtonConf.id)});
  //
  // Update insert button
  let showInsertButton = this.isCommandEnabled(Client.IdfPanel.commands.CMD_INSERT) && this.canInsert;
  showInsertButton = showInsertButton && (!this.locked || this.enableInsertWhenLocked || (this.isDO && this.DOSingleDoc && this.isNewRow()));
  showInsertButton = showInsertButton && (this.status === Client.IdfPanel.statuses.qbe || this.canNavigate());
  let insertButton = Client.eleMap[this.insertButtonConf.id];
  insertButton.updateElement({visible: !!showInsertButton, tooltip: this.getTooltip(this.insertButtonConf.id)});
  //
  // Update duplicate button
  let showDuplicateButton = this.isCommandEnabled(Client.IdfPanel.commands.CMD_DUPLICATE) && this.canInsert && !this.locked && this.canNavigate() && !this.isNewRow();
  let duplicateButton = Client.eleMap[this.duplicateButtonConf.id];
  duplicateButton.updateElement({visible: !!showDuplicateButton, tooltip: this.getTooltip(this.duplicateButtonConf.id)});
  //
  // Update save button
  let showSaveButton = this.isCommandEnabled(Client.IdfPanel.commands.CMD_SAVE);
  showSaveButton = showSaveButton && (this.canInsert || this.canUpdate || (this.DOModified && this.DOCanSave));
  showSaveButton = showSaveButton && !this.locked && this.status !== Client.IdfPanel.statuses.qbe;
  let saveButton = Client.eleMap[this.saveButtonConf.id];
  saveButton.updateElement({visible: !!showSaveButton, tooltip: this.getTooltip(this.saveButtonConf.id)});
  //
  // Update print button
  let showPrintButton = this.isCommandEnabled(Client.IdfPanel.commands.CMD_PRINT) && this.canNavigate() && this.hasBook;
  let printButton = Client.eleMap[this.printButtonConf.id];
  printButton.updateElement({visible: !!showPrintButton, tooltip: this.getTooltip(this.printButtonConf.id)});
  //
  // Update group button
  let showGroupButton = this.isCommandEnabled(Client.IdfPanel.commands.CMD_GROUP) && this.status === Client.IdfPanel.statuses.data;
  showGroupButton = showGroupButton && this.layout === Client.IdfPanel.layouts.list && this.canGroup;
  let groupButton = Client.eleMap[this.groupButtonConf.id];
  groupButton.updateElement({visible: !!showGroupButton, tooltip: this.getTooltip(this.groupButtonConf.id), icon: this.showGroups ? "contract" : "grid"});
  //
  // Update csv button
  let showCsvButton = this.isCommandEnabled(Client.IdfPanel.commands.CMD_CSV) && this.canNavigate() && this.layout === Client.IdfPanel.layouts.list;
  let csvButton = Client.eleMap[this.csvButtonConf.id];
  csvButton.updateElement({visible: !!showCsvButton, tooltip: this.getTooltip(this.csvButtonConf.id)});
  //
  // Update attach button
  let showAttachButton = this.isCommandEnabled(Client.IdfPanel.commands.CMD_ATTACH);
  let attachButton = Client.eleMap[this.attachButtonConf.id];
  attachButton.updateElement({visible: !!showAttachButton, tooltip: this.getTooltip(this.attachButtonConf.id)});
  //
  // Update custom buttons
  let small = this.smallIcons ? " small" : "";
  for (let i = 0; i < this.customCommands.length; i++) {
    let customButton = Client.eleMap[this.customButtonsConf[i].id];
    //
    let className;
    let image = this.customCommands[i].image;
    if (image) {
      Client.Widget.setIconImage(image, customButton);
      className = "generic-btn panel-toolbar-btn " + (Client.Widget.isIconImage(image) ? "" : " image") + " custom-btn" + (i + 1) + small;
    }
    //
    customButton.updateElement({tooltip: this.getTooltip(this.customButtonsConf[i].id, className)});
  }
  //
  // Update fields blob controls
  for (let j = 0; j < this.fields.length; j++)
    this.fields[j].updateBlob();
  //
  // Check mobile buttons
  this.parentIdfView?.checkMobileButtons();
};


/**
 * Get zone for given command
 * @param {Integer} commandIndex
 */
Client.IdfPanel.prototype.getCommandZone = function (commandIndex)
{
  if (Client.mainFrame.isIDF)
    return Client.mainFrame.wep.getCommandZone(commandIndex);
  //
  return this.commandsZones[commandIndex];
};


/**
 * Return true if navigating panel data using navigation button is allowed
 */
Client.IdfPanel.prototype.canNavigate = function ()
{
  return this.status === Client.IdfPanel.statuses.data && (this.allowNavigationWhenModified || !this.DOMaster || !this.DOModified);
};


/**
 * Return true if given command is enabled
 * @param {Integer} cmd
 */
Client.IdfPanel.prototype.isCommandEnabled = function (cmd)
{
  // If cmd is a common command
  if (cmd > 0) {
    let blobCommands = [Client.IdfPanel.commands.CMD_BLOBEDIT, Client.IdfPanel.commands.CMD_BLOBDELETE, Client.IdfPanel.commands.CMD_BLOBNEW, Client.IdfPanel.commands.CMD_BLOBSAVEAS];
    return (this.enabledCommands & cmd) && (this.showToolbar || blobCommands.indexOf(cmd) !== -1) && !this.collapsed;
  }
  else // Otherwise if is a custom command
    return (this.extEnabledCommands & Math.abs(cmd)) && this.showToolbar && !this.collapsed;
};


/**
 * Handle small icons changing css classes to toolbar elements
 */
Client.IdfPanel.prototype.handleSmallIcons = function ()
{
  Client.IdfFrame.prototype.handleSmallIcons.call(this);
  //
  for (let i = 0; i < this.toolbarZonesConfig.length; i++) {
    let toolbarZone = Client.eleMap[this.toolbarZonesConfig[i].id];
    if (!toolbarZone)
      continue;
    //
    // Set "small" class on current toolbar zone children
    for (let j = 0; j < toolbarZone.elements.length; j++) {
      let el = toolbarZone.elements[j];
      if (this.smallIcons)
        el.domObj.classList.add("small");
      else
        el.domObj.classList.remove("small");
      //
      el.updateElement({className: el.domObj.className});
    }
  }
  //
  // Set "small" class on status bar
  let statusbar = Client.eleMap[this.statusbarConf.id];
  if (this.smallIcons)
    statusbar.domObj.classList.add("small");
  else
    statusbar.domObj.classList.remove("small");
  //
  statusbar.updateElement({className: statusbar.domObj.className});
};


/**
 * Handle blob delete
 * @param {Object} event
 */
Client.IdfPanel.prototype.handleBlobDelete = function (event)
{
  let events = [];
  if (Client.mainFrame.isIDF) {
    events.push({
      id: "pantb",
      def: this.toolbarEventDef,
      content: {
        oid: this.id,
        obn: "delblob" + event.content.fieldIndex
      }
    });
  }
  //
  return events;
};


/**
 * Get tooltip for given object
 * @param {String} objId
 */
Client.IdfPanel.prototype.getTooltip = function (objId)
{
  let tooltip = Client.IdfFrame.prototype.getTooltip.call(this, objId);
  //
  if (tooltip)
    return tooltip;
  //
  let wep = Client.mainFrame.wep;
  let title, content, fknum;
  //
  switch (objId) {
    case this.qbeButtonConf.id:
      if (this.qbeTip) {
        title = Client.IdfResources.t("TIP_TITLE_QBETIP");
        content = this.qbeTip;
      }
      break;

    case this.topButtonConf.id:
      title = Client.IdfResources.t("TIP_TITLE_PanelInizio");
      content = wep?.SRV_MSG_PanelStart || Client.IdfResources.t("SRV_MSG_PanelStart");
      break;

    case this.prevButtonConf.id:
      title = Client.IdfResources.t("TIP_TITLE_PanelPaginaPrec");
      content = wep?.SRV_MSG_PanelPrevPage || Client.IdfResources.t("SRV_MSG_PanelPrevPage");
      break;

    case this.nextButtonConf.id:
      title = Client.IdfResources.t("TIP_TITLE_PanelPaginaSucc");
      content = wep?.SRV_MSG_PanelNextPage || Client.IdfResources.t("SRV_MSG_PanelNextPage");
      break;

    case this.bottomButtonConf.id:
      title = Client.IdfResources.t("TIP_TITLE_PanelFine");
      content = wep?.SRV_MSG_PanelEnd || Client.IdfResources.t("SRV_MSG_PanelEnd");
      break;

    case this.searchButtonConf.id:
      title = Client.IdfResources.t("TIP_TITLE_TooltipCerca");
      content = wep?.SRV_MSG_Search || Client.IdfResources.t("SRV_MSG_Search");
      fknum = wep?.FKEnterQBE;
      break;

    case this.findButtonConf.id:
      title = Client.IdfResources.t("TIP_TITLE_TooltipTrova");
      content = wep?.SRV_MSG_Find || Client.IdfResources.t("SRV_MSG_Find");
      fknum = wep?.FKFindData;
      break;

    case this.formListButtonConf.id:
      title = Client.IdfResources.t("TIP_TITLE_TooltipFormList");
      content = wep?.SRV_MSG_FormList || Client.IdfResources.t("SRV_MSG_FormList");
      fknum = wep?.FKFormList;
      break;

    case this.cancelButtonConf.id:
      title = Client.IdfResources.t("TIP_TITLE_TooltipCancel");
      content = wep?.SRV_MSG_Cancel || Client.IdfResources.t("SRV_MSG_Cancel");
      fknum = wep?.FKCancel;
      break;

    case this.refreshButtonConf.id:
      title = Client.IdfResources.t("TIP_TITLE_TooltipRefresh");
      content = wep?.SRV_MSG_Reload || Client.IdfResources.t("SRV_MSG_Reload");
      fknum = wep?.FKRefresh;
      break;

    case this.deleteButtonConf.id:
      title = Client.IdfResources.t("TIP_TITLE_TooltipDelete");
      content = wep?.SRV_MSG_Delete || Client.IdfResources.t("SRV_MSG_Delete");
      fknum = wep?.FKDelete;
      break;

    case this.insertButtonConf.id:
      title = Client.IdfResources.t("TIP_TITLE_TooltipInsert");
      content = wep?.SRV_MSG_Insert || Client.IdfResources.t("SRV_MSG_Insert");
      fknum = wep?.FKInsert;
      break;

    case this.duplicateButtonConf.id:
      title = Client.IdfResources.t("TIP_TITLE_TooltipDuplicate");
      content = wep?.SRV_MSG_Duplicate || Client.IdfResources.t("SRV_MSG_Duplicate");
      fknum = wep?.FKDuplicate;
      break;

    case this.saveButtonConf.id:
      title = Client.IdfResources.t("TIP_TITLE_TooltipUpdate");
      content = wep?.SRV_MSG_Update || Client.IdfResources.t("SRV_MSG_Update");
      fknum = wep?.FKUpdate;
      break;

    case this.printButtonConf.id:
      title = Client.IdfResources.t("TIP_TITLE_Print");
      content = wep?.SRV_MSG_Print || Client.IdfResources.t("SRV_MSG_Print");
      fknum = wep?.FKPrint;
      break;

    case this.groupButtonConf.id:
      title = Client.IdfResources.t("TIP_TITLE_ComandoGruppi");
      content = wep?.SRV_MSG_Group || Client.IdfResources.t("SRV_MSG_Group");
      break;

    case this.csvButtonConf.id:
      title = Client.IdfResources.t("TIP_TITLE_TooltipExport");
      content = wep?.SRV_MSG_Export || Client.IdfResources.t("SRV_MSG_Export");
      break;

    case this.attachButtonConf.id:
      title = Client.IdfResources.t("TIP_TITLE_ComandoAllegati");
      content = wep?.SRV_MSG_Attach || Client.IdfResources.t("SRV_MSG_Attach");
      break;

    default:
      for (let i = 0; i < this.customButtonsConf.length; i++) {
        if (this.customButtonsConf[i].id === objId) {
          title = this.customCommands[i].caption;
          content = this.customCommands[i].tooltip;
          fknum = this.customCommands[i].fknum;
          break;
        }
      }
      break;
  }
  //
  tooltip = Client.Widget.getHTMLTooltip(title, content, fknum);
  //
  return tooltip;
};


/**
 * Update status bar
 */
Client.IdfPanel.prototype.updateStatusbar = function ()
{
  let visible = (this.showStatusbar && !this.collapsed);
  let text = "";
  let className = "";
  //
  let statusbar = Client.eleMap[this.statusbarConf.id];
  let classList = statusbar.getRootObject().classList;
  classList.remove("updated");
  classList.remove("qbe");
  //
  let status = (this.DOModified && this.DOMaster) ? Client.IdfPanel.statuses.updated : this.status;
  //
  // Get status bar text
  switch (status) {
    case Client.IdfPanel.statuses.qbe:
      text = Client.IdfResources.t("SRV_MSG_StatusQBE");
      className = "qbe";
      break;

    case Client.IdfPanel.statuses.data:
      if (this.isNewRow()) {
        if (this.canInsert)
          text = Client.IdfResources.t("SRV_MSG_StatusInsert");
        else
          text = Client.IdfResources.t("SRV_MSG_StatusData1", [this.getActiveRowIndex()]);
      }
      else if (!this.showMultiSelection || this.layout === Client.IdfPanel.layouts.form)
        text = Client.IdfResources.t("SRV_MSG_RowNumOf", [this.getActiveRowIndex(), this.totalRows]);
      else {
        let selectedRows = this.showMultiSelection ? this.getSelectedDataRows() : 0;
        if (selectedRows === 1)
          text = Client.IdfResources.t("PAN_STBAR_SelRow", [this.totalRows]);
        else
          text = Client.IdfResources.t("PAN_STBAR_SelRows", [this.totalRows, selectedRows]);
      }
      break;

    case Client.IdfPanel.statuses.updated:
      text = Client.IdfResources.t("SRV_MSG_StatusUpdated");
      className = "updated";
      break;
  }
  //
  // In case of single doc remove text and className (I don't want to see "Row 1 of 1)
  if (this.DOSingleDoc && (!this.DOMaster || !this.DOModified)) {
    text = "";
    className = "";
  }
  //
  // Update status bar element
  statusbar.updateElement({innerText: text, visible: visible, className: classList.value + " " + className});
};


/**
 * Update layout (list/form)
 */
Client.IdfPanel.prototype.updateLayout = function ()
{
  // If one of layouts is missing, selected page is always 0 (because there's just one page)
  let selectedPage = (!this.hasList || !this.hasForm) ? 0 : this.layout;
  //
  // Update selected page
  let panelContainer = Client.eleMap[this.panelContainerConf.id];
  panelContainer.updateElement({selectedPage});
  //
  // During slide animation, the scrollbars would appear. So set overflow to hidden and reset it after animation end
  let contentEl = Client.eleMap[this.contentContainerConf.id];
  contentEl.updateElement({style: {overflow: "hidden"}});
  setTimeout(function () {
    contentEl.updateElement({style: {overflow: ""}});
  }, 350);
  //
  // Assign subFrame
  for (let i = 0; i < this.fields.length; i++)
    this.fields[i].assignSubFrame();
};


/**
 * Update multiple selection
 * @param {Object} options
 */
Client.IdfPanel.prototype.updateMultiSel = function (options)
{
  options = options || {};
  //
  let index = options.index;
  let value = !!options.value;
  //
  let start = index ?? 1;
  let end = index ?? this.totalRows;
  //
  if (index === undefined && this.selectOnlyVisibleRows && !options.force) {
    start = this.actualPosition;
    end = this.actualPosition + this.getNumRows() - 1;
    //
    if (end > this.totalRows)
      end = this.totalRows;
  }
  //
  for (let i = start; i <= end; i++) {
    this.multiSelStatus[i] = options.invert ? !this.multiSelStatus[i] : value;
    this.fields.forEach(f => f.selectRow(value, i));
  }
  //
  // If I have no index it means I have to update multiple selection on all rows.
  // Since sometimes multiSelStatus length could be greater that totalRows (for example, just after deleting some rows), truncate it at totalRows
  if (index === undefined)
    this.multiSelStatus.length = (this.status !== Client.IdfPanel.statuses.qbe ? this.totalRows + 1 : 1);
  //
  this.updateStatusbar();
};


/**
 * Get header height
 */
Client.IdfPanel.prototype.getHeaderHeight = function ()
{
  return this.headerHeight;
};


/**
 * Get QBE row height
 */
Client.IdfPanel.prototype.getQbeRowHeight = function ()
{
  return this.canUseRowQbe() ? this.getListRowHeight() : 0;
};


/**
 * Get header offset from visual style
 */
Client.IdfPanel.prototype.getHeaderOffset = function ()
{
  let visualStyle = Client.IdfVisualStyle.getByIndex(this.visualStyle);
  //
  return visualStyle ? visualStyle.getHeaderOffset() || 0 : 0;
};


/**
 * Get height of a list row
 */
Client.IdfPanel.prototype.getListRowHeight = function ()
{
  let height = Client.IdfPanel.defaultListRowHeight;
  //
  // If panel has dynamic height rows, list rows have not a specific height.
  // So when I need to know the row height I use the default one
  // (for example when I have to calculate margin top and bottom between two data blocks in the list)
  if (this.hasDynamicHeightRows())
    return height;
  //
  let maxFieldHeight = -1;
  //
  // Ask fields their heights and get the max
  for (let i = 0; i < this.fields.length; i++) {
    let field = this.fields[i];
    if (field.isInList()) {
      let fieldHeight = field.getRects().height;
      //
      if (fieldHeight > maxFieldHeight)
        maxFieldHeight = fieldHeight;
    }
  }
  //
  // If there is a valid max field height, this is the list row height
  if (maxFieldHeight >= 0)
    height = maxFieldHeight;
  //
  return height;
};


/**
 * Get row offset from visual style
 */
Client.IdfPanel.prototype.getListRowOffset = function ()
{
  let visualStyle = Client.IdfVisualStyle.getByIndex(this.visualStyle);
  //
  return visualStyle ? visualStyle.getRowOffset() || 0 : 0;
};


/**
 * Get number of visible rows
 */
Client.IdfPanel.prototype.getNumRows = function ()
{
  if (Client.mainFrame.isIDF)
    return this.numRows;
  //
  let numRows = 0;
  //
  // Get grid element
  let gridDomObj = Client.eleMap[this.gridConf.id].getRootObject();
  //
  // Get visible rows top and bottom
  let visibleRowsTop = gridDomObj.scrollTop + this.getHeaderHeight() + this.getQbeRowHeight();
  let visibleRowsBottom = gridDomObj.scrollTop + gridDomObj.clientHeight;
  //
  for (let i = 1; i <= this.totalRows; i++) {
    // If current data row exists
    let rowEl = this.getRow(i);
    if (!rowEl || this.isRowDetached(i))
      continue;
    //
    let rowDom = rowEl.getRootObject();
    //
    // Get current row top and bottom
    let rowOffset = i === 1 ? 0 : this.getListRowOffset();
    let rowTop = rowDom.offsetTop + rowOffset;
    let rowBottom = rowDom.offsetTop + rowDom.clientHeight;
    //
    // If current row is visible
    if (rowTop >= visibleRowsTop && rowBottom <= visibleRowsBottom)
      numRows++;
    //
    // If I'm out of visible rows window, exit
    if (rowTop > visibleRowsBottom)
      break;
  }
  //
  return numRows;
};


/**
 * Get maximum number of rows
 * @param {Boolean} skipNumRows
 */
Client.IdfPanel.prototype.getMaxRows = function (skipNumRows)
{
  // Get index of last data row. Optimize search reversing rows array and then recalculating last data row index
  let lastDataRowIndex = this.rows.slice().reverse().findIndex(element => element?.isData);
  lastDataRowIndex = this.rows.length - lastDataRowIndex - 1;
  //
  // Get total rows
  let totalRows = this.hasGroupedRows() ? this.groupedRowsRoot.groupedEndingRow : this.totalRows;
  //
  // Last data row index can be greater than total rows count when there are new rows, so get the max among the two values
  let maxIndex = Math.max(totalRows, lastDataRowIndex);
  //
  return maxIndex + (skipNumRows ? 0 : this.getNumRows());
};


Client.IdfPanel.prototype.getFrameList = function (flist)
{
  // TODO:
  // Get all subframes and push them into the list
  // -> the panel is already pushed
};


/**
 * Realize toolbar command set
 * @param {Object} cmsConf
 */
Client.IdfPanel.prototype.realizeCommandSet = function (cmsConf)
{
  // Get commands set zone
  let cmsZoneIdx = this.getCommandZone(Client.IdfPanel.commands.CZ_CMDSET);
  let cmsZoneConf = this.toolbarZonesConfig[cmsZoneIdx];
  let cmsZone = Client.eleMap[cmsZoneConf.id];
  //
  // Since empty zones are not created, add commands set zone if it's missing
  if (!cmsZone)
    cmsZone = Client.eleMap[this.toolbarConf.id].insertBefore({child: cmsZoneConf});
  //
  // Create command set
  cmsZone.insertBefore({child: cmsConf});
};


/**
 * Handle reset cache command
 * @param {Object} options
 */
Client.IdfPanel.prototype.resetCache = function (options)
{
  // Tell my groups to reset cache
  for (let i = 0; i < this.groups.length; i++)
    this.groups[i].resetCache(options);
  //
  // Tell my fields to reset cache
  for (let i = 0; i < this.fields.length; i++)
    this.fields[i].resetCache(options);
  //
  let from = options.from || 1;
  let to = options.to || this.totalRows;
  let dataBlockStart = options.dataBlockStart;
  let dataBlockEnd = options.dataBlockEnd;
  //
  // In case of grouped rows, convert indexes into grouped form
  if (this.hasGroupedRows()) {
    from = this.groupedRowsRoot.realIndexToGroupedIndex(from) - 1;
    to = this.groupedRowsRoot.realIndexToGroupedIndex(to);
    dataBlockStart = this.groupedRowsRoot.realIndexToGroupedIndex(dataBlockStart);
    dataBlockEnd = this.groupedRowsRoot.realIndexToGroupedIndex(dataBlockEnd);
  }
  //
  to = Math.max(to, this.getMaxRows(true));
  //
  if (dataBlockEnd)
    to = Math.max(to, dataBlockEnd);
  //
  // Remove rows elements from dom
  for (let i = from; i <= to; i++) {
    // If I have a data block coming after reset cache, I don't have to remove rows belonging to that block. It's better to reuse them
    if (i >= dataBlockStart && i <= dataBlockEnd)
      continue;
    //
    let row = this.rows[i];
    if (!row)
      continue;
    //
    this.detachRow(i, true);
  }
  //
  this.fillBufferVideo(dataBlockStart, dataBlockEnd);
};


/**
 * Return true if current row is new
 * @param {Integer} index
 */
Client.IdfPanel.prototype.isNewRow = function (index)
{
  return (index ?? this.getActiveRowIndex()) > this.totalRows;
};


/**
 * Update active page
 */
Client.IdfPanel.prototype.updateActivePage = function ()
{
  for (let i = 0; i < this.pages.length; i++) {
    let page = this.pages[i];
    page.updateElement({isActive: page.index === this.activePage});
  }
};


/**
 * Check if this panel can use row QBE
 */
Client.IdfPanel.prototype.canUseRowQbe = function ()
{
  return this.searchMode === Client.IdfPanel.searchModes.row && this.canSearch && this.hasList;
};


/**
 * Handle a resize event
 */
Client.IdfPanel.prototype.handleResize = function ()
{
  let events = [];
  if (!Client.mainFrame.isIDF)
    return events;
  //
  events.push(...Client.IdfFrame.prototype.handleResize.call(this));
  //
  if (this.hasList && this.layout === Client.IdfPanel.layouts.list) {
    rootObject = Client.eleMap[this.gridConf.id].getRootObject();
    let width = rootObject.clientWidth;
    let height = rootObject.clientHeight;
    //
    if (this.canAdaptWidth() && width !== this.lastGridWidth) {
      events.push({
        id: "resize",
        def: Client.IdfMessagesPump.eventTypes.ACTIVE,
        content: {
          oid: this.id,
          obn: "listwidth",
          par1: width,
          par2: 0
        }
      });
    }
    //
    if (this.canAdaptHeight() && height !== this.lastGridHeight) {
      let numRows = Math.floor((height - this.getHeaderHeight() - this.getQbeRowHeight()) / (this.getListRowHeight() + this.getListRowOffset()));
      events.push({
        id: "resize",
        def: Client.IdfMessagesPump.eventTypes.ACTIVE,
        content: {
          oid: this.id,
          obn: "height",
          par1: height,
          par2: numRows
        }
      });
    }
    //
    this.lastGridWidth = width;
    this.lastGridHeight = height;
  }
  //
  // TODO: il ridimensionamento degli oggetti interni al pannello (griglia e campi) viene gestito automaticamente tramite il flex-grow.
  // Quindi per adesso non gestiamo il resize sugli oggetti interni al pannello (griglia e campi)
  // e ci pensiamo quando gestiremo il resize e lo spostamento dei campi a run time.
  //
  // Call onResize method on my fields too
  //for (let i = 0; i < this.fields.length; i++)
  //  events.push(...this.fields[i].handleResize());
  //
  return events;
};


/**
 * Focus near control
 * @param {Object} options
 */
Client.IdfPanel.prototype.focusNearControl = function (options)
{
  let events = [];
  //
  let field = options.fieldValue.parentField;
  let row = options.fieldValue.index;
  if (options.column) {
    let fields = this.getFocusableFields(row);
    let idx = fields.findIndex(f => f === field);
    let nearIdx = idx + options.column;
    //
    // In the form layout I keep cycling through the fields
    if (this.layout === Client.IdfPanel.layouts.form)
      nearIdx = nearIdx % fields.length;
    //
    let nearField = fields[nearIdx];
    if (nearField) {
      // Focus near field
      nearField.focus({absoluteRow: row, selectAll: true});
      return events;
    }
    //
    // I got to the first or last column ... change row
    nearField = fields[options.column > 0 ? 0 : fields.length - 1];
    return this.focusNearControl({fieldValue: nearField.values[row], row: options.column});
  }
  //
  // I don't change line in layout form
  if (this.layout === Client.IdfPanel.layouts.form)
    return events;
  //
  let scrollUp = options.row < 0;
  row += options.row;
  //
  if (this.hasGroupedRows() && this.showGroups)
    row = this.getNextGroupedVisibleRow(row, scrollUp);
  //
  // I can't get out of the panel rows
  if (!this.isRowBetweenLimits(row))
    return events;
  //
  let rowToScroll = row + ((scrollUp ? -1 : 1) * 5);
  //
  if (this.hasGroupedRows() && this.showGroups)
    rowToScroll = this.getNextGroupedVisibleRow(rowToScroll, scrollUp);
  //
  if (this.isRowBetweenLimits(rowToScroll)) {
    // I ask the server to send it to me and I raise the row change event
    if (!field.values[rowToScroll]) {
      events.push(...this.handlePanelScroll(rowToScroll, scrollUp));
      Client.mainFrame.messagesPump.sendEvents(true);
    }
  }
  //
  // If the row I need to move to isn't there yet
  let nearFieldValue = field.values[row];
  if (nearFieldValue) {
    // Focus the field on the near row
    events.push(...this.handleDataRowClick(null, nearFieldValue));
    field.focus({absoluteRow: row, selectAll: true});
  }
  //
  return events;
};


/**
 * Get focusable fields
 * @@param {Number} row
 */
Client.IdfPanel.prototype.getFocusableFields = function (row)
{
  let fields = this.fields.filter(f => f.canHaveFocus(row));
  if (this.advancedTabOrder) {
    let layout = this.layout === Client.IdfPanel.layouts.form ? "form" : "list";
    fields.sort((f1, f2) => f1[layout + "TabOrderIndex"] - f2[layout + "TabOrderIndex"]);
  }
  return fields;
};


/**
 * Check if row is between limits
 * @@param {Number} row
 */
Client.IdfPanel.prototype.isRowBetweenLimits = function (row)
{
  return row >= (this.canUseRowQbe() ? 0 : 1) && row <= Math.max(this.totalRows, this.getNumRows());
};


/**
 * Update the mainContainerConf classes
 */
Client.IdfPanel.prototype.updateClassName = function ()
{
  let el = Client.eleMap[this.mainContainerConf.id];
  //
  let className = el.domObj.className;
  //
  // Clear current status
  className = className.replace("mode-list", "").replace("mode-form", "").replace("has-form", "").replace("has-list", "");
  //
  // Restore status
  className += this.layout === Client.IdfPanel.layouts.list ? " mode-list" : " mode-form";
  className += this.hasForm ? " has-form" : "";
  className += this.hasList ? " has-list" : "";
  //
  el.updateElement({className});
};


/**
 * Get click detail
 * @param {Object} event
 * @param {Widget} srcWidget
 */
Client.IdfPanel.prototype.getClickDetail = function (event, srcWidget)
{
  let detail = Client.IdfFrame.prototype.getClickDetail.call(this, event);
  //
  if (srcWidget instanceof Client.IdfControl)
    srcWidget = srcWidget.parent;
  //
  let field = -1;
  let row = -1;
  if (srcWidget instanceof Client.IdfFieldValue) {
    field = srcWidget.parentField.index;
    row = srcWidget.index;
  }
  else if (srcWidget instanceof Client.IdfField)
    field = srcWidget.index;
  //
  if (Client.mainFrame.isIDF) {
    detail.par4 = field;
    detail.par5 = row;
  }
  else {
    detail.field = field;
    detail.row = row;
  }
  //
  return detail;
};


/**
 * Create placeholder rows in order to avoid empty spaces on scrolling before new rows coming
 * @param {Integer} start
 * @param {Integer} end
 */
Client.IdfPanel.prototype.fillBufferVideo = function (start, end)
{
  // If there is no list, do nothing
  if (!this.hasList)
    return;
  //
  let numRows = this.getNumRows();
  //
  // If all rows are already visible, I don't need placeholder rows
  if (this.totalRows <= numRows)
    return;
  //
  // Set start row
  if (!start) {
    let dataBlockStart = this.actualPosition;
    start = this.hasGroupedRows() ? this.groupedRowsRoot.realIndexToGroupedIndex(dataBlockStart) || 1 : dataBlockStart;
  }
  //
  // Set end row
  if (!end) {
    let dataBlockEnd = this.actualPosition + this.getNumRows();
    end = this.hasGroupedRows() ? this.groupedRowsRoot.realIndexToGroupedIndex(dataBlockEnd) : dataBlockEnd;
  }
  //
  let maxIndex = this.getMaxRows(true);
  //
  let placeholderStart = start - numRows;
  let placeholderEnd = end + numRows;
  //
  if (placeholderStart < 1)
    placeholderStart = 1;
  if (placeholderEnd > maxIndex)
    placeholderEnd = maxIndex;
  //
  // In order to fill buffer video when rows are grouped, I cannot trust placeholder start and end.
  // I have eventually to move them respectively backward and forward to ensure buffer video filling
  if (this.hasGroupedRows()) {
    let boundaries = this.getGroupedBoundaries(placeholderStart, placeholderEnd);
    placeholderStart = boundaries.newPlaceholderStart || placeholderStart;
    placeholderEnd = boundaries.newPlaceholderEnd || placeholderEnd;
  }
  //
  // Attach rows skiping rows belonging to collapsed groups
  for (let i = placeholderStart; i <= placeholderEnd; i++) {
    if (this.hasGroupedRows() && !this.groupedRowsRoot.isRowVisible(i))
      continue;
    //
    this.attachRow(i);
  }
  //
  // Now detach rows out of scroll range
  let max = this.getMaxRows();
  for (let i = 1; i <= max; i++) {
    if (i >= placeholderStart && i <= placeholderEnd)
      continue;
    //
    this.detachRow(i);
  }
  //
  // Finally adjust scrollbar
  this.adjustScrollbar(start, end);
};


/**
 * Create placeholder row configuration
 * @param {Integer} index
 */
Client.IdfPanel.prototype.createPlaceholderRowConf = function (index)
{
  let realIndex = index;
  //
  if (this.hasGroupedRows())
    realIndex = this.groupedRowsRoot.groupedIndexToRealIndex(index);
  //
  // I need a base data row in order to clone it and obtain an equivalent placeholder row.
  // If I have to create an even row, get an even base data row, otherwise get an odd one
  let baseDataRow;
  let start = this.actualPosition;
  let end = start + this.getNumRows();
  for (let i = start; i <= end; i++) {
    let row = this.rows[i];
    //
    if (!row || !row.isData || this.detachedRows[row.id])
      continue;
    //
    let realPos = i;
    if (this.hasGroupedRows())
      realPos = this.groupedRowsRoot.groupedIndexToRealIndex(i);
    //
    if (!!(realPos % 2) === !!(realIndex % 2) && realPos !== this.getActiveRowIndex()) {
      baseDataRow = row;
      break;
    }
  }
  //
  if (!baseDataRow)
    return;
  //
  // Create a placeholder row by cloning base data row
  return this.cloneDataRow({el: Client.eleMap[baseDataRow.id], index, level: 1});
};


/**
 * Create qbe row
 */
Client.IdfPanel.prototype.createQbeRow = function ()
{
  this.qbeRowConf = this.createDataRowConfig(0);
  this.qbeRowConf.className += " panel-list-qbe-row";
  //
  // Qbe row position is "sticky" as well as header row.
  // Thus I have to set its top to header height since I want it to be placed under header row
  this.qbeRowConf.style = {top: this.getHeaderHeight() + "px"};
  //
  // Get grid element
  let grid = Client.eleMap[this.gridConf.id];
  //
  // Create qbe row element and insert it before second row, if any (first row is the header)
  grid.insertBefore({child: this.qbeRowConf, sib: grid.elements[1]?.id});
  //
  for (let i = 0; i < this.fields.length; i++) {
    let f = this.fields[i];
    if (!f.isInList())
      continue;
    //
    f.updateControls({all: true}, 0);
    f.applyVisualStyle(0);
  }
};


/**
 * Called when a Field is moved, movedfield will be moved before nextField
 * @param {Client.IdfField} movedField
 * @param {Client.IdfField} nextField
 */
Client.IdfPanel.prototype.reorderList = function (movedField, nextField)
{
  // If movedField and nextField are the same, do nothing
  if (movedField === nextField)
    return;
  //
  // When dragging a field on another group the target must be the first field of the group
  if (nextField.group && movedField.group !== nextField.group)
    nextField = nextField.group.fields[0];
  //
  // If I moved a group field ALL the group fields MUST be moved
  let mFields = [];
  if (movedField.group && movedField.group !== nextField.group)
    movedField.group.forEach(element => mFields.push(element));
  else
    mFields.push(movedField);
  //
  // Check if next field is fixed
  let isNextFixed = this.isFixedField(nextField);
  //
  let events = [];
  for (let i = 0; i < mFields.length; i++) {
    // Check if moved field is fixed
    let isMovedFixed = this.isFixedField(mFields[i]);
    //
    // If I move a not fixed field before a fixed field, it becomes fixed too.
    // Otherwise if I move a fixed field before a not fixed field, it becomes not fixed too
    if (!isMovedFixed && isNextFixed)
      this.fixedColumns++;
    else if (isMovedFixed && !isNextFixed)
      this.fixedColumns--;
    //
    // First: handle the fields array, remove the target and insert him before the nextfield
    this.fields.splice(this.fields.indexOf(mFields[i]), 1);
    this.fields.splice(this.fields.indexOf(nextField), 0, mFields[i]);
    //
    // Then: same thing with the elements array
    this.elements.splice(this.elements.indexOf(mFields[i]), 1);
    this.elements.splice(this.elements.indexOf(nextField), 0, mFields[i]);
    //
    // If the target AND the source are in the same groups we need to do the same thing on the group
    if (nextField.group && mFields[i].group === nextField.group) {
      // First: handle the fields array
      nextField.group.fields.splice(nextField.group.fields.indexOf(mFields[i]), 1);
      nextField.group.fields.splice(nextField.group.fields.indexOf(nextField), 0, mFields[i]);
      //
      // Then: same thing with the elements array
      nextField.group.elements.splice(nextField.group.elements.indexOf(mFields[i]), 1);
      nextField.group.elements.splice(nextField.group.elements.indexOf(nextField), 0, mFields[i]);
    }
    //
    // Now reorder values rows
    for (let j = 0; j < mFields[i].values.length; j++) {
      // Get j-th values of both moved and next fields
      let movedValue = mFields[i].values[j];
      let nextValue = nextField.values[j];
      //
      // If one of them does not exits, do nothing
      if (!movedValue || !nextValue)
        continue;
      //
      // Move movedValue before nextValue
      let movedValueDomObj = movedValue.listContainer?.getRootObject();
      let nextValueDomObj = nextValue.listContainer?.getRootObject();
      if (movedValueDomObj && nextValueDomObj)
        movedValueDomObj.parentNode.insertBefore(movedValueDomObj, nextValueDomObj);
    }
    //
    if (Client.mainFrame.isIDF) {
      events.push({
        id: "rdcol",
        def: Client.IdfMessagesPump.eventTypes.ACTIVE,
        content: {
          oid: this.id,
          obn: "",
          par1: mFields[i].id,
          par2: nextField.id
        }
      });
    }
    else {
      // TODO
    }
  }
  //
  // Enable advanced tab order and adjust it
  this.advancedTabOrder = true;
  this.fields.forEach((f, i) => f.listTabOrderIndex = i);
  //
  // Now we can send the message to the server AND recreate all the client
  Client.mainFrame.sendEvents(events);
  //
  this.updateStructure();
  this.calcLayout();
};


/**
 * Handle row change
 * @param {number} rowIndex
 * @param {Boolean} rowSelectorClick
 */
Client.IdfPanel.prototype.handleRowChange = function (rowIndex, rowSelectorClick)
{
  let events = [];
  //
  // If this is not a real data row change, do nothing
  if (!rowSelectorClick && rowIndex === this.getActiveRowIndex())
    return events;
  //
  this.updateElement({actualRow: rowIndex - this.actualPosition, skipScroll: true});
  //
  if (rowSelectorClick) {
    if (Client.mainFrame.isIDF) {
      events.push({
        id: "panrs",
        def: this.rowSelectEventDef,
        content: {
          oid: this.id,
          obn: this.actualRow,
          par1: this.hasGroupedRows() ? rowIndex : 0
        }
      });
    }
    else {
      events.push({
        id: "onActivatingRow",
        obj: this.id,
        content: {
          row: this.actualRow
        }
      });
    }
  }
  else {
    if (Client.mainFrame.isIDF) {
      events.push({
        id: "chgrow",
        def: this.scrollEventDef,
        content: {
          oid: this.id,
          par1: this.actualRow,
          par2: this.hasGroupedRows() ? rowIndex : 0
        }
      });
    }
    else {
      events.push({
        id: "chgProp",
        obj: this.id,
        content: {
          name: "actualRow",
          value: rowIndex - 1,
          clid: Client.id
        }
      });
    }
  }
  //
  return events;
};


/**
 * Handle panel scroll
 * @param {number} rowIndex
 * @param {boolean} scrollUp
 */
Client.IdfPanel.prototype.handlePanelScroll = function (rowIndex, scrollUp)
{
  // Request data starting from newActualPosition using a negative or positive offset depending on whether I'm scrolling up or down
  // In this way I avoid empty space while scrolling smoothly
  let newActualPosition = this.hasGroupedRows() ? this.groupedRowsRoot.groupedIndexToRealIndex(rowIndex) : rowIndex;
  let newActualRow = this.getActiveRowIndex() - newActualPosition;
  //
  this.updateElement({actualPosition: newActualPosition, actualRow: newActualRow, skipScroll: true});
  //
  return [{
      id: "panscr",
      def: this.scrollEventDef,
      content: {
        oid: this.id,
        par1: this.actualPosition,
        par2: (scrollUp ? -1 : 1) * this.getNumRows()
      }
    }];
};


/**
 * Check if given field is fixed
 * @param {Client.IdfField} field
 */
Client.IdfPanel.prototype.isFixedField = function (field)
{
  // If there aren't fixed columns, field is not fixed
  if (!this.fixedColumns)
    return;
  //
  // Get fields that can be fixed
  let canBeFixed = this.fields.filter(f => f.isInList() && f.isVisible());
  //
  // Prevent fixedColumns to be greater than number of fields that can be fixed
  let fixedColumns = Math.min(canBeFixed.length, this.fixedColumns);
  //
  for (let i = 0; i < fixedColumns; i++) {
    let f = canBeFixed[i];
    //
    // If current field can adapt in width, it cannot be fixed.
    // Since it interrupts the consecutive fixed fields row and I didn't find given field yet, it's not fixed
    if (f.canAdaptWidth())
      return;
    //
    // If I found given field, it means it's fixed
    if (f === field)
      return true;
  }
};


/**
 * If given field is fixed, get its left
 * @param {Client.IdfField} field
 */
Client.IdfPanel.prototype.getFixedFieldLeft = function (field)
{
  let fixedFieldsWidth = 0;
  //
  if (this.showRowSelector)
    fixedFieldsWidth += Client.eleMap[this.rowSelectorColumnConf.id].getRootObject().offsetWidth;
  //
  for (let i = 0; i < this.fields.length; i++) {
    let f = this.fields[i];
    //
    // If I reach the field I was looking for, eventually add row selector width and then break
    if (f === field)
      break;
    //
    // Add current field width to fixed fields width
    if (this.isFixedField(f))
      fixedFieldsWidth += f.getRects({checkVisibility: true}).width;
  }
  //
  return fixedFieldsWidth;
};


/**
 * Set rows min width in order to handle fixed columns properly
 */
Client.IdfPanel.prototype.setRowsMinWidth = function ()
{
  let gridEl = Client.eleMap[this.gridConf.id];
  if (!gridEl)
    return;
  //
  let minRowWidth = 0;
  //
  for (let i = 0; i < this.fields.length; i++) {
    let field = this.fields[i];
    if (!field.isInList())
      continue;
    //
    minRowWidth += field.canAdaptWidth() ? Client.IdfField.minWidth : field.getRects({checkVisibility: true}).width;
  }
  //
  if (minRowWidth) {
    let gridObj = gridEl.getRootObject();
    for (let i = 0; i < gridObj.childNodes.length; i++)
      gridObj.childNodes[i].style.minWidth = minRowWidth + "px";
  }
};


/**
 * Check if among xmlNode siblings there is a "chg" node referring to a panel
 * @param {XmlNode} xmlNode
 */
Client.IdfPanel.getDataRange = function (xmlNode)
{
  let start;
  let end;
  //
  let panelId = xmlNode.getAttribute("id");
  //
  // After a reset cache command on a panel, server may sends a block of data.
  // Reset cache on a panel should remove values whose index is included into given range ("from" and "to" properties).
  // But if after an "rcache" command there is a "chg" command on the same panel containing some values, I have to avoid removing those values.
  // So look for a "chg" command referring to same panel as "rcache" command and find the range of values that don't have to be removed
  for (let i = 0; i < xmlNode.parentNode.childNodes.length; i++) {
    let sib = xmlNode.parentNode.childNodes[i];
    //
    // If current sibling is not "chg", continue
    if (sib.nodeName !== "chg")
      continue;
    //
    // Get sibling id
    let id = sib.getAttribute("id");
    //
    // If current sibling doesn't refers to a panel or refers to another panel, continue
    if (!id?.startsWith("pan") || id !== panelId)
      continue;
    //
    // Get dataBlockStart and dataBlockEnd from panel configuration
    let panelConf = Client.IdfPanel.createConfigFromXml(sib);
    if (panelConf) {
      start = panelConf.dataBlockStart;
      end = panelConf.dataBlockEnd;
      break;
    }
  }
  //
  return {start, end};
};


/*
 * Called ONE time at the inizialization, decides to show the list filter OR the icon to handle the list field choice
 */
Client.IdfPanel.prototype.initializeListFilters = function ()
{
  this.showListVisisiblityControls = false;
  if (!Client.mainFrame.idfMobile) {
    for (let i = 0; i < this.fields.length; i++)
      this.showListVisisiblityControls ||= this.fields[i].canHideInList;
  }
  //
  let el = Client.eleMap[this.mainContainerConf.id].getRootObject();
  if (this.showListVisisiblityControls)
    el.setAttribute("has-list-hide", "true");
  else if (this.searchMode === Client.IdfPanel.searchModes.header)
    el.setAttribute("has-list-filter", "true");
};


/**
 * Set given rows group as root
 * @param {Client.IdfRowsGroup} rowsGroup
 */
Client.IdfPanel.prototype.setGroupedRowsRoot = function (rowsGroup)
{
  this.groupedRowsRoot = rowsGroup;
};


/**
 * Check if panel is showing rows groups
 */
Client.IdfPanel.prototype.hasGroupedRows = function ()
{
  return !!this.groupedRowsRoot;
};


/**
 * Reset rows groups
 */
Client.IdfPanel.prototype.resetGroupedRows = function ()
{
  if (!this.groupedRowsRoot)
    return;
  //
  // Existing rows have an index that takes in account groups. So convert it to real format
  let newRows = [];
  for (let i = 0; i < this.rows.length; i++) {
    let row = this.rows[i];
    //
    if (!row)
      continue;
    //
    if (!row.isData) {
      this.detachRow(i, true);
      continue;
    }
    //
    let realIndex = this.groupedRowsRoot.groupedIndexToRealIndex(i);
    newRows[realIndex] = row;
  }
  //
  // Replace grouped rows array with its not grouped version
  this.rows = newRows.slice();
  //
  // Close groups
  this.groupedRowsRoot.close();
  //
  delete this.groupedRowsRoot;
  //
  this.fillBufferVideo();
};


/**
 * Reset margin top and margin bottom of list rows
 */
Client.IdfPanel.prototype.resetRowsMargins = function ()
{
  let grid = Client.eleMap[this.gridConf.id];
  //
  for (let i = 0; i < grid.elements.length; i++) {
    let row = grid.elements[i];
    //
    if (row.id === this.gridHeaderConf.id || row.id === this.qbeRowConf?.id || row.id === this.aggregateRowConf.id)
      continue;
    //
    row.updateElement({style: {marginTop: "", marginBottom: ""}});
    Client.Widget.updateElementClassName(row, "first-row", true);
  }
};


/**
 * Get first not detached row
 */
Client.IdfPanel.prototype.getFirstGridRow = function ()
{
  let grid = Client.eleMap[this.gridConf.id].getRootObject();
  //
  let firstRow;
  for (let i = 0; i < grid.childNodes.length; i++) {
    let rowObj = grid.childNodes[i];
    //
    if (rowObj.id === this.gridHeaderConf.id || rowObj.id === this.qbeRowConf?.id || rowObj.id === this.aggregateRowConf.id)
      continue;
    //
    firstRow = Client.eleMap[rowObj.id];
    Client.Widget.updateElementClassName(firstRow, "first-row");
    break;
  }
  //
  return firstRow;
};


/**
 * Get last not detached row
 */
Client.IdfPanel.prototype.getLastGridRow = function ()
{
  let grid = Client.eleMap[this.gridConf.id].getRootObject();
  //
  let lastRow;
  for (let i = grid.childNodes.length - 1; i >= 0; i--) {
    let rowObj = grid.childNodes[i];
    //
    if (rowObj.id === this.gridHeaderConf.id || rowObj.id === this.qbeRowConf?.id || rowObj.id === this.aggregateRowConf.id)
      continue;
    //
    lastRow = Client.eleMap[rowObj.id];
    break;
  }
  //
  return lastRow;
};


/**
 * Get margin top for given row
 * @param {Object} row
 */
Client.IdfPanel.prototype.getRowMarginTop = function (row)
{
  let marginTop;
  //
  if (!row)
    return marginTop;
  //
  let rowIndex = this.rows.findIndex(rowObj => rowObj?.id === row.id);
  //
  let rowHeight = this.getListRowHeight();
  //
  if (this.hasGroupedRows())
    marginTop = this.groupedRowsRoot.getRowMarginTop(rowIndex, rowHeight);
  else
    marginTop = (rowIndex - 1) * rowHeight;
  //
  return marginTop;
};


/**
 * Get margin bottom for given row
 * @param {Object} row
 */
Client.IdfPanel.prototype.getRowMarginBottom = function (row)
{
  let marginBottom;
  //
  if (!row)
    return marginBottom;
  //
  let rowIndex = this.rows.findIndex(rowObj => rowObj?.id === row.id);
  //
  let rowHeight = this.getListRowHeight();
  //
  if (this.hasGroupedRows())
    marginBottom = this.groupedRowsRoot.getRowMarginBottom(rowIndex, rowHeight);
  else {
    marginBottom = (this.totalRows - rowIndex) * rowHeight;
    marginBottom = marginBottom < 0 ? 0 : marginBottom;
  }
  //
  return marginBottom;
};


/**
 * Get start and end boundaries in order to ensure filling buffer video when grouping is enabled
 * @param {Integer} placeholderStart
 * @param {Integer} placeholderEnd
 */
Client.IdfPanel.prototype.getGroupedBoundaries = function (placeholderStart, placeholderEnd)
{
  // Count how many visible rows there are in given boundaries
  let visibleRows = 0;
  for (let i = placeholderStart; i <= placeholderEnd; i++) {
    if (this.groupedRowsRoot.isRowVisible(i))
      visibleRows++;
  }
  //
  // Get boundaries size
  let size = (placeholderEnd - placeholderStart) + 1;
  //
  // If there are residual rows, it means that some rows in given boundaries are not visible (i.e. their groups are collapsed)
  let residualRows = size - visibleRows;
  //
  // If no residual rows, do nothing
  if (residualRows === 0)
    return {};
  //
  let maxIndex = this.getMaxRows(true);
  //
  // If there are residual rows, try to move boundaries in order to include more visible rows
  let newPlaceholderStart;
  let newPlaceholderEnd;
  //
  // Split residual rows in two half in order to include rows both on bottom side and on top side
  let halfBottom = parseInt(residualRows / 2);
  let halfTop = residualRows - halfBottom;
  //
  // Try to move placeholderStart backward until I include all half bottom rows
  for (let i = placeholderStart - 1; i >= 1; i--) {
    if (halfBottom === 0)
      break;
    //
    if (!this.groupedRowsRoot.isRowVisible(i))
      continue;
    //
    halfBottom--;
    newPlaceholderStart = i;
  }
  //
  // Try to move placeholderEnd forward until I include all half top rows
  for (let i = placeholderEnd + 1; i <= maxIndex; i++) {
    if (halfTop === 0)
      break;
    //
    if (!this.groupedRowsRoot.isRowVisible(i))
      continue;
    //
    halfTop--;
    newPlaceholderEnd = i;
  }
  //
  // If there is no more space on top, try to move the residual halfTop on bottom
  if (halfTop > 0 && halfBottom === 0) {
    placeholderStart = newPlaceholderStart;
    //
    for (let i = placeholderStart - 1; i >= 1; i--) {
      if (halfTop === 0)
        break;
      //
      if (!this.groupedRowsRoot.isRowVisible(i))
        continue;
      //
      halfTop--;
      newPlaceholderStart = i;
    }
  }
  else if (halfBottom > 0 && halfTop === 0) { // Otherwise try to move the residual halfBottom on top
    placeholderEnd = newPlaceholderEnd;
    for (let i = placeholderEnd + 1; i <= maxIndex; i++) {
      if (halfBottom === 0)
        break;
      //
      if (!this.groupedRowsRoot.isRowVisible(i))
        continue;
      //
      halfBottom--;
      newPlaceholderEnd = i;
    }
  }
  //
  return {newPlaceholderStart, newPlaceholderEnd};
};


/**
 * Get rows group by index
 * @param {Integer} index
 */
Client.IdfPanel.prototype.getRowsGroupByIndex = function (index)
{
  return Client.eleMap[this.groupedRowsRoot.groupsIds[index]];
};


/**
 * Get number of visible rows in grouped mode
 * @param {Integer} limit
 */
Client.IdfPanel.prototype.getVisibleGroupedRows = function (limit)
{
  let visibleRows = 0;
  let rowsCount = this.groupedRowsRoot.groupedEndingRow;
  for (let i = 1; i <= rowsCount; i++) {
    if (i > limit)
      break;
    //
    // If current row exists and it's not detached, it is the row I was finding
    if (this.rows[i]?.id && this.groupedRowsRoot.isRowVisible(i))
      visibleRows++;
  }
  //
  return visibleRows;
};


/**
 * Get index of next grouped visible row starting from start
 * @param {Integer} start
 * @param {Boolean} scrollUp
 */
Client.IdfPanel.prototype.getNextGroupedVisibleRow = function (start, scrollUp)
{
  if (start <= 0)
    return start;
  //
  let maxRows = this.getMaxRows(true);
  //
  // Convert real index to grouped one
  let rowIndex = this.groupedRowsRoot.realIndexToGroupedIndex(start);
  //
  // Search first visible data row starting from rowIndex
  while (this.rows[rowIndex]?.isGroupHeader || !this.groupedRowsRoot.isRowVisible(rowIndex)) {
    rowIndex = scrollUp ? rowIndex - 1 : rowIndex + 1;
    //
    if (rowIndex < 0 || rowIndex > maxRows)
      break;
  }
  //
  // Convert grouped index to real
  rowIndex = rowIndex < 0 ? -1 : this.groupedRowsRoot.groupedIndexToRealIndex(rowIndex);
  //
  return rowIndex;
};


/**
 * Open list control popup
 * @param {Client.IdfField} selectedField
 */
Client.IdfPanel.prototype.openControlListPopup = function (selectedField)
{
  let callback = function (r, values) {
    let events = [];
    //
    // Check the resut : if r is NULL the user clicked outside the popup to close it, in this case we need to apply the field visibility
    // 1 - open filter popup
    if (r === 1)
      selectedField.openFilterPopup();
    else if (r === 2)
      events.push(...selectedField.handleSort(Client.IdfField.sortModes.DESC));
    else if (r === 3)
      events.push(...selectedField.handleSort(Client.IdfField.sortModes.ASC));
    else if (r === 4)
      events.push(...selectedField.handleSort(Client.IdfField.sortModes.NONE));
    else if (r === 5)
      events.push(...selectedField.handleGrouping(Client.IdfField.groupingModes.DESC));
    else if (r === 6)
      events.push(...selectedField.handleGrouping(Client.IdfField.groupingModes.ASC));
    else if (r === 7)
      events.push(...selectedField.handleGrouping(Client.IdfField.groupingModes.NONE));
    else if (!r) {
      // Apply the field configurations
      for (let k in values) {
        let f = Client.eleMap[k];
        f.hiddenInList = !values[k];
      }
      //
      this.updateStructure();
      this.calcLayout();
    }
    //
    if (events.length > 0)
      Client.mainFrame.sendEvents(events);
  }.bind(this);
  //
  let buttons = [];
  let inputs = [];
  let alertOpts = {
    style: "popup-list-controls",
    inputs,
    buttons,
    rect: {
      refId: Client.eleMap[selectedField.listContainerId].getRootObject().id
    }
  };
  //
  // Add the command to show the filter
  if (this.status === Client.IdfPanel.statuses.data && !this.DOModified && this.searchMode === Client.IdfPanel.searchModes.header && this.canSearch && selectedField.enabledInQbe && selectedField.type !== Client.IdfField.types.STATIC && selectedField.dataType !== Client.IdfField.dataTypes.BLOB)
    buttons.push({id: 1, text: Client.IdfResources.t("LFIL_FILTER_CAPT"), icon: "funnel"});
  if (selectedField.canSort) {
    buttons.push({id: 2, text: Client.IdfResources.t("LFIL_SORT_DESC"), icon: "arrow-dropdown"});
    buttons.push({id: 3, text: Client.IdfResources.t("LFIL_SORT_ASC"), icon: "arrow-dropup"});
    buttons.push({id: 4, text: Client.IdfResources.t("LFIL_SORT_CLEAR"), icon: "close-circle-outline"});
    //
    if (this.canGroup && this.showGroups) {
      buttons.push({id: 5, text: Client.IdfResources.t("LFIL_GROUP_LBL"), icon: "arrow-dropdown"});
      buttons.push({id: 6, text: Client.IdfResources.t("LFIL_GROUP_LBL_D"), icon: "arrow-dropup"});
      buttons.push({id: 7, text: Client.IdfResources.t("LFIL_DEGROUP_LBL"), icon: "close-circle-outline"});
    }
  }
  //
  for (let i = 0; i < this.fields.length; i++) {
    let f = this.fields[i];
    if (f.isInList() && f.visible && f.canHideInList)
      inputs.push({id: f.id, type: "checkbox", label: f.listHeader, checked: !f.hiddenInList, value: f.id});
  }
  //
  Client.IonHelper.createAlert(alertOpts, callback);
};