/*
 * Instant Developer Cloud
 * Copyright Pro Gamma Spa 2000-2021
 * All rights reserved
 */

var Client = Client || {};


/**
 * @class A panel field
 * @param {Object} widget
 * @param {View|Element} parent - the parent element
 * @param {View} view
 */
Client.IdfField = function (widget, parent, view)
{
  // Add this field to panel fields array
  parent.addField(this);
  //
  this.values = [];
  this.visibleValuesCount = 0;
  //
  // Set default values
  widget = Object.assign({
    listTop: 0,
    listLeft: 0,
    formTop: 0,
    formLeft: 0,
    maxLength: 255,
    sortMode: Client.IdfField.sortModes.NONE,
    groupingMode: Client.IdfField.groupingModes.NONE,
    type: Client.IdfField.types.MASTER,
    visible: true,
    showInList: true,
    showInForm: true,
    inList: true,
    showListHeader: true,
    listHeaderAbove: false,
    showFormHeader: true,
    listNumRows: 1,
    formNumRows: 1,
    formHeaderAbove: false,
    causeValidation: true,
    canActivate: false,
    hasValueSource: false,
    alignment: -1,
    enabled: true,
    activableDisabled: false,
    superActive: false,
    unbound: false,
    canSort: true,
    editorType: Client.IdfField.editorTypes.NORMAL,
    scale: 0,
    showHtmlEditorToolbar: true,
    multiUpload: false,
    maxUploadSize: 10 * 1024 * 1024,
    maxUploadFiles: 0,
    uploadExtensions: "*.*",
    uploadDescription: "",
    image: "",
    mask: "",
    imageResizeMode: Client.IdfField.stretches.REPEAT,
    autoLookup: false,
    smartLookup: false,
    optional: true,
    pageIndex: 0,
    enabledInQbe: true,
    comboMultiSel: true,
    comboSeparator: ";",
    notifySelectionChange: false,
    aggregateOfField: -1,
    listTabOrderIndex: -1,
    formTabOrderIndex: -1,
    controlType: !Client.mainFrame.isIDF ? Client.IdfField.controlTypes.EDIT : undefined,
    clickEventDef: Client.IdfMessagesPump?.eventTypes.ACTIVE,
    changeEventDef: Client.IdfMessagesPump?.eventTypes.DEFERRED
  }, widget);
  //
  // Set original dimensions and position
  this.orgListWidth = widget.listWidth;
  this.orgListWidthPerc = widget.listWidthPerc;
  this.orgListHeight = widget.listHeight;
  this.orgListHeightPerc = widget.listHeightPerc;
  this.orgListLeft = widget.listLeft;
  this.orgListLeftPerc = widget.listLeftPerc;
  this.orgListTop = widget.listTop;
  this.orgListTopPerc = widget.listTopPerc;
  this.orgFormWidth = widget.formWidth;
  this.orgFormWidthPerc = widget.formWidthPerc;
  this.orgFormHeight = widget.formHeight;
  this.orgFormHeightPerc = widget.formHeightPerc;
  this.orgFormLeft = widget.formLeft;
  this.orgFormLeftPerc = widget.formLeftPerc;
  this.orgFormTop = widget.formTop;
  this.orgFormTopPerc = widget.formTopPerc;
  //
  widget.children = widget.children || [];
  //
  // If parent panel use row qbe, create a special field value (having "0" as index) to handle filter on parent field
  if (parent.canUseRowQbe() && widget.inList) {
    let rowQbeValue = this.createElementConfig({c: "IdfFieldValue", index: 0, isRowQbe: true});
    widget.children.push(rowQbeValue);
  }
  //
  // If this is a static field, create a child IdfFieldValue (IDF do not send me a value in this case)
  if (widget.type === Client.IdfField.types.STATIC) {
    // Static field doesn't have to show activator
    widget.activatorWidth = 0;
    //
    let fieldValue = this.createElementConfig({c: "IdfFieldValue", index: 1});
    widget.children.push(fieldValue);
    //
    // If field has a subFrame, it is located in parent idfView's subFrames
    if (widget.subFrameId) {
      let subFrameConf = parent.parentIdfView?.getSubFrame(widget.subFrameId);
      if (subFrameConf) {
        this.subFrameConf = subFrameConf;
        this.subFrameConf.isSubFrame = true;
      }
    }
  }
  //
  Client.Widget.call(this, widget, parent, view);
  //
  // Append the subframe between my children after the base constructor
  // because elements are initialized in it
  if (this.subFrame)
    this.elements.push(this.subFrame);
};


// Make Client.IdfField extend Client.Widget
Client.IdfField.prototype = new Client.Widget();


Client.IdfField.transPropMap = {
  idx: "index",
  vis: "visible",
  inl: "showInList",
  inf: "showInForm",
  pag: "pageIndex",
  lli: "inList",
  hdr: "header",
  lih: "listHeader",
  lwi: "listWidth",
  lhe: "listHeight",
  lle: "listLeft",
  lto: "listTop",
  lhr: "listResizeWidth",
  lvr: "listResizeHeight",
  foh: "formHeader",
  fwi: "formWidth",
  fhe: "formHeight",
  fle: "formLeft",
  fto: "formTop",
  fhr: "formResizeWidth",
  fvr: "formResizeHeight",
  hdl: "showListHeader",
  hla: "listHeaderAbove",
  lhs: "listHeaderSize",
  hdf: "showFormHeader",
  hfa: "formHeaderAbove",
  fhs: "formHeaderSize",
  lnr: "listNumRows",
  fnr: "formNumRows",
  dat: "dataType",
  max: "maxLength",
  smo: "sortMode",
  gro: "groupingMode",
  idp: "type",
  cva: "causeValidation",
  act: "canActivate",
  aci: "activatorImage",
  acw: "activatorWidth",
  qvs: "hasValueSource",
  aln: "alignment",
  bkc: "backColor",
  frc: "color",
  msk: "mask",
  ftm: "fontModifiers",
  ena: "enabled",
  acd: "activableDisabled",
  sac: "superActive",
  unb: "unbound",
  srt: "canSort",
  edi: "editorType",
  fsc: "scale",
  uet: "showHtmlEditorToolbar",
  mup: "multiUpload",
  mus: "maxUploadSize",
  muf: "maxUploadFiles",
  uex: "uploadExtensions",
  uds: "uploadDescription",
  img: "image",
  irm: "imageResizeMode",
  alo: "autoLookup",
  lke: "smartLookup",
  opt: "optional",
  sub: "subFrameId",
  gru: "groupId",
  qen: "enabledInQbe",
  cms: "comboMultiSel",
  aof: "aggregateOfField",
  qbf: "qbeFilter",
  vfl: "visualFlags",
  cvs: "comboSeparator",
  chg: "changeEventDef",
  wtm: "placeholder",
  uts: "notifySelectionChange",
  lta: "listTabOrderIndex",
  fta: "formTabOrderIndex",
  oqbf: "needFilterPopup",
  ocb: "needToOpenCombo",
  cmd: "command"
};


Client.IdfField.resizeModes = {
  NONE: 1,
  MOVE: 2,
  STRETCH: 3
};


Client.IdfField.types = {
  STATIC: -1,
  MASTER: 0,
  LOOKUP: 1
};


Client.IdfField.editorTypes = {
  NORMAL: 0,
  HTMLEDITOR: 1
};


Client.IdfField.dataTypes = {
  UNSPECIFIED: 0,
  INTEGER: 1,
  FLOAT: 2,
  DECIMAL: 3,
  CURRENCY: 4,
  CHARACTER: 5,
  DATE: 6,
  TIME: 7,
  DATETIME: 8,
  TEXT: 9,
  BLOB: 10,
  BOOLEAN: 11,
  FIXED_CHARACTER: 12,
  OBJECT: 13
};


Client.IdfField.stretches = {
  REPEAT: 1,
  CENTER: 2,
  FIT: 3
};


Client.IdfField.sortModes = {
  ASC: -1,
  NONE: 0,
  DESC: 1
};


Client.IdfField.groupingModes = {
  ASC: -1,
  NONE: 0,
  DESC: 1
};


Client.IdfField.defaultHeight = 32;
Client.IdfField.minWidth = 20;

Client.IdfField.controlTypes = {AUTO: 1, EDIT: 2, COMBO: 3, CHECK: 4, OPTION: 5, BUTTON: 6, HTMLEDITOR: 7, CUSTOM: 8, BLOB: 10, CHECKLIST: 11, LISTGROUPHEADER: 111};


Object.defineProperty(Client.IdfField.prototype, "index", {
  get: function () {
    return this.parent.fields.indexOf(this);
  }
});


/**
 * Create element configuration from xml
 * @param {XmlNode} xml
 */
Client.IdfField.createConfigFromXml = function (xml)
{
  let config = {};
  //
  let attrList = xml.attributes;
  for (let i = 0; i < attrList.length; i++) {
    if (attrList[i].nodeName === "pdp") {
      // Array of properties that handle percentage dimensions
      let percNames = [
        "listLeftPerc", "listTopPerc", "formLeftPerc", "formTopPerc",
        "listWidthPerc", "listHeightPerc", "listHeaderSizePerc",
        "formWidthPerc", "formHeightPerc", "formHeaderSizePerc"
      ];
      //
      // Get percentage values
      let percValues = attrList[i].nodeValue.split(",");
      //
      // Assign percentage value to the equivalent percentage name, skipping "-1" values
      for (let j = 0; j < percValues.length; j++) {
        let perc = parseInt(percValues[j]);
        if (perc === -1)
          continue;
        //
        // Server sends me "110" for "11%", so divide value by 10 to obtain percentage value
        config[percNames[j]] = perc / 10;
      }
      break;
    }
  }
  //
  // Pre-calculate values config to make it easier for IdfPanel to find the beginning and end block indexes
  // when a new set of data comes from server
  config.valuesConfig = [];
  for (let i = 0; i < xml.childNodes.length; i++) {
    let child = xml.childNodes[i];
    //
    // I'm only interested in "val" nodes
    if (child.nodeName !== "val")
      continue;
    //
    // Create val configuration
    let valConfig = {};
    attrList = child.attributes;
    for (let j = 0; j < attrList.length; j++) {
      let attrNode = attrList[j];
      valConfig[attrNode.nodeName] = attrNode.nodeValue;
    }
    //
    // Push it into valuesConfig
    config.valuesConfig.push(valConfig);
  }
  //
  return config;
};


/**
 * Convert properties values
 * @param {Object} props
 */
Client.IdfField.convertPropValues = function (props)
{
  props = props || {};
  //
  for (let p in props) {
    switch (p) {
      case Client.IdfField.transPropMap.idx:
      case Client.IdfField.transPropMap.pag:
      case Client.IdfField.transPropMap.max:
      case Client.IdfField.transPropMap.smo:
      case Client.IdfField.transPropMap.gro:
      case Client.IdfField.transPropMap.idp:
      case Client.IdfField.transPropMap.lwi:
      case Client.IdfField.transPropMap.lhe:
      case Client.IdfField.transPropMap.lle:
      case Client.IdfField.transPropMap.lto:
      case Client.IdfField.transPropMap.lhr:
      case Client.IdfField.transPropMap.lvr:
      case Client.IdfField.transPropMap.lhs:
      case Client.IdfField.transPropMap.fwi:
      case Client.IdfField.transPropMap.fhe:
      case Client.IdfField.transPropMap.fle:
      case Client.IdfField.transPropMap.fto:
      case Client.IdfField.transPropMap.fhr:
      case Client.IdfField.transPropMap.fvr:
      case Client.IdfField.transPropMap.fhs:
      case Client.IdfField.transPropMap.lnr:
      case Client.IdfField.transPropMap.fnr:
      case Client.IdfField.transPropMap.dat:
      case Client.IdfField.transPropMap.acw:
      case Client.IdfField.transPropMap.aln:
      case Client.IdfField.transPropMap.edi:
      case Client.IdfField.transPropMap.mus:
      case Client.IdfField.transPropMap.muf:
      case Client.IdfField.transPropMap.irm:
      case Client.IdfField.transPropMap.vfl:
      case Client.IdfField.transPropMap.chg:
      case Client.IdfField.transPropMap.fsc:
      case Client.IdfField.transPropMap.lta:
      case Client.IdfField.transPropMap.fta:
      case Client.IdfField.transPropMap.aof:
        props[p] = parseInt(props[p]);
        break;

      case Client.IdfField.transPropMap.vis:
      case Client.IdfField.transPropMap.inl:
      case Client.IdfField.transPropMap.inf:
      case Client.IdfField.transPropMap.lli:
      case Client.IdfField.transPropMap.hdl:
      case Client.IdfField.transPropMap.hla:
      case Client.IdfField.transPropMap.hdf:
      case Client.IdfField.transPropMap.hfa:
      case Client.IdfField.transPropMap.cva:
      case Client.IdfField.transPropMap.act:
      case Client.IdfField.transPropMap.qvs:
      case Client.IdfField.transPropMap.ena:
      case Client.IdfField.transPropMap.acd:
      case Client.IdfField.transPropMap.sac:
      case Client.IdfField.transPropMap.unb:
      case Client.IdfField.transPropMap.srt:
      case Client.IdfField.transPropMap.uet:
      case Client.IdfField.transPropMap.mup:
      case Client.IdfField.transPropMap.alo:
      case Client.IdfField.transPropMap.lke:
      case Client.IdfField.transPropMap.opt:
      case Client.IdfField.transPropMap.qen:
      case Client.IdfField.transPropMap.cms:
      case Client.IdfField.transPropMap.uts:
      case Client.IdfField.transPropMap.oqbf:
      case Client.IdfField.transPropMap.ocb:
        props[p] = props[p] === "1";
        break;
    }
  }
};


/**
 * Create elements configuration
 * @param {Object} widget
 */
Client.IdfField.prototype.createElementsConfig = function (widget)
{
  // Each field has a "list" and a "form" configuration
  let config = {};
  //
  // 1) Create list container configuration
  if (this.isShown()) {
    config.list = this.createContainerConfig();
    config.aggregate = this.createAggregateContainerConfig();
  }
  //
  // 2) Create form container configuration
  if (this.isShown(true))
    config.form = this.createContainerConfig(true);
  //
  return config;
};


/**
 * Create elements configuration for list/form mode
 * @param {Boolean} form
 */
Client.IdfField.prototype.createContainerConfig = function (form)
{
  let containerConf, headerConf, headerButtonConf, headerTextConf;
  //
  if (!form && this.isInList()) {
    // Create list container configuration
    let offsetCol = this.parent.getHeaderOffset() ? " offset-col" : "";
    containerConf = this.createElementConfig({c: "IonCol", className: "panel-list-col" + offsetCol, customid: this.id.replace(/:/g, "_") + "_lc", events: ["onClick"]});
    //
    // Create list header configuration
    headerButtonConf = this.createElementConfig({c: "IonButton", className: "generic-btn field-header-btn", visible: false});
    containerConf.children.push(headerButtonConf);
    //
    headerTextConf = this.createElementConfig({c: "IonText", type: "span"});
    containerConf.children.push(headerTextConf);
    //
    // Save elements ids
    this.listContainerId = containerConf.id;
    this.listHeaderButtonId = headerButtonConf.id;
    this.listHeaderTextId = headerTextConf.id;
  }
  else {
    // Create out list container configuration
    containerConf = this.createElementConfig({c: "Container", className: "panel-form-row"});
    //
    // Create out list header configuration (static field has no header)
    if (!this.isStatic()) {
      let numRows = form ? this.formNumRows : this.listNumRows;
      //
      headerConf = this.createElementConfig({c: "Container", className: "panel-form-col-header " + (numRows > 1 ? "" : " fixed-height-col"), customid: this.id.replace(/:/g, "_") + "_lc"});
      containerConf.children.push(headerConf);
      //
      headerButtonConf = this.createElementConfig({c: "IonButton", className: "generic-btn field-header-btn", visible: false});
      headerConf.children.push(headerButtonConf);
      //
      headerTextConf = this.createElementConfig({c: "IonText", type: "span"});
      headerConf.children.push(headerTextConf);
    }
    //
    let controlConfig = this.createControlConfig(form);
    //
    // Create out list value container configuration
    let valueContainerConf = this.createElementConfig(controlConfig.container);
    containerConf.children.push(valueContainerConf);
    //
    let outListControlConf = this.createElementConfig(controlConfig.control);
    valueContainerConf.children.push(outListControlConf);
    //
    // Save elements ids
    if (form) {
      this.formContainerId = containerConf.id;
      this.formHeaderId = headerConf?.id;
      this.formHeaderButtonId = headerButtonConf?.id;
      this.formHeaderTextId = headerTextConf?.id;
      this.formValueId = valueContainerConf.id;
      this.formControlId = outListControlConf.id;
    }
    else {
      this.outListContainerId = containerConf.id;
      this.outListHeaderId = headerConf?.id;
      this.outListHeaderButtonId = headerButtonConf?.id;
      this.outListHeaderTextId = headerTextConf?.id;
      this.outListValueId = valueContainerConf.id;
      this.outListControlId = outListControlConf.id;
    }
  }
  //
  return containerConf;
};


/**
 * Create aggregate container configuration
 */
Client.IdfField.prototype.createAggregateContainerConfig = function ()
{
  if (!this.isInList())
    return;
  //
  // Create aggregate container
  this.aggregateContainerConf = this.createElementConfig({c: "IonCol", className: "panel-list-col"});
  this.aggregateContainerId = this.aggregateContainerConf.id;
  //
  return this.aggregateContainerConf;
};


/**
 * Create control configuration
 * @param {Boolean} form
 * @param {Number} index
 */
Client.IdfField.prototype.createControlConfig = function (form, index)
{
  let config = {};
  //
  // Define base control configuration
  config.control = {c: "IdfControl", dataType: this.dataType};
  //
  if (!form && this.isInList()) {
    // Define list container configuration
    let fixedHeightCol = this.parent.hasDynamicHeightRows() ? "" : " fixed-height-col";
    let offsetCol = this.parent.getListRowOffset() ? " offset-col" : "";
    let cid = this.id.replace(/:/g, "_") + "_lv" + index;
    config.container = {c: "IonCol", className: "panel-list-col" + fixedHeightCol + offsetCol, customid: cid};
    //
    // Set specific in list control properties
    config.control.badgeInside = true;
    config.control.heightResize = this.parent.hasDynamicHeightRows();
  }
  else {
    let numRows = form ? this.formNumRows : this.listNumRows;
    let cid = this.id.replace(/:/g, "_") + (form ? "_fv" : (this.isStatic() ? "_lc" : "_lv0"));
    config.container = {c: "Container", className: "panel-form-col-value" + (numRows > 1 ? "" : " fixed-height-col"), customid: cid};
  }
  //
  config.container.events = ["onClick", "onDblclick", "onContextmenu", "onFocusin", "onFocusout"];
  //
  return config;
};


/**
 * Check if given data type is a text one
 * @param {Integer} dt
 */
Client.IdfField.isText = function (dt)
{
  return [
    Client.IdfField.dataTypes.UNSPECIFIED,
    Client.IdfField.dataTypes.TEXT,
    Client.IdfField.dataTypes.CHARACTER,
    Client.IdfField.dataTypes.FIXED_CHARACTER
  ].includes(dt);
};


/**
 * Check if given data type is a numeric one
 * @param {Integer} dt
 */
Client.IdfField.isNumeric = function (dt)
{
  return [
    Client.IdfField.dataTypes.INTEGER,
    Client.IdfField.dataTypes.FLOAT,
    Client.IdfField.dataTypes.DECIMAL,
    Client.IdfField.dataTypes.CURRENCY
  ].includes(dt);
};


/**
 * Check if given data type is date
 * @param {Integer} dt
 */
Client.IdfField.isDate = function (dt)
{
  return [
    Client.IdfField.dataTypes.DATE,
    Client.IdfField.dataTypes.DATETIME
  ].includes(dt);
};


/**
 * Check if given data type is time
 * @param {Integer} dt
 */
Client.IdfField.isTime = function (dt)
{
  return [
    Client.IdfField.dataTypes.TIME,
    Client.IdfField.dataTypes.DATETIME
  ].includes(dt);
};


/**
 * Check if given data type is date, time or datetime
 * @param {Integer} dt
 */
Client.IdfField.isDateOrTime = function (dt)
{
  return [
    Client.IdfField.dataTypes.DATE,
    Client.IdfField.dataTypes.TIME,
    Client.IdfField.dataTypes.DATETIME
  ].includes(dt);
};


/**
 * Get widget requirements
 * @param {Object} widget
 */
Client.IdfField.getRequirements = function (widget)
{
  let prefix = Client.mainFrame.isIDF ? "fluid/" : "";
  let req = {};
  if (widget.editorType !== Client.IdfField.editorTypes.HTMLEDITOR)
    return req;
  //
  // Add Html editor requirements
  req[prefix + "jquery.min.js"] = {type: "jc", name: "JQuery"};
  req[prefix + "objects/htmleditor/htmlEditor.js"] = {type: "jc", name: "htmlEditorJS"};
  req[prefix + "objects/htmleditor/plugins/emoji/emojify/emojify.css"] = {type: "cs", name: "emojifyCSS"};
  req[prefix + "objects/htmleditor/plugins/emoji/emojify/emojify.js"] = {type: "jc", name: "emojifyJS"};
  req[prefix + "objects/htmleditor/trumbowyg.min.js"] = {type: "jc", name: "trumbowygJS"};
  req[prefix + "objects/htmleditor/ui/icons.svg"] = {type: "sv", name: "trumbowygSVG"};
  req[prefix + "objects/htmleditor/ui/trumbowyg.min.css"] = {type: "cs", name: "trumbowygCSS"};
  //
  return req;
};


/**
 * Realize widget UI
 * @param {Object} widget
 * @param {View|Element|Widget} parent
 * @param {View} view
 */
Client.IdfField.prototype.realize = function (widget, parent, view)
{
  // Create elements configuration
  let config = this.createElementsConfig(widget);
  //
  let fieldEl;
  //
  // Create list version of this field
  if (config.list) {
    // Create field element and append it to parent (panel rootObject). Then it will be appended in the proper object
    fieldEl = view.createElement(config.list, parent, view);
    this.mainObjects.push(fieldEl);
    //
    // Create aggregate field element and append it to parent (panel rootObject). Then it will be appended in the proper object
    if (config.aggregate) {
      let aggregateFieldEl = view.createElement(config.aggregate, parent, view);
      this.mainObjects.push(aggregateFieldEl);
    }
  }
  //
  // Create form version of this field
  if (config.form) {
    // Create field element and append it to parent (panel rootObject). Then it will be appended in the proper object
    fieldEl = view.createElement(config.form, parent, view);
    this.mainObjects.push(fieldEl);
  }
  //
  // Create widget children
  this.createChildren(widget);
};


/**
 * Append field dom objects to their own column
 */
Client.IdfField.prototype.place = function ()
{
  if (this.isShown() && this.parent.getListFieldColumn(this.id)) {
    this.placeListForm();
    //
    // If field is in list also place its aggregate container
    if (this.isInList())
      this.placeListForm({aggregate: true});
  }
  //
  if (this.isShown(true) && this.parent.getFormFieldColumn(this.id))
    this.placeListForm({form: true});
};


/**
 * Append list/form field dom object to its own column
 * @param {Object} options
 *                        - form
 *                        - aggregate: true if I have to place aggregate container
 */
Client.IdfField.prototype.placeListForm = function (options)
{
  options = options || {};
  let form = options.form;
  let aggregate = options.aggregate;
  //
  let parentColumnId, fieldContainerId;
  if (!form) {
    let parentColumnConf = this.parent.getListFieldColumn(this.id, aggregate).conf;
    parentColumnId = parentColumnConf.id;
    //
    if (!aggregate)
      this.listParentColConf = parentColumnConf;
    //
    fieldContainerId = this.isInList() ? (aggregate ? this.aggregateContainerId : this.listContainerId) : this.outListContainerId;
  }
  else {
    this.formParentColConf = this.parent.getFormFieldColumn(this.id).conf;
    parentColumnId = this.formParentColConf.id;
    //
    fieldContainerId = this.formContainerId;
  }
  //
  // Get field parent column
  let fieldCol = Client.eleMap[parentColumnId];
  let fieldEl = Client.eleMap[fieldContainerId];
  //
  // Append field to its parent column
  fieldCol.getRootObject().appendChild(fieldEl.getRootObject());
  fieldCol.elements.push(fieldEl);
  fieldEl.parent = fieldCol;
};


/**
 * Remove field dom objects from their parent column
 */
Client.IdfField.prototype.unplace = function ()
{
  if (this.isShown()) {
    this.unplaceListForm();
    //
    // If field is in list, also unplace its aggregate container
    if (this.isInList())
      this.unplaceListForm({aggregate: true});
  }
  //
  if (this.isShown(true))
    this.unplaceListForm({form: true});
};


/**
 * Remove field dom object from its parent column
 * @param {Object} options
 *                        - form
 *                        - aggregate: true if I have to place aggregate container
 */
Client.IdfField.prototype.unplaceListForm = function (options)
{
  options = options || {};
  let form = options.form;
  let aggregate = options.aggregate;
  //
  // Get proper field container configuration
  let fieldContainerId;
  if (!form)
    fieldContainerId = this.isInList() ? (aggregate ? this.aggregateContainerId : this.listContainerId) : this.outListContainerId;
  else
    fieldContainerId = this.formContainerId;
  //
  // Get field element and its root object
  let fieldEl = Client.eleMap[fieldContainerId];
  let fieldRootObject = fieldEl.getRootObject();
  //
  // Detach root object
  fieldRootObject.parentNode.removeChild(fieldRootObject);
  //
  // Remove field element from parent
  let index = fieldEl.parent.elements.findIndex(el => el.id === fieldEl.id);
  fieldEl.parent.elements.splice(index, 1);
};


/**
 * Write index-th value on field dom objects
 * @param {Integer} index
 * @param {Boolean} skipList
 */
Client.IdfField.prototype.writeValue = function (index, skipList)
{
  this.updateControls({text: true, skipList}, index);
};


/**
 * Assign out list value container and form value container to index-th value
 * @param {Integer} index
 */
Client.IdfField.prototype.assignControls = function (index)
{
  // Since static fields have just one value, I always have to assign controls to it
  if (this.isStatic())
    index = 1;
  //
  // Clear all values' out list and form controls
  for (let i = 1; i <= this.values.length; i++) {
    if (!this.values[i])
      continue;
    //
    this.values[i].clearControls();
  }
  //
  // If index-th values does not exist, do nothing else
  if (!this.values[index])
    return;
  //
  let params = {};
  //
  if (this.isShown() && !this.isInList()) {
    params.outListContainer = Client.eleMap[this.outListValueId];
    params.outListControl = Client.eleMap[this.outListControlId];
  }
  //
  if (this.isShown(true)) {
    params.formContainer = Client.eleMap[this.formValueId];
    params.formControl = Client.eleMap[this.formControlId];
  }
  //
  this.values[index].assignControls(params);
};


/**
 * Update element properties
 * @param {Object} props
 */
Client.IdfField.prototype.updateElement = function (props)
{
  props = props || {};
  //
  let calcLayout, updateStructure, applyVisualStyle;
  let propsToUpdate = {};
  //
  Client.Widget.prototype.updateElement.call(this, props);
  //
  if (props.visible !== undefined) {
    this.visible = props.visible;
    calcLayout = true;
  }
  //
  if (props.showInList !== undefined) {
    // Do nothing: this property always maintains its initial value
  }
  //
  if (props.showInForm !== undefined) {
    // Do nothing: this property always maintains its initial value
  }
  //
  if (props.maxLength !== undefined) {
    this.maxLength = props.maxLength;
    propsToUpdate.maxLength = true;
  }
  //
  if (props.sortMode !== undefined) {
    this.sortMode = props.sortMode;
    this.updateSortMode();
  }
  //
  if (props.groupingMode !== undefined)
    this.groupingMode = props.groupingMode;
  //
  if (props.scale !== undefined) {
    this.scale = props.scale;
    propsToUpdate.scale = true;
  }
  //
  if (props.type !== undefined) {
    this.type = props.type;
    propsToUpdate.type = true;
  }
  //
  if (props.isPassword !== undefined) {
    this.isPassword = props.type;
    propsToUpdate.isPassword = true;
  }
  //
  if (props.inList !== undefined)
    this.inList = props.inList;
  //
  if (props.listHeader !== undefined) {
    this.listHeader = props.listHeader;
    this.updateHeader();
  }
  //
  if (props.formHeader !== undefined) {
    this.formHeader = props.formHeader;
    this.updateHeader(true);
  }
  //
  if (props.header !== undefined) {
    this.header = props.header;
    //
    if (this.isStatic())
      this.updateHeader();
  }
  //
  if (props.showListHeader !== undefined) {
    this.showListHeader = props.showListHeader;
    this.showHeader();
  }
  //
  if (props.showFormHeader !== undefined) {
    this.showFormHeader = props.showFormHeader;
    this.showHeader(true);
  }
  //
  if (props.listHeaderAbove !== undefined) {
    this.listHeaderAbove = props.listHeaderAbove;
    this.setHeaderAbove();
    calcLayout = true;
  }
  //
  if (props.formHeaderAbove !== undefined) {
    this.formHeaderAbove = props.formHeaderAbove;
    this.setHeaderAbove(true);
    calcLayout = true;
  }
  //
  if (props.listHeaderSize !== undefined) {
    this.listHeaderSize = isNaN(props.listHeaderSize) ? undefined : props.listHeaderSize;
    calcLayout = true;
  }
  //
  if (props.listHeaderSizePerc !== undefined) {
    this.listHeaderSizePerc = isNaN(props.listHeaderSizePerc) ? undefined : props.listHeaderSizePerc;
    calcLayout = true;
  }
  //
  if (props.formHeaderSize !== undefined) {
    this.formHeaderSize = isNaN(props.formHeaderSize) ? undefined : props.formHeaderSize;
    calcLayout = true;
  }
  //
  if (props.formHeaderSizePerc !== undefined) {
    this.formHeaderSizePerc = isNaN(props.formHeaderSizePerc) ? undefined : props.formHeaderSizePerc;
    calcLayout = true;
  }
  //
  if (props.listWidth !== undefined) {
    this.listWidth = isNaN(props.listWidth) ? undefined : props.listWidth;
    this.orgListWidth = this.listWidth;
    updateStructure = true;
  }
  //
  if (props.listWidthPerc !== undefined) {
    this.listWidthPerc = isNaN(props.listWidthPerc) ? undefined : props.listWidthPerc;
    this.orgListWidthPerc = this.listWidthPerc;
    updateStructure = true;
  }
  //
  if (props.listResizeWidth !== undefined) {
    this.listResizeWidth = props.listResizeWidth;
    calcLayout = true;
  }
  //
  if (props.listHeight !== undefined) {
    this.listHeight = isNaN(props.listHeight) ? undefined : props.listHeight;
    this.orgListHeight = this.listHeight;
    updateStructure = true;
  }
  //
  if (props.listHeightPerc !== undefined) {
    this.listHeightPerc = isNaN(props.listHeightPerc) ? undefined : props.listHeightPerc;
    this.orgListHeightPerc = this.listHeightPerc;
    updateStructure = true;
  }
  //
  if (props.listResizeHeight !== undefined) {
    this.listResizeHeight = props.listResizeHeight;
    calcLayout = true;
  }
  //
  if (props.listLeft !== undefined) {
    this.listLeft = isNaN(props.listLeft) ? undefined : props.listLeft;
    this.orgListLeft = this.listLeft;
    updateStructure = true;
  }
  //
  if (props.listLeftPerc !== undefined) {
    this.listLeftPerc = isNaN(props.listLeftPerc) ? undefined : props.listLeftPerc;
    this.orgListLeftPerc = this.listLeftPerc;
    updateStructure = true;
  }
  //
  if (props.listTop !== undefined) {
    this.listTop = isNaN(props.listTop) ? undefined : props.listTop;
    this.orgListTop = this.listTop;
    updateStructure = true;
  }
  //
  if (props.listTopPerc !== undefined) {
    this.listTopPerc = isNaN(props.listTopPerc) ? undefined : props.listTopPerc;
    this.orgListTopPerc = this.listTopPerc;
    updateStructure = true;
  }
  //
  if (props.formWidth !== undefined) {
    this.formWidth = isNaN(props.formWidth) ? undefined : props.formWidth;
    this.orgFormWidth = this.formWidth;
    updateStructure = true;
  }
  //
  if (props.formWidthPerc !== undefined) {
    this.formWidthPerc = isNaN(props.formWidthPerc) ? undefined : props.formWidthPerc;
    this.orgFormWidthPerc = this.formWidthPerc;
    updateStructure = true;
  }
  //
  if (props.formResizeWidth !== undefined) {
    this.formResizeWidth = props.formResizeWidth;
    calcLayout = true;
  }
  //
  if (props.formHeight !== undefined) {
    this.formHeight = isNaN(props.formHeight) ? undefined : props.formHeight;
    this.orgFormHeight = this.formHeight;
    updateStructure = true;
  }
  //
  if (props.formHeightPerc !== undefined) {
    this.formHeightPerc = isNaN(props.formHeightPerc) ? undefined : props.formHeightPerc;
    this.orgFormHeightPerc = this.formHeightPerc;
    updateStructure = true;
  }
  //
  if (props.formResizeHeight !== undefined) {
    this.formResizeHeight = props.formResizeHeight;
    calcLayout = true;
  }
  //
  if (props.formLeft !== undefined) {
    this.formLeft = isNaN(props.formLeft) ? undefined : props.formLeft;
    this.orgFormLeft = this.formLeft;
    updateStructure = true;
  }
  //
  if (props.formLeftPerc !== undefined) {
    this.formLeftPerc = isNaN(props.formLeftPerc) ? undefined : props.formLeftPerc;
    this.orgFormLeftPerc = this.formLeftPerc;
    updateStructure = true;
  }
  //
  if (props.formTop !== undefined) {
    this.formTop = isNaN(props.formTop) ? undefined : props.formTop;
    this.orgFormTop = this.formTop;
    updateStructure = true;
  }
  //
  if (props.formTopPerc !== undefined) {
    this.formTopPerc = isNaN(props.formTopPerc) ? undefined : props.formTopPerc;
    this.orgFormTopPerc = this.formTopPerc;
    updateStructure = true;
  }
  //
  if (props.aggregateOfField !== undefined) {
    this.aggregateOfField = props.aggregateOfField;
    updateStructure = true;
  }
  //
  if (props.listNumRows !== undefined) {
    this.listNumRows = props.listNumRows;
    propsToUpdate.listNumRows = true;
  }
  //
  if (props.formNumRows !== undefined) {
    this.formNumRows = props.formNumRows;
    propsToUpdate.formNumRows = true;
  }
  //
  if (props.dataType !== undefined)
    this.dataType = props.dataType;
  //
  if (props.valueList !== undefined) {
    // If value list is for qbe, I don't have to update field value list
    if (props.valueList.qbe) {
      // If value list was required by filter popup, give to it
      if (props.valueList.popup) {
        let popupFilter = Client.eleMap[props.valueList.popup];
        if (popupFilter)
          popupFilter.updateValueList(props.valueList);
      }
      else if (this.parent.canUseRowQbe() && this.isInList()) // Otherwise give value list to qbe value
        this.values[0].updateElement({valueList: props.valueList});
      //
    }
    else {
      this.valueList = props.valueList;
      propsToUpdate.valueList = true;
    }
  }
  //
  if (props.smartLookup !== undefined) {
    this.smartLookup = props.smartLookup;
    propsToUpdate.smartLookup = true;
  }
  //
  if (props.hasValueSource !== undefined) {
    this.hasValueSource = props.hasValueSource;
    propsToUpdate.hasValueSource = true;
  }
  //
  if (props.autoLookup !== undefined) {
    this.autoLookup = props.autoLookup;
    propsToUpdate.autoLookup = true;
  }
  //
  if (props.optional !== undefined) {
    this.optional = props.optional;
    propsToUpdate.optional = true;
  }
  //
  if (props.enabled !== undefined) {
    this.enabled = props.enabled;
    propsToUpdate.enabled = true;
    //
    applyVisualStyle = true;
  }
  //
  if (props.enabledInQbe !== undefined) {
    this.enabledInQbe = props.enabledInQbe;
    propsToUpdate.enabled = true;
    //
    applyVisualStyle = true;
  }
  //
  if (props.canActivate !== undefined) {
    this.canActivate = props.canActivate;
    propsToUpdate.canActivate = true;
  }
  //
  if (props.activableDisabled !== undefined) {
    this.activableDisabled = props.activableDisabled;
    propsToUpdate.activableDisabled = true;
  }
  //
  if (props.superActive !== undefined) {
    this.superActive = props.superActive;
    propsToUpdate.superActive = true;
  }
  //
  if (props.canSort !== undefined) {
    this.canSort = props.canSort;
    propsToUpdate.canSort = true;
  }
  //
  if (props.alignment !== undefined) {
    this.alignment = props.alignment;
    //
    if (this.isStatic())
      this.values[1].updateElement({alignment: this.alignment});
    //
    propsToUpdate.alignment = true;
  }
  //
  if (props.backColor !== undefined) {
    this.backColor = props.backColor;
    propsToUpdate.backColor = true;
  }
  //
  if (props.color !== undefined) {
    this.color = props.color;
    propsToUpdate.color = true;
  }
  //
  if (props.mask !== undefined) {
    this.mask = props.mask || (this.isStatic() ? "=" : "");
    propsToUpdate.mask = true;
  }
  //
  if (props.fontModifiers !== undefined) {
    this.fontModifiers = props.fontModifiers;
    propsToUpdate.fontModifiers = true;
  }
  //
  if (props.className !== undefined) {
    this.className = props.className;
    //
    // The className can have a responsive grid, in that case we must extract it
    let cls = Client.Widget.extractGridClasses(this.className);
    this.className = cls.className;
    this.gridClass = cls.gridClass;
    //
    // There are special classes that must be set also on the list header
    if (this.isInList() && (this.className.indexOf("lg-visible") >= 0 || this.className.indexOf("md-visible") >= 0)) {
      let hl = Client.eleMap[this.listHeaderTextId];
      Client.Widget.updateElementClassName(hl, this.className.indexOf("md-visible") >= 0 ? "md-visible" : "lg-visible");
    }
    //
    propsToUpdate.className = true;
  }
  //
  if (props.badge !== undefined)
    propsToUpdate.badge = true;
  //
  if (props.activatorWidth !== undefined) {
    this.activatorWidth = props.activatorWidth;
    propsToUpdate.activatorWidth = true;
  }
  //
  if (props.activatorImage !== undefined) {
    this.activatorImage = props.activatorImage;
    propsToUpdate.activatorImage = true;
  }
  //
  if (props.editorType !== undefined) {
    this.editorType = props.editorType;
    propsToUpdate.editorType = true;
  }
  //
  if (props.showHtmlEditorToolbar !== undefined) {
    this.showHtmlEditorToolbar = props.showHtmlEditorToolbar;
    propsToUpdate.showHtmlEditorToolbar = true;
  }
  //
  if (props.multiUpload !== undefined) {
    this.multiUpload = props.multiUpload;
    propsToUpdate.multiUpload = true;
  }
  //
  if (props.uploadExtensions !== undefined) {
    this.uploadExtensions = props.uploadExtensions;
    propsToUpdate.uploadExtensions = true;
  }
  //
  if (props.comboMultiSel !== undefined) {
    this.comboMultiSel = props.comboMultiSel;
    propsToUpdate.comboMultiSel = true;
  }
  //
  if (props.comboSeparator !== undefined) {
    this.comboSeparator = props.comboSeparator;
    propsToUpdate.comboSeparator = true;
  }
  //
  if (props.controlType !== undefined) {
    this.controlType = props.controlType;
    propsToUpdate.controlType = true;
  }
  //
  if (props.qbeFilter !== undefined) {
    this.qbeFilter = props.qbeFilter;
    propsToUpdate.qbeFilter = true;
    //
    for (let i = 0; i <= this.values.length; i++) {
      if (!this.values[i])
        continue;
      //
      this.values[i].updateQbeFilter();
    }
  }
  //
  // "visualFlags" property exists just on IDF. Thus, transform variation on visualFlags into variations on each single property
  if (props.visualFlags !== undefined) {
    this.visualFlags = props.visualFlags;
    //
    props.canSortFlag = (this.visualFlags & 0x1) !== 0;
    props.showOnlyIcon = (this.visualFlags & 0x8) !== 0;
    props.showActivator = (this.visualFlags & 0x10) !== 0;
    props.isHyperLink = (this.visualFlags & 0x80) !== 0;
    props.slidePad = (this.visualFlags & 0x40) !== 0;
    props.autoTab = (this.visualFlags & 0x200) !== 0;
    props.usePlaceholderasNull = (this.visualFlags & 0x00000400) !== 0;
    props.handleTabOrder = (this.visualFlags & 0x00000800) !== 0;
    props.canHideInList = (this.visualFlags & 0x00001000) !== 0;
    props.hiddenInList = (this.visualFlags & 0x00002000) !== 0;
  }
  //
  if (props.canSortFlag !== undefined) {
    this.canSortFlag = props.canSortFlag;
    propsToUpdate.canSort = true;
  }
  //
  if (props.showOnlyIcon !== undefined)
    this.showOnlyIcon = props.showOnlyIcon;
  //
  if (props.showActivator !== undefined)
    this.showActivator = props.showActivator;
  //
  if (props.isHyperLink !== undefined) {
    this.isHyperLink = props.isHyperLink;
    propsToUpdate.isHyperLink = true;
  }
  //
  if (props.slidePad !== undefined)
    this.slidePad = props.slidePad;
  //
  if (props.autoTab !== undefined)
    this.autoTab = props.autoTab;
  //
  if (props.usePlaceholderasNull !== undefined)
    this.usePlaceholderasNull = props.usePlaceholderasNull;
  //
  if (props.handleTabOrder !== undefined)
    this.handleTabOrder = props.handleTabOrder;
  //
  if (props.canHideInList !== undefined)
    this.canHideInList = props.canHideInList;
  //
  if (props.hiddenInList !== undefined)
    this.hiddenInList = props.hiddenInList;
  //
  if (props.image !== undefined) {
    this.image = props.image;
    //
    // If image has to be shown in value, simply remember to update control image
    if (this.parent.showFieldImageInValue)
      propsToUpdate.image = true;
    else { // Otherwise show image in header
      if (this.parent.hasList)
        this.updateImage();
      //
      if (this.parent.hasForm)
        this.updateImage(true);
    }
  }
  //
  if (props.imageResizeMode !== undefined) {
    // Both IdfField and IdfSpan have their own resize modes list.
    // The two resize modes have different names, but similar behaviour.
    // Thus standardize modes using control resize modes list.
    let imgResizeMode;
    switch (props.imageResizeMode) {
      case Client.IdfField.stretches.REPEAT:
        imgResizeMode = Client.IdfControl.stretches.NONE;
        break;

      case Client.IdfField.stretches.FIT:
        imgResizeMode = Client.IdfControl.stretches.FILL;
        break;

      case Client.IdfField.stretches.CENTER:
        imgResizeMode = Client.IdfControl.stretches.ENLARGE;
        break;
    }
    //
    this.imageResizeMode = imgResizeMode;
    propsToUpdate.imageResizeMode = true;
    //
    // If image has not to be shown in value, update header image resize mode
    if (!this.parent.showFieldImageInValue) {
      if (this.parent.hasList)
        this.updateImageResizeMode();
      //
      if (this.parent.hasForm)
        this.updateImageResizeMode(true);
    }
  }
  //
  if (props.placeholder !== undefined) {
    this.placeholder = props.placeholder;
    propsToUpdate.placeholder = true;
  }
  //
  if (props.notifySelectionChange !== undefined)
    this.notifySelectionChange = props.notifySelectionChange;
  //
  if (props.listTabOrderIndex !== undefined)
    this.listTabOrderIndex = props.listTabOrderIndex;
  //
  if (props.formTabOrderIndex !== undefined)
    this.formTabOrderIndex = props.formTabOrderIndex;
  //
  if (props.needFilterPopup !== undefined)
    this.openFilterPopup();
  //
  if (props.needToOpenCombo !== undefined)
    this.openCombo();
  //
  if (props.changeEventDef !== undefined)
    this.changeEventDef = props.changeEventDef;
  //
  // If I have to update parent panel structure and layout, do it now
  if (!this.realizing)
    this.parent.updateObjects({structure: updateStructure, layout: calcLayout || updateStructure});
  //
  // If I have to apply visual style, do it now
  if (applyVisualStyle)
    this.applyVisualStyle();
  //
  if (propsToUpdate.canSort)
    this.updateHeader();
  //
  // Update controls
  this.updateControls(propsToUpdate);
};


/**
 * Handle an event
 * @param {Object} event
 */
Client.IdfField.prototype.onEvent = function (event)
{
  let events = Client.Widget.prototype.onEvent.call(this, event);
  //
  switch (event.id) {
    case "onClick":
      if (event.obj === this.listContainerId) {
        if (this.parent.showListVisisiblityControls)
          this.parent.openControlListPopup(this);
        else if (this.parent.searchMode === Client.IdfPanel.searchModes.header)
          this.openFilterPopup();
        else if (this.parent.canGroup && this.parent.showGroups)
          events.push(...this.handleGrouping());
        else if (this.isSortable())
          events.push(...this.handleSort());
      }
      break;
  }
  //
  return events;
};


/**
 * Apply visual style
 * @param {Integer} index
 */
Client.IdfField.prototype.applyVisualStyle = function (index)
{
  // I don't need to apply visual style while field is realizing. When parent panel will be realized,
  // it will apply visual style to itself and to its children (i.e. fields and groups)
  if (this.realizing)
    return;
  //
  let layouts = ["list", "form"];
  //
  for (let i = 0; i < layouts.length; i++) {
    let form = (layouts[i] === "form");
    //
    if (!this.isShown(form) || this.isStatic())
      continue;
    //
    let notNull = !this.optional && this.parent.status !== Client.IdfPanel.statuses.qbe && !this.parent.locked && !this.isStatic();
    let visOptions = {objType: "fieldHeader", notNull};
    visOptions.list = form ? false : this.isInList();
    //
    let headerContainerId = form ? this.formHeaderId : (this.isInList() ? this.listContainerId : this.outListHeaderId);
    let headerContainer = Client.eleMap[headerContainerId];
    this.addVisualStyleClasses(headerContainer, visOptions);
    //
    // Get field header alignment
    let headerAlignment = this.getHeaderAlignment(form);
    headerAlignment = Client.IdfVisualStyle.getTextAlign(headerAlignment);
    //
    // Set field header alignment
    let headerId = form ? this.formHeaderTextId : (this.isInList() ? this.listHeaderTextId : this.outListHeaderTextId);
    let header = Client.eleMap[headerId];
    header.updateElement({style: {textAlign: headerAlignment, justifyContent: headerAlignment}});
  }
  //
  // Apply visual style on field values
  let start = index ?? this.parent.getDataBlockStart();
  let end = index ?? this.getDataBlockEnd();
  for (let i = start; i <= end; i++) {
    if (!this.values[i])
      continue;
    //
    this.values[i].applyVisualStyle();
  }
};


/**
 * Update values controls
 * @param {Object} propsToUpdate - example {visualStyle: true, editorType: true, ...}
 * @param {Integer} index
 */
Client.IdfField.prototype.updateControls = function (propsToUpdate, index)
{
  let start = index ?? this.parent.getDataBlockStart();
  let end = index ?? this.getDataBlockEnd();
  //
  for (let i = start; i <= end; i++) {
    if (!this.values[i])
      continue;
    //
    this.values[i].updateControls(propsToUpdate);
  }
};


/**
 * Return true if I'm right aligned
 */
Client.IdfField.prototype.isRightAligned = function ()
{
  // Get my visual style
  let vis = Client.IdfVisualStyle.getByIndex(this.getVisualStyle());
  //
  // Get my alignment
  let alignment = this.alignment;
  //
  // If I have no specific alignment, ask visual style for value alignment (see IdfVisualStyle.getAlignment without parameters)
  if (alignment === -1)
    alignment = vis ? vis.getAlignment() : -1;
  //
  // If my alignment is RIGHT, I'm right aligned
  if (alignment === Client.IdfVisualStyle.alignments.RIGHT)
    return true;
  //
  // If I'm not numeric, I'm not right aligned
  if (!Client.IdfField.isNumeric(this.dataType))
    return false;
  //
  // A control different from EDIT with an associated value list is not right aligned
  if (this.valueList && (!vis || vis.getControlType() !== Client.IdfField.controlTypes.EDIT))
    return false;
  //
  // Autolookup fields are not right aligned
  if (this.autoLookup)
    return false;
  //
  // If alignment is not auto, it's not right alignment
  if (alignment !== Client.IdfVisualStyle.alignments.AUTO)
    return false;
  //
  return true;
};


/**
 * Return true if this field is in list
 */
Client.IdfField.prototype.isInList = function ()
{
  return this.inList;
};


/**
 * Calculate layout rules to handle resize mode
 * @param {Integer} index
 */
Client.IdfField.prototype.calcLayout = function (index)
{
  // Calculate layout for list mode
  if (this.isShown() && this.parent.getListFieldColumn(this.id))
    this.calcListFormLayout(false, index);
  //
  // Calculate layout for form mode
  if (this.isShown(true) && this.parent.getFormFieldColumn(this.id))
    this.calcListFormLayout(true, index);
};


/**
 * Calculate layout for list or form mode
 * @param {Boolean} form
 * @param {Integer} index
 */
Client.IdfField.prototype.calcListFormLayout = function (form, index)
{
  let headerStyle = {};
  let fieldStyle = {};
  let outListColStyle = {};
  //
  let xs = this.canAdaptWidth(form) ? "" : "auto";
  //
  let rects = this.getRects({form});
  let width = rects.width, height = rects.height, left = rects.left, top = rects.top;
  //
  let parentWidth = (!form && this.isInList()) ? this.parent.gridWidth : this.parent.originalWidth;
  let parentHeight = (!form && this.isInList()) ? this.parent.gridHeight : this.parent.originalHeight;
  //
  // Use flex to handle width resize. Aggregate of field behaves like an in list field
  if (!form && (this.isInList() || this.aggregateOfField !== -1)) {
    fieldStyle.flexGrow = this.canAdaptWidth() ? "1" : "0";
    fieldStyle.flexShrink = this.canAdaptWidth() ? "1" : "0";
    fieldStyle.flexBasis = width !== undefined ? width + "px" : "auto";
    //
    if (width !== undefined && this.group && !this.canAdaptWidth())
      fieldStyle.minWidth = width + "px";
    //
    if (this.aggregateOfField !== -1) {
      // If I'm a values aggregation of another field, I have to be as wider as parent field
      fieldStyle.flexBasis = "100%";
      //
      // Use my own height
      fieldStyle.height = height + "px";
      fieldStyle.minHeight = height + "px";
    }
    else {
      let listRowHeight = this.parent.getListRowHeight();
      //
      // If panel hasn't dynamic height rows, assign fixed height to field
      if (!this.parent.hasDynamicHeightRows())
        fieldStyle.height = listRowHeight + "px";
      else
        fieldStyle.minHeight = listRowHeight + "px";
    }
    //
    // Assign fixed height to field header
    headerStyle = Object.assign({}, fieldStyle);
    //
    // If this field belongs to a group, field has no height. Group has
    headerStyle.height = this.group ? "" : this.parent.getHeaderHeight() + "px";
    headerStyle.minHeight = this.group ? "" : this.parent.getHeaderHeight() + "px";
  }
  else {
    let headerAbove = ((form && this.formHeaderAbove) || (!form && this.listHeaderAbove));
    //
    // Use percentage header size or normal one
    let headerSize;
    let headerSizePerc = form ? this.formHeaderSizePerc : this.listHeaderSizePerc;
    if (headerSizePerc !== undefined)
      headerSize = (this.parent.originalWidth * headerSizePerc / 100);
    else
      headerSize = form ? this.formHeaderSize : this.listHeaderSize;
    //
    // If header is above value, set header size as min-height
    if (headerAbove)
      headerStyle.minHeight = headerSize ? headerSize + "px" : "";
    else // Otherwise set it as width
      headerStyle.width = headerSize ? headerSize + "px" : "";
    //
    outListColStyle.padding = "0px";
    outListColStyle.flexBasis = width !== undefined ? width + "px" : "auto";
    outListColStyle.flexGrow = this.canAdaptWidth(form) ? "1" : "0";
    outListColStyle.minWidth = this.canAdaptWidth(form) ? "0" : width + "px";
    //
    // Default col height is "auto"
    outListColStyle.height = "auto";
    //
    // Set absolute height if there is an height and this field cannot adapt its height
    if (height !== undefined && !this.canAdaptHeight(form)) {
      outListColStyle.height = height + "px";
      fieldStyle.height = height + "px";
    }
    //
    // Get field's group dimensions and position
    let groupLeft = 0;
    let groupTop = 0;
    if (this.group) {
      groupLeft = form ? this.group.formLeft : this.group.listLeft;
      groupTop = form ? this.group.formTop : this.group.listTop;
      //
      parentWidth = form ? this.group.formWidth : this.group.listWidth;
      parentHeight = form ? this.group.formHeight : this.group.listHeight;
    }
    //
    // Get field parent column
    let fieldColumn = form ? this.parent.getFormFieldColumn(this.id) : this.parent.getListFieldColumn(this.id);
    //
    // Tell field column if it can adapt its width and height
    fieldColumn.canAdaptWidth = this.canAdaptWidth(form);
    fieldColumn.canAdaptHeight = this.canAdaptHeight(form);
    //
    // Calculate margin left
    let fieldColumnLeft = fieldColumn.rect.left || 0;
    let deltaLeft = fieldColumn.isMostLeft ? left - groupLeft : left - fieldColumnLeft;
    outListColStyle.marginLeft = deltaLeft + "px";
    //
    // Calculate margin right
    let deltaRight = fieldColumn.rect.deltaRight;
    if (fieldColumn.isMostRight)
      deltaRight = this.group ? (parentWidth - width - (left - groupLeft)) : (parentWidth - width - left);
    outListColStyle.marginRight = deltaRight + "px";
    //
    // Calculate margin top
    let fieldColumnTop = fieldColumn.rect.top || 0;
    let deltaTop = fieldColumn.isMostTop ? top - groupTop : top - fieldColumnTop;
    outListColStyle.marginTop = deltaTop + "px";
    //
    // Get panel toolbar height
    let toolbarHeight = this.parent.getToolbarHeight();
    //
    // Calculate margin bottom
    let deltaBottom = fieldColumn.rect.deltaBottom;
    if (fieldColumn.isMostBottom)
      deltaBottom = this.group ? (parentHeight - height - (top - groupTop)) : (parentHeight - height - top - toolbarHeight);
    outListColStyle.marginBottom = deltaBottom + "px";
  }
  //
  // Update in list field
  if (!form && this.isInList()) {
    // Update field column element
    let el = Client.eleMap[this.listContainerId];
    if (el) {
      let fixedLeft = this.parent.isFixedField(this) ? this.parent.getFixedFieldLeft(this) + "px" : "";
      //
      headerStyle.left = fixedLeft;
      el.updateElement({style: headerStyle, xs: xs});
      //
      Client.Widget.updateElementClassName(el, "fixed-col", !fixedLeft);
      //
      // Update aggregate column element
      el = Client.eleMap[this.aggregateContainerId];
      el.updateElement({style: headerStyle, xs: xs});
      Client.Widget.updateElementClassName(el, "fixed-col", !fixedLeft);
    }
    //
    // Update field values elements
    let start = index ?? this.parent.getDataBlockStart();
    let end = index ?? this.getDataBlockEnd();
    for (let i = start; i <= end; i++) {
      if (!this.values[i])
        continue;
      //
      this.values[i].setListLayout({style: fieldStyle, xs: xs});
    }
  }
  else { // Otherwise update out list field
    // Update header style (static field has no header)
    if (!this.isStatic()) {
      let headerId = form ? this.formHeaderId : this.outListHeaderId;
      Client.eleMap[headerId]?.updateElement({style: headerStyle});
    }
    //
    let valueContainerId = form ? this.formValueId : this.outListValueId;
    let parentColId = form ? this.formParentColConf.id : this.listParentColConf.id;
    //
    let el = Client.eleMap[valueContainerId];
    if (el)
      el.updateElement({style: fieldStyle});
    //
    el = Client.eleMap[parentColId];
    if (el)
      el.updateElement({style: outListColStyle});
  }
};


/**
 * Check if this field can apdapt its list/form width
 * @param {Boolean} form
 */
Client.IdfField.prototype.canAdaptWidth = function (form)
{
  let resizeWidth = form ? this.formResizeWidth : this.listResizeWidth;
  //
  // A field can adapt its width if it and its parent view have an adaptable width
  let canAdapt = this.parentIdfView?.resizeWidth !== Client.IdfView.resizeModes.NONE && resizeWidth === Client.IdfField.resizeModes.STRETCH;
  //
  // If field is in list, I need also panel to have an adaptable width
  if (!form && this.isInList())
    canAdapt = canAdapt && this.parent.resizeWidth === Client.IdfPanel.resizeModes.stretch;
  //
  return canAdapt;
};


/**
 * Check if this field can apdapt its list/form height
 * @param {Boolean} form
 */
Client.IdfField.prototype.canAdaptHeight = function (form)
{
  let resizeHeight = form ? this.formResizeHeight : this.listResizeHeight;
  //
  // A field can adapt its height if it and its parent view have an adaptable height
  let canAdapt = this.parentIdfView?.resizeHeight !== Client.IdfView.resizeModes.NONE && resizeHeight === Client.IdfField.resizeModes.STRETCH;
  //
  // If field is in list, it can't adapt its height
  if (!form && this.isInList())
    canAdapt = false;
  //
  return canAdapt;
};


/**
 * Check if this field can move left
 */
Client.IdfField.prototype.canMoveLeft = function ()
{
  // A field can move left if it's parent view has an adaptable width and it has MOVE has resize width mode
  let canMove = this.parentIdfView?.resizeWidth !== Client.IdfView.resizeModes.NONE && this.listResizeWidth === Client.IdfField.resizeModes.MOVE;
  //
  // If field is in list, it can't move left
  if (this.isInList())
    canMove = false;
  //
  return canMove;
};


/**
 * Check if this field can move top
 */
Client.IdfField.prototype.canMoveTop = function ()
{
  // A field can move top if it's parent view has an adaptable height and it has MOVE has resize height mode
  let canMove = this.parentIdfView?.resizeHeight !== Client.IdfView.resizeModes.NONE && this.listResizeHeight === Client.IdfField.resizeModes.MOVE;
  //
  // If field is in list, it can't move top
  if (this.isInList())
    canMove = false;
  //
  return canMove;
};


/**
 * Get field rects
 * @param {Object} options
 *                  - form
 *                  - real: true if need browser rects
 *                  - checkVisibility
 */
Client.IdfField.prototype.getRects = function (options)
{
  options = options || {};
  //
  let form = options.form;
  //
  if (options.checkVisibility && !this.isVisible(form))
    return {width: 0, height: 0, left: 0, top: 0};
  //
  let width, height, left, top;
  //
  let parentWidth = (!form && this.isInList()) ? this.parent.gridWidth : this.parent.originalWidth;
  let parentHeight = (!form && this.isInList()) ? this.parent.gridHeight : this.parent.originalHeight;
  //
  // Use width percentage or normal width
  let widthPerc = form ? this.formWidthPerc : this.listWidthPerc;
  if (widthPerc !== undefined)
    width = (parentWidth * widthPerc / 100);
  else
    width = form ? this.formWidth : this.listWidth;
  //
  // Use height percentage or normal height
  let heightPerc = form ? this.formHeightPerc : this.listHeightPerc;
  if (heightPerc !== undefined)
    height = (parentHeight * heightPerc / 100);
  else
    height = form ? this.formHeight : this.listHeight;
  //
  // Use left percentage or normal left
  let leftPerc = form ? this.formLeftPerc : this.listLeftPerc;
  if (leftPerc !== undefined)
    left = (this.parent.originalWidth * leftPerc / 100);
  else
    left = form ? this.formLeft : this.listLeft;
  //
  // Use top percentage or normal top
  let topPerc = form ? this.formTopPerc : this.listTopPerc;
  if (topPerc !== undefined)
    top = (this.parent.originalHeight * topPerc / 100);
  else
    top = form ? this.formTop : this.listTop;
  //
  if (options.real) {
    // In case of in list field I cannot get real height, top and left values because they are different for every values.
    // I just can get real width, if required, because every values of in list field have the same width
    if (!form && this.isInList())
      width = Client.eleMap[this.listContainerId].getRootObject().clientWidth;
    else {
      let fieldContainerId = form ? this.formContainerId : this.outListContainerId;
      let rootObject = Client.eleMap[fieldContainerId].getRootObject();
      let fieldRects = rootObject.getBoundingClientRect();
      //
      width = rootObject.clientWidth;
      height = rootObject.clientHeight;
      left = fieldRects.left;
      top = fieldRects.top;
    }
  }
  //
  return {width, height, left, top};
};


/**
 * Get index-th value
 * @param {Integer} index
 */
Client.IdfField.prototype.getValueByIndex = function (index)
{
  return this.values[index];
};


/**
 * Get data block end. I need this to handle empty values that are not included into panel totalRows property
 */
Client.IdfField.prototype.getDataBlockEnd = function ()
{
  return Math.max(this.parent.getDataBlockEnd(), this.values.length);
};


/**
 * Get header alignment
 * @param {Boolean} form
 */
Client.IdfField.prototype.getHeaderAlignment = function (form)
{
  // Default header alignment is left
  let headerAlignment = Client.IdfVisualStyle.alignments.LEFT;
  //
  // In case of "out list" field, header is right aligned just if it's above value and value is right aligned
  if (!form && !this.isInList()) {
    headerAlignment = this.isRightAligned() && this.listHeaderAbove ? Client.IdfVisualStyle.alignments.RIGHT : Client.IdfVisualStyle.alignments.LEFT;
    return headerAlignment;
  }
  // Form field header is left aligned if it's not above
  if (form && !this.formHeaderAbove)
    return headerAlignment;
  //
  // Get header alignment
  headerAlignment = this.alignment;
  //
  // If I have not specific alignment or it's AUTO, ask to visual style
  if (headerAlignment === -1 || headerAlignment === Client.IdfVisualStyle.alignments.AUTO) {
    let vis = Client.IdfVisualStyle.getByIndex(this.getVisualStyle());
    headerAlignment = vis ? vis.getAlignment(form ? "formHeader" : "listHeader") : -1;
  }
  //
  // If there isn't an alignment yet or it's AUTO again, get the default one based on data type
  if (headerAlignment === -1 || headerAlignment === Client.IdfVisualStyle.alignments.AUTO)
    headerAlignment = this.isRightAligned() ? Client.IdfVisualStyle.alignments.RIGHT : Client.IdfVisualStyle.alignments.LEFT;
  //
  return headerAlignment;
};


/**
 * Update my header
 * @param {Boolean} form
 */
Client.IdfField.prototype.updateHeader = function (form)
{
  // If I have to update header for a form/list field and parent panel has not form/list or field is not visible in form/list, do nothing
  if (!this.isShown(form))
    return;
  //
  // If this is a static field, I have to update text of its unique value
  if (this.isStatic())
    this.values[1].updateElement({text: this.header});
  else { // Otherwise update innerText of header container
    let headerText, headerButton;
    //
    if (form) {
      headerText = Client.eleMap[this.formHeaderTextId];
      headerButton = Client.eleMap[this.formHeaderButtonId];
    }
    else {
      headerText = this.isInList() ? Client.eleMap[this.listHeaderTextId] : Client.eleMap[this.outListHeaderTextId];
      headerButton = this.isInList() ? Client.eleMap[this.listHeaderButtonId] : Client.eleMap[this.outListHeaderButtonId];
      //
      if (this.isInList())
        Client.Widget.updateElementClassName(Client.eleMap[this.listContainerId], "field-header-clickable", !this.isSortable());
    }
    //
    // Extract the image from the caption if present
    let {caption, icon, color} = Client.Widget.extractCaptionData(form ? this.formHeader : this.listHeader);
    //
    // If there is an icon, set it into header button
    if (icon)
      Client.Widget.setIconImage(icon, headerButton, false, color);
    //
    headerButton.updateElement({visible: !!icon});
    headerText.updateElement({innerText: caption, visible: !!caption});
  }
};


/**
 * Show/hide header
 * @param {Boolean} form
 */
Client.IdfField.prototype.showHeader = function (form)
{
  // If I have to show header for a form/list field and parent panel has not form/list or field is not visible in form/list, do nothing
  if (!this.isShown(form))
    return;
  //
  // Static field has no header
  if (this.isStatic())
    return;
  //
  let show = (form ? this.showFormHeader : this.showListHeader && this.parent.getHeaderHeight() > 0) && this.isVisible(form);
  //
  let headerContainer;
  //
  if (form)
    headerContainer = Client.eleMap[this.formHeaderId];
  else
    headerContainer = this.isInList() ? Client.eleMap[this.listContainerId] : Client.eleMap[this.outListHeaderId];
  //
  if (!form && this.isInList()) {
    // If field is visible use "visibility" style property because header has to take up space also when it's not visible
    if (this.isVisible())
      headerContainer.updateElement({style: {display: "flex", visibility: show ? "visible" : "hidden"}});
    else // Otherwise use "display" property to make it totally disappear when not visible
      headerContainer.updateElement({style: {visibility: "visible", display: show ? "flex" : "none"}});
  }
  else
    headerContainer.updateElement({style: {display: show ? "flex" : "none"}});
};


/**
 * Set header above value
 * @param {Boolean} form
 */
Client.IdfField.prototype.setHeaderAbove = function (form)
{
  // If I have to set header above for a form/list field and parent panel has not form/list or field is "in-list" or it's not visible in form/list, do nothing
  if (!this.isShown(form) || (!form && this.isInList()))
    return;
  //
  // Static field has no header
  if (this.isStatic())
    return;
  //
  let headerAbove = form ? this.formHeaderAbove : this.listHeaderAbove;
  //
  // Get row container
  let rowContainer = form ? Client.eleMap[this.formContainerId] : Client.eleMap[this.outListContainerId];
  //
  // Get old class name
  let className = rowContainer.getRootObject().className || "";
  //
  // Remove "header-above" class
  className = className.split(" ").filter((c) => !c.startsWith("header-above")).join(" ");
  //
  // Add "header-above" class if needed
  className = className + " " + (headerAbove ? "header-above" : "");
  //
  className = className.trim();
  //
  rowContainer.updateElement({className});
};


/**
 * Update my image
 * @param {Boolean} form
 */
Client.IdfField.prototype.updateImage = function (form)
{
  // If I have to update header image for a form/list field and parent panel has not form/list or field is not visible in form/list, do nothing
  if (!this.isShown(form))
    return;
  //
  // Static field has no header
  if (this.isStatic())
    return;
  //
  let headerContainer;
  //
  if (form)
    headerContainer = Client.eleMap[this.formHeaderId];
  else
    headerContainer = this.isInList() ? Client.eleMap[this.listContainerId] : Client.eleMap[this.outListHeaderId];
  //
  let src = (Client.mainFrame.isIDF ? "images/" : "") + this.image;
  headerContainer.updateElement({style: {backgroundImage: "url('" + src + "')"}});
};


/**
 * Update image resize mode
 * @param {Boolean} form
 */
Client.IdfField.prototype.updateImageResizeMode = function (form)
{
  // Static field has no header
  if (this.isStatic())
    return;
  //
  let headerContainer;
  //
  if (form)
    headerContainer = Client.eleMap[this.formHeaderId];
  else
    headerContainer = this.isInList() ? Client.eleMap[this.listContainerId] : Client.eleMap[this.outListHeaderId];
  //
  let className = headerContainer.getRootObject().className || "";
  //
  // Remove old resize mode class
  className = className.split(" ").filter((c) => !c.startsWith("control-blob-img")).join(" ");
  //
  switch (this.imageResizeMode) {
    case Client.IdfControl.stretches.FILL:
      className += " control-blob-img-fill";
      break;

    case Client.IdfControl.stretches.ENLARGE:
      className += " control-blob-img-enlarge";
      break;
  }
  //
  headerContainer.updateElement({className});
};


/**
 * Update sort mode
 */
Client.IdfField.prototype.updateSortMode = function ()
{
  let listHeaderText = Client.eleMap[this.listHeaderTextId];
  if (!listHeaderText)
    return;
  //
  switch (this.sortMode) {
    case Client.IdfField.sortModes.NONE:
      Client.Widget.updateElementClassName(listHeaderText, "sort-field-indicator-up", true);
      Client.Widget.updateElementClassName(listHeaderText, "sort-field-indicator-down", true);
      break;

    case Client.IdfField.sortModes.ASC:
      Client.Widget.updateElementClassName(listHeaderText, "sort-field-indicator-down");
      Client.Widget.updateElementClassName(listHeaderText, "sort-field-indicator-up", true);
      break;

    case Client.IdfField.sortModes.DESC:
      Client.Widget.updateElementClassName(listHeaderText, "sort-field-indicator-down", true);
      Client.Widget.updateElementClassName(listHeaderText, "sort-field-indicator-up");
      break;
  }
};


/**
 * Handle sort
 * @param {Integer} sortMode
 */
Client.IdfField.prototype.handleSort = function (sortMode)
{
  // If I don't have new sort mode, calculate it
  if (sortMode === undefined) {
    switch (this.sortMode) {
      case Client.IdfField.sortModes.NONE:
        sortMode = Client.IdfField.sortModes.DESC;
        break;

      case Client.IdfField.sortModes.ASC:
        sortMode = Client.IdfField.sortModes.DESC;
        break;

      case Client.IdfField.sortModes.DESC:
        sortMode = Client.IdfField.sortModes.ASC;
        break;
    }
  }
  //
  let yck;
  let ctrlPress;
  //
  switch (sortMode) {
    case Client.IdfField.sortModes.NONE:
      ctrlPress = -1;
      break;

    case Client.IdfField.sortModes.ASC:
      yck = 10;
      break;

    case Client.IdfField.sortModes.DESC:
      yck = 20;
      break;
  }
  //
  // Apply sort
  return [{id: "clk", def: this.clickEventDef, content: {oid: this.id, par1: "cap", yck, shp: -1, ctp: ctrlPress}}];
};


/**
 * Handle grouping
 * @param {Integer} groupingMode
 */
Client.IdfField.prototype.handleGrouping = function (groupingMode)
{
  // If I don't have new grouping mode, calculate it
  if (groupingMode === undefined) {
    switch (this.groupingMode) {
      case Client.IdfField.groupingModes.NONE:
        groupingMode = Client.IdfField.groupingModes.DESC;
        break;

      case Client.IdfField.groupingModes.ASC:
        groupingMode = Client.IdfField.groupingModes.DESC;
        break;

      case Client.IdfField.groupingModes.DESC:
        groupingMode = Client.IdfField.groupingModes.ASC;
        break;
    }
  }
  //
  let yck;
  //
  switch (groupingMode) {
    case Client.IdfField.groupingModes.NONE:
      yck = 2;
      break;

    case Client.IdfField.groupingModes.ASC:
      yck = 1;
      break;

    case Client.IdfField.groupingModes.DESC:
      yck = 30;
      break;
  }
  //
  // Apply grouping
  return [{id: "clk", def: this.clickEventDef, content: {oid: this.id, par1: "cap", yck, shp: -1}}];
};


/**
 * Return true if I'm enabled
 * @param {Integer} valueIndex - for in list field, index of value to check if it's enabled
 */
Client.IdfField.prototype.isEnabled = function (valueIndex)
{
  let qbeStatus = this.parent.status === Client.IdfPanel.statuses.qbe;
  let rowQbe = this.parent.searchMode === Client.IdfPanel.searchModes.row;
  //
  // If I have a value index to check and panel status is QBE or
  // value index is "0" and qbe mode is ROW
  if ((qbeStatus && valueIndex !== undefined) || (rowQbe && valueIndex === 0)) {
    // Just values belonging to first row are enabled in QBE status
    if (qbeStatus && valueIndex > 1)
      return false;
    //
    // If I'm a simple lookup field (not auto lookup nor smart lookup), I'm not enabled
    if (this.isLookup() && !this.autoLookup && !this.smartLookup)
      return false;
    //
    // Check if I'm enabled when status is QBE
    if (!this.enabledInQbe)
      return false;
    //
    return true;
  }
  //
  // If parent panel is locked and I'm not static, I'm not enabled
  if (this.parent.locked && !this.isStatic() && this.causeValidation)
    return false;
  //
  // If I belong to a page that is not enabled, I'm not enabled
  if (this.page && !this.page.isEnabled())
    return false;
  //
  // If I belong to a group that is not enabled, I'm not enabled
  if (this.group && !this.group.isEnabled())
    return false;
  //
  let enabled = this.enabled;
  //
  // If I'm not static and I have to check a specific row index, check if value is enabled
  if (enabled && !this.isStatic() && valueIndex) {
    // If value is not enabled, I'm not enabled
    let val = this.getValueByIndex(valueIndex);
    if (val && !val.isEnabled())
      return false;
    //
    // Check if value index refers to a new row
    let isNew = valueIndex > this.parent.totalRows;
    //
    // If parent panel can insert but it cannot update, check if value index refers to a row having an inserted document.
    // In this case I have to consider that row as a new row. Use row selector type to check it
    if (!isNew && this.parent.canInsert && !this.parent.canUpdate) {
      // Row selector belongs to first field, so get it
      let firstField = this.parent.fields[0];
      let firstFieldVal = firstField.getValueByIndex(valueIndex);
      //
      // If first field value has an "inserted document" row selector, consider valueIndex-th row as new
      if (firstFieldVal) {
        let isInsertedDoc = firstFieldVal.rowSelectorType === Client.IdfFieldValue.rowSelectorTypes.INSERTED_DOC_ERROR;
        isInsertedDoc = isInsertedDoc || firstFieldVal.rowSelectorType === Client.IdfFieldValue.rowSelectorTypes.INSERTED_DOC_UPDATED;
        //
        if (isInsertedDoc)
          isNew = true;
      }
    }
    //
    // If parent panel cannot update, I'm not enabled for not new rows
    if (!isNew && !this.parent.canUpdate)
      enabled = false;
    //
    // If parent panel cannot insert, I'm not enabled for new rows
    if (isNew && !this.parent.canInsert)
      enabled = false;
    //
    // Cannot update unbound column on row inserting
    if (isNew && this.unbound)
      enabled = false;
  }
  //
  return enabled;
};


/**
 * Return true if I'm visible in given layout
 * @param {Boolean} form
 */
Client.IdfField.prototype.isVisible = function (form)
{
  if (!this.isShown(form))
    return false;
  //
  if (!this.visible)
    return false;
  //
  if (!form && this.hiddenInList)
    return false;
  //
  // If there are no visible values, I'm not visible
  if (!this.visibleValuesCount)
    return false;
  //
  // If I belong to a page that is not visible, I'm not visible
  if (this.page && !this.page.isVisible())
    return false;
  //
  // If I belong to a group that is not visible, I'm not visible
  if (this.group && !this.group.isVisible(form))
    return false;
  //
  return true;
};


/**
 * Return true if this field is static
 */
Client.IdfField.prototype.isStatic = function ()
{
  return this.type === Client.IdfField.types.STATIC;
};


/**
 * Return true if this is a lookup field
 */
Client.IdfField.prototype.isLookup = function ()
{
  return this.type >= Client.IdfField.types.LOOKUP;
};


/**
 * Add given value to values array
 * @param {IdfFieldValue} value
 */
Client.IdfField.prototype.addValue = function (value)
{
  this.values[value.index] = value;
};


/**
 * Show or hide row selectors
 * @param {Boolean} show
 */
Client.IdfField.prototype.updateRowSelectorsVisibility = function (show)
{
  for (let i = 0; i <= this.values.length; i++) {
    if (!this.values[i])
      continue;
    //
    this.values[i].updateRowSelectorsVisibility(show);
  }
};


/**
 * Show or hide multiple selection checkbox
 * @param {Boolean} show
 */
Client.IdfField.prototype.updateMultiSelVisibility = function (show)
{
  for (let i = 1; i <= this.values.length; i++) {
    if (!this.values[i])
      continue;
    //
    this.values[i].updateMultiSelVisibility(show);
  }
};


/**
 * Select or unselect index-th row
 * @param {Boolean} value
 * @param {Integer} index
 */
Client.IdfField.prototype.selectRow = function (value, index)
{
  let fieldValue = this.values[index];
  if (!fieldValue)
    return;
  //
  fieldValue.selectRow(value);
};


/**
 * Check if this field can sort
 */
Client.IdfField.prototype.isSortable = function ()
{
  return this.canSort && this.canSortFlag;
};


/**
 * Check if this field can use header QBE
 */
Client.IdfField.prototype.canUseHeaderQbe = function ()
{
  let headerQbe = this.parent.searchMode === Client.IdfPanel.searchModes.header;
  let enabledInQbe = this.enabledInQbe && (!this.isLookup() || this.autoLookup || this.smartLookup);
  //
  return headerQbe && this.parent.canSearch && (this.isSortable() || enabledInQbe) && this.dataType !== Client.IdfField.dataTypes.BLOB;
};


/**
 * Check if this field can use row QBE
 */
Client.IdfField.prototype.canUseRowQbe = function ()
{
  return this.parent.searchMode === Client.IdfPanel.searchModes.row && this.parent.canSearch && this.dataType !== Client.IdfField.dataTypes.BLOB;
};


/**
 * Handle reset cache command
 * @param {Object} options
 */
Client.IdfField.prototype.resetCache = function (options)
{
  if (this.isStatic())
    return;
  //
  let from = options.from || 1;
  let to = options.to || this.values.length;
  //
  for (let i = from; i <= to; i++) {
    let value = this.values[i];
    if (!value)
      continue;
    //
    // If I have a data block coming after reset cache, I don't have to remove values belonging to that block. It's better to reuse them, emptying their text
    if (i >= options.dataBlockStart && i <= options.dataBlockEnd) {
      value.updateElement({text: ""});
      continue;
    }
    //
    // Clear controls assigned to current value
    value.clearControls();
    //
    // Close current value and remove it from values array
    value.close(true);
    this.values[i] = undefined;
  }
};


/**
 * Update blob controls
 */
Client.IdfField.prototype.updateBlob = function ()
{
  // If data type is not BLOB, do nothing
  if (this.dataType !== Client.IdfField.dataTypes.BLOB)
    return;
  //
  this.updateControls({updateBlobCommands: true});
};


/**
 * Update visibility
 * @param {Integer} index
 */
Client.IdfField.prototype.updateVisibility = function (index)
{
  let modes = ["list", "form"];
  //
  for (let i = 0; i < modes.length; i++) {
    let form = modes[i] === "form";
    let parentColumn = form ? this.parent.getFormFieldColumn(this.id) : this.parent.getListFieldColumn(this.id);
    //
    if (!this.isShown(form) || !parentColumn)
      continue;
    //
    // Update header visibility
    this.showHeader(form);
    //
    // Update aggregate container visibility
    if (!form)
      Client.eleMap[this.aggregateContainerId]?.updateElement({style: {visibility: this.isVisible() ? "flex" : "none"}});
    //
    // Update values visibility
    let start = index ?? this.parent.getDataBlockStart();
    let end = index ?? this.getDataBlockEnd();
    for (let j = start; j <= end; j++) {
      if (!this.values[j])
        continue;
      //
      this.values[j].updateVisibility(form);
    }
  }
};


/**
 * Assign sub frame to proper control
 */
Client.IdfField.prototype.assignSubFrame = function ()
{
  for (let i = 0; i <= this.values.length; i++) {
    if (!this.values[i])
      continue;
    //
    this.values[i].assignSubFrame();
  }
};


/**
 * Notified on parent panel status change
 */
Client.IdfField.prototype.onPanelStatusChange = function ()
{
  this.updateControls({isInQbe: true});
};


/**
 * Return true if this field is shown in given layout
 * @param {Boolean} form
 */
Client.IdfField.prototype.isShown = function (form)
{
  if (form)
    return this.parent.hasForm && this.showInForm;
  else
    return this.parent.hasList && this.showInList;
};


/**
 * Handle a resize event
 */
Client.IdfField.prototype.handleResize = function ()
{
  let events = [];
  let rects;
  let width, height, left, top;
  //
  if (this.isVisible() && this.parent.layout === Client.IdfPanel.layouts.list) {
    rects = this.getRects({real: true});
    width = isNaN(rects.width) ? undefined : Math.floor(rects.width);
    height = isNaN(rects.height) ? undefined : Math.floor(rects.height);
    left = isNaN(rects.left) ? undefined : Math.floor(rects.left);
    top = isNaN(rects.top) ? undefined : Math.floor(rects.top);
    //
    if (width && height) {
      if (this.lastListWidth !== width || this.lastListHeight !== height || this.lastListLeft !== left || this.lastListTop !== top) {
        events.push({
          id: "resize",
          def: Client.IdfMessagesPump.eventTypes.ACTIVE,
          content: {
            oid: this.id,
            obn: "list",
            par1: width,
            par2: height,
            par3: left,
            par4: top
          }
        });
      }
      //
      this.lastListWidth = width;
      this.lastListHeight = height;
      this.lastListLeft = left;
      this.lastListTop = top;
    }
  }
  //
  if (this.isVisible(true) && this.parent.layout === Client.IdfPanel.layouts.form) {
    rects = this.getRects({form: true, real: true});
    width = isNaN(rects.width) ? undefined : Math.floor(rects.width);
    height = isNaN(rects.height) ? undefined : Math.floor(rects.height);
    left = isNaN(rects.left) ? undefined : Math.floor(rects.left);
    top = isNaN(rects.top) ? undefined : Math.floor(rects.top);
    //
    if (width && height) {
      if (this.lastFormWidth !== width || this.lastFormHeight !== height || this.lastFormLeft !== left || this.lastFormTop !== top) {
        events.push({
          id: "resize",
          def: Client.IdfMessagesPump.eventTypes.ACTIVE,
          content: {
            oid: this.id,
            obn: "form",
            par1: width,
            par2: height,
            par3: left,
            par4: top
          }
        });
      }
      //
      this.lastFormWidth = width;
      this.lastFormHeight = height;
      this.lastFormLeft = left;
      this.lastFormTop = top;
    }
  }
  //
  return events;
};


/**
 * Handle selection change event
 * @param {Object} event
 */
Client.IdfField.prototype.handleSelectionChange = function (event)
{
  let events = [];
  //
  if (!this.notifySelectionChange || this.parent.status === Client.IdfPanel.statuses.qbe)
    return events;
  //
  let range = document.getSelection().getRangeAt(0);
  if (Client.mainFrame.isIDF)
    events.push({
      id: "txtsel",
      def: this.parent.selectionChangeEventDef,
      delay: 250,
      content: {
        oid: this.id,
        par1: range.startOffset,
        par2: range.endOffset
      }
    });
  //
  return events;
};


/**
 * Give focus to the element
 * @param {Object} options
 */
Client.IdfField.prototype.focus = function (options)
{
  options = options || {};
  //
  let row;
  if (options.row === undefined || this.parent.layout === Client.IdfPanel.layouts.form)
    row = this.parent.getActiveRowIndex();
  else if (options.absoluteRow !== undefined)
    row = options.absoluteRow;
  else
    row = this.parent.actualPosition + options.row;
  //
  delete options.row;
  delete options.absoluteRow;
  //
  this.values[row].focus(options);
};


Client.IdfField.prototype.isDraggable = function (element)
{
  // A Field is draggable IF
  // - you can reorder the columns AND you are on the list header
  // - the panel can drag AND you are on a disablef field (form/list)
  if (this.parent.canReorderColumn && this.isInList() && Client.Utils.isMyParent(element.getRootObject(), this.listContainerId))
    return true;
  //
  return false;
};


Client.IdfField.prototype.canResizeW = function (element)
{
  return this.parent.canResizeColumn && this.isInList() && Client.Utils.isMyParent(element.getRootObject(), this.listContainerId);
};


Client.IdfField.prototype.applyDragDropCursor = function (cursor)
{
  // Apply the resize cursor only on the list header
  let obj = Client.eleMap[this.listContainerId]?.getRootObject();
  if (!obj)
    return;
  //
  if (cursor === "e-resize" || cursor === "w-resize") {
    obj.setAttribute("opnt", "dd");
    obj.style.cursor = cursor;
    obj.classList.add("system-cursor");
    //
    // Clear the cursor on mouse leave
    if (!obj.onmouseleave)
      obj.onmouseleave = Client.Widget.ddClearPointer;
  }
  else if (obj.getAttribute("opnt")) {
    // I already set a cursor on the object BUT now i have no operation : clear the cursor
    obj.style.cursor = "";
    obj.setAttribute("opnt", "");
    obj.classList.remove("system-cursor");
  }
};


Client.IdfField.prototype.canBeDroppedOn = function (widget, targetDomElement, draggedDomElement)
{
  // If the drag operation was started from the list header and the panel can reorder columns we can drop only onto other fields list headers
  if (this.isInList() && this.parent.canReorderColumn && Client.Utils.isMyParentEl(draggedDomElement, Client.eleMap[this.listContainerId])) {
    if (widget instanceof Client.IdfField && widget.isInList() && this.parent === widget.parent && Client.Utils.isMyParentEl(targetDomElement, Client.eleMap[widget.listContainerId]))
      return true;
    else
      return false;
  }
  //
  // Normally we can drop everywhere
  return true;
};


Client.IdfField.prototype.acceptsDrop = function (widget, targetDomElement)
{
  // A field can accept the drop if
  // - the widget is a list field, i'm a list field AND the panel can reorder list and we are of the same panel and the target IS the field header
  // - the panel accepts a generic drop
  if (this.parent.canReorderColumn && widget instanceof Client.IdfField && this.isInList() && widget.isInList() && widget.parent === this.parent && Client.Utils.isMyParentEl(targetDomElement, Client.eleMap[this.listContainerId]))
    return true;
  //
  return false;
};


Client.IdfField.prototype.handleDrop = function (dragWidget, droppedElement, x, y, ev)
{
  this.parent.reorderList(dragWidget, this);
};


Client.IdfField.prototype.onTransform = function (options)
{
  this.updateElement({listWidth: options.w});
  //
  // tell the server the new dimensions
  let events = [];
  if (Client.mainFrame.isIDF) {
    events.push({
      id: "rescol",
      def: Client.IdfMessagesPump.eventTypes.ACTIVE,
      content: {
        oid: this.id,
        obn: "",
        par1: Math.round(options.w)
      }
    });
  }
  else {
    // TODO
  }
  Client.mainFrame.sendEvents(events);
};


/**
 * Sometimes the server sends an ID with this format:
 * this.id +
 *           ":fv" - the form value control
 *           ":lv1 ... :lv99" - the list control of the requested ROW
 *           ":lc" - list caption
 *           ":fc" - form caption
 * @param {String} id
 *
 * @returns {DomNode} the domObj relative to the ID
 */
Client.IdfField.prototype.getDomObjFromId = function (id)
{
  if (id.includes(":fv") || (id.includes(":fc") && this.isStatic()))
    return Client.eleMap[this.formValueId].getRootObject();
  else if (id.includes(":fc"))
    return Client.eleMap[this.formHeaderId].getRootObject();
  else if (id.includes(":lc"))
    return Client.eleMap[this.listContainerId].getRootObject();
  else if (id.includes(":lv")) {
    let pos = parseInt(id.substring(id.indexOf(":lv") + 3));
    return this.values[pos].listContainer.getRootObject();
  }
};


/**
 * Return the target to use for opening a popup on this widget
 * @param {String} targetId
 * @returns {DomNode}
 */
Client.IdfField.prototype.getPopupTarget = function (targetId)
{
  // targetId could be a string that refers to a particular object: form value (fv), list value (lv1...99) or captions (lc, fc)
  // So if there is a targetId and its different from my id, I have to get the proper object
  if (targetId && targetId !== this.id)
    return this.getDomObjFromId(targetId);
  //
  if (this.isStatic())
    return this.values[1].getPopupTarget();
};


/**
 * Return true if the field can be focused
 * @param {Number} row
 */
Client.IdfField.prototype.canHaveFocus = function (row)
{
  if (!this.isVisible(this.parent.layout))
    return false;
  //
  // In the search line only the fields in the list
  if (row === 0 && this.parent.layout === Client.IdfPanel.layouts.list && !this.isInList())
    return false;
  //
  return true;
};


/**
 * Open filter popup
 */
Client.IdfField.prototype.openFilterPopup = function ()
{
  // If I cannot use header or row qbe, don't open popup
  if (!this.canUseHeaderQbe() && !this.canUseRowQbe())
    return;
  //
  // If panel mode is LIST and I'm not shown in list mode, don't open popup
  if (this.parent.layout === Client.IdfPanel.layouts.list && !this.isShown())
    return;
  //
  // If panel mode is FORM and I'm not shown in form mode, don't open popup
  if (this.parent.layout === Client.IdfPanel.layouts.form && !this.isShown(true))
    return;
  //
  // Open filter popup
  new Client.IdfFilterPopup({id: "filter-popup", field: this}, this.view, this.view);
};


/**
 * Open combo
 */
Client.IdfField.prototype.openCombo = function ()
{
  // If panel mode is LIST and I'm not shown in list mode, don't open combo
  if (this.parent.layout === Client.IdfPanel.layouts.list && !this.isShown())
    return;
  //
  // If panel mode is FORM and I'm not shown in form mode, don't open combo
  if (this.parent.layout === Client.IdfPanel.layouts.form && !this.isShown(true))
    return;
  //
  // Get active row field value and open combo
  let activeValue = this.values[this.parent.getActiveRowIndex()];
  activeValue.openCombo();
};


/**
 * Handle function keys
 * @param {Object} event
 */
Client.IdfField.prototype.handleFunctionKeys = function (event)
{
  // If I have a command attached, I see if the key is for it
  let events = [];
  //
  let form = this.parent.layout === Client.IdfPanel.layouts.form;
  if (this.command && this.isEnabled(form) && this.isVisible(form))
    events.push(...Client.eleMap[this.command].handleFunctionKeys(event, -1, -1));
  //
  return events;
};


/**
 * Update visible values count
 * @param {Boolean} visible
 */
Client.IdfField.prototype.updateVisibleValuesCount = function (visible)
{
  this.visibleValuesCount += visible ? 1 : -1;
};