/*
 * Instant Developer Cloud
 * Copyright Pro Gamma Spa 2000-2021
 * All rights reserved
 */

var Client = Client || {};


/**
 * @class A frame object
 * @param {Object} widget
 * @param {View|Element} parent - the parent element
 * @param {View} view
 */
Client.IdfFrame = function (widget, parent, view)
{
  // If no widget do nothing.
  // I need this protection because there are some classes (ex: idfButtonBar) that extend this class
  if (!widget)
    return;
  //
  this.children = widget.children;
  //
  // Set default values
  widget = Object.assign({
    caption: "",
    icon: "",
    vertical: false,
    onlyContent: !this.isLeaf(),
    showBorder: false,
    collapsible: true,
    collapsed: false,
    lockable: false,
    locked: !Client.mainFrame.isIDF,
    visible: true,
    enabled: true,
    showToolbar: true,
    showStatusbar: false,
    showScrollbar: Client.IdfFrame.scrollbarTypes.both,
    smallIcons: false,
    className: "",
    canDrag: false,
    canDrop: false,
    //
    // Set default events definition
    collapseEventDef: (Client.mainFrame.isIDF ? Client.IdfMessagesPump.eventTypes.ACTIVE : undefined),
    mouseClickEventDef: (Client.mainFrame.isIDF ? Client.IdfMessagesPump.eventTypes.CLIENTSIDE : undefined),
    mouseDoubleClickEventDef: (Client.mainFrame.isIDF ? Client.IdfMessagesPump.eventTypes.CLIENTSIDE : undefined),
    collapseAnimationDef: Client.IdfWebEntryPoint.getAnimationDefault("frame")
  }, widget);
  //
  this.defaultGridClass = Client.mainFrame?.wep?.defaultResponsiveClass || Client.Utils.getCSSVarValue("--default-responsive-class");
  //
  let realHandledKeys = widget.handledKeys;
  widget.handledKeys = -1;
  //
  Client.Widget.call(this, widget, parent, view);
  //
  this.handledKeys = realHandledKeys;
};


// Make Client.IdfFrame extend Client.Widget
Client.IdfFrame.prototype = new Client.Widget();


Client.IdfFrame.transPropMap = {
  ver: "vertical",
  fr1: "frame1",
  fr2: "frame2",
  orw: "originalWidth",
  orh: "originalHeight",
  cms: "commandSet",
  ocn: "onlyContent",
  frb: "showBorder",
  clp: "collapsible",
  col: "collapsed",
  lkb: "lockable",
  lok: "locked",
  vis: "visible",
  ena: "enabled",
  img: "image",
  stb: "showToolbar",
  ssb: "showStatusbar",
  smi: "smallIcons",
  scr: "showScrollbar",
  dcl: "deleteFrame",
  clc: "collapseEventDef",
  mck: "mouseClickEventDef",
  mdk: "mouseDoubleClickEventDef",
  cla: "collapseAnimationDef",
  dra: "canDrag",
  dro: "canDrop"
};


Client.IdfFrame.scrollbarTypes = {
  none: 0,
  horizontal: 1,
  vertical: 2,
  both: 3
};


/**
 * Create element configuration from xml
 * @param {XmlNode} xml
 */
Client.IdfFrame.createConfigFromXml = function (xml)
{
  return {isSubFrame: xml.nodeName === "suf"};
};


/**
 * Convert properties values
 * @param {Object} props
 */
Client.IdfFrame.convertPropValues = function (props)
{
  props = props || {};
  //
  for (let p in props) {
    switch (p) {
      case Client.IdfFrame.transPropMap.ver:
      case Client.IdfFrame.transPropMap.ocn:
      case Client.IdfFrame.transPropMap.frb:
      case Client.IdfFrame.transPropMap.clp:
      case Client.IdfFrame.transPropMap.col:
      case Client.IdfFrame.transPropMap.lkb:
      case Client.IdfFrame.transPropMap.lok:
      case Client.IdfFrame.transPropMap.vis:
      case Client.IdfFrame.transPropMap.ena:
      case Client.IdfFrame.transPropMap.stb:
      case Client.IdfFrame.transPropMap.ssb:
      case Client.IdfFrame.transPropMap.dcl:
      case Client.IdfFrame.transPropMap.dra:
      case Client.IdfFrame.transPropMap.dro:
        props[p] = props[p] === "1";
        break;

      case Client.IdfFrame.transPropMap.orw:
      case Client.IdfFrame.transPropMap.orh:
      case Client.IdfFrame.transPropMap.clc:
      case Client.IdfFrame.transPropMap.scr:
      case Client.IdfFrame.transPropMap.mck:
      case Client.IdfFrame.transPropMap.mdk:
        props[p] = parseInt(props[p]);
        break;
    }
  }
};


/**
 * Get root object. Root object is the object where children will be inserted
 * @param {Boolean} el - if true, get the element itself istead of its domObj
 */
Client.IdfFrame.prototype.getRootObject = function (el)
{
  let rootObject = this.mainObjects[0].elements[1];
  return el ? rootObject : rootObject.domObj;
};


/**
 * Create elements configuration
 * @param {Object} widget
 */
Client.IdfFrame.prototype.createElementsConfig = function (widget)
{
  // Create main container configuration
  let className = "frame-container collapsible-container";
  if (!this.isLeaf())
    className += " wireframe";
  if (this.isSubFrame)
    className += " subframe-container";
  if (this.inPreview)
    className += " inpreview";
  //
  this.mainContainerConf = this.createElementConfig({c: "Container", className, events: ["onClick", "onDblclick", "onContextmenu"]});
  this.mainContainerConf.animations = [
    {trigger: "animation", prop: "collapseElement", duration: (widget.collapseAnimationDef.indexOf("none") === 0 ? 0 : 250)},
    {trigger: "animation", prop: "expandElement", duration: (widget.collapseAnimationDef.indexOf("none") === 0 ? 0 : 250)}
  ];
  //
  // Create toolbar configuration
  this.createToolbarConfig();
  //
  // Create content container configuration
  this.contentContainerConf = this.createElementConfig({c: "Container", className: "frame-content"});
  this.mainContainerConf.children.push(this.contentContainerConf);
};


/**
 * Create toolbar configuration
 */
Client.IdfFrame.prototype.createToolbarConfig = function ()
{
  // Create toolbar configuration
  this.toolbarConf = this.createElementConfig({c: "IonItem", type: "header", className: "frame-toolbar" + (this.isSubFrame ? " subframe-toolbar" : "")});
  this.mainContainerConf.children.push(this.toolbarConf);
  //
  // Create menubutton configuration
  this.menuButtonConf = this.createElementConfig({c: "IonButton", icon: "menu", className: "generic-btn frame-toolbar-btn frame-menu-btn", events: ["onClick"], visible: false});
  this.toolbarConf.children.push(this.menuButtonConf);
  //
  // Create collapse button configuration
  this.collapseButtonConf = this.createElementConfig({c: "IonButton", icon: "arrow-dropup", className: "generic-btn frame-toolbar-btn frame-collapse-btn", events: ["onClick"]});
  this.toolbarConf.children.push(this.collapseButtonConf);
  //
  // Create lock button configuration
  this.lockButtonConf = this.createElementConfig({c: "IonButton", icon: "lock", className: "generic-btn frame-toolbar-btn frame-lock-btn", events: ["onClick"]});
  this.toolbarConf.children.push(this.lockButtonConf);
  //
  // Create icon button configuration
  this.iconButtonConf = this.createElementConfig({c: "IonButton", className: "generic-btn frame-toolbar-btn frame-icon-btn"});
  this.toolbarConf.children.push(this.iconButtonConf);
  //
  // Create title configuration
  this.titleConf = this.createElementConfig({c: "IonLabel", className: "frame-title"});
  this.toolbarConf.children.push(this.titleConf);
  //
  // Create caption configuration
  this.captionConf = this.createElementConfig({c: "Span", className: "frame-caption"});
  this.titleConf.children.push(this.captionConf);
};


/**
 * Realize widget UI
 * @param {Object} widget
 * @param {View|Element|Widget} parent
 * @param {View} view
 */
Client.IdfFrame.prototype.realize = function (widget, parent, view)
{
  // Create elements configuration
  this.createElementsConfig(widget);
  //
  // Create the main container
  this.mainObjects.push(view.createElement(this.mainContainerConf, parent, view));
  this.mainObjects[0].getRootObject()?.setAttribute("spellcheck", "false");
  //
  // Create widget children
  this.createChildren(widget);
};


/**
 * Create children elements
 * @param {Object} el
 */
Client.IdfFrame.prototype.createChildren = function (el)
{
  Client.Widget.prototype.createChildren.call(this, el);
  //
  if (this.elements[0] instanceof Client.IdfCustomElement) {
    let controlConf = this.createElementConfig({c: "IdfControl", customElement: this.customElement, container: this.getRootObject(true)});
    this.control = this.insertBefore({child: controlConf});
  }
  //
};


/**
 * Update inner elements properties
 * @param {Object} props
 */
Client.IdfFrame.prototype.updateElement = function (props)
{
  let methodsToCall = {};
  //
  props = props || {};
  //
  Client.Widget.prototype.updateElement.call(this, props);
  //
  let mainContainer = Client.eleMap[this.mainContainerConf.id];
  //
  if (props.width !== undefined) {
    this.width = isNaN(props.width) ? undefined : props.width;
    methodsToCall.calcLayout = true;
  }
  //
  if (props.height !== undefined) {
    this.height = isNaN(props.height) ? undefined : props.height;
    methodsToCall.calcLayout = true;
  }
  //
  if (props.caption !== undefined)
    methodsToCall.updateToolbar = true;
  //
  if (props.vertical !== undefined) {
    this.vertical = props.vertical;
    //
    let el = Client.eleMap[this.contentContainerConf.id];
    el.updateElement({className: "frame-content " + (!props.vertical ? "horizontal" : (this.frame1 ? "vertical" : ""))});
  }
  //
  if (props.onlyContent !== undefined) {
    this.onlyContent = props.onlyContent;
    //
    let el = Client.eleMap[this.toolbarConf.id];
    el.updateElement({visible: !props.onlyContent});
  }
  //
  if (props.showBorder !== undefined) {
    this.showBorder = props.showBorder;
    Client.Widget.updateElementClassName(mainContainer, "frame-border", !this.showBorder);
  }
  //
  if (props.collapsible !== undefined) {
    this.collapsible = props.collapsible;
    methodsToCall.updateToolbar = true;
  }
  //
  if (props.collapsed !== undefined) {
    this.setCollapsed(props.collapsed);
    methodsToCall.updateToolbar = true;
  }
  //
  if (props.lockable !== undefined) {
    this.lockable = props.lockable;
    methodsToCall.updateToolbar = true;
  }
  //
  if (props.locked !== undefined) {
    // Remove old class
    Client.Widget.updateElementClassName(mainContainer, this.locked ? "locked" : "unlocked", true);
    //
    this.locked = props.locked;
    methodsToCall.updateToolbar = true;
    //
    // Add new class
    Client.Widget.updateElementClassName(mainContainer, this.locked ? "locked" : "unlocked");
    //
    methodsToCall.applyVisualStyle = true;
  }
  //
  if (props.visible !== undefined) {
    this.visible = props.visible;
    mainContainer.updateElement({visible: this.visible});
  }
  //
  if (props.enabled !== undefined)
    this.enabled = props.enabled;
  if (props.canDrag !== undefined)
    this.canDrag = props.canDrag;
  if (props.canDrop !== undefined)
    this.canDrop = props.canDrop;
  //
  if (props.image !== undefined) {
    this.image = props.image;
    methodsToCall.updateToolbar = true;
  }
  //
  if (props.showToolbar !== undefined) {
    this.showToolbar = props.showToolbar;
    methodsToCall.updateToolbar = true;
  }
  //
  if (props.showStatusbar !== undefined) {
    this.showStatusbar = props.showStatusbar;
    methodsToCall.updateToolbar = true;
  }
  //
  if (props.smallIcons !== undefined) {
    this.smallIcons = props.smallIcons;
    this.handleSmallIcons();
    methodsToCall.calcLayout = true;
  }
  if (props.showScrollbar !== undefined) {
    // Remove old class
    let scrollClasses = ["scroll-none", "scroll-horizontal", "scroll-vertical", "scroll-both"];
    Client.Widget.updateElementClassName(mainContainer, scrollClasses[this.showScrollbar], true);
    //
    this.showScrollbar = props.showScrollbar;
    //
    // Add new class
    Client.Widget.updateElementClassName(mainContainer, scrollClasses[this.showScrollbar]);
  }
  //
  if (props.className !== undefined) {
    // Remove old class
    Client.Widget.updateElementClassName(mainContainer, this.className, true);
    if (this.gridClass)
      Client.Widget.updateElementClassName(mainContainer, this.gridClass, true);
    //
    this.className = props.className;
    //
    // The className can have a responsive grid, in that case we must extract it
    let cls = Client.Widget.extractGridClasses(this.className);
    this.className = cls.className;
    this.gridClass = cls.gridClass || this.defaultGridClass;
    //
    // Add new class
    Client.Widget.updateElementClassName(mainContainer, this.className);
    if (this.gridClass)
      Client.Widget.updateElementClassName(mainContainer, this.gridClass);
  }
  //
  if (props.collapseEventDef !== undefined)
    this.collapseEventDef = props.collapseEventDef;
  if (props.mouseClickEventDef !== undefined)
    this.mouseClickEventDef = props.mouseClickEventDef;
  if (props.mouseDoubleClickEventDef !== undefined)
    this.mouseDoubleClickEventDef = props.mouseDoubleClickEventDef;
  //
  let methods = Object.keys(methodsToCall);
  for (let i = 0; i < methods.length; i++) {
    let method = methods[i];
    //
    if (method === "calcLayout") {
      // Skip calcLayout if I'm realizing and my parent is a frame or a view.
      // In this case my parent will calculate layout when all children will have been created
      if (this.realizing && (this.parent instanceof Client.IdfFrame || this.parent instanceof Client.IdfView))
        continue;
    }
    //
    // If I'm an extended class having current method defined in my prototype, I'll call that method during my updateElement
    if (this[method] !== Client.IdfFrame.prototype[method])
      props[method] = true;
    else // Otherwise call base method
      this[method]();
  }
  //
  if (props.deleteFrame)
    this.close(true);
};


/**
 * Handle an event
 * @param {Object} event
 */
Client.IdfFrame.prototype.onEvent = function (event)
{
  let events = [];
  //
  if (event.content instanceof Array && this.customElement) {
    events.push(...this.customElement.onEvent(event));
    events.forEach(e => e.content.oid = this.id);
  }
  else {
    events.push(...Client.Widget.prototype.onEvent.call(this, event));
    //
    switch (event.id) {
      case "chgProp":
        if (this.customElement)
          events.push(...this.customElement.onEvent(event));
        break;

      case "onClick":
      case "onDblclick":
      case "onContextmenu":
        events.push(...this.handleFrameClick(event));
        //
        if (event.id === "onClick") {
          if (event.obj === this.collapseButtonConf.id)
            events.push(...this.handleCollapseButtonClick(event));
          //
          if (event.obj === this.menuButtonConf.id)
            events.push(...this.handleMenuButtonClick(event));
        }
        break;
    }
  }
  //
  return events;
};


/**
 * Handle click on collapse button
 * @param {Object} event
 */
Client.IdfFrame.prototype.handleCollapseButtonClick = function (event)
{
  let events = [];
  //
  // If I'm on IDF, change collapsed status just if collapse event has to be handled client side too
  if (!Client.mainFrame.isIDF || Client.IdfMessagesPump.isClientSideEvent(this.collapseEventDef))
    this.updateElement({collapsed: !this.collapsed});
  //
  // Send collapse event
  if (Client.mainFrame.isIDF)
    events.push({
      id: "col",
      def: this.collapseEventDef,
      content: {
        oid: this.id,
        obn: this.collapsed ? "col" : "exp",
        xck: event.content.offsetX,
        yck: event.content.offsetY
      }
    });
  else // On IDC send onCollapse event
    events.push({
      id: "onCollapse",
      obj: this.id,
      content: this.collapsed
    });
  //
  return events;
};


/**
 * Handle click on menu button
 * @param {Object} event
 */
Client.IdfFrame.prototype.handleMenuButtonClick = function (event)
{
  let events = [];
  //
  if (!this.owner) // Menu button: Show the menu
    Client.mainFrame.wep?.commandList.toggleMenu();
  else if (this.parentIdfView)    // BackButton: close this view
    events.push(...this.parentIdfView.handleCloseButtonClick());
  //
  return events;
};


/**
 * Get widget requirements
 * @param {Object} w
 */
Client.IdfFrame.getRequirements = function (w)
{
  return Client.IdfView.getRequirements(w);
};


/**
 * Calculate layout rules to handle resize mode
 */
Client.IdfFrame.prototype.calcLayout = function ()
{
  let style = {};
  //
  // Get default min height
  let minHeight = this.onlyContent ? 0 : this.getToolbarHeight();
  //
  if (this.isSubFrame && this.parent instanceof Client.IdfPanel)
    style["min-height"] = minHeight + "px";
  else {
    // If there is a width, set it using resize mode rules
    if (this.originalWidth !== undefined) {
      if (this.parentIdfView?.resizeWidth === Client.IdfView.resizeModes.NONE) {
        style["max-width"] = this.originalWidth + "px";
        style.width = this.originalWidth + "px";
      }
      else if (this.parentIdfView?.resizeWidth === Client.IdfView.resizeModes.EXTEND && !this.isLeaf())
        style["min-width"] = this.originalWidth + "px";
    }
    //
    // If width has not be set, set it using percentage
    if (!style.width) {
      let widthPerc = 100;
      //
      if (this.parent.getChildPercentageWidth)
        widthPerc = this.parent.getChildPercentageWidth(this.originalWidth);
      //
      style.width = widthPerc + "%";
      style["max-width"] = widthPerc + "%";
      //
      let mainContainer = Client.eleMap[this.mainContainerConf.id].getRootObject();
      this.width = (widthPerc * (this.parent.width || mainContainer.clientWidth)) / 100;
    }
    //
    // If there is a height, set it using resize mode rules
    if (this.originalHeight !== undefined) {
      if (this.parentIdfView?.resizeHeight === Client.IdfView.resizeModes.NONE) {
        style.height = this.originalHeight + "px";
        style["max-height"] = this.originalHeight + "px";
      }
      else if (this.parentIdfView?.resizeHeight === Client.IdfView.resizeModes.EXTEND && !this.isLeaf())
        style["min-height"] = (this.originalHeight < minHeight ? minHeight : this.originalHeight) + "px";
    }
    //
    // If min height has not be set, set it using default one
    if (!style["min-height"])
      style["min-height"] = minHeight + "px";
    //
    // If height has not be set, set it using percentage
    if (!style.height) {
      let heightPerc = 100;
      //
      if (this.parent.getChildPercentageHeight)
        heightPerc = this.parent.getChildPercentageHeight(this.originalHeight);
      //
      style.height = heightPerc + "%";
      //
      let mainContainer = Client.eleMap[this.mainContainerConf.id].getRootObject();
      this.height = (heightPerc * (this.parent.height || mainContainer.clientHeight)) / 100;
    }
  }
  //
  // Update main container style
  let el = Client.eleMap[this.mainContainerConf.id];
  if (el)
    el.updateElement({style: style});
  //
  // Don't calculate children layout in case of IdfPanel. The panel itself will handle its children
  if (this instanceof Client.IdfPanel)
    return;
  //
  // Tell my children to calculate their layout
  for (let i = 0; i < this.elements.length; i++) {
    if (this.elements[i].calcLayout)
      this.elements[i].calcLayout();
  }
};


/**
 * Get child width as a percentage of parent width
 * @param {Integer} childWidth
 * @param {Integer} parentWidth
 */
Client.IdfFrame.prototype.getChildPercentageWidth = function (childWidth, parentWidth)
{
  parentWidth = parentWidth || this.width;
  if (parentWidth === undefined) {
    let mainContainer = Client.eleMap[this.mainContainerConf.id].getRootObject();
    parentWidth = mainContainer.clientWidth;
  }
  //
  // If child has not an explicit width, calculate it
  if (childWidth === undefined) {
    // If this frame has an horizontal alignment, assign an equal portion of residual width to all children having no width
    if (!this.vertical) {
      let noWidthChildren = 0;
      //
      // Calculate residual width subtracting children widths from parent width
      let residualWidth = parentWidth;
      for (let i = 0; i < this.elements.length; i++) {
        let elWidth = this.elements[i].width;
        residualWidth -= (elWidth || 0);
        //
        // Count children having no width
        if (elWidth === undefined)
          noWidthChildren++;
      }
      //
      // Set child width as a portion of residual width
      childWidth = residualWidth / noWidthChildren;
    }
    else // Otherwise use parent width has child width
      childWidth = parentWidth;
  }
  //
  let percentageWidth = 100;
  if (parentWidth)
    percentageWidth = (childWidth / parentWidth) * 100;
  //
  return percentageWidth;
};


/**
 * Get child height as a percentage of parent height
 * @param {Integer} childHeight
 * @param {Integer} parentHeight
 */
Client.IdfFrame.prototype.getChildPercentageHeight = function (childHeight, parentHeight)
{
  parentHeight = parentHeight || this.height;
  if (parentHeight === undefined) {
    let mainContainer = Client.eleMap[this.mainContainerConf.id].getRootObject();
    parentHeight = mainContainer.clientHeight;
  }
  //
  // If child has not an explicit height, calculate it
  if (childHeight === undefined) {
    // If this frame has a vertical alignment, assign an equal portion of residual heigth to all children having no height
    if (this.vertical) {
      let noHeightChildren = 0;
      //
      // Calculate residual height subtracting children heights from parent height
      let residualHeight = parentHeight;
      for (let i = 0; i < this.elements.length; i++) {
        let elHeight = this.elements[i].height;
        residualHeight -= (elHeight || 0);
        //
        // Count children having no height
        if (elHeight === undefined)
          noHeightChildren++;
      }
      //
      // Set child height as a portion of residual height
      childHeight = residualHeight / noHeightChildren;
    }
    else // Otherwise use parent height has child height
      childHeight = parentHeight;
  }
  //
  let percentageHeight = 100;
  if (parentHeight)
    percentageHeight = (childHeight / parentHeight) * 100;
  //
  // If there is only one child, it has to fill all available height
  if (this.elements.length === 1)
    percentageHeight = 100;
  //
  return percentageHeight;
};


/**
 * Returns true if this frame is a leaf
 */
Client.IdfFrame.prototype.isLeaf = function ()
{
  let isLeaf = true;
  //
  // If at least on of my children is an IdfFrame (or a derived class), it means I'm not a leaf
  let children = this.children || [];
  for (let i = 0; i < children.length; i++) {
    if (Client.Widget.isFrameClass(children[i].c) || Client.Widget.isFrameClass(children[i].class)) {
      isLeaf = false;
      break;
    }
  }
  //
  return isLeaf;
};


/**
 * Get toolbar height
 */
Client.IdfFrame.prototype.getToolbarHeight = function ()
{
  let isEditing = Client.mainFrame?.isEditing();
  let toolbarId = (isEditing ? "dmo_" : "") + this.toolbarConf.id;
  //
  return document.getElementById(toolbarId).offsetHeight;
};


/**
 * Collapse/expand frame content
 * @param {Boolean} collapsed
 */
Client.IdfFrame.prototype.setCollapsed = function (collapsed)
{
  let mainContainer = Client.eleMap[this.mainContainerConf.id];
  //
  // Remove old class
  Client.Widget.updateElementClassName(mainContainer, this.collapsed ? "collapsed" : "expanded", true);
  //
  this.collapsed = collapsed;
  //
  // Add new class
  Client.Widget.updateElementClassName(mainContainer, this.collapsed ? "collapsed" : "expanded");
  //
  clearTimeout(this.overflowTimer);
  let contentEl = Client.eleMap[this.contentContainerConf.id];
  if (this.collapsed) {
    contentEl.updateElement({style: {overflow: "hidden"}});
    if (this.inPreview)
      setTimeout(() => this.parent.removeChild(this), 250);
  }
  else
    this.overflowTimer = setTimeout(() => contentEl.updateElement({style: {overflow: ""}}), 250);
};


/**
 * Update toolbar
 */
Client.IdfFrame.prototype.updateToolbar = function ()
{
  let small = this.smallIcons ? " small" : "";
  //
  // Update collapse button
  let collapsible = this.collapsible;
  //
  // If my parent is an IdfTabbedView, I can never collapse
  if (this.parent instanceof Client.IdfTabbedView || Client.mainFrame.idfMobile)
    collapsible = false;
  //
  let collapseButton = Client.eleMap[this.collapseButtonConf.id];
  collapseButton.updateElement({visible: collapsible, icon: this.collapsed ? "arrow-dropdown" : "arrow-dropup", tooltip: this.getTooltip(this.collapseButtonConf.id)});
  //
  // Update lock button
  let lockButton = Client.eleMap[this.lockButtonConf.id];
  lockButton.updateElement({visible: (this.lockable && !this.collapsed), icon: this.locked ? "lock" : "unlock", tooltip: this.getTooltip(this.lockButtonConf.id)});
  //
  // Update icon button
  let iconButton = Client.eleMap[this.iconButtonConf.id];
  //
  if (this.image) {
    Client.Widget.setIconImage(this.image, iconButton);
    iconButton.updateElement({className: "generic-btn frame-toolbar-btn frame-icon-btn" + (Client.Widget.isIconImage(this.image) ? "" : " image") + small});
  }
  //
  // If there is an icon and I'm not collapsed, show icon element
  iconButton.updateElement({visible: !!this.image && !this.collapsed});
  //
  // Update caption
  let caption = this.caption;
  //
  // If I have a caption and status bar is visible and I'm not collapsed, add ":" to caption
  if (this.caption && this.showStatusbar && !this.collapsed)
    caption += ":";
  //
  // Set caption on caption element
  let captionEl = Client.eleMap[this.captionConf.id];
  captionEl.updateElement({innerHTML: this.getHTMLIcon(caption)});
};


/**
 * Handle small icons changing css classes to toolbar elements
 */
Client.IdfFrame.prototype.handleSmallIcons = function ()
{
  // Set "small" class on toolbar
  let toolbar = Client.eleMap[this.toolbarConf.id];
  if (this.smallIcons)
    toolbar.domObj.classList.add("small");
  else
    toolbar.domObj.classList.remove("small");
  //
  toolbar.updateElement({className: toolbar.domObj.className});
  //
  // Set "small" class on toolbar children
  for (let i = 0; i < toolbar.elements.length; i++) {
    let el = toolbar.elements[i];
    if (this.smallIcons)
      el.domObj.classList.add("small");
    else
      el.domObj.classList.remove("small");
    //
    el.updateElement({className: el.domObj.className});
  }
  //
  // Set "small" class on title
  let title = Client.eleMap[this.titleConf.id];
  if (this.smallIcons)
    title.domObj.classList.add("small");
  else
    title.domObj.classList.remove("small");
  //
  title.updateElement({className: title.domObj.className});
  //
  // Set "small" class on caption
  let caption = Client.eleMap[this.captionConf.id];
  if (this.smallIcons)
    caption.domObj.classList.add("small");
  else
    caption.domObj.classList.remove("small");
  //
  caption.updateElement({className: caption.domObj.className});
};


/**
 * Get tooltip for given object
 * @param {String} objId
 */
Client.IdfFrame.prototype.getTooltip = function (objId)
{
  let wep = Client.mainFrame.wep;
  let tooltip;
  //
  let title, content, fknum;
  switch (objId) {
    case this.collapseButtonConf.id:
      if (this.collapsed) {
        title = Client.IdfResources.t("TIP_TITLE_MostraRiquadro");
        content = wep?.SRV_MSG_ShowFrame || Client.IdfResources.t("SRV_MSG_ShowFrame");
      }
      else {
        title = Client.IdfResources.t("TIP_TITLE_NascondiRiquadro");
        content = wep?.SRV_MSG_HideFrame || Client.IdfResources.t("SRV_MSG_HideFrame");
      }
      break;

    case this.lockButtonConf.id:
      fknum = Client.mainFrame.wep?.FKLocked;
      if (this.locked) {
        title = Client.IdfResources.t("TIP_TITLE_TooltipUnlock");
        content = wep?.SRV_MSG_Unlock || Client.IdfResources.t("SRV_MSG_Unlock");
      }
      else {
        title = Client.IdfResources.t("TIP_TITLE_TooltipLock");
        content = wep?.SRV_MSG_Lock || Client.IdfResources.t("SRV_MSG_Lock");
      }
      break;
  }
  //
  tooltip = Client.Widget.getHTMLTooltip(title, content, fknum);
  //
  return tooltip;
};


/**
 * Apply visual style
 */
Client.IdfFrame.prototype.applyVisualStyle = function ()
{
};


/**
 * Get frame list
 * @param {Array} flist
 */
Client.IdfFrame.prototype.getFrameList = function (flist)
{
  for (let f = 0; f < this.elements.length; f++) {
    if (this.elements[f] instanceof Client.IdfFrame) {
      flist.push(this.elements[f]);
      this.elements[f].getFrameList(flist);
    }
    else if (this.elements[f] instanceof Client.IdfTab && this.elements[f].elements && this.elements[f].elements.length > 0) {
      flist.push(this.elements[f].elements[0]);
      this.elements[f].elements[0].getFrameList(flist);
    }
  }
};


/**
 * Realize toolbar command set
 * @param {Object} cmsConf
 */
Client.IdfFrame.prototype.realizeCommandSet = function (cmsConf)
{
  Client.eleMap[this.toolbarConf.id].insertBefore({child: cmsConf});
};


/**
 * Handle click on frame container
 * @param {Object} event
 */
Client.IdfFrame.prototype.handleFrameClick = function (event)
{
  let events = [];
  double = (event.id === "onDblclick");
  //
  // If the event is client side do nothing
  if (Client.mainFrame.isIDF && ((!double && this.mouseClickEventDef === Client.IdfMessagesPump.eventTypes.CLIENTSIDE) || (double && this.mouseDoubleClickEventDef === Client.IdfMessagesPump.eventTypes.CLIENTSIDE)))
    return events;
  //
  let srcWidget;
  let srcElement = event.content.srcEvent?.srcElement;
  while (srcElement) {
    srcWidget = Client.eleMap[srcElement.id]?.parentWidget;
    if (srcWidget === this || srcWidget?.parentIdfFrame === this)
      break;
    //
    srcElement = srcElement.parentNode;
  }
  //
  let detail = this.getClickDetail(event, srcWidget || this);
  //
  // Give event IDF format
  if (Client.mainFrame.isIDF)
    events.push({
      id: "rawclk",
      def: (double ? this.mouseDoubleClickEventDef : this.mouseClickEventDef),
      content: {
        oid: this.id,
        obn: double,
        par1: (event.content.button || 0),
        par2: Math.floor(detail.xb) + "-" + Math.floor(detail.yb),
        par3: Math.floor(detail.x) + "-" + Math.floor(detail.y),
        par4: detail.par4,
        par5: detail.par5
      }
    });
  else
    events.push({
      obj: this.id,
      id: double ? "onMouseDoubleClick" : "onMouseClick",
      content: {
        ...detail,
        double,
        button: (event.content.button || 0)
      }
    });
  //
  return events;
};


/**
 * Handle function keys
 * @param {Object} event
 */
Client.IdfFrame.prototype.handleFunctionKeys = function (event)
{
  let events = [];
  if (event.content.type !== "keydown")
    return events;
  //
  // First I check the frame toolbar
  events.push(...Client.mainFrame.wep?.commandList.handleFunctionKeys(event, this.parentIdfView.index/*, this.parentIdfView.getFrameIndex(this) + 1*/) || []);
  return events;
};


/**
 * Get click detail
 * @param {Object} event
 * @param {Widget} srcWidget
 */
Client.IdfFrame.prototype.getClickDetail = function (event, srcWidget)
{
  event.content = event.content || {};
  //
  let rct = Client.eleMap[this.contentContainerConf.id].domObj.getBoundingClientRect();
  //
  let scrX = event.content.clientX;
  let scrY = event.content.clientY;
  let ret = {
    xb: scrX,
    yb: scrY,
    x: scrX - rct.left,
    y: scrY - rct.top
  };
  //
  if (ret.x < 0)
    ret.x = 0;
  if (ret.y < 0)
    ret.y = 0;
  //
  return ret;
};


/**
 * Update internal controls
 * @param {Object} propsToUpdate - example {visualStyle: true, editorType: true, ...}
 */
Client.IdfFrame.prototype.updateControls = function (propsToUpdate)
{
  if (!this.control)
    return false;
  //
  propsToUpdate = propsToUpdate || {};
  //
  let controlProps = {};
  Object.keys(propsToUpdate).forEach(p => controlProps[p] = this.customElement[p]);
  //
  this.control.updateElement(controlProps);
  return true;
};


/**
 * onresize Message
 * @param {Event} ev - the event occured when the browser window was resized
 */
Client.IdfFrame.prototype.onResize = function (ev)
{
  Client.mainFrame.sendEvents(this.handleResize());
  Client.Widget.prototype.onResize.call(this, ev);
};


/**
 * Handle a resize event
 */
Client.IdfFrame.prototype.handleResize = function ()
{
  delete this.delayResize;
  //
  let events = [];
  if (!Client.mainFrame.isIDF)
    return events;
  //
  if (this.inPreview)
    return events;
  //
  // If the tab is not exposed don't send its resize info to the server, send it when selected
  if (this.parent instanceof Client.IdfTab && !this.parent.isActiveTab()) {
    this.delayResize = true;
    return events;
  }
  //
  let rootObject = Client.eleMap[this.getRootObject(true).id].getRootObject();
  //
  let compStyle = getComputedStyle(rootObject);
  let width = parseInt(rootObject.clientWidth);
  width -= parseInt(compStyle.paddingLeft) + parseInt(compStyle.paddingRight);
  width -= parseInt(compStyle.marginLeft) + parseInt(compStyle.marginRight);
  let height = parseInt(rootObject.clientHeight);
  height -= parseInt(compStyle.paddingTop) + parseInt(compStyle.paddingBottom);
  height -= parseInt(compStyle.marginTop) + parseInt(compStyle.marginBottom);
  if (width !== this.lastWidth || height !== this.lastHeight) {
    events.push({
      id: "resize",
      def: Client.IdfMessagesPump.eventTypes.ACTIVE,
      content: {
        oid: this.id,
        par1: width,
        par2: height
      }
    });
    //
    this.lastWidth = width;
    this.lastHeight = height;
  }
  //
  return events;
};


Client.IdfFrame.prototype.acceptsDrop = function (element)
{
  return this.canDrop;
};


Client.IdfFrame.prototype.canResizeW = function (element)
{
  return this.isLeaf() && Client.mainFrame.wep?.resizableFrames && this.parent instanceof Client.IdfFrame && !this.parent.vertical;
};


Client.IdfFrame.prototype.canResizeH = function (element)
{
  return this.isLeaf() && Client.mainFrame.wep?.resizableFrames && this.parent instanceof Client.IdfFrame && this.parent.vertical;
};


Client.IdfFrame.prototype.getSupportedTransformOperation = function (x, y, element, root)
{
  let op = Client.Widget.prototype.getSupportedTransformOperation.call(this, x, y, element, Client.eleMap[this.mainContainerConf.id].getRootObject());
  //
  // I need to check if i'm the first or second child of my parent
  let firstChild = false;
  let children = this.parent.children || [];
  for (let i = 0; i < children.length; i++) {
    if (Client.Widget.isFrameClass(children[i].c) || Client.Widget.isFrameClass(children[i].class)) {
      firstChild = children[i].id === this.id;
      break;
    }
  }
  //
  // The TOP resize is admitted only on the second child of a !vertical frame
  // (the vertical thing is already checked by the resize, so we need only to check the result operation)
  if (op === Client.Widget.transformOperation.RESIZETOP && firstChild)
    op = Client.Widget.transformOperation.NONE;
  //
  // The BOTTOM resize is admitted only on the first child of a !vertical frame
  if (op === Client.Widget.transformOperation.RESIZEBOTTOM && !firstChild)
    op = Client.Widget.transformOperation.NONE;
  //
  // The LEFT resize is admitted only on the second child of a vertical frame
  if (op === Client.Widget.transformOperation.RESIZELEFT && firstChild)
    op = Client.Widget.transformOperation.NONE;
  //
  // The RIGHT resize is admitted only on the first child of a vertical frame
  if (op === Client.Widget.transformOperation.RESIZERIGHT && !firstChild)
    op = Client.Widget.transformOperation.NONE;
  //
  return op;
};


Client.IdfFrame.prototype.getTransformOperationTargetWidget = function (operation, element)
{
  return operation !== Client.Widget.transformOperation.DRAG && this.parent instanceof Client.IdfTab ? this.parent.parent : this;
};


Client.IdfFrame.prototype.getTransformOperationTargetObj = function (operation, element)
{
  return Client.eleMap[this.mainContainerConf.id].getRootObject();
};


Client.IdfFrame.prototype.applyDragDropCursor = function (cursor)
{
  // Apply the resize cursor only on the list header
  let obj = Client.eleMap[this.mainContainerConf.id].getRootObject();
  //
  if (cursor) {
    obj.setAttribute("opnt", "dd");
    obj.style.cursor = cursor;
    //
    // Clear the cursor on mouse leave
    if (!obj.onmouseleave)
      obj.onmouseleave = Client.Widget.ddClearPointer;
  }
  else if (obj.getAttribute("opnt")) {
    // I already set a cursor on the object BUT now i have no operation : clear the cursor
    obj.style.cursor = "";
    obj.setAttribute("opnt", "");
  }
};


Client.IdfFrame.prototype.onTransform = function (options)
{
  if (!(this.parent instanceof Client.IdfFrame))
    return;
  //
  // Get the delta
  let currentRect = Client.eleMap[this.mainContainerConf.id].getRootObject().getBoundingClientRect();
  let deltaH = options.h - currentRect.height;
  let deltaW = options.w - currentRect.width;
  //
  // The delta is calculated on the current dimensions that are not the height of the frames that the calcLayout uses to derive
  // the percentages, so we need to calculate the delta as a percentage of the current dimension AND then recalculate it by converting
  // the percentage of the server height
  let deltaHperc = (deltaH / currentRect.height) * 100;
  deltaH = (this.originalHeight / 100) * deltaHperc;
  let deltaWperc = (deltaW / currentRect.width) * 100;
  deltaW = (this.originalWidth / 100) * deltaWperc;
  //
  // Get my brother
  let bro;
  let children = this.parent.elements || [];
  children.forEach((ele) => {
    if (!bro && Client.Widget.isFrameClass(ele.class) && ele.id !== this.id)
      bro = ele;
  });
  //
  if (bro)
    bro.resizeFrame(-deltaH, -deltaW);
  this.resizeFrame(deltaH, deltaW);
  //
  // Update the layout of the view
  this.parentIdfView?.calcLayout();
};


Client.IdfFrame.prototype.resizeFrame = function (deltaH, deltaW)
{
  this.height = this.height + deltaH;
  this.originalHeight = this.height;
  //
  this.width = this.width + deltaW;
  this.originalWidth = this.width;
  //
  let frames = [];
  let children = this.elements || [];
  children.forEach((ele) => {
    if (Client.Widget.isFrameClass(ele.class))
      frames.push(ele);
  });
  //
  if (frames.length > 0) {
    let chDeltaH = this.vertical ? Math.round(deltaH / frames.length) : deltaH;
    let chDeltaW = this.vertical ? deltaW : Math.round(deltaW / frames.length);
    //
    frames.forEach((ele) => {
      ele.resizeFrame(chDeltaH, chDeltaW);
    });
  }
};
