/*
 * Instant Developer Cloud
 * Copyright Pro Gamma Spa 2000-2021
 * All rights reserved
 */

var Client = Client || {};


/**
 * @class A frame containing buttons
 * @param {Object} widget
 * @param {View|Element} parent - the parent element
 * @param {View} view
 */
Client.IdfTreeNode = function (widget, parent, view)
{
  // Set default values
  widget = Object.assign({
    expandable: true,
    expanded: false,
    alreadyExpanded: false,
    selectable: true,
    selected: false,
    draggable: true,
    droppable: true,
    activable: true,
    useHTML: Client.mainFrame.isIDF,
    showLabel: true
  }, widget);
  //
  this.level = (parent instanceof Client.IdfTree) ? 1 : parent.level + 1;
  //
  if (Client.mainFrame.isEditing())
    widget.expanded = true;
  //
  Client.Widget.call(this, widget, parent, view);
  //
  if (this.parentTree.activeNode === this.id)
    this.setActiveNode(true);
};


// Make Client.IdfTreeNode extend Client.Widget
Client.IdfTreeNode.prototype = new Client.Widget();


Client.IdfTreeNode.transPropMap = Object.assign({}, Client.Widget.transPropMap, {
  cap: "label",
  aex: "alreadyExpanded",
  exp: "expanded",
  cch: "selectable",
  sel: "selected",
  img: "image"
});


/**
 * Convert properties values
 * @param {Object} props
 */
Client.IdfTreeNode.convertPropValues = function (props)
{
  props = props || {};
  //
  for (let p in props) {
    switch (p) {

      case Client.IdfTreeNode.transPropMap.aex:
      case Client.IdfTreeNode.transPropMap.exp:
      case Client.IdfTreeNode.transPropMap.cch:
      case Client.IdfTreeNode.transPropMap.sel:
        props[p] = (props[p] === "1" || props[p] === true);
        break;
    }
  }
};


/**
 * Create elements configuration
 */
Client.IdfTreeNode.prototype.createElementsConfig = function ()
{
  this.treeContainerConf = this.createElementConfig({c: "Container", className: "treenode-container"});
  //
  this.headerContainerConf = this.createElementConfig({c: "IonItem", className: "treenode-header", tabIndex: 0, events: ["onClick", "onFocusin"]});
  this.treeContainerConf.children.push(this.headerContainerConf);
  //
  // Create collapse button configuration
  this.collapseButtonConf = this.createElementConfig({c: "IonButton", iconPosition: "only", clear: "true", className: "treenode-exp-icon"});
  this.headerContainerConf.children.push(this.collapseButtonConf);
  //
  // This is the spacer that will be visible if the node is not expandable
  this.spacerButtonConf = this.createElementConfig({c: "Container", visible: false, className: "treenode-spacer"});
  let spacerContentConf = this.createElementConfig({c: "Container", className: "treenode-spacer-inner"});
  this.spacerButtonConf.children.push(spacerContentConf);
  this.headerContainerConf.children.push(this.spacerButtonConf);
  //
  // Create the check
  this.checkConf = this.createElementConfig({c: "IonCheckbox", className: "treenode-check"});
  this.headerContainerConf.children.push(this.checkConf);
  //
  // Create the icon
  this.nodeIconConf = this.createElementConfig({c: "IonButton", visible: false, className: "generic-btn treenode-img"});
  this.headerContainerConf.children.push(this.nodeIconConf);
  //
  // Create the label
  this.nodeLabelConf = this.createElementConfig({c: "IonLabel", className: "treenode-header-label"});
  this.headerContainerConf.children.push(this.nodeLabelConf);
  //
  // Create the badge
  this.nodeBadgeConf = this.createElementConfig({c: "IonBadge", visible: false, className: "treenode-badge"});
  this.headerContainerConf.children.push(this.nodeBadgeConf);
  //
  // This is the children container
  if (!Client.mainFrame.idfMobile) {
    this.contentBoxConf = this.createElementConfig({c: "IonList", className: "treenode-content collapsible-container", noLines: true, style: {"max-height": 0}});
    this.contentBoxConf.animations = [{trigger: "animation", prop: "collapseElement", duration: 250},
      {trigger: "animation", prop: "expandElement", duration: 250}];
    this.treeContainerConf.children.push(this.contentBoxConf);
    //
    // Add desktop events
    this.collapseButtonConf.events = ["onClick"];
    this.nodeLabelConf.events = ["onClick", "onContextmenu", "onDblclick"];
  }
  else
    this.contentBoxConf = this.createElementConfig({c: "IonList", className: "treenode-content", noLines: false});
};


/**
 * Realize widget UI
 * @param {Object} widget
 * @param {View|Element|Widget} parent
 * @param {View} view
 */
Client.IdfTreeNode.prototype.realize = function (widget, parent, view)
{
  // Create elements configuration
  this.createElementsConfig(widget);
  //
  // Create the main container
  let treeContainer = view.createElement(this.treeContainerConf, parent, view);
  treeContainer.enableKeyEvent({inputs: true, type: "down"});
  this.mainObjects.push(treeContainer);
  //
  let customLabelId = widget.children?.find(c => c.class !== "IdfTreeNode")?.id;
  //
  // Create the children container into the tree level container
  if (Client.mainFrame.idfMobile)
    view.createElement(this.contentBoxConf, this.parentTree.getLevelContainer(this.level + 1), view);
  //
  // Create widget children
  this.createChildren(widget);
  //
  this.customLabel = Client.eleMap[customLabelId];
  //
  // Tell the parent to update it's icon if needed
  if (parent instanceof Client.IdfTreeNode)
    parent.updateExpansionIconVisibility();
};


/**
 * Update inner elements properties
 * @param {Object} props
 */
Client.IdfTreeNode.prototype.updateElement = function (props)
{
  props = props || {};
  //
  Client.Widget.prototype.updateElement.call(this, props);
  //
  let handleExpansion = false;
  let updateExpansionIconVisibility = false;
  for (let p in props) {
    let v = props[p];
    switch (p) {
      case "alreadyExpanded":
        this.alreadyExpanded = v;
        handleExpansion = true;
        updateExpansionIconVisibility = true;
        break;

      case "expanded":
        this.expanded = v;
        handleExpansion = true;
        break;

      case "expandable":
        this.expandable = v;
        handleExpansion = true;
        updateExpansionIconVisibility = true;
        break;

      case "image":
      case "icon":
        this.setImage(v);
        break;

      case "label":
        this.setLabel(v);
        break;

      case "tooltip":
        this.setTooltip(v);
        break;

      case "selectable":
        this.setSelectable(v);
        break;

      case "selected":
        this.setSelected(v);
        break;

      case "badge":
        this.setBadge(v);
        break;

      case "className":
        this.setClassName(v);
        break;

      case "draggable":
        this.draggable = v;
        this.updateDraggable();
        break;

      case "droppable":
        this.setDroppable(v);
        break;

      case "activable":
        this.setActivable(v);
        break;

      case "showLabel":
        this.showLabel = v;
        Client.Widget.updateElementClassName(Client.eleMap[this.headerContainerConf.id], "hidden-treenode-header-label", this.showLabel);
        break;

      case "style":
        this.style = v;
        this.applyStyleProp([Client.eleMap[this.headerContainerConf.id]], "style", v);
        break;
    }
  }
  //
  // Show or hide the expansion icon
  if (updateExpansionIconVisibility)
    this.updateExpansionIconVisibility();
  //
  if (handleExpansion)
    this.handleExpansion();
};


Client.IdfTreeNode.prototype.handleExpansion = function ()
{
  if ((!Client.mainFrame.isEditing() && this.expanded && !this.alreadyExpanded) || (Client.mainFrame.isEditing() && !this.expandable))
    return;
  //
  let collapseButton = Client.eleMap[this.collapseButtonConf.id];
  collapseButton.updateElement({icon: Client.mainFrame.idfMobile ? "arrow-dropright" : (this.expanded ? "remove" : "add")});
  //
  let contentBox = Client.eleMap[this.contentBoxConf.id];
  if (!Client.mainFrame.idfMobile) {
    Client.Widget.updateElementClassName(contentBox, "expanded", !this.expanded);
    Client.Widget.updateElementClassName(contentBox, "collapsed", this.expanded);
  }
  else {
    if (!this.expanded)
      contentBox.updateElement({visible: false});
    else
      setTimeout(() => contentBox.updateElement({ visible: true }), 0);
    //
    // Navigate to the page (if expanded without children do nothing)
    if ((this.expanded && this.childrenTreeNodes.length) || !this.expanded)
      this.parentTree.handleNodeExpansion(this);
  }
};


Client.IdfTreeNode.prototype.setImage = function (value)
{
  this.image = value || "";
  //
  let nodeIcon = Client.eleMap[this.nodeIconConf.id];
  //
  Client.Widget.setIconImage({el: nodeIcon, image: this.image});
  Client.Widget.updateObject(nodeIcon, {visible: !!this.image});
};


Client.IdfTreeNode.prototype.setLabel = function (value)
{
  this.label = value;
  Client.eleMap[this.nodeLabelConf.id].updateElement({
    [this.useHTML ? "innerHTML" : "innerText"]: Client.Widget.getHTMLForCaption(this.label)
  });
};


Client.IdfTreeNode.prototype.setTooltip = function (value)
{
  let tooltip = this.tooltip ? Client.Widget.getHTMLTooltip(this.tooltip) : null;
  //
  let el = this.showLabel ? Client.eleMap[this.nodeLabelConf.id] : this.customLabel;
  Client.Widget.updateObject(el, {tooltip});
};


Client.IdfTreeNode.prototype.setSelectable = function (value)
{
  this.selectable = value;
  Client.eleMap[this.checkConf.id].updateElement({disabled: !this.parentTree.enabled || !this.selectable});
};


Client.IdfTreeNode.prototype.setSelected = function (value)
{
  this.selected = value;
  Client.eleMap[this.checkConf.id].updateElement({checked: this.selected});
};


Client.IdfTreeNode.prototype.setBadge = function (value)
{
  Client.eleMap[this.nodeBadgeConf.id].updateElement({innerText: this.badge, visible: !!this.badge});
};


Client.IdfTreeNode.prototype.setClassName = function (value)
{
  let treeContainer = Client.eleMap[this.treeContainerConf.id];
  Client.Widget.updateElementClassName(treeContainer, this.className, true);
  this.className = value;
  Client.Widget.updateElementClassName(treeContainer, this.className);
};


Client.IdfTreeNode.prototype.updateDraggable = function (recursive)
{
  let draggable = this.parentTree.dragDrop && this.parentTree.canDrag && this.parentTree.enabled && this.draggable;
  //
  let nodeLabel = Client.eleMap[this.headerContainerConf.id];
  Client.Widget.updateObject(nodeLabel, {draggable});
  Client.Widget.updateElementClassName(nodeLabel, "tree-draggable", !draggable);
  //
  // Tell all the children
  if (recursive)
    this.childrenTreeNodes.forEach(tn => tn.updateDraggable(true));
};


Client.IdfTreeNode.prototype.setDroppable = function (value)
{
  this.droppable = value;
  //
  let nodeLabel = Client.eleMap[this.headerContainerConf.id];
  Client.Widget.updateElementClassName(nodeLabel, "tree-droppable", !value);
};


Client.IdfTreeNode.prototype.updateEnabled = function (recursive)
{
  let checkBox = Client.eleMap[this.checkConf.id];
  checkBox.updateElement({disabled: !this.parentTree.enabled || !this.selectable});
  //
  let headerBox = Client.eleMap[this.headerContainerConf.id];
  Client.Widget.updateElementClassName(headerBox, "disabled", this.parentTree.enabled && this.activable);
  //
  // Tell all the children
  if (recursive)
    this.childrenTreeNodes.forEach(tn => tn.updateEnabled(true));
};


Client.IdfTreeNode.prototype.setActivable = function (value)
{
  this.activable = value;
  //
  let nodeLabel = Client.eleMap[this.nodeLabelConf.id];
  Client.Widget.updateElementClassName(nodeLabel, "disabled", this.parentTree.enabled && this.activable);
  //
  this.updateEnabled();
};


/**
 * Check if this node is the active node and add the class.
 * If not cycle on the children
 * @param {Boolean} value
 */
Client.IdfTreeNode.prototype.setActiveNode = function (value)
{
  Client.Widget.updateElementClassName(Client.eleMap[this.headerContainerConf.id], "treenode-sel-item", !value);
  //
  if (value)
    this.focus();
};


/**
 * Append a child DOM Object to root object DOM
 * @param {Element} child - child element that requested the insertion
 * @param {HTMLElement} domObj - child DOM object to add
 */
Client.IdfTreeNode.prototype.appendChildObject = function (child, domObj)
{
  let parentEl = Client.eleMap[this.contentBoxConf.id];
  //
  // Server side elements have to be append to header container
  if (!Client.mainFrame.isIDF && !child.parentWidget)
    parentEl = Client.eleMap[this.headerContainerConf.id];
  //
  parentEl.appendChildObject(child, domObj);
  parentEl.elements.push(child);
  child.parent = parentEl;
};


/**
 * Show or hide the expansion icon
 */
Client.IdfTreeNode.prototype.updateExpansionIconVisibility = function ()
{
  let sh = this.childrenTreeNodes.length > 0 || Client.mainFrame.isEditing() || (!this.alreadyExpanded && this.expandable);
  //
  let el = Client.eleMap[this.collapseButtonConf.id];
  el.updateElement({visible: sh});
  //
  el = Client.eleMap[this.spacerButtonConf.id];
  el.updateElement({visible: !sh});
};


/**
 * Handle an event
 * @param {Object} event
 */
Client.IdfTreeNode.prototype.onEvent = function (event)
{
  let events = Client.Widget.prototype.onEvent.call(this, event);
  //
  if (!this.parentTree.enabled)
    return events;
  //
  switch (event.id) {
    case "onClick":
      let isHeader = event.obj === this.headerContainerConf.id || event.obj === this.nodeLabelConf.id;
      let isLeaf = this.alreadyExpanded && !this.childrenTreeNodes.length;
      //
      // On Mobile expand when clicking the header
      if (Client.mainFrame.idfMobile && isHeader && !isLeaf)
        event.obj = this.collapseButtonConf.id;
      //
      if (event.obj === this.collapseButtonConf.id)
        events.push(...this.handleCollapseButtonClick(event));
      //
      if (isHeader && (!Client.mainFrame.idfMobile || isLeaf))
        events.push(...this.handleLabelClick(event));
      break;

    case "chgProp":
      if (event.obj === this.checkConf.id)
        events.push(...this.handleCheckChange(event));
      break;

    case "onContextmenu":
      events.push(...this.handleContextMenu(event));
      break;

    case "onDblclick":
      events.push(...this.handleDoubleClick(event));
      break;

    case "onKey":
      events.push(...this.handleKey(event));
      break;

    case "onFocusin":
      this.handleFocus(event);
      break;
  }
  //
  return events;
};


/**
 * Handle click on collapse button
 * @param {Object} event
 */
Client.IdfTreeNode.prototype.handleCollapseButtonClick = function (event)
{
  return this.reverseExpansion();
};


/**
 * Expand/collapse the node
 */
Client.IdfTreeNode.prototype.reverseExpansion = function ()
{
  let events = [];
  //
  if (!this.expandable)
    return events;
  //
  // If I'm expanded I can collapse, if I'm collapsed I can expand if I've no children and no previous expansion (so maybe I have children, but the server
  // hasn't said it) or if I have children
  let wasExpanded = this.expanded;
  if (this.expanded || this.alreadyExpanded || this.childrenTreeNodes.length)
    this.updateElement({expanded: !this.expanded});
  //
  if (this.parentTree.activateOnExpand)
    this.parentTree.setActiveNode(this.id);
  //
  // Send expand event
  if (Client.mainFrame.isIDF) {
    events.push({
      id: "trnexp",
      def: this.parentTree.expandEventDef,
      content: {
        oid: this.id
      }
    });
  }
  else {
    events.push({
      id: "chgProp",
      obj: this.id,
      content: {
        name: "expanded",
        value: !wasExpanded,
        clid: Client.id
      }
    });
  }
  //
  return events;
};


/**
 * Handle click on label
 * @param {Object} event
 */
Client.IdfTreeNode.prototype.handleLabelClick = function (event)
{
  let events = [...this.activate()];
  //
  if (!Client.mainFrame.isIDF && this.events.includes("onClick")) {
    events.push({
      ...event,
      obj: this.id
    });
  }
  //
  return events;
};


/**
 * Activate the node
 */
Client.IdfTreeNode.prototype.activate = function ()
{
  let events = [];
  if (!this.parentTree.enabled || !this.activable)
    return events;
  //
  // We have already the node, no need to cycle the list
  this.parentTree.setActiveNode(this.id);
  //
  if (Client.mainFrame.isIDF)
    events.push({
      id: "clk",
      def: this.parentTree.clickEventDef,
      content: {
        oid: this.id
      }
    });
  else {
    events.push({
      id: "chgProp",
      obj: this.parentTree.id,
      content: {
        name: "activeNode",
        value: this.id,
        clid: Client.id
      }
    });
  }
  //
  return events;
};


/**
 * Handle change of check
 * @param {Object} event
 */
Client.IdfTreeNode.prototype.handleCheckChange = function (event)
{
  return this.select(!!event.content.value);
};


/**
 * Select/unselect the node
 * @@param {Boolean} selected
 */
Client.IdfTreeNode.prototype.select = function (selected)
{
  let events = [];
  //
  if (!this.parentTree.showMultipleSelection || !this.parentTree.enabled || !this.selectable)
    return events;
  //
  this.updateElement({selected});
  //
  if (Client.mainFrame.isIDF)
    events.push({
      obj: this.id,
      def: this.parentTree.checkEventDef,
      id: "chg",
      content: {
        oid: this.id,
        obn: "check",
        par1: selected ? "on" : ""
      }
    });
  else
    events.push({
      id: "chgProp",
      obj: this.id,
      content: {
        name: "selected",
        value: selected,
        clid: Client.id
      }
    });
  //
  return events;
};


Client.IdfTreeNode.prototype.handleContextMenu = function (event)
{
  let events = [];
  //
  if (Client.mainFrame.isIDF && !this.parentTree.popupMenu)
    return events;
  //
  // If it's already being managed by one of the child nodes I do nothing
  if (event.content.srcEvent.justHandled)
    return events;
  //
  event.content.srcEvent.justHandled = true;
  if (Client.mainFrame.isIDF)
    events.push({
      id: "rclk",
      def: this.parentTree.clickEventDef,
      content: {
        oid: this.id,
        par1: this.parentTree.popupMenu
      }
    });
  else if (this.events.includes("onContextmenu"))
    events.push({
      ...event,
      obj: this.id
    });
  //
  return events;
};


Client.IdfTreeNode.prototype.handleDoubleClick = function (event)
{
  let events = [];
  //
  if (Client.mainFrame.isIDF)
    return events;
  //
  if (this.events.includes("onDblclick"))
    events.push({
      ...event,
      obj: this.id
    });
  //
  return events;
};


Client.IdfTreeNode.prototype.handleKey = function (event)
{
  let events = [];
  //
  // Discard the event if it occurred on a child node
  if (Client.Widget.getWidgetByElement(Client.Utils.findElementFromDomObj(event.content.srcEvent.target)) !== this)
    return events;
  //
  switch (event.content.keyCode) {
    case 13:
      events.push(...this.activate());
      break;

    case 32:
      events.push(...this.select(!this.selected));
      //
      // When the page does not have active focus on an interactive element (such as an input, a textarea, or a button),
      // pressing the spacebar will scroll the page down by approximately the visible size of the viewport (as if you pressed the "Page" key Down").
      event.content.srcEvent.preventDefault();
      break;

    case 37: // Left arrow
      if (this.expanded)
        events.push(...this.reverseExpansion());
      else if (this.parent instanceof Client.IdfTreeNode)
        this.parentTreeNode.focus();
      event.content.srcEvent.preventDefault();
      break;

    case 38: // Up arrow
      this.nearBefore?.focus();
      event.content.srcEvent.preventDefault();
      break;

    case 39: // Right arrow
      if (!this.expanded)
        events.push(...this.reverseExpansion());
      else
        this.childrenTreeNodes[0]?.focus();
      event.content.srcEvent.preventDefault();
      break;

    case 40: // Down arrow
      ((this.expanded && this.childrenTreeNodes[0]) || this.nearAfter)?.focus();
      event.content.srcEvent.preventDefault();
      break;
  }
  //
  return events;
};


Object.defineProperty(Client.IdfTreeNode.prototype, "parentTree", {
  get() {
    if (this.parent instanceof Client.IdfTree)
      return this.parent;
    else
      return this.parentTreeNode?.parentTree;
  }
});


Object.defineProperty(Client.IdfTreeNode.prototype, "parentTreeNode", {
  get() {
    if (this.parent instanceof Client.IdfTreeNode)
      return this.parent;
  }
});

Object.defineProperty(Client.IdfTreeNode.prototype, "childrenTreeNodes", {
  get() {
    return this.elements.filter(el => el instanceof Client.IdfTreeNode);
  }
});

Object.defineProperty(Client.IdfTreeNode.prototype, "previousSibling", {
  get() {
    let parent = this.parentTreeNode || this.parentTree;
    return parent.elements[parent.elements.indexOf(this) - 1];
  }
});

Object.defineProperty(Client.IdfTreeNode.prototype, "nextSibling", {
  get() {
    let parent = this.parentTreeNode || this.parentTree;
    return parent.elements[parent.elements.indexOf(this) + 1];
  }
});

Object.defineProperty(Client.IdfTreeNode.prototype, "lastLeaf", {
  get() {
    return (this.expanded && this.childrenTreeNodes.slice(-1)[0]?.lastLeaf) || this;
  }
});

Object.defineProperty(Client.IdfTreeNode.prototype, "nearBefore", {
  get() {
    return this.previousSibling?.lastLeaf || this.parentTreeNode;
  }
});

Object.defineProperty(Client.IdfTreeNode.prototype, "nearAfter", {
  get() {
    return this.nextSibling || this.parentTreeNode?.nearAfter;
  }
});


/**
 * Create view/element configuration from xml
 * @param {XmlNode} xml
 */
Client.IdfTreeNode.createConfigFromXml = function (xml)
{
  let config = {};
  //
  if (xml.childNodes && xml.childNodes.length > 0) {
    config.childrenNodes = [];
    //
    for (let j = 0; j < xml.childNodes.length; j++) {
      let nodeConfig = Client.Widget.createConfigFromXml(xml.childNodes[j]);
      if (!nodeConfig)
        continue;
      //
      config.childrenNodes.push(nodeConfig);
    }
  }
  //
  return config;
};


Client.IdfTreeNode.prototype.resetCache = function ()
{
  this.childrenTreeNodes.slice().forEach(tn => this.removeChild(tn));
};

Client.IdfTreeNode.prototype.getPopupTarget = function ()
{
  return Client.eleMap[this.nodeLabelConf.id].getRootObject();
};

Client.IdfTreeNode.prototype.isDraggable = function (element)
{
  // If the tree has the old drag&drop use that, it must win
  return this.parentTree.canDrag && this.parentTree.enabled && !this.parentTree.dragDrop && this.draggable;
};

Client.IdfTreeNode.prototype.acceptsDrop = function (widget)
{
  // If the tree has the old drag&drop use that, it must win
  return this.parentTree.canDrop && this.parentTree.enabled && !this.parentTree.dragDrop && this.droppable;
};

Client.IdfTreeNode.prototype.getTransformOperationTargetObj = function (operation, element)
{
  let headerContainer = Client.eleMap[this.headerContainerConf.id];
  Client.Widget.updateElementClassName(headerContainer, "dd-multiple-selection", !this.parentTree.showMultipleSelection);
  //
  return headerContainer.getRootObject();
};


/*
 * On mobile: Collapse all expanded nodes, deep-first
 */
Client.IdfTreeNode.prototype.collapseBranch = function ()
{
  this.childrenTreeNodes.forEach(tn => tn.collapseBranch());
  this.updateElement({expanded: false});
};


/**
 * Insert a child element before another element
 * @param {Object} content - contains the element to insert and the id of the existing element. The new element is inserted before this element
 */
Client.IdfTreeNode.prototype.insertBefore = function (content)
{
  let e = Client.Widget.prototype.insertBefore.call(this, content);
  //
  if (this.childrenTreeNodes.length === 1)
    this.updateExpansionIconVisibility();
  //
  return e;
};


/**
 * Remove a child from the element
 * @param {Object} content - an object with the id of the element to remove
 */
Client.IdfTreeNode.prototype.removeChild = function (content)
{
  Client.Widget.prototype.removeChild.call(this, content);
  //
  if (!this.childrenTreeNodes.length)
    this.updateExpansionIconVisibility();
};


/**
 * Give focus to the node
 * @param {Object} options
 */
Client.IdfTreeNode.prototype.focus = function (options)
{
  // I can't use the focus of Element beacuse IonItem give the focus to it's fist child focusable
  Client.eleMap[this.headerContainerConf.id].domObj.focus();
};


/**
 * Handle focus event
 * @param {Object} event
 */
Client.IdfTreeNode.prototype.handleFocus = function (event)
{
  // Focus the control only if focus occurred on external zone of control
  if (Client.Utils.findElementFromDomObj(event.content.srcEvent.target)?.parentWidget === this)
    this.focus();
};
