/*
 * Instant Developer Cloud
 * Copyright Pro Gamma Spa 2000-2021
 * All rights reserved
 */

/* global Client */

/**
 * @class A swipe menu
 * @param {Object} element - the element description
 * @param {View|Element} parent - the parent element
 * @param {View} view - the view containing the dialog
 * @extends Client.Container
 */
Client.SwipeMenu = function (element, parent, view)
{
  Client.Container.call(this, element, parent, view);
  //
  // The div element is not attached to the DOM, it will be attach when needed
  this.domObj.parentNode.removeChild(this.domObj);
  //
  this.initPos = 0;
  this.lastSwipeDirection = Hammer.DIRECTION_LEFT;
  this.swipeMenuOpen = false;
  this.swipeStatus = Client.SwipeMenu.statusMap.CLOSED;
};


// Make Client.SwipeMenu extend Client.Dialog
Client.SwipeMenu.prototype = new Client.Container();


/**
 * Map of the swipe animation status
 */
Client.SwipeMenu.statusMap = {
  OPENING: 1,
  OPENED: 2,
  CLOSING: 3,
  CLOSED: 4
};


/**
 * Update element properties
 * @param {Object} el - properties to update
 */
Client.SwipeMenu.prototype.updateElement = function (el)
{
  if (!el)
    return;
  //
  // Delete changes that was generated by this client
  this.purgeMyProp(el);
  //
  if (el.commands) {
    this.commands = el.commands;
    this.createSwipeMenu(this.commands);
    delete el.commands;
  }
  //
  if (el.default) {
    if (this.commands) {
      for (var i = 0; i < this.commands.length; i++) {
        if (this.commands[i].n === el.default) {
          this.swipeMenuDefault = this.commands[i].v;
          break;
        }
      }
    }
  }
  //
  if (el.swipeMenuOpen !== undefined) {
    if (el.swipeMenuOpen && !this.swipeMenuOpen) {
      this.show();
    }
    else if (!el.swipeMenuOpen && this.swipeMenuOpen)
      this.closeSwipeMenu(this.parent);
    delete (el.swipeMenuOpen);
  }
  //
  if (el.visible !== undefined) {
    this.visible = el.visible;
    delete el.visible;
    if (!this.visible && this.swipeStatus !== Client.SwipeMenu.statusMap.CLOSED) {
      this.closeSwipeMenu(this.parent);
    }
  }
  //
  Client.Container.prototype.updateElement.call(this, el);
};


/**
 * Attach events handler
 * @param {Array} events - array of the events to handle
 */
Client.SwipeMenu.prototype.attachEvents = function (events)
{
  // Get the global hammer instance
  var hammer = Client.mainFrame.getHammerManager();
  //
  // Add the onSwipeSelected event, if needed
  if (events) {
    var idxic = events.indexOf("onSwipeSelected");
    if (idxic > -1) {
      events.splice(idxic, 1);
      this.swipeSelectedEvent = true;
    }
    //
    if (!hammer.get("tap")) {
      var recognizer = new Hammer.Tap({threshold: 20, time: 500});
      hammer.add(recognizer);
    }
    //
    var pthis = this;
    this.tapf = function (ev) {
      var cName = ev.srcEvent.target.className;
      if (typeof cName === "string" && cName.indexOf("swipe-menu") > -1) {
        if (pthis.domObj.contains(ev.srcEvent.target)) {
          if (pthis.swipeSelectedEvent) {
            if (pthis.lastEventTimestamp && ev.timeStamp - pthis.lastEventTimestamp < 250)
              return;
            pthis.lastEventTimestamp = ev.timeStamp;
            //
            var obj = ev.srcEvent.target;
            while (obj && !obj.value) {
              obj = obj.parentNode;
            }
            var v = obj ? obj.value : undefined;
            //
            var e = [{obj: pthis.id, id: "onSwipeSelected", content: v}];
            Client.mainFrame.sendEvents(e);
            pthis.closeSwipeMenu(pthis.parent);
          }
        }
      }
      else {
        var sMenu = document.getElementsByClassName("swipe-menu");
        for (var i = 0; i < sMenu.length; i++) {
          var sm = Client.Utils.findElementFromDomObj(sMenu[i]);
          var lItem = sm.parent;
          pthis.closeSwipeMenu(lItem, sm);
        }
      }
    };
    hammer.on("tap", this.tapf);
  }
  //
  // Create the pan recognizer
  var pan = hammer.get("pan");
  if (!pan) {
    pan = new Hammer.Pan({direction: Hammer.DIRECTION_HORIZONTAL, threshold: 10});
    hammer.add(pan);
  }
  //
  // If there is a default command, create the swipe recognizer
  if (this.swipeMenuDefault) {
    var swipe = hammer.get("swipe");
    if (!swipe) {
      swipe = new Hammer.Swipe();
      hammer.add(swipe);
    }
    if (pan && swipe)
      pan.recognizeWith(swipe);
  }
  //
  // Add the pan event. It will recognize left and right pans.
  var pthis = this;
  this.panlf = function (ev) {
    // Pan on the row OR if this is the already active swipe menu pan on everything
    var pan = false;
    if (Client.Utils.isMyParent(ev.target, pthis.parent.id))
      pan = true;
    if (pthis.parent && pthis.parent.parent && pthis.parent.parent.swipeOpen && pthis.parent.parent.swipeOpen === pthis)
      pan = true;
    if (pan)
      pthis.panOnItem(ev);
  };
  hammer.on("panleft", this.panlf);
  //
  this.panrf = function (ev) {
    // Pan on the row OR if this is the already active swipe menu pan on everything
    var pan = false;
    if (Client.Utils.isMyParent(ev.target, pthis.parent.id))
      pan = true;
    if (pthis.parent && pthis.parent.parent && pthis.parent.parent.swipeOpen && pthis.parent.parent.swipeOpen === pthis)
      pan = true;
    if (pan)
      pthis.panOnItem(ev);
  };
  hammer.on("panright", this.panrf);
  //
  // Add the pan end event
  this.panef = function (ev) {
    // Pan on the row OR if this is the already active swipe menu pan on everything
    var pan = false;
    if (Client.Utils.isMyParent(ev.target, pthis.parent.id))
      pan = true;
    if (pthis.parent && pthis.parent.parent && pthis.parent.parent.swipeOpen && pthis.parent.parent.swipeOpen === pthis)
      pan = true;
    if (pan) {
      Client.mainFrame.preventClick();
      pthis.panendOnItem(ev);
    }
  }
  hammer.on("panend", this.panef);
  //
  // If there is a default command, add the swipe event handler
  if (this.swipeMenuDefault) {
    this.swipf = function (ev) {
      if (ev.target.id === pthis.parent.id) {
        if (pthis.swipeSelectedEvent) {
          if (pthis.lastEventTimestamp && ev.timeStamp - pthis.lastEventTimestamp < 250)
            return;
          pthis.lastEventTimestamp = ev.timeStamp;
          var e = [{obj: pthis.id, id: "onSwipeSelected", content: pthis.swipeMenuDefault}];
          Client.mainFrame.sendEvents(e);
          pthis.closeSwipeMenu(pthis.parent);
        }
      }
    };
    hammer.on("swipe", this.swipf);
  }
  //
  this.parent.domObj.addEventListener("mouseleave", function (ev) {
    // If the mouse exits from the container during swipe we close the swipemenu, but only if the target is not a brother
    var toTarget = ev.relatedTarget ? ev.relatedTarget : ev.toElement;
    if (toTarget) {
      var obj = toTarget;
      while (obj) {
        if (obj === pthis.parent.parent.domObj)
          return;
        obj = obj.parentNode;
      }
    }
    //
    if (pthis.swipeMenuOpen)
      pthis.panendOnItem(ev);
  });
  //
  // When the list scrolls close the menu if is opened, but use a single scroll handler for each swipe menu
  if (this.parent && this.parent.parent && !this.parent.parent.swipeScrollHandled) {
    this.parent.parent.swipeScrollHandled = true;
    //
    this.scrollFunction = function (ev) {
      // Get all the swipemenu dom objects within the scrolled object
      var sMenu = ev.target.getElementsByClassName("swipe-menu");
      for (var i = 0; i < sMenu.length > 0; i++) {
        // Get the corrisponding javascript object
        var sm = Client.Utils.findElementFromDomObj(sMenu[i]);
        if (sm) {
          // On Scroll we close the menu
          if (sm.swipeMenuOpen)
            sm.closeSwipeMenu(sm.parent);
        }
      }
    };
    this.parent.parent.domObj.addEventListener("scroll", this.scrollFunction);
  }
  //
  // When a swipe menu is open all the touches must be blocked if they are not on the swipe menu
  if (!this.onTouchFunction) {
    this.onTouchFunction = function (ev) {
      // Check if the touched object is inside a swipe menu
      var obj = ev.target;
      var inside = false;
      while (obj) {
        if (obj === pthis.domObj) {
          inside = true;
          break;
        }
        // If found the list break, no need to go further
        if (obj === pthis.parent.parent.domObj)
          break;
        //
        obj = obj.parentNode;
      }
      //
      // If clicked outside the swipemenu stop the action && start closing the swipemenu
      if (!inside) {
        ev.preventDefault();
        ev.stopPropagation();
        //
        if (pthis.swipeMenuOpen)
          pthis.closeSwipeMenu(pthis.parent);
      }
    };
  }
  //
  Client.Container.prototype.attachEvents.call(this, events);
};


/**
 * Build the swipe menu
 * @param {ValueList} commands - commands of the swipe menu
 */
Client.SwipeMenu.prototype.createSwipeMenu = function (commands)
{
  // Delete any previous command
  while (this.domObj.firstChild)
    this.domObj.removeChild(this.domObj.firstChild);
  //
  // Create the menu items
  for (var i = 0; i < commands.length; i++) {
    var smitem = commands[i];
    //
    // Extract command properties
    var icon = smitem.src || smitem.img || smitem.icon;
    var label = smitem.n || smitem.name || smitem.label;
    var code = smitem.v || smitem.code || smitem.cmd;
    var style = smitem.s || smitem.style || smitem.class || smitem.cls || smitem.className;
    //
    var item = document.createElement("div");
    item.classList.add("swipe-menu-item");
    item.value = code;

    this.domObj.appendChild(item);
    //
    // Add image
    if (icon) {
      var img;
      if (icon.substring(0, 4) === "svg:") {
        img = document.createElementNS("http://www.w3.org/2000/svg", "svg");
        var use = document.createElementNS("http://www.w3.org/2000/svg", "use");
        use.setAttributeNS("http://www.w3.org/1999/xlink", "xlink:href", "#" + icon.substring(4));
        img.appendChild(use);
        img.setAttribute("class", "swipe-menu-img");

      }
      else if (icon.substring(0, 4) === "cls:") {
        var cn = icon.substring(4).split(",");
        img = document.createElement("div");
        img.className = "swipe-menu-img " + cn[0];
        if (cn.length)
          img.textContent = cn[1];
      }
      else if (icon.substring(0, 4) === "ion:" && Client.IonHelper) {
        img = document.createElement("ion-icon");
        Client.IonHelper.setIonIcon(icon.substring(4), img, "swipe-menu-img");
      }
      else {
        img = document.createElement("img");
        img.src = icon;
        img.className = "swipe-menu-img";
      }
      item.appendChild(img);
    }
    //
    // Add label
    if (label) {
      var text = document.createElement("span");
      text.classList.add("swipe-menu-item-value");
      text.innerText = label;
      item.appendChild(text);
    }
    //
    // Add style / class
    if (typeof style === "object") {
      for (var s in style)
        item.style[s] = style[s];
    }
    else if (typeof style === "string") {
      if (style.indexOf(":") > 0)
        item.style.cssText = style;
      else {
        var cl = style.split(" ");
        for (var j = 0; j < cl.length; j++)
          item.classList.add(cl[j]);
      }
    }
  }
};


/**
 * Show the swipe menu
 */
Client.SwipeMenu.prototype.show = function ()
{
  if (!this.visible)
    return;
  //
  var listItem = this.parent;
  var listBox = listItem.parent;
  //
  var sMenu = document.getElementsByClassName("swipe-menu");
  for (var i = 0; i < sMenu.length > 0; i++) {
    var sm = Client.Utils.findElementFromDomObj(sMenu[i]);
    if (sm) {
      var lItem = sm.parent;
      this.closeSwipeMenu(lItem, sm);
    }
  }
  //
  var elemPos = Client.Utils.findElemPos(listItem.domObj, listItem.parent.domObj);
  //
  // Attach the swipe menu to the list item parent
  listBox.domObj.insertBefore(this.domObj, listItem.domObj);
  //
  // Position the swipe menu
  this.domObj.classList.add("swipe-menu");
  this.domObj.style.top = (elemPos.top + listItem.parent.domObj.scrollTop) + "px";
  this.domObj.style.left = (elemPos.right - this.domObj.offsetWidth) + "px";
  this.domObj.style.height = (elemPos.bottom - elemPos.top) + "px";
  //
  // If the list container position is static, make it relative
  var listCntStyle = window.getComputedStyle(listItem.parent.domObj, null);
  if (listCntStyle.position === "static") {
    listItem.parent.domObj.style.position = "relative";
    this.listContainerPosition = listCntStyle.position;
  }
  //
  listBox.swipeOpen = this;
  //
  document.body.addEventListener("touchstart", this.onTouchFunction, true);
  document.body.addEventListener("mousedown", this.onTouchFunction, true);
};


/**
 * Move the item on pan
 * @param {MouseEvent} ev
 */
Client.SwipeMenu.prototype.panOnItem = function (ev)
{
  // If not visible, do not scroll
  if (!this.visible)
    return;
  //
  // Block swipe as click are disabled
  if (Client.mainFrame.isClickPrevented())
    return;
  //
  // Accept also diagonal movement, but only when the menu is open (for opening the menu the use must move horizontally)
  if (!this.swipeMenuOpen && (ev.deltaY > 10 || ev.deltaY < -10))
    return;
  //
  // Only right or left pan
  if (ev.direction !== Hammer.DIRECTION_LEFT && ev.direction !== Hammer.DIRECTION_RIGHT)
    return;
  //
  // If the swipe menu is already open or it's closing ignore the gesture
  if (this.swipeStatus === Client.SwipeMenu.statusMap.CLOSING || this.swipeStatus === Client.SwipeMenu.statusMap.OPENED)
    return;
  //
  var obj = this.parent;
  //
  // A swipeMenu is already open.. do nothing (only one)
  if (obj && obj.parent && obj.parent.swipeOpen && obj.parent.swipeOpen !== this)
    return;
  if (!this.swipeMenuOpen)
    this.show();
  //
  // Memorize the direction
  this.lastSwipeDirection = ev.direction;
  var old = this.swipeMenuOpen;
  if (!this.swipeMenuOpen) {
    this.swipeMenuOpen = true;
    this.swipeStatus = Client.SwipeMenu.statusMap.OPENING;
  }
  //
  // Get the delta movement
  var mov = 0 + ev.deltaX;
  //
  // When the delta is left and larger than the menu slow down the expansion
  if (mov < 0 && -mov > this.domObj.offsetWidth) {
    // There is a bug with hammer when the panEnd doesn't trigger, in that case there will be a gap, so in touch (where the bug is)
    // we show at max the menu size
    if (ev?.pointerType !== "touch")
      mov = 0 - this.domObj.offsetWidth + (ev.deltaX + this.domObj.offsetWidth) / 2;
    else
      mov = 0 - this.domObj.offsetWidth;
  }
  //
  // No right movement
  if (mov >= 0)
    mov = 0;
  //
  // Apply the transformation
  var t = "transform";
  obj.domObj.style[t] = "translate3d(" + mov + "px, 0px, 0px)";
  //
  // Tell the server what has changed
  if (old !== this.swipeMenuOpen) {
    var e = [];
    e.push({obj: this.id, id: "chgProp", content: {name: "swipeMenuOpen", value: this.swipeMenuOpen, clid: Client.id}});
    Client.mainFrame.sendEvents(e);
  }
  //
  ev.preventDefault();
};


/**
 * Event raised when the pan ends
 * @param {MouseEvent} ev
 */
Client.SwipeMenu.prototype.panendOnItem = function (ev)
{
  // If the user panned on a closing menu when the pan ends ignore the gesture
  if (this.swipeStatus === Client.SwipeMenu.statusMap.CLOSING)
    return;
  //
  var obj = this.parent;
  //
  // If I have to close the swipe menu
  // - moved to the right
  // - moved not more than the width/2
  // - no commands (??)
  if (this.domObj.offsetWidth === 0 || ev.deltaX >= 0 || (ev.deltaX < 0 && -ev.deltaX < this.domObj.offsetWidth / 2)) {
    // Move the list item
    var itemPos = Client.Utils.findElemPos(obj.domObj);
    var parentPos = Client.Utils.findElemPos(obj.parent.domObj);
    this.swipeStatus = Client.SwipeMenu.statusMap.CLOSING;
    //
    // Find the current left position of the list item relative to his parent
    var start = itemPos.left - parentPos.left;
    //
    var from = {style_transform: "translate3d(" + start + "px, 0px, 0px)"};
    var to = {style_transform: "translate3d(0px, 0px, 0px)"};
    var cnt = {repetitions: 1, segments: [{from: from, to: to, duration: 500, easing: "easeTo"}]};
    cnt.start = start;
    cnt.end = 0;
    this.animateSwipeMenu(cnt);
    //
    // Send the message to other clients, so they can start the animation (add clid identfier so we can skip thios client)
    cnt.clid = Client.id;
    var e = [{obj: this.id, id: "animateSwipeMenu", content: cnt, bc: true}];
    Client.mainFrame.sendEvents(e);
    //
    // The swipe menu is now invisible
    this.swipeMenuOpen = false;
    this.initPos = 0;
    //
    // Remove the swipe menu when the animation ends
    var pthis = this;
    var timeLength = (start === 0 ? 0 : 500);
    setTimeout(function () {
      pthis.swipeStatus = Client.SwipeMenu.statusMap.CLOSED;
      if (obj.parent.domObj.contains(pthis.domObj))
        obj.parent.domObj.removeChild(pthis.domObj);
      //
      document.body.removeEventListener("touchstart", pthis.onTouchFunction, true);
      document.body.removeEventListener("mousedown", pthis.onTouchFunction, true);
    }, timeLength);
    //
    var e = [{obj: this.id, id: "chgProp", content: {name: "swipeMenuOpen", value: this.swipeMenuOpen, clid: Client.id}}];
    Client.mainFrame.sendEvents(e);
    //
    // i'm not more the opened swipe menu
    if (this.parent && this.parent.parent && this.parent.parent.swipeOpen && this.parent.parent.swipeOpen === this)
      this.parent.parent.swipeOpen = null;
  }
  else {
    // If the pan direction is right and and the swipe menu is not visible then the list item is in its start position and there is nothing to do
    if (this.lastSwipeDirection === Hammer.DIRECTION_RIGHT && !this.swipeMenuOpen)
      return;
    //
    // Move the list item
    this.uncoverSwipeMenu();
    this.swipeStatus = Client.SwipeMenu.statusMap.OPENED;
  }
};


/**
 * Move the swipe menu parent
 */
Client.SwipeMenu.prototype.uncoverSwipeMenu = function ()
{
  var obj = this.parent;
  var myPos = Client.Utils.findElemPos(this.domObj);
  var itemPos = Client.Utils.findElemPos(obj.domObj);
  var parentPos = Client.Utils.findElemPos(obj.parent.domObj);
  //
  // Find the current left position of the list item relative to his parent
  var start = itemPos.left - parentPos.left;
  //
  // Calculate the left end position of the list item.
  // It is the swipe menu left position relative to the list item parent minus the width of the list item
  var end = (myPos.left - parentPos.left) - (itemPos.right - itemPos.left);
  //
  // Find the margin left of the item
  var s = window.getComputedStyle(obj.domObj, null);
  var marginLeft = parseInt(s.marginLeft);
  if (!isNaN(marginLeft))
    end -= marginLeft;
  //
  var from = {style_transform: "translate3d(" + start + "px, 0px, 0px)"};
  var to = {style_transform: "translate3d(" + end + "px, 0px, 0px)"};
  var cnt = {repetitions: 1, segments: [{from: from, to: to, duration: 500, easing: "easeTo"}]};
  this.animateSwipeMenu(cnt);
  //
  // Send the message to other clients, so they can start the animation (add clid identfier so we can skip thios client)
  cnt.clid = Client.id;
  var e = [{obj: this.id, id: "animateSwipeMenu", content: cnt, bc: true}];
  Client.mainFrame.sendEvents(e);
  //
  this.initPos = -this.domObj.offsetWidth;
};


/**
 * Close the swipe menu
 * @param {Element} obj
 */
Client.SwipeMenu.prototype.closeSwipeMenu = function (obj, sm)
{
  // If the menu is closed nothing to do
  if (this.swipeStatus === Client.SwipeMenu.statusMap.CLOSING || this.swipeStatus === Client.SwipeMenu.statusMap.CLOSED)
    return;
  //
  if (!sm)
    sm = this;
  this.swipeMenuOpen = false;
  this.swipeStatus = Client.SwipeMenu.statusMap.CLOSING;
  Client.mainFrame.preventClick();
  //
  // The start position is the width of the swipemenu - the list item
  var start = obj.domObj.offsetWith - obj.parent.domObj.offsetWith;
  //
  var from = {style_transform: "translate3d(" + start + "px, 0px, 0px)"};
  var to = {style_transform: "translate3d(0px, 0px, 0px)"};
  var cnt = {repetitions: 1, segments: [{from: from, to: to, duration: 500, easing: "easeTo"}]};
  cnt.start = start;
  cnt.end = 0;
  this.animateSwipeMenu(cnt, obj);
  //
  // Send the message to other clients, so they can start the animation (add clid identfier so we can skip thios client)
  cnt.clid = Client.id;
  cnt.targetId = obj.id;
  var e = [{obj: this.id, id: "animateSwipeMenu", content: cnt, bc: true}];
  Client.mainFrame.sendEvents(e);
  //
  // Remove the swipe menu when the animation finishes
  var pthis = this;
  var timeLength = (start === 0 ? 0 : 500);
  setTimeout(function () {
    if (obj.parent.domObj.contains(sm.domObj))
      obj.parent.domObj.removeChild(sm.domObj);
    pthis.swipeStatus = Client.SwipeMenu.statusMap.CLOSED;
    //
    document.body.removeEventListener("touchstart", pthis.onTouchFunction, true);
    document.body.removeEventListener("mousedown", pthis.onTouchFunction, true);
  }, timeLength);
  //
  var e = [];
  for (var i = 0; i < obj.ne(); i++) {
    var o = obj.elements[i];
    var cName = o.domObj.className;
    if (typeof cName === "string" && cName.indexOf("swipe-menu") > -1) {
      o.swipeMenuOpen = false;
      o.initPos = 0;
      e.push({obj: o.id, id: "chgProp", content: {name: "swipeMenuOpen", value: this.swipeMenuOpen}, clid: Client.id});
      break;
    }
  }
  if (e.length > 0)
    Client.mainFrame.sendEvents(e);
  //
  // Restore the position property of the list container style if it was changed when the swipe menu was shown
  if (this.listContainerPosition) {
    obj.parent.domObj.style.position = this.listContainerPosition;
    delete this.listContainerPosition;
  }
  if (this.parent && this.parent.parent && this.parent.parent.swipeOpen && this.parent.parent.swipeOpen === this)
    this.parent.parent.swipeOpen = null;
};


/**
 * Close the element
 */
Client.SwipeMenu.prototype.close = function ()
{
  for (var i = 0; i < this.ne(); i++)
    this.elements[i].close(false);
  //
  delete Client.eleMap[this.id];
  //
  if (this.domObj && this.domObj.parentNode)
    this.domObj.parentNode.removeChild(this.domObj);
  //
  if (this.parent && this.parent.parent && this.scrollFunction) {
    this.parent.parent.domObj.removeEventListener("scroll", this.scrollFunction);
    this.parent.parent.swipeScrollHandled = false;
  }
  //
  // Remove listener
  var hammer = Client.mainFrame.getHammerManager();
  if (this.tapf) {
    hammer.off("tap", this.tapf);
    delete this.tapf;
  }
  if (this.panlf) {
    hammer.off("panleft", this.panlf);
    delete this.panlf;
  }
  if (this.panrf) {
    hammer.off("panright", this.panrf);
    delete this.panrf;
  }
  if (this.panef) {
    hammer.off("panend", this.panef);
    delete this.panef;
  }
  if (this.swipf) {
    hammer.off("swipe", this.swipf);
    delete this.swipf;
  }
};


Client.SwipeMenu.prototype.getSwipeMenu = function (el)
{
  for (var j = 0; j < el.ne(); j++) {
    if (el.elements[j].domObj.className.indexOf("swipe-menu") > -1)
      return el.elements[j];
  }
  return null;
};


Client.SwipeMenu.prototype.animateSwipeMenu = function (animation, target)
{
  // The message is broadcasted to all clients, also the original:
  // in this case we use clid for skipping the message in the original client
  if (animation.clid && animation.clid === Client.id)
    return;
  //
  // Nothing to animate
  if (animation.start === 0 && animation.end === 0)
    return;
  //
  // Get the target of the animation (if a target is specified use that, otherwise use the parent)
  // if the animations contains a targetId use that
  var obj = target ? target : this.parent;
  if (animation.targetId)
    obj = Client.eleMap[animation.targetId];
  var anim = new Client.ClientAnimation(animation, obj);
  anim.play(false);
};