/*
 * Instant Developer Cloud
 * Copyright Pro Gamma Spa 2000-2021
 * All rights reserved
 */

/* global Hammer */

var Client = Client || {};
/**
 * @class Base class for every object in the client
 * @param {Object} element - the element description
 * @param {View|Element} parent - the parent element
 * @param {View} view - the view containing the dialog
 */
Client.Element = function (element, parent, view)
{
  // The constructor will be called twice during object initialization
  // first time, without any parameter, then with them.
  if (element === undefined)
    return;
  if (view === undefined)
    return;
  //
  // Some objects only exist client side and thus they don't have an id (ids are generated server side).
  // But this is weird because into Client.eleMap you will find "undefined" as object key!
  // So in this case generate an id. I need to add a special charachter in order to avoid collisions with ids generated server side
  if (element.id === undefined)
    element.id = Client.mainFrame.generateCounter() + "_";
  //
  // Get the value of element properties and put the element in the map
  this.name = element.name;
  this.id = element.id;
  this.class = element.class;
  this.view = view;
  this.parent = parent;
  this.parentWidget = element.parentWidget;
  this.cloned = element.cloned;
  delete element.parentWidget;
  delete element.cloned;
  //
  Client.eleMap[this.id] = this;
};


Client.Element.EventProperties = {y: 1, x: 1, offsetY: 1, offsetX: 1, button: 1, metaKey: 1, altKey: 1, shiftKey: 1,
  ctrlKey: 1, clientY: 1, clientX: 1, screenY: 1, screenX: 1, which: 1, charCode: 1, keyCode: 1, pageY: 1, pageX: 1,
  layerY: 1, layerX: 1, detail: 1, target: 2, relatedTarget: 2, key: 1};

Client.Element.ExcludedProp = {name: true, ds: true, clid: true, visible: true, animations: true, pid: true, children: true, events: true, class: true, startHidden: true};

Client.Element.fakeEmptyValue = String.fromCharCode(0xa0);

/**
 * Update element properties
 * @param {Object} el - properties to update
 */
Client.Element.prototype.updateElement = function (el)
{
  // Without any DOM object, cannot do anything
  if (!this.domObj)
    return;
  //
  // Delete changes that was generated by this client
  this.purgeMyProp(el);
  //
  var isEditing = Client.mainFrame.isEditing();
  //
  // item relative position (it is an ionic property, but every widget needs it)
  if (el.itemSide !== undefined) {
    this.itemSide = el.itemSide;
    delete el.itemSide;
  }
  //
  // At the beginning remove the empty element class if present
  if (isEditing && this.domObj && this.domObj.classList && this.domObj.classList.contains("emptycontainer"))
    this.domObj.classList.remove("emptycontainer");
  //
  // Setting the property removes the children, if we are in editing recreate them after setting the property
  if (isEditing && this.ne() && (el.innerText !== undefined || el.textContent !== undefined || el.innerHTML !== undefined)) {
    if (el.innerText !== undefined) {
      this.domObj.innerText = el.innerText;
      delete el.innerText;
    }
    if (el.textContent !== undefined) {
      this.domObj.textContent = el.textContent;
      delete el.textContent;
    }
    if (el.innerHTML !== undefined) {
      this.domObj.innerHTML = el.innerHTML;
      delete el.innerHTML;
    }
    //
    for (var elcnt = 0; elcnt < this.ne(); elcnt++) {
      // For a subview i must add its children, the DomObj is the same as the parent (me)
      if (this.elements[elcnt] instanceof Client.View) {
        var subFrm = this.elements[elcnt];
        for (var subel = 0; subel < subFrm.ne(); subel++)
          if (subFrm.elements[subel].domObj)
            this.domObj.appendChild(subFrm.elements[subel].domObj);
      }
      else if (this.elements[elcnt].domObj)
        this.domObj.appendChild(this.elements[elcnt].domObj);
    }
  }
  //
  // Apply properties contained in el
  var k = Object.keys(el);
  for (var i = 0; i < k.length; i++) {
    var p = k[i];
    //
    // Some properties should not be handled
    if (Client.Element.ExcludedProp[p])
      continue;
    //
    // Editing mode will handle id in a different way
    if (isEditing && p === "id")
      continue;
    //
    // generate a client exception
    if (p === "exception")
      throw el[p];
    //
    // Some properties are for the element itself, not for the domObj
    if (p === "rownum" && this[p] !== el[p]) {
      this[p] = el[p];
      this.updateListPosition();
      continue;
    }
    //
    // Set if use the popup
    if (p === "usePopupError") {
      this[p] = el[p];
      continue;
    }
    //
    // Set the error on the element
    if (p === "errorText") {
      this.setError(el.errorText, true);
      continue;
    }
    //
    // Set the error on the element
    if (p === "debouncingTimeout") {
      this.debouncingTimeout = el[p];
      continue;
    }
    //
    if (p === "animate") {
      this.animate = el.animate;
      continue;
    }
    //
    if (p === "tooltip") {
      this.addTooltip(el.tooltip);
      continue;
    }
    if (p === "rowpos") {
      this.rowPosition = el.rowpos;
      continue;
    }
    //
    if (p === "innerHTML" && Client.resourceHome) {
      // maybe the new html contains images/urls that should be absolutized
      // here we handle the "resources" case
      var v = el[p];
      var j = -1;
      while (true) {
        j = v.indexOf("/resources/", j + 1);
        if (j >= 0) {
          var l1 = v.lastIndexOf("'", j);
          var m1 = Math.max(v.lastIndexOf('"', j), l1);
          var l2 = v.indexOf("'", j);
          var m2 = Math.max(v.indexOf('"', j), l2);
          var s = Client.Utils.abs(v.substring(m1 + 1, m2));
          v = v.substring(0, m1 + 1) + s + v.substring(m2);
          j = m1 + s.length;
        }
        else
          break;
      }
      el[p] = v;
    }
    //
    // Setting style property
    if (p === "style") {
      if (el.style !== "") {
        if (typeof el.style === "string") {
          try {
            el.style = JSON.parse(el.style);
          }
          catch (ex) {
            this.domObj.style.cssText = el.style;
            continue;
          }
        }
        //
        // Clear old style (only in IDE mode)
        // and not when the style change comes from an animation: in that case apply only the style change
        if (isEditing && !el.fromanim && !this.parentWidget)
          this.domObj.style.cssText = "";
        //
        // Set new style
        var kk = Object.keys(el.style);
        for (var j = 0; j < kk.length; j++) {
          var pr = kk[j];
          var v = el.style[pr];
          //
          if (Client.Utils.requireAbs(pr))
            v = Client.Utils.absStyle(v);
          //
          var tgt = this.domObj;
          if (el.fromanim)
            tgt = this.getRootObject();
          //
          // If the user wrote !important in the value we should use the setProperty (no IE<9)
          if (v && v.indexOf && v.indexOf("!important") > 0)
            tgt.style.setProperty(pr, v.replace("!important", ""), "important");
          else
            tgt.style[pr] = v;
        }
      }
      else {
        // Resetting object style
        this.domObj.style.cssText = "";
      }
      //
      continue;
    }
    if (p === "autoFocus" && el[p] === true) {
      if (this.view && this.view.canAutoFocus())
        this.focus();
      continue;
    }
    if (p === "customNavigation") {
      this.customNavigation = el.customNavigation;
      continue;
    }
    if (p === "customid") {
      this.domObj.setAttribute(p, el[p]);
      continue;
    }
    //
    // unconmment this to see if there are other properties to exclude
    //console.log("updateElement " + p);
    //
    var v = el[p];
    //
    // Absolutize relative url for inplace apps
    if (p === "src" || p === "href")
      v = Client.Utils.abs(v);
    //
    this.domObj[p] = v;
    //
    //
    if (this.changeTrigger && (p === "innerText" || p === "textContent" || p === "innerHTML"))
      this.onAnimationTrigger(null, "change");
  }
  //
  // Handling object animations
  if (el.animations) {
    this.attachAnimations(el.animations);
    delete el.animations;
  }
  //
  // Handle element mask
  if (el.mask !== undefined) {
    if (Client.Utils.isNodeEditable(this.domObj))
      this.setMask(el.mask);
    //
    delete el.mask;
  }
  //
  if (el.maskType !== undefined) {
    this.maskType = el.maskType;
    delete el.maskType;
  }
  //
  // The element is asked to be hidden on creation and triggering the entering animation
  // only if this element has the entering animation and is visible
  if (el.startHidden && this.enterTrigger && el.visible) {
    // Start hiding the object
    el.visible = false;
    //
    // Memorize that the object is invisibile because of the temporary entering animation
    this.startHidden = true;
  }
  //
  var rootObj = this.getRootObject();
  //
  // If the server set the visibility when in editing we must remove the class that will hide this object
  if (isEditing && el.visible && !el.forhighlight)
    rootObj.classList.remove("element-hidden-highlighted");
  //
  // Handling object visibility
  if (el.visible !== undefined && el.visible !== this.visible) {
    this.visible = el.visible;
    if (el.visible === false) {
      if (rootObj.style.display !== "none") {
        // Must hide the domObj, use the animation if a trigger is defined and the object is in the DOM
        // and this is not the creation animation of an object (in that case we don't want the exiting animation to be triggered at the beginnning)
        if (this.animate !== false && this.exitTrigger && rootObj.parentNode && !this.startHidden) {
          this.onAnimationTrigger(null, "exit");
        }
        else {
          this.oldDisplay = rootObj.style.display;
          rootObj.style.display = "none";
        }
      }
    }
    else {
      var display = rootObj.style.display;
      if ((display === undefined || display === "none") && display !== this.oldDisplay) {
        // Must show the domObj, use the animation if a trigger is defined and the object is in the DOM
        if (this.animate !== false && this.enterTrigger && rootObj.parentNode) {
          this.onAnimationTrigger(null, "enter");
        }
        else {
          rootObj.style.display = this.oldDisplay === undefined ? "" : this.oldDisplay;
          delete this.oldDisplay;
        }
      }
    }
    //
    this.visibilityChanged(this.visible);
  }
  //
  // Hilight object, if the view editor requested it
  if (this.hilightObj) {
    var hlobj = document.getElementById("hlo-" + this.id);
    if (hlobj) {
      var pthis = this;
      window.setTimeout(function () {
        // Check if the element exists
        if (!Client.eleMap[pthis.id])
          return;
        pthis.hilightObj(true);
      }, 0);
    }
    //
    if (isEditing) {
      if (!this.domObj.id)
        this.domObj.id = "dmo_" + this.id;
      this.domObj.setAttribute("readonly", "true");
    }
  }
  //
  if (isEditing) {
    // If the domObj is hidden add the 'custom class'
    var _this = this;
    window.setTimeout(function () {
      // Check if the element has some children and some of these are visibile, in that case we should not apply the empty-container
      var hasVisibleChildren = false;
      if (_this.elements) {
        for (var ce = 0; ce < _this.elements.length && !hasVisibleChildren; ce++)
          if (_this.elements[ce].visible || _this.elements[ce].visible === undefined) // undefined = visible
            hasVisibleChildren = true;
      }
      if (hasVisibleChildren)
        return;
      //
      if (_this.domObj && _this.domObj.classList) {
        // Cases:
        // clientWidth or clientHeight 0 and classname === "" and textContent != ""
        // in this case if the user set some properties using the class we choose to reflect the real UI, and if
        // textContent is != "" the span has clientWidth or clientHeight 0.. in that case we skip also the hilite
        // if the Element has visible children we not set the empty container (maybe they are visibile due to the overflow)
        if ((_this.domObj.clientWidth === 0 || _this.domObj.clientHeight === 0) &&
                (_this.domObj.className === "" || _this.domObj.className === "hilightClass") &&
                (!_this.domObj.textContent || _this.domObj.textContent === "") &&
                !_this.domObj.getAttribute("src") &&
                _this.domObj.style.display !== "none") {
          _this.domObj.classList.add("emptycontainer");
          if (_this.hilighted)
            _this.hilightObj(true);
        }
      }
    }, 50);
  }
  //
  if (this.widgetLevel === 1 && this.createDropTargets) {
    window.setTimeout(function () {
      this.createDropTargets();
    }.bind(this), 15);
  }
};


/**
 * Attach events handler
 * @param {Array} events - array of the events to handle
 */
Client.Element.prototype.attachEvents = function (events)
{
  if (!events)
    return;
  //
  this.events = events;
  //
  // Delete some not-native events
  let pos = events.indexOf("onEndAnimation");
  if (pos >= 0) {
    events.splice(pos, 1);
    this.sendEndAnimation = true;
  }
  //
  pos = events.indexOf("onKey");
  if (pos >= 0)
    events.splice(pos, 1);
  //
  // Add touch events
  let eventsToCheck = [];
  if (Client.mainFrame.hammerEnabled) {
    if (Client.mainFrame.device.isMobile) {
      eventsToCheck.push({ideName: "onDblclick", constr: "Tap", name: "doubleTap", opt: {event: "doubleTap", taps: 2}});
      eventsToCheck.push({ideName: "onContextmenu", constr: "Press", name: "press"});
    }
    eventsToCheck.push({ideName: "onSwipe", constr: "Swipe", name: "swipe", opt: {event: "swipe", velocity: 0.1}});
    eventsToCheck.push({ideName: "onPinch", constr: "Pinch", name: "pinch"});
    eventsToCheck.push({ideName: "onRotate", constr: "Rotate", name: "rotate"});
    eventsToCheck.push({ideName: "onPan", constr: "Pan", name: "pan"});
  }
  //
  for (let i = 0; i < eventsToCheck.length; i++) {
    let x = eventsToCheck[i];
    let idxic = events.indexOf(x.ideName);
    if (idxic >= 0) {
      events.splice(idxic, 1);
      //
      // Create the touch events manager
      let hammer = Client.mainFrame.getHammerManager();
      //
      // Add the event
      this.addTouchEvent(hammer, x.ideName, x.constr, x.name, x.opt);
    }
  }
  //
  let eventHandler = this.onEvent.bind(this);
  //
  for (let i = 0; i < events.length; i++) {
    let eventName = events[i];
    let name = eventName.toLowerCase();
    if (name === "onclick")
      this.domObj.setAttribute("click-delay", 40);
    //
    // Some events (such as focusin and focusout) can only be attacked with the addEventListener
    if (this.domObj[name] === undefined)
      this.domObj.addEventListener(name.substring(2).toLowerCase(), eventHandler);
    else
      this.domObj[name] = eventHandler;
  }
  //
  this.domObj.addEventListener("focus", () => {
    if (!this.errorPopup && this.usePopupError === true)
      this.drawErrorPopup();
  }, {passive: true, capture: false});
  //
  this.domObj.addEventListener("blur", () => {
    if (this.errorPopup) {
      this.errorPopup.close();
      delete this.errorPopup;
    }
  }, {passive: true, capture: false});
};


/**
 * Handler of event
 * @param {Event} ev
 */
Client.Element.prototype.onEvent = function (ev)
{
  // If the element is disabled there's nothing to do
  if ((ev.type === "click" || ev.type === "dblclick") && this.domObj.disabled)
    return;
  //
  // Don't send events if they are recognized near a scroll event
  if (ev.type === "click" || ev.type === "dblclick") {
    if (Client.mainFrame.isClickPrevented())
      return;
  }
  // Debouncing
  if (ev.type === "click" && this.debouncingTimeout) {
    let t = new Date().getTime();
    if (t - this.lastClickTime < this.debouncingTimeout)
      return;
    this.lastClickTime = t;
  }
  //
  let events = [];
  switch (ev.type) {
    case "scroll":
      // Do not set this properties on a virtual list
      if (!this.rowCount) {
        events.push({obj: this.id, id: "chgProp", content: {name: "clientWidth", value: this.domObj.clientWidth, clid: Client.id}});
        events.push({obj: this.id, id: "chgProp", content: {name: "clientHeight", value: this.domObj.clientHeight, clid: Client.id}});
        events.push({obj: this.id, id: "chgProp", content: {name: "scrollTop", value: this.domObj.scrollTop, clid: Client.id}});
      }
      break;
  }
  //
  let en = "on" + ev.type.substring(0, 1).toUpperCase() + ev.type.substr(1);
  if (en === "onContextmenu")
    ev.preventDefault();
  //
  // Don't propagate click events when there is a listening child element
  // Some elements needs to propagate, so we can "prevent" in another way
  // We want the click events to propagate within the widgets
  if (ev.type === "click" || ev.type === "dblclick") {
    if (this.preventClickPolicy && this.preventClickPolicy() === "prevent")
      Client.mainFrame.preventClick();
    else if (!this.parent || !this.parent.parentWidget) {
      ev.stopPropagation();
      //
      // We need to stop the normal propagation, but during editing we need to call the editor global click
      // handler
      if (Client.mainFrame.isEditing())
        Client.eleMap["editm"].onClick(ev);
    }
  }
  //
  if (Client.mainFrame.isEditing() && ev.type === "focusin")
    return;
  //
  events.push({obj: this.id, id: en, content: this.saveEvent(ev)});
  Client.mainFrame.sendEvents(events);
};


/**
 * Return a copy of the element
 * @param {Object} config
 * @param {Client.Element} parent
 * @param {Map} referencesMap
 */
Client.Element.prototype.clone = function (config, parent, referencesMap)
{
  config = config || {};
  //
  let baseClass = this instanceof Client.Widget ? Client.Widget : Client.Element;
  let el = new baseClass(Object.assign({cloned: true}, config), undefined, this.view);
  Object.setPrototypeOf(el, Object.getPrototypeOf(this));
  //
  let prop;
  let keys = Object.keys(this);
  //
  // Clone entire root domObj
  if (!parent) {
    referencesMap = new Map();
    Client.Utils.cloneDomObj(this.getRootObject(), referencesMap);
  }
  //
  // Cloning properties
  for (let k = 0; k < keys.length; k++) {
    prop = keys[k];
    //
    if (["id", "name", "cloned", "elements", "mainObjects"].includes(prop))
      continue;
    //
    if (config && config[prop] !== undefined)
      continue;
    //
    // Skip Element and View
    if (this[prop] instanceof Client.Element || this[prop] instanceof Client.View)
      continue;
    //
    if (this[prop] instanceof HTMLElement) {
      el[prop] = referencesMap.get(this[prop]);
      //
      // If I didn't find el[prop] it means it's not into rootObject hierarchy, so clone it separately
      if (!el[prop])
        el[prop] = Client.Utils.cloneDomObj(this[prop], referencesMap);
      //
      // If domObj has an id, change it to be the same as cloned object id
      if (prop === "domObj" && el.domObj.id)
        el.domObj.id = el.id;
    }
    else if (prop === "events")
      el.events = JSON.parse(JSON.stringify(this.events));
    else if (prop === "animations")
      el.animations = JSON.parse(JSON.stringify(this.animations));
    else if (typeof this[prop] === "object") {
      try {
        el[prop] = JSON.parse(JSON.stringify(this[prop]));
      }
      catch (ex) {

      }
    }
    else  // Copying simple properties
      el[prop] = this[prop];
  }
  //
  el.parent = parent;
  //
  if (this.elements) {
    el.elements = [];
    //
    // For each child, create a clone of it
    for (let i = 0; i < this.elements.length; i++) {
      let child = this.elements[i];
      //
      // Get linked element. If I find it, it means it has already been cloned
      let newChild, newChildDomObj;
      if (child instanceof Client.Widget) {
        newChildDomObj = referencesMap.get(child.mainObjects[0]?.domObj);
        newChild = Client.eleMap[newChildDomObj?.id];
        if (newChild && newChild !== child)
          newChild = newChild.parentWidget;
      }
      else {
        newChildDomObj = referencesMap.get(child.domObj);
        newChild = Client.eleMap[newChildDomObj?.id];
      }
      //
      // If current child has not been cloned yet, clone it
      if (!newChild?.cloned || newChild === child)
        newChild = child.clone(undefined, el, referencesMap);
      //
      el.elements.push(newChild);
      //
      if (el instanceof Client.Widget)
        newChild.parentWidget = el;
      //
      newChild.parent = el;
    }
  }
  //
  if (el.events?.length)
    el.attachEvents(el.events);
  //
  if (el.animations?.length)
    el.attachAnimations(el.animations);
  //
  // Add listeners for client events
  el.addEventsListeners();
  //
  return el;
};


/**
 * Add events listeners
 */
Client.Element.prototype.addEventsListeners = function ()
{

};


/**
 * Add a touch event handler
 * @param {type} hammer - the touch event manager
 * @param {type} ideName - the name of the event in the ide
 * @param {type} constr - the name of the constructor of the event
 * @param {type} name - the name of the event
 * @param {type} opt - the options
 */
Client.Element.prototype.addTouchEvent = function (hammer, ideName, constr, name, opt)
{
  // Get the name of the event
  var n = name;
  //
  // If there is a custom name, get it instead
  if (opt && opt.event)
    n = opt.event;
  //
  // If the hammer manager has not still this recognizer, add it
  if (!hammer.get(n)) {
    var recognizer = new Hammer[constr](opt);
    hammer.add(recognizer);
  }
  //
  // Add the event to the touch events list
  if (!this.touchEventsList)
    this.touchEventsList = [];
  this.touchEventsList.push(name);
  //
  // Add the event listener
  var pthis = this;
  hammer.on(n, function (ev) {
    //
    // hammer has some problems with pinch and isFinal... better to help it
    if (ev.type === "pinch" && ev.srcEvent && (ev.srcEvent.type === "touchend" || ev.srcEvent.type === "pointerup"))
      ev.isFinal = true;
    //
    // If the element is disabled there's nothing to do
    if (n === "doubleTap" && pthis.domObj.disabled)
      return;
    //
    var objId;
    var targetId = ev.target.id;
    //
    // The touch event was fired by the element
    if (targetId === pthis.domObj.id)
      objId = targetId;
    else {
      // Check if the touch event was fired by one of my children
      var parent = ev.target.parentNode;
      while (parent) {
        // The event was fired by one of my children.
        // If the child has its own event handler, let him manage the event.
        if (parent.id === pthis.domObj.id) {
          var el = Client.eleMap[targetId];
          if (!el || !el.touchEventsList || el.touchEventsList.indexOf(name) === -1)
            objId = parent.id;
          break;
        }
        parent = parent.parentNode;
      }
    }
    //
    // If I've found the element that fired the event
    if (objId) {
      // Don't send events if they are recognized near a scroll event
      if (Client.mainFrame.isClickPrevented() && !ev.isFinal) {
        return;
      }
      //
      var obj = pthis.saveEvent(ev.srcEvent);
      var event = pthis.saveTouchEvent(ev, obj);
      var e = [{obj: objId, id: ideName, content: event}];
      Client.mainFrame.sendEvents(e);
    }
  });
};


/**
 * Append a child DOM Object to this element DOM Object
 * Usually the child element is not contained in this element array
 * it will be pushed there just after this function call.
 * @param {Element} child child element that requested the insertion
 * @param {HTMLElement} domObj child DOM object to add
 */
Client.Element.prototype.appendChildObject = function (child, domObj)
{
  this.domObj.appendChild(domObj);
};


/**
 * Draw a popup to show the text of the error
 */
Client.Element.prototype.drawErrorPopup = function ()
{
  // Get the error text
  var msg = this.domObj.validationMessage ? this.domObj.validationMessage : this.errorMessage;
  if (msg && msg.length > 0) {
    // Create the popup
    this.errorPopup = new Client.Dialog({options: {ref: {id: this.id, whisker: true}}}, this, this.view);
    this.errorPopup.domObj.classList.add("error-popup");
    //
    // Add a div for the image
    var img = document.createElement("div");
    img.className = "error-popup-img";
    this.errorPopup.domObj.appendChild(img);
    //
    // Add the text
    var txt = document.createElement("div");
    txt.innerText = msg;
    this.errorPopup.domObj.appendChild(txt);
    //
    this.errorPopup.positionElement();
  }
};


/**
 * Close the popup that shows the error
 */
Client.Element.prototype.closePopup = function ()
{
  if (this.errorPopup) {
    this.errorPopup.close();
    this.errorPopup = null;
  }
};


/**
 * Delete properties that where originated by this client
 * @param {Object} el - the properties to delete
 */
Client.Element.prototype.purgeMyProp = function (el)
{
  if (el.clid) {
    // el.clid contains properties generated by clients.
    // example: el.clid = {value:"012345"}, if the value property was generated by the 012345 client
    for (var pid in el.clid) {
      if (el.clid[pid] === Client.id)
        delete el[pid];
    }
  }
};


/**
 * Save event property to send it to server
 * @param {Event} ev - the event to save
 */
Client.Element.prototype.saveEvent = function (ev)
{
  if (ev === undefined)
    return;
  //
  // If the event is a custom one, no filtering is needed
  if (!(ev instanceof Event))
    return ev;
  //
  var ris = {srcEvent: ev};
  //
  // Cannot use KEYS here because needed properties are in
  // prototypes. So IN is fine.
  for (var p in ev) {
    var pt = Client.Element.EventProperties[p];
    if (pt === 1)
      ris[p] = ev[p];
    if (pt === 2) {
      var obj = ev[p];
      while (obj && !obj.id)
        obj = obj.parentNode;
      if (obj && obj !== window)
        ris[p] = obj.id;
    }
    //
    // Add the first element.id if the event happened on a non-element dom object
    if (ris.target && !Client.eleMap[ris.target]) {
      var obj = ev.target;
      while (obj && !Client.eleMap[obj.id]) {
        obj = obj.parentNode;
      }
      if (obj)
        ris.targetElement = obj.id;
    }
  }
  return ris;
};


/**
 * Save touch event properties to send to server
 * @param {Event} ev - the event to save
 * @param {Object} obj - the object to put the properties in
 * @returns {Object}
 */
Client.Element.prototype.saveTouchEvent = function (ev, obj)
{
  if (!obj)
    obj = {};
  //
  // Cannot use KEYS here because needed properties are in
  // prototypes. So IN is fine.
  for (var p in ev) {
    if (p === "srcEvent" || p === "center" || p === "preventDefault" || p === "target" || p === "changedPointers" || p === "pointers")
      continue;
    //
    obj[p] = ev[p];
  }
  //
  return obj;
};


/**
 * Create child elements
 * @param {Object} el - property of the children to create
 */
Client.Element.prototype.createChildren = function (el)
{
  if (el.children) {
    this.elements = [];
    for (var i = 0; i < el.children.length; i++) {
      if (el.children[i].child) {
        this.insertBefore(el.children[i]);
      }
      else {
        var e = this.view.createElement(el.children[i], this, this.view);
        this.elements.push(e);
      }
    }
  }
};


/**
 * Remove the element and its children from the element map
 * @param {boolean} firstLevel - if true remove the dom of the element too
 * @param {boolean} triggerAnimation - if true and on firstLevel trigger the animation of 'removing'
 */
Client.Element.prototype.close = function (firstLevel, triggerAnimation)
{
  this.clearPendingFocus();
  //
  if (Client.Element.lastFocusedElement === this)
    delete Client.Element.lastFocusedElement;
  //
  if (this.view?.activeElement === this)
    delete this.view.activeElement;
  //
  // Tell my parent
  this.parent?.onRemoveChildObject(this);
  //
  for (let i = 0; i < this.ne(); i++)
    this.elements[i].close(false);
  //
  delete Client.eleMap[this.id];
  //
  let root = this.getRootObject();
  //
  if (firstLevel && root?.parentNode) {
    // There is a trigger for removing and we must invoke him. The parameter triggerAnimation is used only for this trigger because the DOM removal in
    // a DataMap is done also on refreshing, in that case we don't want to trigger the animation so the 'remove' animation is special for the Datamap Row and
    // the animation must be triggered explicitly.
    // If the element has an Exit animation that animation must be triggered also on removal
    if (this.animate !== false && this.removeTrigger && triggerAnimation) {
      this.onAnimationTrigger(null, "remove");
    }
    else if (this.animate !== false && this.exitTrigger) {
      this.removeElement = true;
      this.onAnimationTrigger(null, "exit");
    }
    else
      root.remove();
  }
  //
  if (this.focusTimeout) {
    clearTimeout(this.focusTimeout);
    delete this.focusTimeout;
  }
  if (this.tooltip) {
    this.tooltip.destroy();
    delete this.tooltip;
  }
  if (this.slipInstance) {
    this.slipInstance.detach();
    delete this.slipInstance;
  }
  if (this.hedown)
    this.enableKeyEvent({type: "down", enable: false});
  if (this.heup)
    this.enableKeyEvent({type: "up", enable: false});
  if (this.hepress)
    this.enableKeyEvent({type: "press", enable: false});
};


/**
 * Insert a child element before another element
 * @param {Object} content - contains the element to insert and the id of the existing element. The new element is inserted before this element
 */
Client.Element.prototype.insertBefore = function (content)
{
  // let's see if we can handle requirements
  if (!Client.mainFrame.loadClientRequirements(content.child))
    return;
  //
  var e, i;
  if (content.child.type === "view") {
    e = new Client.View(content.child, this);
    //
    if (content.id) {
      // the ID of this subview is MY ID as I'm the reference to it.
      delete Client.eleMap[e.id];
      e.id = content.id;
    }
    //
    Client.eleMap[e.id] = e;
  }
  else {
    // Search if the element is already a children (check if contains children)
    for (i = 0; i < this.ne(); i++) {
      var el = this.elements[i];
      if (el.id === content.child.id) {
        // The element is already a children, remove it from the array of the elements
        // so it can moved in the new position
        e = el;
        this.elements.splice(i, 1);
        break;
      }
    }
    //
    if (!e) {
      // Verify if the element is already child of another element
      e = Client.eleMap[content.child.id];
      if (e) {
        e.parent.onRemoveChildObject(content.child);
        //
        for (i = 0; i < e.parent.ne(); i++) {
          if (e.parent.elements[i].id === content.child.id) {
            e.parent.elements.splice(i, 1);
            break;
          }
        }
        //
        this.appendChildObject(e, e.getRootObject());
        e.parent = this;
      }
    }
    //
    if (!e) {
      // Tell the element that it should start hidden if has an enter animation
      content.child.startHidden = true;
      e = this.view.createElement(content.child, this, this.view);
    }
  }
  //
  if (this.elements === undefined)
    this.elements = [];
  //
  if (content.sib) {
    for (i = 0; i < this.ne(); i++) {
      var el = this.elements[i];
      if (el.id === content.sib) {
        //
        // Insert the child in the right position in the elements array
        this.elements.splice(i, 0, e);
        //
        // Insert the child in the right position in the document
        var sibling;
        if (el instanceof Client.View) {
          if (el.elements[0] && el.elements[0].getRootObject)
            sibling = el.elements[0].getRootObject();
          else
            sibling = el.domObj;
        }
        else if (e instanceof Client.IdfFrame)
          sibling = el.mainObjects[0]?.domObj;
        else
          sibling = el.getRootObject();
        //
        var parent = sibling.parentNode;
        if (parent) {
          if (e instanceof Client.IdfFrame)
            parent.insertBefore(e.mainObjects[0]?.domObj, sibling);
          else if (e instanceof Client.Element)
            parent.insertBefore(e.getRootObject(), sibling);
          else {
            for (var j = 0; j < e.ne(); j++)
              parent.insertBefore(e.elements[j].getRootObject(), sibling);
          }
        }
        //
        this.onPositionChildObject(i);
        //
        break;
      }
    }
  }
  //
  // If I can't insert the child before any element, append it
  if (!content.sib || i >= this.ne())
    this.elements.push(e);
  //
  // I've a child, the emptycontainer is not needed anymore (if needed the child will have it)
  if (Client.mainFrame.isEditing() && this.domObj && this.domObj.classList && this.domObj.classList.contains("emptycontainer"))
    this.domObj.classList.remove("emptycontainer");
  if (e && e.startHidden) {
    delete e.startHidden;
    e.updateElement({visible: true});
  }
  //
  return e;
};


/**
 * Remove a child from the element
 * @param {Object} content - an object with the id of the element to remove
 */
Client.Element.prototype.removeChild = function (content)
{
  var id = content.id;
  for (var i = 0; i < this.ne(); i++) {
    if (this.elements[i].id === id)
    {
      this.elements[i].close(true, content.triggerAnimation);
      this.elements.splice(i, 1);
      break;
    }
  }
};


/**
 * Ask the listbox to move this object to match new list position
 */
Client.Element.prototype.updateListPosition = function ()
{
  var o = this.parent;
  while (o) {
    if (o.rowCount) {
      o.moveToListPosition(this);
      break;
    }
    o = o.parent;
  }
};


/**
 * Move this object on the list
 * @param {int} n
 */
Client.Element.prototype.setRownum = function (n)
{
  this.rownum = n;
  this.updateListPosition();
};


/**
 * Abort pending focus to child elements
 */
Client.Element.prototype.clearPendingFocus = function ()
{
  if (Client.Element.lastFocusedElement?.focusTimeout && this.getRootObject().contains(Client.Element.lastFocusedElement.getRootObject())) {
    clearTimeout(Client.Element.lastFocusedElement.focusTimeout);
    delete Client.Element.lastFocusedElement.focusTimeout;
    delete Client.Element.lastFocusedElement;
  }
};


/**
 * Check if element is focused
 * @param {Boolean} focusing
 */
Client.Element.prototype.hasFocus = function (focusing)
{
  if (this.getRootObject().contains(document.activeElement))
    return true;
  //
  if (focusing && Client.Element.lastFocusedElement?.focusTimeout)
    return this.getRootObject().contains(Client.Element.lastFocusedElement.getRootObject());
  //
  return false;
};


/**
 * Give focus to the element
 * @param {Object} options
 */
Client.Element.prototype.focus = function (options)
{
  // If the focus for this element is already planned I don't do anything
  if (this.focusTimeout)
    return;
  //
  // If it's a further attempt to set focus and another element
  // has been set on fire in the meantime, I'll let it go
  if (options?.retry && Client.Element.lastFocusedElement && Client.Element.lastFocusedElement !== this)
    return;
  //
  Client.Element.lastFocusedElement = this;
  this.view.justFocused = true;
  //
  if (this.domObj) {
    // Check if the objetc is focusable:
    // - the domObj and all its parents must be in the DOM
    // - in the chain no node must have 'display:none'
    // - in the chain no node must have an animation (transition)
    let canfocus = true;
    //
    let focusBlockedTransform = new Map();
    let appui = document.getElementById("app-ui");
    let obj = this.domObj;
    while (obj) {
      if (obj === appui || obj === document)
        break;
      //
      // To know if an animation is ended we use the currentStyle that is live updated during the transition.
      // So if we have a transition we memorize the currentStyle Transition, if at the next tick is the same we could set the focus
      // because the trasnision is probably ended
      let s = obj.style;
      if (!obj.parentNode || s.display === "none")
        canfocus = false;
      else if (!options?.skipAnimationCheck) {
        let currentTransform = window.getComputedStyle ? getComputedStyle(obj).transform : obj.currentStyle["transform"];
        if (s.transform)
          focusBlockedTransform.set(obj, currentTransform);
      }
      obj = obj.parentNode;
    }
    //
    if (canfocus && focusBlockedTransform.size) {
      if (!this.focusBlockedTransform)
        canfocus = false;
      else if (focusBlockedTransform.size === this.focusBlockedTransform.size) {
        for (let [key, value] of focusBlockedTransform) {
          if (!this.focusBlockedTransform.has(key) || this.focusBlockedTransform.get(key) !== value) {
            canfocus = false;
            break;
          }
        }
      }
      else
        canfocus = false;
    }
    this.focusBlockedTransform = focusBlockedTransform;
    //
    if (canfocus) {
      delete this.focusBlockedTransform;
      delete this.nFocusTry;
      //
      let objToFocus = this.getFocusableObj();
      //
      let focus = () => {
        if (document.activeElement !== objToFocus) {
          // Force blur of active element if different from this one
          // This is useful on a mobile device to close the keyboard just by focusing a different element
          // even if it cannot handle focus
          document.activeElement?.blur();
          //
          objToFocus.focus({preventScroll: options?.skipScroll});
          if (document.activeElement !== objToFocus)
            return;
          //
          if (!options?.skipScroll)
            objToFocus.scrollIntoView({block: "nearest", inline: "nearest"});
        }
        //
        if (this.isSelectable() && options?.selectionStart !== undefined && options?.selectionEnd !== undefined) {
          objToFocus.selectionStart = options.selectionStart;
          objToFocus.selectionEnd = options.selectionEnd;
        }
        //
        Client.lastActiveElement = objToFocus;
        return true;
      };
      //
      if (!focus())
        setTimeout(focus, 0);
      //
      if (this.focusTimeout) {
        clearTimeout(this.focusTimeout);
        delete this.focusTimeout;
      }
      return;
    }
  }
  //
  if (!this.focusTimeout) {
    if (!this.nFocusTry)
      this.nFocusTry = 0;
    else
      this.nFocusTry++;
    //
    // To much focus try, skip the operation
    if (this.nFocusTry > 15) {
      delete this.nFocusTry;
      return;
    }
    //
    this.focusTimeout = setTimeout(() => {
      delete this.focusTimeout;
      options = Object.assign({retry: true}, options);
      this.focus(options);
    }, 100);
  }
};


/**
 * Returns the object that can be focused
 */
Client.Element.prototype.getFocusableObj = function ()
{
  if (this.hasFocus())
    return document.activeElement;
  //
  let objToFocus = this.domObj;
  if (!Client.Utils.isNodeEditable(objToFocus)) {
    // Maybe we have a composite element and we can focus an input inside it
    let el = objToFocus.getElementsByTagName("INPUT");
    if (!el.length)
      el = objToFocus.getElementsByTagName("TEXTAREA");
    if (!el.length)
      el = objToFocus.querySelectorAll("span[contenteditable=true]");
    if (!el.length)
      el = objToFocus.querySelectorAll("button.item-cover");
    if (!el.length)
      el = objToFocus.querySelectorAll("div.range-knob-handle");
    if (el.length)
      objToFocus = el[0];
  }
  return objToFocus;
};


/**
 * Remove focus from the element
 */
Client.Element.prototype.blur = function ()
{
  if (this.domObj)
    this.domObj.blur();
  //
  if (this.focusTimeout) {
    clearTimeout(this.focusTimeout);
    delete this.focusTimeout;
    delete this.focusBlockedTransition;
    delete this.nFocusTry;
  }
};


/**
 * click the element
 */
Client.Element.prototype.click = function ()
{
  if (this.domObj)
    this.domObj.click();
};


/**
 * Sends properties to server
 */
Client.Element.prototype.sendProp = function ()
{
  var e = [];
  for (var i = 0; i < arguments.length; i++) {
    var x = this[arguments[i]];
    if (x === undefined)
      x = this.domObj[arguments[i]];
    e.push({obj: this.id, id: "chgProp", content: {name: arguments[i], value: x, clid: Client.id}});
  }
  //
  return e;
};


/**
 * onresize Message
 * @param {Event} ev - the event occured when the browser window was resized
 */
Client.Element.prototype.onResize = function (ev)
{
  if (this.elements) {
    for (var i = 0; i < this.ne(); i++) {
      this.elements[i].onResize(ev);
    }
  }
  if (this.rownum !== undefined)
    this.updateListPosition();
};


/**
 * Activate the element
 */
Client.Element.prototype.activate = function ()
{
  this.domObj.focus();
  if (this.domObj.type === "button" || this.domObj.tagName.indexOf("BUTTON") > -1)
    this.domObj.click();
};


/**
 * Check if the element or one of its children is the element to activate
 * @param {type} key - the key pressed
 * @param {Object} checkedElems - elements already checked
 * @returns {Client.Element|undefined}
 */
Client.Element.prototype.findElementToActivate = function (key, checkedElems)
{
  if (!checkedElems)
    checkedElems = {};
  //
  var el;
  //
  // The element was already checked
  if (checkedElems[this.id])
    return;
  //
  // It's the first time I see this element, add it to the checked elements map
  checkedElems[this.id] = true;
  //
  // Do not activate invisible elements or children of invisible elements
  if (this.isVisible() === false)
    return;
  //
  // Do not activate disabled elements or children of disabled elements
  if (this.enabled === false)
    return;
  //
  if (this.cmdKey === key)
    el = this;
  else if (this.elements) {
    for (var i = 0; i < this.ne(); i++) {
      el = this.elements[i].findElementToActivate(key, checkedElems);
      if (el)
        break;
    }
  }
  //
  return el;
};


/**
 * Check if the element has to handle key down
 * @param {type} ev - key down event
 * @returns {Boolean}
 */
Client.Element.prototype.handleKeyDown = function (ev)
{
  var ret = false;
  if (Client.Utils.isNodeEditable(this.domObj)) {
    // If it's CTRL-A, CTRL-Y, CTRL-Z, CTRL-X, CTRL-C, CTRL-V
    if (ev.ctrlKey || ev.metaKey) {
      var c = ["A", "Y", "Z", "X", "C", "V"];
      if (c.indexOf(String.fromCharCode(ev.keyCode)) > -1)
        ret = true;
    }
    else if (ev.keyCode === 37 || ev.keyCode === 39) { // arrow left and right
      //
      // Get cursor current position
      var curPos = Client.Utils.getCursorPos(this.domObj);
      //
      // If cursor is in the middle
      if (curPos > 0 && curPos < this.domObj.value.length)
        ret = true;
      else if (curPos === 0 && ev.keyCode !== 37) {
        // If cursor is at the beginning and the key pressed is not the left arrow
        ret = true;
      }
      else if (curPos === this.domObj.value.length && ev.keyCode !== 39) {
        // If cursor is at the end and key pressed is not the right arrow
        ret = true;
      }
    }
  }
  //
  return ret;
};


/**
 * Return the first child element that can get focus
 * @returns {Client.Element}
 */
Client.Element.prototype.getFirstFocusableElement = function ()
{
  var el;
  for (var i = 0; i < this.ne(); i++) {
    if (Client.Utils.isNodeEditable(this.elements[i].domObj)) {
      el = this.elements[i];
      break;
    }
  }
  //
  return el;
};


/**
 * Return the last child element that can get focus
 * @returns {Client.Element}
 */
Client.Element.prototype.getLastFocusableElement = function ()
{
  var el;
  for (var i = this.ne() - 1; i >= 0; i--) {
    if (Client.Utils.isNodeEditable(this.elements[i].domObj)) {
      el = this.elements[i];
      break;
    }
  }
  //
  return el;
};


/**
 * Get the previous element that can receive focus
 * @param {Client.Element} currEl
 * @returns {Client.Element|undefined}
 */
Client.Element.prototype.getPrevFocusableElement = function (currEl)
{
  var previousEl;
  for (var i = 0; i < this.ne(); i++) {
    if (this.elements[i] !== currEl)
      previousEl = this.elements[i];
    else
      break;
  }
  //
  return previousEl;
};


/**
 * Get the next element that can receive focus
 * @param {Client.Element} currEl
 * @returns {Client.Element|undefined}
 */
Client.Element.prototype.getNextFocusableElement = function (currEl)
{
  var nextEl;
  for (var i = 0; i < this.ne(); i++) {
    if (this.elements[i] === currEl) {
      if (i + 1 < this.ne())
        nextEl = this.elements[i + 1];
      break;
    }
  }
  //
  return nextEl;
};


/**
 * Set custom error message
 * @param {string} message - the error message
 * @param {bool} srv
 */
Client.Element.prototype.setError = function (message, srv)
{
  if (!message && this.errorPopup)
    this.errorPopup.close();
  //
  if (this.errorMessage !== message) {
    this.errorMessage = message;
    //
    if (this.errorMessage)
      this.domObj.classList.add("element-invalid");
    else
      this.domObj.classList.remove("element-invalid");
  }
  //
  if (srv) {
    if (message) {
      let errval = (this.domObj.value !== undefined) ? this.domObj.value : this.value;
      if ((this.domObj.type === "date" || this.domObj.type === "datetime-local") && !this.domObj.value && this.domObj.validationMessage && this.domObj.validity && this.domObj.validity.badInput)
        errval = "badinput";
      this.srvError = {message: message, value: errval};
    }
    else
      delete this.srvError;
  }
};


/**
 * Check & Reset custom error message
 *  * @param {bool} final
 */
Client.Element.prototype.checkError = function (final)
{
  // if the error is the same as before, reset it (called on "oninput")
  if (this.srvError && this.domObj.value === this.srvError.value)
    this.setError(this.srvError.message);
  else {
    let errorMsg = final ? this.domObj.validationMessage : "";
    if (final && (this.domObj.type === "date" || this.domObj.type === "datetime-local") && document.activeElement === this.domObj && !this.domObj.value)
      errorMsg = "";
    this.setError(errorMsg, final);
  }
};


/**
 * Play an animation on this element
 * @param {Object} content - animation to play
 */
Client.Element.prototype.playAnimation = function (content)
{
  if (this.currentAnimation)
    this.currentAnimation.stop();
  //
  var cnt = content;
  if (typeof cnt === "string")
    cnt = JSON.parse(content);
  //
  this.currentAnimation = new Client.ClientAnimation(cnt, this);
  this.currentAnimation.play(false);
};


/**
 * Play an animation on this element, reverted
 * @param {Object} content - animation to play
 *                           (optional, if the animation was already played on the Element this parameter can be skipped;
 *                           in that case the animation to be reverted will be the last animation played)
 */
Client.Element.prototype.revertAnimation = function (content)
{
  if (this.currentAnimation)
    this.currentAnimation.stop();
  //
  // If an animation is sent from the server use that
  if (content && content !== "") {
    var cnt = content;
    if (typeof cnt === "string")
      cnt = JSON.parse(content);
    //
    this.currentAnimation = new Client.ClientAnimation(cnt, this);
  }
  //
  // If the animation is present play it in reverted order
  if (this.currentAnimation)
    this.currentAnimation.play(true);
};


/**
 * Pause an animation on this element
 */
Client.Element.prototype.pauseAnimation = function ()
{
  if (this.currentAnimation)
    this.currentAnimation.pause();
};


/**
 * Stop an animation on this element
 * @param {boolean} gotoEnd - if true goes to the final status
 */
Client.Element.prototype.stopAnimation = function (gotoEnd)
{
  if (this.currentAnimation)
    this.currentAnimation.stop(gotoEnd);
};


/**
 * Resumes an animation on this element
 */
Client.Element.prototype.resumeAnimation = function ()
{
  if (this.currentAnimation)
    this.currentAnimation.resume();
};


/**
 * Reset an animation, the element is reset on the initial status before the animation.
 * If the animation is ongoing it will be stopped
 */
Client.Element.prototype.resetAnimation = function ()
{
  if (this.currentAnimation)
    this.currentAnimation.reset();
};


/**
 * An animation/segment is ended, send the event to the other side
 * @param {Integer} segment Segment that ha ended or -1 in case of animation stopped
 * @param {Boolean} final final step
 * @param {Boolean} stopped true if the user stopped the animation
 * @param {String} id id of the animation
 */
Client.Element.prototype.onEndAnimation = function (segment, final, stopped, id)
{
  // Send the event only if is needed and we are the master
  if (this.sendEndAnimation && (Client.clientType === undefined || Client.clientType === "owner")) {
    var ee = [{obj: this.id, id: "onEndAnimation", content: {segment: segment, final: final, stopped: stopped, id: (id === undefined ? "" : id)}}];
    Client.mainFrame.sendEvents(ee);
  }
};


/**
 * Add to this element an animation definition
 * @param {Array} animationsList - array of animations
 */
Client.Element.prototype.attachAnimations = function (animationsList)
{
  if (!this.animations)
    this.animations = [];
  //
  for (var i = 0; i < animationsList.length; i++) {
    var def = animationsList[i];
    if (typeof def === "string")
      def = JSON.parse(def);
    //
    // Enable the events to respond to trigger changes
    var pthis = this;
    switch (def.trigger) {
      case "hover" :
        if (!this.overTrigger)
          this.domObj.addEventListener("mouseover", function (ev) {
            pthis.onAnimationTrigger(ev, "mouseover");
          }); // jshint ignore:line
        if (!this.outTrigger)
          this.domObj.addEventListener("mouseout", function (ev) {
            pthis.onAnimationTrigger(ev, "mouseout");
          }); // jshint ignore:line
        this.overTrigger = true;
        this.outTrigger = true;
        break;

      case "active" :
        if (!this.downTrigger) {
          if (Client.mainFrame.device.isMobile)
            this.domObj.addEventListener("touchstart", function (ev) {
              // Click will be prevented
              if (Client.mainFrame.isClickPrevented())
                return;
              //
              // Try to detect the offset of the touch in respect of the element
              var x = ev.targetTouches && ev.targetTouches.length > 0 ? ev.targetTouches[0].clientX : -1;
              var y = ev.targetTouches && ev.targetTouches.length > 0 ? ev.targetTouches[0].clientY : -1;
              if (x !== -1 && y !== -1 && pthis.domObj.getBoundingClientRect) {
                var rect = pthis.domObj.getBoundingClientRect();
                ev.offsetX = x - rect.left;
                ev.offsetY = y - rect.top;
              }
              pthis.onAnimationTrigger(ev, "mousedown");
            }, {passive: true}); // jshint ignore:line
          else
            this.domObj.addEventListener("mousedown", function (ev) {
              // Click will be prevented
              if (Client.mainFrame.isClickPrevented())
                return;
              //
              pthis.onAnimationTrigger(ev, "mousedown");
            }, {passive: true}); // jshint ignore:line
        }
        if (!this.upTrigger && def.type !== "ripple") {
          if (Client.mainFrame.device.isMobile)
            this.domObj.addEventListener("touchend", function (ev) {
              // Click will be prevented
              if (Client.mainFrame.isClickPrevented())
                return;
              //
              pthis.onAnimationTrigger(ev, "mouseup");
            }, {passive: true}); // jshint ignore:line
          else
            this.domObj.addEventListener("mouseup", function (ev) {
              // Click will be prevented
              if (Client.mainFrame.isClickPrevented())
                return;
              //
              pthis.onAnimationTrigger(ev, "mouseup");
            }, {passive: true}); // jshint ignore:line
        }
        this.downTrigger = true;
        this.upTrigger = true;
        break;

      case "change" :
        this.addChangeTrigger(def);
        break;

      case "scroll" :
        if (!this.scrollTrigger)
          this.domObj.addEventListener("scroll", function (ev) {
            pthis.onAnimationTrigger(ev, "scroll");
          }, {passive: true}); // jshint ignore:line
        this.scrollTrigger = true;
        break;

      case "enter" :
        this.enterTrigger = true;
        if (def.autoreverse)
          this.exitTrigger = true;
        break;

      case "exit" :
        this.exitTrigger = true;
        break;

      case "remove" :
        this.removeTrigger = true;
        break;

      case "changeprop" :
        this.changePropTrigger = true;
        if (!this.changePropTriggerList)
          this.changePropTriggerList = [];
        this.changePropTriggerList.push(def.prop);
        break;

      case "click" :
        if (!this.clickTrigger) {
          if (Client.mainFrame.device.isMobile && Client.mainFrame.hammerEnabled) {
            // "doubleTap" "Press"
            var hammer = Client.mainFrame.getHammerManager();
            //
            // If the hammer manager has not still this recognizer, add it
            if (!hammer.get("tap"))
              hammer.add(new Hammer.Tap({threshold: 20, time: 500}));
            hammer.on("tap", function (ev) {
              // Click inside me
              if (Client.Utils.isMyParent(ev.target, pthis.id) || ev.target === pthis.getRootObject()) {
                var x = 0, y = 0;
                if (ev.pointers && ev.pointers.length > 0) {
                  // Calculate the offset of the click respect the object
                  var rect = pthis.getRootObject().getBoundingClientRect();
                  if (ev.pointers[0].clientX > rect.left)
                    x = ev.pointers[0].clientX - rect.left;
                  if (ev.pointers[0].clientY > rect.top)
                    y = ev.pointers[0].clientY - rect.top;
                }
                ev.offsetX = x;
                ev.offsetY = y;
                // Click will be prevented
                if (Client.mainFrame.isClickPrevented())
                  return;
                pthis.onAnimationTrigger(ev, "click");
              }
            });
          }
          else
            this.domObj.addEventListener("click", function (ev) {
              // Click will be prevented
              if (Client.mainFrame.isClickPrevented())
                return;
              pthis.onAnimationTrigger(ev, "click");
            });
        }
        //
        this.domObj.setAttribute("click-delay", 40);
        //
        this.clickTrigger = true;
        break;

      case "dblclick" :
        if (!this.dblClickTrigger) {
          if (Client.mainFrame.device.isMobile && Client.mainFrame.hammerEnabled) {
            // "doubleTap" "Press"
            var hammer = Client.mainFrame.getHammerManager();
            //
            // If the hammer manager has not still this recognizer, add it
            if (!hammer.get("press"))
              hammer.add(new Hammer.Press());
            hammer.on("press", function (ev) {
              // Click inside me
              if (Client.Utils.isMyParent(ev.target, pthis.id) || ev.target === pthis.getRootObject()) {
                var x = 0, y = 0;
                if (ev.pointers && ev.pointers.length > 0) {
                  // Calculate the offset of the click respect the object
                  var rect = pthis.getRootObject().getBoundingClientRect();
                  if (ev.pointers[0].clientX > rect.left)
                    x = ev.pointers[0].clientX - rect.left;
                  if (ev.pointers[0].clientY > rect.top)
                    y = ev.pointers[0].clientY - rect.top;
                }
                ev.offsetX = x;
                ev.offsetY = y;
                // Click will be prevented
                if (Client.mainFrame.isClickPrevented())
                  return;
                pthis.onAnimationTrigger(ev, "dblclick");
              }
            });
          }
          else
            this.domObj.addEventListener("dblclick", function (ev) {
              // Click will be prevented
              if (Client.mainFrame.isClickPrevented())
                return;
              pthis.onAnimationTrigger(ev, "dblclick");
            });
        }
        this.dblClickTrigger = true;
        break;

      case "animation" :
        this.domObj.setAttribute(def.prop, "handle");
        this.domObj.addEventListener("animationstart", function (ev) {
          if (ev.target.getAttribute(ev.animationName) === "handle" && ev.animationName === "collapseElement") {
            // Already done
            if (ev.target.style.maxHeight === "0px")
              return;
            //
            // Clear and set the starting value
            ev.target.style.transition = "";
            ev.target.style.maxHeight = ev.target.offsetHeight + "px";
            //
            // Set the animation
            let duration = def.duration !== undefined ? def.duration : 250;
            ev.target.style.transition = `max-height ${duration}ms`;
            let t = ev.target.offsetHeight;
            //
            // Animate to the destination
            ev.target.style.maxHeight = "0px";
          }
          if (ev.target.getAttribute(ev.animationName) === "handle" && ev.animationName === "expandElement") {
            // we need to expand only if the element was collapsed, so if the maxHeight is not 0 we are already expanded
            if (ev.target.style.maxHeight !== "0px")
              return;
            //
            // Clear
            ev.target.style.transition = "";
            ev.target.style.maxHeight = "";
            //
            // Get the target height
            let t = this?.parentWidget?.maxHeight ? this.parentWidget.maxHeight : ev.target.offsetHeight;
            //
            // Set the stage
            ev.target.style.maxHeight = "0px";
            let duration = def.duration !== undefined ? def.duration : 250;
            ev.target.style.transition = `max-height ${duration}ms`;
            //
            // Force the refow so appliyng the height uses the transition
            let t2 = ev.target.offsetHeight;
            //
            // Animate to the destination value
            ev.target.style.maxHeight = t + "px";
          }
        }.bind(this));
        this.domObj.addEventListener("animationend", function (ev) {
          let maxH = "";
          if (this.parentWidget && this.parentWidget.isLeaf && !this.parentWidget.isLeaf())
            maxH = this?.parentWidget?.hasMaxHeight() ? this.parentWidget.getMaxHeight() + "px" : "";
          else
            maxH = this?.parentWidget?.maxHeight ? this.parentWidget.maxHeight + "px" : "";
          //
          if (ev.target.getAttribute(ev.animationName) === "handle" && ev.animationName === "collapseElement") {
            ev.target.style.transition = "";
          }
          if (ev.target.getAttribute(ev.animationName) === "handle" && ev.animationName === "expandElement") {
            // Already done
            if (ev.target.style.maxHeight === maxH || this.domObj !== ev.target)
              return;
            ev.target.style.transition = "";
            ev.target.style.maxHeight = maxH;
          }
        }.bind(this));
        break;
    }
    //
    // Add the definition to the list
    this.animations.push(def);
  }
};


/**
 * Adds an animation with a change trigger
 * @param {Object} animation - animation definition object
 */
Client.Element.prototype.addChangeTrigger = function (animation)
{
  if (!this.changeTrigger) {
    var pthis = this;
    //
    // Add Onchange
    this.domObj.addEventListener("change", function (ev) {
      pthis.onAnimationTrigger(ev, "change");
    });
    //
    this.changeTrigger = true;
  }
};


/**
 * Function called when a condition must trigger the animations
 * @param {MouseEvent} ev -
 * @param {String} trigger - trigger that has been fired
 * @param {String} par1 - optional parameter:
 *   1) in case of "changeprop" trigger is the property changed
 */
Client.Element.prototype.onAnimationTrigger = function (ev, trigger, par1)
{
  // No animation for this object
  if (this.animate === false)
    return;
  //
  var triggerAnimation = trigger;
  var tempFrom;
  //
  // Adapt the trigger for special cases
  switch (trigger) {
    case "mouseover":
    case "mouseout":
      triggerAnimation = "hover";
      break;

    case "mousedown":
    case "mouseup":
      triggerAnimation = "active";
      break;
  }
  var enterHandled = false;
  var exitHandled = false;
  for (var a = 0; a < this.animations.length; a++) {
    // Check if this animation must be triggered
    if (this.animations[a].trigger === triggerAnimation || (triggerAnimation === "exit" && this.animations[a].trigger === "enter" && this.animations[a].autoreverse === true)) {
      var def = this.animations[a];
      var targetObject = this;
      //
      // If the target of the animation is another object check if is present
      if (def.target !== undefined && def.target !== "this") {
        targetObject = Client.eleMap[def.target];
        //
        // Skip the animation if the target is not found
        if (!targetObject)
          continue;
      }
      //
      // If we handle the enter/exit animation on this object ok, otherwise we must apply the standard visibility
      if (trigger === "enter" && targetObject === this)
        enterHandled = true;
      if (trigger === "exit" && targetObject === this)
        exitHandled = true;
      //
      // Each type of animation has a special function to generate the segments and can have different defaults based on the trigger
      var animDef;
      switch (def.type) {
        case "slide" :
          // The exit/mouseout/mouseup triggers must play the slide in the reverse order, the other triggers plays it as defined
          if (def.movelength) {
            if (trigger === "exit" || trigger === "mouseout" || trigger === "mouseup") {
              var oldMove;
              if (trigger === "exit" && def.trigger === "enter" && def.autoreverse === true) {
                oldMove = def.movelength;
                def.startPosition = def.movelength;
                def.movelength = 0;
              }
              animDef = Client.ClientAnimation.translate(def);
              if (trigger === "exit" && def.trigger === "enter" && def.autoreverse === true) {
                def.startPosition = 0;
                def.movelength = oldMove;
              }
            }
            else
              animDef = Client.ClientAnimation.translate(def);
          }
          else {
            if (trigger === "exit" || trigger === "mouseout" || trigger === "mouseup")
              animDef = Client.ClientAnimation.slideOut(def, targetObject);
            else
              animDef = Client.ClientAnimation.slideIn(def, targetObject);
          }
          break;

        case "fade" :
          if (trigger === "exit") {
            if (!def.from)
              def.from = 1;
            if (!def.to)
              def.to = 0;
          }
          else if (trigger === "enter") {
            if (!def.from)
              def.from = 0;
            if (!def.to)
              def.to = 1;
          }
          //
          animDef = Client.ClientAnimation.fade(def, targetObject);
          //
          // The mouseout, mouseup and autoreverted triggers must play the reverse animation
          if (trigger === "mouseout" || trigger === "mouseup" || (trigger === "exit" && def.trigger === "enter" && def.autoreverse === true) || (trigger === "enter" && def.trigger === "exit" && def.autoreverse === true)) {
            tempFrom = animDef.segments[0].from;
            animDef.segments[0].from = animDef.segments[0].to;
            animDef.segments[0].to = tempFrom;
          }
          break;

        case "zoom" :
          if (trigger === "exit" || trigger === "remove") {
            if (!def.from)
              def.from = 1;
            if (!def.to)
              def.to = 0;
          }
          else if (trigger === "enter") {
            if (!def.from)
              def.from = 0;
            if (!def.to)
              def.to = 1;
          }
          animDef = Client.ClientAnimation.zoom(def, targetObject);
          //
          // The mouseout and mouseup triggers must play the reverse animation
          if (trigger === "mouseout" || trigger === "mouseup") {
            tempFrom = animDef.segments[0].from;
            animDef.segments[0].from = animDef.segments[0].to;
            animDef.segments[0].to = tempFrom;
          }
          break;

        case "flip" :
          if (trigger === "exit") {
            def.from = 0;
            def.to = 180;
          }
          else if (trigger === "enter") {
            def.from = 180;
            def.to = 0;
          }
          animDef = Client.ClientAnimation.flip(def, targetObject);
          //
          // The mouseout and mouseup triggers must play the reverse animation
          if (trigger === "mouseout" || trigger === "mouseup") {
            tempFrom = animDef.segments[0].from;
            animDef.segments[0].from = animDef.segments[0].to;
            animDef.segments[0].to = tempFrom;
          }
          break;

        case "expand" :
          if (trigger === "exit" || trigger === "remove") {
            if (!def.from)
              def.from = "100%";
            if (!def.to)
              def.to = "0px";
            //
            // Memorize 'original' dimensions, to restore in case of 'enter'
            targetObject.oldExpandedHeigh = targetObject.domObj.style.height;
            targetObject.oldExpandedWidth = targetObject.domObj.style.width;
          }
          else if (trigger === "enter") {
            if (!def.from)
              def.from = "0px";
            if (!def.to)
              def.to = "100%";
          }
          // Handle autoreverse
          if (trigger === "exit" && def.trigger === "enter" && def.autoreverse === true) {
            var oto = def.to;
            def.to = def.from;
            def.from = oto;
          }
          animDef = Client.ClientAnimation.expand(def, targetObject);
          //
          if (trigger === "exit" && def.trigger === "enter" && def.autoreverse === true) {
            var oto = def.to;
            def.to = def.from;
            def.from = oto;
          }
          //
          // Now we can clear the old dimensions cache
          if (trigger === "enter") {
            delete targetObject.oldExpandedHeigh;
            delete targetObject.oldExpandedWidth;
          }
          //
          // The mouseout and mouseup triggers must play the reverse animation
          if (trigger === "mouseout" || trigger === "mouseup") {
            tempFrom = animDef.segments[0].from;
            animDef.segments[0].from = animDef.segments[0].to;
            animDef.segments[0].to = tempFrom;
          }
          break;

        case "ripple":
          // The ripple effect is executed only one time for multiple triggers
          if (trigger === "mouseup" || trigger === "mouseout")
            break;
          //
          // If the ripple is triggered by a mouse down the center of the ripple must be the point pressed/touched
          if (trigger === "mousedown" || trigger === "click" || trigger === "dblclick") {
            // In web we must calculate the offset of the click in respect to the object rect, in mobile
            // the handler has already calculated that
            if (Client.mainFrame.device.isMobile && Client.mainFrame.hammerEnabled) {
              def.posX = ev.offsetX;
              def.posY = ev.offsetY;
            }
            else {
              var brct = this.domObj.getBoundingClientRect();
              def.posX = ev.clientX - brct.left;
              def.posY = ev.clientY - brct.top;
            }
          }
          //
          if (targetObject.executeRipple)
            targetObject.executeRipple(def);
          break;

        case "change" :
          animDef = Client.ClientAnimation.change(def, targetObject);
          //
          // The mouseout and mouseup triggers must play the reverse animation
          if (trigger === "mouseout" || trigger === "mouseup") {
            tempFrom = animDef.segments[0].from;
            animDef.segments[0].from = animDef.segments[0].to;
            animDef.segments[0].to = tempFrom;
          }
          break;

        case "custom" :
          animDef = Client.ClientAnimation.custom(def, targetObject);
          //
          // The mouseout and mouseup triggers must play the reverse animation
          if (trigger === "mouseout" || trigger === "mouseup") {
            tempFrom = animDef.segments[0].from;
            animDef.segments[0].from = animDef.segments[0].to;
            animDef.segments[0].to = tempFrom;
          }
          break;

        case "class" :
          // Set the classes on the object after the delay
          window.setTimeout(function () {
            // Handle autoreverse
            if (trigger === "exit" && def.trigger === "enter" && def.autoreverse === true) {
              var oadd = def.addclasses;
              def.addclasses = def.removeclasses;
              def.removeclasses = oadd;
            }
            if (targetObject.domObj.classList && targetObject.domObj.classList.add && def.addclasses)
              targetObject.domObj.classList.add(def.addclasses);
            if (targetObject.domObj.classList && targetObject.domObj.classList.remove && def.removeclasses)
              targetObject.domObj.classList.remove(def.removeclasses);
            //
            // Handle autoreverse
            if (trigger === "exit" && def.trigger === "enter" && def.autoreverse === true) {
              var oadd = def.addclasses;
              def.addclasses = def.removeclasses;
              def.removeclasses = oadd;
            }
            //
            // Set the default 'visibility' handling
            if (targetObject === this && trigger === "exit")
              targetObject.domObj.style.display = "none";
            if (targetObject === this && trigger === "enter")
              targetObject.domObj.style.display = "";
          }, def.delay);
          //
          // If we must apply the 'restore' after the animation end set the timer to remove the added classes and add the removed classes
          if (def.restore && def.duration > 0) {
            window.setTimeout(function () {
              // Handle autoreverse
              if (trigger === "exit" && def.trigger === "enter" && def.autoreverse === true) {
                var oadd = def.addclasses;
                def.addclasses = def.removeclasses;
                def.removeclasses = oadd;
              }
              //
              if (targetObject.domObj.classList && targetObject.domObj.classList.remove && def.addclasses)
                targetObject.domObj.classList.remove(def.addclasses);
              if (targetObject.domObj.classList && targetObject.domObj.classList.add && def.removeclasses)
                targetObject.domObj.classList.add(def.removeclasses);
              //
              // Handle autoreverse
              if (trigger === "exit" && def.trigger === "enter" && def.autoreverse === true) {
                var oadd = def.addclasses;
                def.addclasses = def.removeclasses;
                def.removeclasses = oadd;
              }
            }, def.delay + def.duration);
          }
          break;
      }
      //
      if (animDef) {
        var transformName = "style_transform";
        //
        // At the last step the object must be hidden (only if we are animating this object,
        // otherwise we cannot change its visibility)
        if (trigger === "exit" && targetObject === this) {
          animDef.finalState = {style_display: "none", style_opacity: ""};
          if (def.restore)
            animDef.finalState[transformName] = "";
          //
          // At the end of the expand reset the dimensions to the base dimensionsions
          if (def.type === "expand") {
            var eleObj = targetObject.getRootObject();
            if (def.expanddirection === "vertical") {
              animDef.finalState["style_height"] = eleObj.getAttribute("originalH");
              animDef.finalState["style_minHeight"] = eleObj.style.minHeight ? eleObj.style.minHeight : "";  // at the end restore the min height
            }
            else {
              animDef.finalState["style_width"] = eleObj.getAttribute("originalW");
              animDef.finalState["style_minWidth"] = eleObj.style.minWidth ? eleObj.style.minWidth : "";
            }
          }
          //
          // For handling expansion we must remove the min height during the animation
          if (def.type === "expand") {
            if (def.expanddirection === "vertical")
              animDef.startingState = {style_minHeight: "0px"};
            else
              animDef.startingState = {style_minWidth: "0px"};
          }
          targetObject.oldDisplay = targetObject.domObj.style.display;
        }
        //
        // At the first step the object must be visible (only if we are animating this object,
        // otherwise we cannot change its visibility)
        if (trigger === "enter" && targetObject === this) {
          animDef.startingState = {style_display: this.oldDisplay};
          if (def.restore) {
            animDef.finalState = {style_opacity: ""};
            animDef.finalState[transformName] = "";
          }
          //
          // At the end of the expand reset the dimensions to the base dimensionsions
          var eleObj = targetObject.getRootObject();
          if (def.restore && def.type === "expand") {
            if (def.expanddirection === "vertical")
              animDef.finalState["style_height"] = eleObj.getAttribute("originalH");
            else
              animDef.finalState["style_width"] = eleObj.getAttribute("originalW");
          }
          delete targetObject.oldDisplay;
          //
          if (def.type === "expand") {
            if (!animDef.finalState)
              animDef.finalState = {};
            //
            if (def.expanddirection === "vertical") {
              animDef.startingState["style_minHeight"] = "0px";
              animDef.finalState["style_height"] = eleObj.getAttribute("originalH");
              animDef.finalState["style_minHeight"] = eleObj.style.minHeight ? eleObj.style.minHeight : "";  // at the end restore the min height
            }
            else {
              animDef.startingState["style_minWidth"] = "0px";
              animDef.finalState["style_width"] = eleObj.getAttribute("originalW");
              animDef.finalState["style_minWidth"] = eleObj.style.minWidth ? eleObj.style.minWidth : "";
            }
          }
        }
        //
        // At the end of this animation the domObj must be removed
        if (trigger === "remove" || this.removeElement) {
          var pthis = targetObject;
          animDef.endcallback = function () {
            pthis.domObj.parentNode.removeChild(pthis.domObj);
          };
        }
        //
        // Set the ID of the animation triggered and play it on this element
        animDef.id = def.id;
        var triggeredAnimation = new Client.ClientAnimation(animDef, targetObject);
        triggeredAnimation.play(false);
      }
    }
  }
  //
  // If an 'enter' trigger is on another object still we have to handle the visibility of this object
  if (trigger === "enter" && !enterHandled) {
    this.domObj.style.display = this.oldDisplay === undefined ? "" : this.oldDisplay;
    delete this.oldDisplay;
  }
  if (trigger === "exit" && !exitHandled) {
    this.oldDisplay = this.domObj.style.display;
    this.domObj.style.display = "none";
  }
};


/**
 * Executes a ripple on this element
 * @param {Object} params - parameters of the animation
 */
Client.Element.prototype.executeRipple = function (params)
{
  // Read the parameters
  var easing = params.easing ? params.easing : "easeOutQuad";
  var duration = params.duration ? params.duration : 400;
  var radius = params.radius ? params.radius : 150;
  var color = params.color ? params.color : "rgba(255,255,255,0.3)";
  var posX = params.posX ? params.posX : "50%";
  var posY = params.posY ? params.posY : "50%";
  //
  // Handle function names
  if (easing === "ease")
    easing = "easeOutQuad";
  if (easing === "ease-in")
    easing = "easeFrom";
  if (easing === "ease-in-out")
    easing = "easeFrom";
  if (easing === "ease-out")
    easing = "easeTo";
  //
  // Remove old ripples from the element and stop the animation
  if (this.rippleAnimation) {
    this.rippleAnimation.stop();
    this.rippleAnimation = null;
  }
  if (this.rippleElement) {
    this.rippleElement.parentNode.removeChild(this.rippleElement);
    this.rippleElement = null;
  }
  if (this.oldRipplePos) {
    this.domObj.style.position = this.oldRipplePos;
    delete this.oldRipplePos;
  }
  //
  // Create SVG and Circle and add the SVG to the element (in the first position, so it not 'cover' other childs)
  this.rippleElement = document.createElement("SVG");
  if (this.domObj.firstChild)
    this.domObj.insertBefore(this.rippleElement, this.domObj.firstChild);
  else
    this.domObj.appendChild(this.rippleElement);
  //
  // For the browser to create the correct SVG we must create it inline (Chrome bug?)
  this.rippleElement.outerHTML = "<SVG width='100%' height='100%' class='animation-ripple-container' ><CIRCLE cx='" +
          posX + "' cy='" + posY + "' r='0%' fill='" + color + "'></CIRCLE></SVG>";
  //
  // Now we must re-obtain the correct object
  for (var ch = 0; ch < this.domObj.childNodes.length; ch++)
    if (this.domObj.childNodes.item(ch).tagName === "svg" && this.domObj.childNodes.item(ch).getAttribute("class") === "animation-ripple-container") {
      this.rippleElement = this.domObj.childNodes.item(ch);
      break;
    }
  var circleElement = this.rippleElement.firstChild;
  //
  // Check if the element is static positioned, in this case the position is set to relative (with top and left not set the position reamins the same)
  // -> the svg is absolute positioned in this element to cover it (100%), so we need a position different from static to the element
  var pos = "static";
  try {
    var so = window.getComputedStyle(this.domObj, null);
    pos = so.position;
  }
  catch (ex) {
  }
  if (pos === "static") {
    this.oldRipplePos = this.domObj.style.position;
    this.domObj.style.position = "relative";
  }
  //
  // Create the step and final animation functions
  var pthis = this;
  var stepRipple = function (state) {
    // Apply the new radius to the circle
    circleElement.setAttribute("r", state.r);
    //
    // Apply the fading
    pthis.rippleElement.style.opacity = state.o;
  };
  var finishRipple = function (state) {
    // When the ripple ha finished we must remove the animation instance, the SVG and reset the position of the domObj
    if (pthis.rippleAnimation)
      pthis.rippleAnimation = null;
    if (pthis.rippleElement) {
      pthis.rippleElement.parentNode.removeChild(pthis.rippleElement);
      pthis.rippleElement = null;
    }
    if (pthis.oldRipplePos) {
      pthis.domObj.style.position = pthis.oldRipplePos;
      delete pthis.oldRipplePos;
    }
    //
    // Send the event to the server if requested
    pthis.onEndAnimation(1, true, false, params.id ? params.id : "ripple");
  };
  //
  // Create the animation and apply it
  this.rippleAnimation = new Tweenable();
  var config = {
    from: {r: "0%", o: 1},
    to: {r: radius, o: 0},
    duration: duration,
    easing: easing,
    step: stepRipple,
    finish: finishRipple
  };
  this.rippleAnimation.tween(config);
};


/**
 * Return the dom root object of this element
 * @returns {DomElement}
 */
Client.Element.prototype.getRootObject = function ()
{
  return this.domObj;
};


/**
 * Change/Create the tag of this domobj
 * @param {string} tag
 */
Client.Element.prototype.changeTag = function (tag)
{
  if (this.domObj && tag.toLowerCase() === this.domObj.tagName.toLowerCase())
    return;
  //
  var oldDom = this.domObj;
  this.domObj = document.createElement(tag);
  //
  while (oldDom && oldDom.firstChild)
    this.domObj.appendChild(oldDom.firstChild);
  //
  // Copy old class, inline style and ID
  if (oldDom.className)
    this.domObj.className = oldDom.className;
  //
  var s = oldDom.getAttribute("style");
  if (s)
    this.domObj.setAttribute("style", s);
  //
  this.domObj.id = oldDom.id;
  //
  if (oldDom.parentNode)
    oldDom.parentNode.replaceChild(this.domObj, oldDom);
};


/**
 * returns object position and size
 * @param {string} cbId - callback id
 */
Client.Element.prototype.getBoundingClientRect = function (cbId)
{
  var ele = this.getRootObject();
  var rect = ele.getBoundingClientRect();
  var ris = {left: rect.left, top: rect.top, right: rect.right, bottom: rect.bottom, width: rect.width, height: rect.height};
  var e = [{obj: this.id, id: "cb", content: {res: ris, cbId: cbId}}];
  Client.mainFrame.sendEvents(e);
};


/**
 * One of my children element has been removed
 * @param {Element} child
 */
Client.Element.prototype.onRemoveChildObject = function (child)
{
};


/**
 * One of my children element has been moved by the insertBefore function as the sibling position was required
 * The child object was first added to this object, then it is positioned at the right place
 * @param {int} position
 */
Client.Element.prototype.onPositionChildObject = function (position)
{
};


/**
 * Enable tooltip on this element
 * @param {string|object} tooltip
 */
Client.Element.prototype.addTooltip = function (tooltip)
{
  // Close previous instance
  if (this.tooltip)
    this.tooltip.destroy();
  //
  if (tooltip) {
    var opt = {inlinePositioning: true, duration: 100, delay: [750, 100]};
    Object.assign(opt, Client.mainFrame.theme.tippy);
    //
    if (typeof tooltip === "string") {
      opt.content = tooltip;
    }
    else if (typeof tooltip === "object") {
      if (tooltip.text) {
        // previous version
        opt.content = tooltip.text;
        opt.placement = tooltip.position || "auto";
        if (tooltip.animate == "false")
          opt.duration = 0;
      }
      else {
        Object.assign(opt, tooltip);
      }
    }
    //
    this.tooltip = tippy(this.domObj, opt);
  }
};


/**
 * @returns {integen} children count
 */
Client.Element.prototype.ne = function ()
{
  return this.elements ? this.elements.length : 0;
};


/**
 * Return any elements
 * @param {object} type
 * @returns {array} any elements that match criteria
 */
Client.Element.prototype.getElements = function (type)
{
  var ar = [];
  if (!type || this instanceof type)
    ar.push(this);
  if (this.elements) {
    for (var i = 0; i < this.elements.length; i++) {
      var a = this.elements[i].getElements(type);
      for (var j = 0; j < a.length; j++)
        ar.push(a[j]);
    }
  }
  return ar;
};


/**
 * Check if the element or one of its parents is not visible
 * @returns {Boolean}
 */
Client.Element.prototype.isVisible = function ()
{
  let visible = !(this.visible === false);
  //
  // During animation, element could be not visible yet, while its dom obj is already visible
  // So in this case I treat the element as visible
  if (!visible && this.currentAnimation?.status === Client.ClientAnimation.statusMap.ONGOING && this.getRootObject().style.display !== "none")
    visible = true;
  //
  if (!visible)
    return false;
  //
  if (this.parent)
    return Client.Element.prototype.isVisible.call(this.parent);
  //
  return true;
};


/**
 * Tell to the children that the visibility has changed
 * @param {Boolean} visible
 */
Client.Element.prototype.visibilityChanged = function (visible)
{
  if (Client.mainFrame.device.fullscreen && visible)
    this.handleFullscreen();
  //
  if (!this.elements)
    return;
  //
  for (var i = 0; i < this.elements.length; i++)
    this.elements[i].visibilityChanged(visible);
};


/**
 * Get list of properties a testauto can check
 */
Client.Element.prototype.getTestProperties = function ()
{
  return ["innerText", "innerHTML", "value", "src", "visible", "className", "disabled", "enabled", "style"];
};


/**
 * Get element default property
 */
Client.Element.prototype.getDefaultProp = function ()
{
  var prop;
  //
  if (this.domObj) {
    // If domObj has an input tag, value is element default property
    if (this.domObj.tagName === "INPUT" || this.domObj.getElementsByTagName("INPUT").length > 0)
      prop = "value";
    else if (this.domObj.tagName === "IMG" || this.domObj.getElementsByTagName("IMG").length > 0) // If it has an img tag, get src
      prop = "src";
    else // Otherwise get innerText as default property
      prop = "innerText";
  }
  //
  return prop;
};


/**
 * Set given property and fire events if needed
 * @param {Object} options
 *                        - propName
 *                        - propValue
 *                        - events - events to dispatch
 *                        - domObj - dom object to dispatch event on
 */
Client.Element.prototype.setProperty = function (options)
{
  options = options || {};
  var events = options.events || [];
  var domObj = options.domObj || this.domObj;
  var name = options.propName || "";
  var value = options.propValue || "";
  //
  // If there isn't a dom object or name of property to change is missing, do nothing..
  if (!domObj || !name)
    return;
  //
  // If there isn't an event called  "on" + name, probably name is really a property name.
  // So I have to update the element
  if (!domObj["on" + name]) {
    // Just value and checked property are allowed
    if (["value", "checked"].indexOf(name) === -1)
      return;
    //
    var obj = {};
    obj[name] = value;
    this.updateElement(obj);
  }
  else // Otherwise I have to dispatch the requested event
    events.push(name);
  //
  // Create and dispatch all requested events
  var ev;
  for (var i = 0; i < events.length; i++) {
    ev = new Event(events[i], {
      bubbles: true,
      cancelable: true
    });
    //
    domObj.dispatchEvent(ev);
  }
};


/**
 * Handle a registered event
 * @param {Object} ev
 */
Client.Element.prototype.handleRegisteredEvent = function (ev)
{
};


/**
 * Set the class name of the domObj
 * @param {Object} cn
 */
Client.Element.prototype.setClassName = function (cn)
{
  this.domObj.className = cn;
};


/**
 * Set an attrribute on this object or sub-object
 * @param {string} attrName
 * @param {string} value
 */
Client.Element.prototype.setAttribute = function (attrName, value)
{
  // Check for subobjects
  var as = attrName.split(" ");
  var objList = [this.domObj];
  if (as.length > 1) {
    attrName = as[as.length - 1];
    var sel = as[0];
    //
    if (sel === "HTML" || sel === "BODY")
      objList = document.getElementsByTagName(sel);
    else if (sel.startsWith("#"))
      objList = [document.getElementById(sel.substring(1))];
    else {
      objList = this.domObj.getElementsByTagName(as[0]);
      if (!objList.length)
        objList = this.domObj.getElementsByClassName(as[0]);
    }
  }
  //
  for (var i = 0; i < objList.length; i++) {
    var obj = objList[i];
    if (!obj)
      continue;
    if (value === null || value === undefined) {
      obj.removeAttribute(attrName);
    }
    else if (attrName === "class+") {
      obj.classList.add(value);
    }
    else if (attrName === "style+") {
      obj.style.cssText += value;
    }
    else {
      obj.setAttribute(attrName, value);
    }
  }
};

/**
 * Check if the element must use the custom navigation
 */
Client.Element.prototype.handleCustomNavigation = function ()
{
  if (this.parentWidget)
    return false;
  //
  // If the user set the value return it, if not
  // - ask the parent
  // - if i've not parent use the default (TRUE)
  if (this.customNavigation !== undefined)
    return this.customNavigation;
  else if (this.parent && this.parent.handleCustomNavigation) {
    this.customNavigation = this.parent.handleCustomNavigation();
    return this.customNavigation;
  }
  else
    return true;
};

/**
 * Enable onKey events
 * @param {object} options
 * options.target = [self*,id,app,body]
 * options.type = [down,up,press*]
 * options.enable= [true*,false]
 * options.keys = [*|key-list]
 * options.capture = [true,false*]
 * options.inputs = [true,false*]
 * shift = |
 * command/ctrl = ^
 * alt = /
 * not on input = @
 * Example: [^c,^v] -> Copy and Paste
 * Example: [arrow*] -> arrowLeft, arrowTop, arrowRight, arrowBottom
 */
Client.Element.prototype.enableKeyEvent = function (options)
{
  options = options || {};
  //
  var obj;
  if (options.target === "self" || options.target === undefined)
    obj = this.domObj;
  else if (options.target === "app")
    obj = document.getElementById("app-ui");
  else if (options.target === "body")
    obj = document.body;
  else
    obj = document.getElementById(options.target);
  //
  if (!obj)
    return;
  //
  if (options.keys)
    options.keys = options.keys.split(",");
  //
  var he = function (ev) {
    var ok = true;
    var inp = Client.Utils.isNodeEditable(document.activeElement);
    //
    // Check key
    if (options.keys) {
      var k = ev.key;
      if (k === ",")
        k = "Comma";
      if (ev.ctrlKey || ev.metaKey)
        k = "^" + k;
      if (ev.shiftKey)
        k = "|" + k;
      if (ev.altKey)
        k = "/" + k;
      //
      ok = options.keys.includes(k);
      if (!ok) {
        var ok2 = options.keys.includes("@" + k);
        if (ok2 && !inp)
          ok = true;
      }
    }
    // Check key on inputs
    if (options.inputs != true && inp) {
      ok = false;
    }
    //
    if (ok) {
      let data = this.saveEvent(ev);
      data.type = type;
      data.input = inp;
      Client.mainFrame.sendEvents([{obj: this.id, id: "onKey", content: data}]);
    }
    //
    if (ok && options.cancel) {
      ev.preventDefault();
      ev.stopPropagation();
      return false;
    }
  }.bind(this);
  //
  var type = "key" + (options.type || "press");
  //
  if (options.enable == false) {
    if (this["he" + type]) {
      for (var i = 0; i < this["he" + type].length; i++) {
        obj.removeEventListener(type, this["he" + type][i], {capture: true});
        obj.removeEventListener(type, this["he" + type][i], {capture: false});
      }
    }
    delete this["he" + type];
  }
  else {
    obj.addEventListener(type, he, {capture: options.capture});
    if (!this["he" + type])
      this["he" + type] = [];
    this["he" + type].push(he);
  }
};


/**
 * Set mask
 * @param {String} mask
 * @param {String} type
 */
Client.Element.prototype.setMask = function (mask, type)
{
  this.mask = mask;
  this.maskType = this.getMaskType(type);
  //
  if (this.mask === ">")
    this.domObj.style.textTransform = "uppercase";
  else if (this.mask === "<")
    this.domObj.style.textTransform = "lowercase";
  else if (!this.mask)
    this.domObj.style.textTransform = "";
};


/**
 * Get mask type
 */
Client.Element.prototype.getMaskType = function ()
{
  return "A";
};


/**
 * Check if I have to handle mask
 * @param {Object} ev
 */
Client.Element.prototype.handleMask = function (ev)
{
  // If target is not editable, do nothing
  if (!Client.Utils.isNodeEditable(ev.target))
    return;
  //
  // Check if my parent widget (if any) is an IdfControl
  let isIdfControl = this.parentWidget && (this.parentWidget instanceof Client.IdfControl);
  //
  // Check if mask is a case mask. maskedinput doesn't have to handle case mask
  let isCaseMask = this.mask === ">" || this.mask === "<";
  //
  if (!this.mask || isCaseMask || (isIdfControl && !this.parentWidget.enabled))
    return;
  //
  switch (ev.type) {
    case "keydown":
      let ok = hk(ev);
      if (!ok)
        ev.preventDefault();
      break;

    case "focus":
      if (!ev.target.disabled)
        mc(this.mask, this.maskType, ev);
      break;

    case "blur":
      if (!ev.target.disabled) {
        umc(ev);
        //
        // Ok, if the value is changed we must notify the onchange, since the masking sets the value the event is not triggered
        let oldv = ev.target.getAttribute("idmaskoldvalue");
        if (oldv !== ev.target.value)
          ev.target.onchange(ev);
        ev.target.setAttribute("idmaskoldvalue", ev.target.value);
      }
      break;
  }
};


Client.Element.prototype.applyStyleProp = function (elementList, stylePropName, styleprop)
{
  // Check and create the container of the applied properties for this style
  if (!this.appliedStyle)
    this.appliedStyle = {};
  if (!this.appliedStyle[stylePropName])
    this.appliedStyle[stylePropName] = {};
  //
  let appliedStyle = this.appliedStyle[stylePropName];
  //
  // In editing the setting is not differential, so we need to clear all the set styles AND reset them
  if (Client.mainFrame?.isEditing() && appliedStyle) {
    let styleprp = Object.keys(appliedStyle);
    for (let j = 0; j < styleprp.length; j++) {
      elementList.forEach((el) => {
        if (el && el.getRootObject())
          el.getRootObject().style[styleprp[j]] = "";
      });
      delete appliedStyle[styleprp[j]];
    }
  }
  //
  if (styleprop !== "") {
    if (typeof styleprop === "string") {
      try {
        styleprop = JSON.parse(styleprop);
      }
      catch (ex) {
        styleprop = {};
      }
    }
    //
    let styleprp = Object.keys(styleprop);
    for (let j = 0; j < styleprp.length; j++) {
      var pr = styleprp[j];
      var v = styleprop[pr];
      //
      if (Client.Utils.requireAbs(pr))
        v = Client.Utils.absStyle(v);
      //
      elementList.forEach((el) => {
        if (el && el.getRootObject())
          el.getRootObject().style[pr] = v;
      });
      appliedStyle[pr] = v;
    }
  }
  else {
    // Remove all set values
    let styleprp = Object.keys(appliedStyle);
    for (let j = 0; j < styleprp.length; j++) {
      elementList.forEach((el) => {
        if (el && el.getRootObject())
          el.getRootObject().style[styleprp[j]] = "";
      });
      delete appliedStyle[styleprp[j]];
    }
  }
};


Client.Element.isSelectable = function (domObj)
{
  return typeof domObj.selectionStart === "number";
};
Client.Element.prototype.isSelectable = function ()
{
  return Client.Element.isSelectable(this.domObj);
};


/**
 * Make editable span simulate input behaviour
 * @param {Object} options
 */
Client.Element.simulateInput = function (options)
{
  let {domObj, showHTML, fixEmpty} = options;
  //
  let getValue = () => showHTML ? domObj.innerHtml : domObj.textContent;
  let setValue = newValue => {
    if (showHTML)
      domObj.innerHtml = newValue;
    else
      domObj.textContent = newValue;
    //
    if (domObj.maxLength)
      domObj.lastValue = domObj.value.replaceAll(Client.Element.fakeEmptyValue, " ");
  };
  let removeFakeEmptyValue = () => {
    if (getValue() === Client.Element.fakeEmptyValue) {
      setValue("");
      return true;
    }
  };
  //
  let checkEmptyValue = () => {
    if (!getValue())
      setValue(Client.Element.fakeEmptyValue);
  };
  //
  // In order to handle mask I need to define some properties (value, selectionStart, selectionEnd) on span
  // because maskedinp.js works on those properties (it has been implemented on input only)
  // and I don't want to change the whole maskedinp to handle span case
  Object.defineProperty(domObj, "value", {
    get: function () {
      let v = getValue();
      return (v === Client.Element.fakeEmptyValue ? "" : v);
    },
    set: function (newValue) {
      setValue(newValue);
    }
  });
  //
  Object.defineProperty(domObj, "selectionStart", {
    get: function () {
      let start = Client.Element.getSelection(domObj).start;
      if (start === 1 && getValue() === Client.Element.fakeEmptyValue)
        start = 0;
      return start;
    },
    set: function (newValue) {
      domObj.selStart = newValue;
    }
  });
  //
  Object.defineProperty(domObj, "selectionEnd", {
    get: function () {
      let end = Client.Element.getSelection(domObj).end;
      if (end === 1 && getValue() === Client.Element.fakeEmptyValue)
        end = 0;
      return end;
    },
    set: function (newValue) {
      Client.Element.setSelection(domObj, domObj.selStart, newValue);
    }
  });
  //
  // I also need to define these function to make maskedinput work as expected
  domObj.select = function () {};
  domObj.onchange = function () {};
  //
  if (!fixEmpty)
    return;
  //
  // Flex centered contenteditable spans have a weird bug:
  // the cursor is not vertical centered when they are empty.
  // To avoid this behaviour, add a space on focus and on input when it's empty, and remove it on blur
  domObj.addEventListener("beforeinput", ev => {
    if (ev.data && removeFakeEmptyValue()) {
      // On iOS devices, setting value in beforeinput event cause oninput event to not be triggered.
      // So I have to prevent default and set value explicitly
      if (Client.mainFrame.device.operatingSystem === "ios") {
        ev.preventDefault();
        setValue(ev.data);
      }
      //
      domObj.selectionStart = Infinity;
      domObj.selectionEnd = Infinity;
    }
  });
  //
  domObj.addEventListener("input", ev =>
  {
    let maxLength = domObj.maxLength;
    let currentValue = domObj.value;
    //
    // If there is a maxLength and current value exceeds maxLength
    if (maxLength && currentValue.length > maxLength && ev.inputType !== "deleteContentBackward") {
      let lastValue = domObj.lastValue || "";
      //
      // Calculate diff between last value and current value
      let diff = Client.Utils.getStringsDiff(lastValue, currentValue);
      //
      // Calculate number of characters that exceed maxLength
      let overflow = currentValue.length > maxLength ? currentValue.length - maxLength : 0;
      //
      // Remove overflow characters from diff text
      diff.text = diff.text.slice(0, diff.text.length - overflow);
      //
      // Calculate new value and set it
      let newValue = lastValue.slice(0, diff.start) + diff.text + lastValue.slice(diff.start);
      if (newValue.length > maxLength)
        newValue = newValue.slice(0, maxLength);
      //
      setValue(newValue);
      //
      // Set cursor position
      let newCursorPosition = diff.start + diff.text.length;
      Client.Element.setSelection(domObj, newCursorPosition, newCursorPosition);
    }
    //
    // If there is a maxLength save the last value
    if (maxLength)
      domObj.lastValue = domObj.value.replaceAll(Client.Element.fakeEmptyValue, " ");
  });
  //
  domObj.addEventListener("input", checkEmptyValue);
  domObj.addEventListener("focus", checkEmptyValue);
  domObj.addEventListener("blur", removeFakeEmptyValue);
};


/**
 * Get selection
 * @param {HTMLElement} domObj
 */
Client.Element.getSelection = function (domObj)
{
  let nodes = domObj.childNodes;
  //
  let start = 0, end = 0;
  let selection = document.getSelection();
  //
  if (!selection.rangeCount)
    return {start, end};
  //
  // Don't mess with visible cursor
  let range = selection.getRangeAt(0);
  let inSelection = false;
  for (let node of nodes) {
    let length = 0;
    switch (node.nodeName) {
      case "#text":
        length = node.nodeValue === Client.Element.fakeEmptyValue ? 0 : node.length;
        break;

      case "BR":
        length = 1;
        break;
    }
    //
    if (!inSelection) {
      if (range.startContainer === node) {
        inSelection = true;
        start += range.startOffset;
      }
      else
        start += length;
    }
    //
    if (inSelection && range.endContainer === node) {
      end += range.endOffset;
      break;
    }
    else
      end += length;
  }
  //
  return {start, end};
};


/**
 * Set selection
 * @param {HTMLElement} domObj
 * @param {Integer} start
 * @param {Integer} end
 */
Client.Element.setSelection = function (domObj, start, end)
{
  let nodes = domObj.childNodes;
  if (start === -1 || end === -1 || nodes.length === 0)
    return;
  //
  let length = domObj.innerText.length;
  if (start > length)
    start = length;
  if (end > length)
    end = length;
  //
  let range = document.createRange();
  let startContainer, endContainer;
  let startOffset, endOffset;
  let pos = 0;
  for (let node of nodes) {
    let length;
    switch (node.nodeName) {
      case "#text":
        if (node.nodeValue === Client.Element.fakeEmptyValue) {
          length = 0;
          start = 0;
          end = 0;
        }
        else
          length = node.length;
        break;

      case "BR":
        length = 1;
        break;
    }
    //
    if (start >= pos && start <= pos + length) {
      startContainer = node;
      startOffset = start - pos;
    }
    if (end >= pos) {
      endContainer = node;
      endOffset = end - pos;
    }
    pos += length;
    if (end < pos)
      break;
  }
  //
  range.setStart(startContainer, startOffset);
  range.setEnd(endContainer, Math.min(endContainer.length, endOffset));
  //
  let sel = document.getSelection();
  sel.removeAllRanges();
  sel.addRange(range);
};


/**
 * Handle fullscreen
 */
Client.Element.prototype.handleFullscreen = function ()
{
  // If I don't need fullscreen padding, do nothing
  if (!Client[this.class]?.needsFullscreenPadding)
    return;
  //
  let rootObject = this.getRootObject();
  let elRects = {top: 0, bottom: 0, left: 0, right: 0};
  let bodyRects = document.body.getBoundingClientRect();
  //
  let domObj = rootObject;
  do {
    elRects.top += domObj.offsetTop;
    elRects.left += domObj.offsetLeft;
    domObj = domObj.offsetParent;
  } while (domObj);
  //
  elRects.right = elRects.left + rootObject.offsetWidth;
  elRects.bottom = elRects.top + rootObject.offsetHeight;
  //
  let classList = (this.className || "").split(" ");
  classList = new Set(classList);
  //
  // Add/remove padding classes based on whether element overlaps the safe area edges
  if (Client.mainFrame.device.safeAreaInsets.top && (elRects.top - bodyRects.top <= Client.mainFrame.device.safeAreaInsets.top))
    classList.add("fullscreen-padding-top");
  else
    classList.delete("fullscreen-padding-top");
  //
  if (Client.mainFrame.device.safeAreaInsets.left && (elRects.left - bodyRects.left <= Client.mainFrame.device.safeAreaInsets.left))
    classList.add("fullscreen-padding-left");
  else
    classList.delete("fullscreen-padding-left");
  //
  if (Client.mainFrame.device.safeAreaInsets.right && (bodyRects.right - elRects.right <= Client.mainFrame.device.safeAreaInsets.right))
    classList.add("fullscreen-padding-right");
  else
    classList.delete("fullscreen-padding-right");
  //
  if (Client.mainFrame.device.safeAreaInsets.bottom && (bodyRects.bottom - elRects.bottom <= Client.mainFrame.device.safeAreaInsets.bottom))
    classList.add("fullscreen-padding-bottom");
  else
    classList.delete("fullscreen-padding-bottom");
  //
  this.updateElement({className: Array.from(classList).join(" ")});
};


/**
 * Check if element is editable in editing mode
 */
Client.Element.prototype.isEditable = function (domObj)
{
  // If edit mode is not enabled, I'm not editable
  if (!Client.mainFrame.isEditing())
    return;
  //
  // On IDC all elements are editable
  if (!Client.mainFrame.isIDF)
    return true;
  //
  let editableClasses = [
    Client.IdfControl,
    Client.IdfFieldValue,
    Client.IdfField,
    Client.IdfGroup,
    Client.IdfBox
  ];
  //
  // If I have an ancestor fieldValue, I'm not editable if its index is different than 1
  let ele = this;
  if (domObj) {
    var eleid = Client.Utils.getDomObjRealId(domObj.id);
    ele = Client.eleMap[eleid];
  }
  let parentFieldValue = Client.Utils.getElementParentWidget(ele, Client.IdfFieldValue);
  if (parentFieldValue && parentFieldValue.index !== 1)
    return;
  //
  let obj = this;
  //
  // If I'm not a widget, get my parentWidget
  if (!(this instanceof Client.Widget))
    obj = Client.Utils.getElementParentWidget(this);
  //
  let foundClass = editableClasses.find(cls => obj instanceof cls);
  //
  return !!foundClass;
};


/**
 * Check if element is resizable in editing mode
 */
Client.Element.prototype.isEditorResizable = function (x, y, domObj)
{
  // If edit mode is not enabled, I'm not resizable +
  // On IDC all elements are not resizable
  if (!Client.mainFrame.isEditing() || !Client.mainFrame.isIDF)
    return 0;
  //
  let editableClasses = [
    Client.IdfField,
    Client.IdfBox,
    Client.IdfFrame,
    Client.IdfSection
  ];
  //
  // If i'm a field check the object that the mouse hovers, if is in list we can apply the resize ONLY on the first row
  if (domObj) {
    var eleid = Client.Utils.getDomObjRealId(domObj.id);
    var ele = Client.eleMap[eleid];
    let parentFieldValue = Client.Utils.getElementParentWidget(ele, Client.IdfFieldValue);
    if (parentFieldValue && parentFieldValue.index !== 1)
      return 0;
  }
  //
  let obj = this;
  //
  // If I'm not a widget, get my parentWidget
  if (!(this instanceof Client.Widget))
    obj = Client.Utils.getElementParentWidget(this);
  //
  let foundClass = editableClasses.find(cls => obj instanceof cls);
  //
  if (!foundClass)
    return 0;
  //
  // A section is resizable only vertically if the report is not in edit mode
  if (obj instanceof Client.IdfSection)
    return (!Client.mainFrame.editReport || y !== 1) ? Client.ViewEdit.resize.NONE : Client.ViewEdit.resize.BOTTOM;
  //
  // If we are on a box mastro (direct child of the page) we can resize only if we are in mastro editing
  if (obj instanceof Client.IdfBox && obj.parent instanceof Client.IdfBookPage && Client.mainFrame.editReport)
    return Client.ViewEdit.resize.NONE;
  //
  // lista cursori:
  // ["", "ew-resize", "ew-resize", "ns-resize", "ns-resize", "se-resize", "sw-resize", "ne-resize", "nw-resize"];
  if (x === 0 && y === 0)
    return Client.ViewEdit.resize.NONE; // "" (default cursor)
  else if (x === -1 && y === 0)
    return Client.ViewEdit.resize.LEFT; // "ew-resize" (right side)
  else if (x === 1 && y === 0)
    return Client.ViewEdit.resize.RIGHT; // "ew-resize" (left side)
  else if (x === 0 && y === -1)
    return Client.ViewEdit.resize.TOP; // "ns-resize" (top side)
  else if (x === 0 && y === 1)
    return Client.ViewEdit.resize.BOTTOM; // "ns-resize" (bottom side)
  else if (x === -1 && y === 1)
    return Client.ViewEdit.resize.BOTTOMLEFT; // "se-resize" (bottom-left corner)
  else if (x === 1 && y === 1)
    return Client.ViewEdit.resize.BOTTOMRIGHT; // "sw-resize" (bottom-right corner)
  else if (x === -1 && y === -1)
    return Client.ViewEdit.resize.TOPLEFT; // "ne-resize" (top-left corner)
  else if (x === 1 && y === -1)
    return Client.ViewEdit.resize.TOPRIGHT; // "nw-resize" (top-right corner)
  //
  return 0;
};
