// ************************************************
// Instant Developer RD3 Framework for Bootstrap
// (c) 1999-2016 Pro Gamma Srl - All rights reserved
//
// Classe WebEntryPoint: gestisce il modello in memoria
// dell'intera pagina browser. Rappresenta l'intera
// pagina dell'applicazione
// ************************************************

function WebEntryPoint()
{
  // Proprieta' dell'HEADER
  this.Identifier = "wep";        // Identificatore dell'oggetto
  this.MainCaption = "";          // Titolo della pagina browser, titolo mostrato nell'Header
  this.MainImage = "";            // Immagine mostrata nell'header
  this.HelpFile = "";             // Help da mostrare quando si preme F1
  this.DebugType = 0;             // Tipo di debug da utilizzare
  this.CmdPrompt = "Cmd: ";       // Titolo del campo CMD (se vuota allora il campo comando non appare)
  this.ProgressBarFile = "";      // File da cercare sul server per gestire le operazioni interrompibili
  this.EntryPoint = "";           // Entry Point dell'applicazione (nome applicazione.htm o .aspx)
  this.SessionID = "";            // ID di sessione
  this.Language = "ITA";          // Lingua dell'applicazione
  this.ShowLogoff = true;         // Mostrare l'icona di logoff?
  this.SideMenuWidth = 0;       // Largezza del menu laterale dell'applicazione
  this.VisualFlags = 0x00000EFF;  // Flag visuali
  //
  // Altre proprieta' del WEP
  this.RefreshInterval = 0;       // Valore in millisecondi dell'intervallo di refresh automatico
  this.RefreshLocation = 0;       // Valore in millisecondi dell'intervallo di refresh della getlocation
  this.WidgetMode = false;        // Se vero occorre renderizzare solo la videata attiva e non il resto della pagina
  this.ActiveForm = null;         // Oggetto WebForm che rappresenta la form attiva
  this.UseDecimalDot = false;     // Indica se occorre rappresentare i numeri con il punto decimale o la virgola
  this.WelcomePage = "qhelp.htm"; // URL da caricare come pagina di benvenuto (da mostrare quando non ci sono altre form aperte)
  this.HandledKeys = 0;           // Tasti da intercettare a livello di applicazione
  this.AccentColor = RD3_Glb.IsMobile7() ? "rgb(21, 125, 251)" : "rgb(0, 129, 194)";   // Color di accento di default
  //
  this.MenuType = RD3_Glb.MENUTYPE_LEFTSB; // Tipo di menu' applicativo
  //
  this.MotionThreshold = {accel: 999, rot: 999};
  //
  // Oggetti secondari gestiti da questo oggetto di modello
  this.CmdObj = new CommandList();    // Gestore dei comandi e command set
  this.VSList = new VisAttrList();    // Gestore dei visual styles
  this.IndObj = new IndHandler();     // Gestore degli indicatori
  this.TimerObj = new TimerHandler(); // Gestore dei timer
  this.VoiceObj = new VoiceHandler(); // Gestore dei comandi vocali
  this.StackForm = new Array();      // Elenco delle form aperte
  this.CustomCommands = new Array();  // Elenco dei custom commands di pannello
  this.CommandZones = new Array();    // Mappa dei comandi di pannello nelle zone
  this.DelayDialog = new PopupDelay();  // Delay o Progress Window
  this.ExecuteRitardati = new Array();  // Array di comandi ritardati arrivati dal server
  //
  this.ScreenZones = new Array();  // Array delle ScreenZone
  //
  // Variabili di classe, gestite localmente
  this.RefreshTimerId = 0;    // Timer utilizzato per il refresh
  this.FocusBoxTimerId = 0;   // Timer utilizzato per il nascondere il focs box
  this.ResizeTimerId = 0;     // Timer utilizzato per il resize
  this.Redirecting = false;   // Indica che la pagina sta cambiando con un altra
  this.CloseFormId = "";      // ID della form da chiudere
  this.MenuScrollDir = 0;     // Direzione dello scroll attuale (1 down, -1 up, 0 nessuno)
  this.WatchLocationId = 0;   // Watch utilizzato per leggere la location
  this.LastWatchTime = 0;     // Tempo in cui e' stata inviata l'ultima informazione
  this.DisableOnClick = 0;    // Utilizzato per disabilitare la gestione dell'onclick che chiude i popup
  //
  // Struttura per la definizione delle caratteristiche degli eventi di questo oggetto
  this.CloseAllEventDef = RD3_Glb.EVENT_SERVERSIDE | RD3_Glb.EVENT_IMMEDIATE;   // Click sul pulsante chiudi tutto
  this.CloseAppEventDef = RD3_Glb.EVENT_SERVERSIDE | RD3_Glb.EVENT_IMMEDIATE;   // Click sul pulsante chiudi applicazione
  //
  // Variabili di collegamento con il DOM
  this.Realized = false;        // Se vero, gli oggetti del DOM sono gia' stati creati
  this.InResponse = false;      // Se vero stavo gestendo una risposta del server
  this.RefreshCommands = false; // Se vero durante l'end process devo fare rinfrescare i comandi per aggiornare le toolbar
  this.RecalcLayout = false;    // Se vero devo forzare l'adaptlayout alla fine della gestione della risposta
  //
  this.WepBox = null;         // Contenitore dell'intero WebEntryPoint
  this.HeaderBox = null;      // Div che contiene l'header
  this.SideMenuBox = null;    // Div che contiene il menu e la form list
  this.FormsBox = null;       // Div che contiene le form
  this.LeftDockedBox = null;  // Div che contiene le form docked left
  this.TopDockedBox = null;   // Div che contiene le form docked top
  this.RightDockedBox = null; // Div che contiene le form docked right
  this.BottomDockedBox = null;// Div che contiene le form docked bottom
  this.WelcomeBox = null;     // Div che contiene la welcome page
  this.WelcomeForm = null;    // Form che contiene la welcome page in caso mobile
  this.TrayletBox = null;     // Il DIV che contiene l'IFRAME per inviare messaggi alla traylet
  this.CalPopup = null;       // l'IFRAME del calendario
  this.BlobFrame = null;      // l'IFRAME per il caricamento dei blob
  //
  this.MenuBox = null;        // Div che contiene il menu' laterale (senza la from list)
  this.MenuScrollUp = null;   // Div superiore che compare se il menu deve scrollare (al posto della scrollbar)
  this.MenuScrollDown = null; // Div inferiore che compare se il menu deve scrollare (al posto della scrollbar)
  this.FormListBox = null;    // Div che contiene la lista delle form, il titolo ed il pulsante di chiusura
  this.FormListHeader = null; // Div che contiene il titolo della FormList
  this.FormListTitle = null;  // Div che contiene i nomi delle form aperte
  this.CloseAllBox = null;    // Div che contiene il pulsante di chiusura di tutte le form
  this.CloseAllButton = null; // Span che racchiude l'intero pulsante Chiudi tutto e inserito dentro CloseAllBox sul lato destro
  this.CloseAllImg = null;    // Immagine del pulsante chiudi tutto
  this.CloseAllTxt = null;    // Testo del pulsante chiudi tutto
  //
  // Oggetti per la definizione dell'header
  this.SuppressMenuBox = null;  // IMG che contiene il bottone per la soppressione del menu'
  this.MainImageBox = null;     // IMG che contiene l'immagine dell'applicazione
  this.MainCaptionBox = null;   // SPAN che contiene la caption dell'applicazione
  this.DividerBox = null;       // SPAN utilizzato per spaziare i comandi nell'header
  this.DebugImageBox = null;    // IMG utilizzato il bottone di open debug
  this.HelpFileBox = null;      // IMG utilizzato il bottone di help
  this.CommandBox = null;       // SPAN che contiene il prompt e l'input box per il comando
  this.CloseAppBox = null;      // IMG che contiene il bottone di chiusura dell'applicazione
  this.CommandCaption = null;   // TEXT che contiene il propmt dei comandi
  this.CommandInput = null;     // INPUT box che contiene l'input per i comandi dell'utente
  this.ComImg = null;           // IMG da mostrare in caso di cominicazione Ajax in corso
  //
  // Oggetti per la definizione della task-bar
  this.TaskbarTable = null;        // TABLE contenuta nella sidebar per la costruzione della taskbar
  this.TaskbarStartCell = null;    // TD che contiene il bottone di start della takbar
  this.TaskbarQuickCell = null;    // TD che contiene la toolbar "quick start" della taskbar
  this.TaskbarFormListCell = null; // TD che contiene la form list della task bar
  this.TaskbarTrayCell = null;     // TD che contiene la tray area della task bar
  this.TaskbarMenuBox = null;      // DIV posizionato che contiene il menu task bar
  //
  this.FocusBox = null;         // DIV che viene posizionato nel campi del pannello per indicare che sono attivi
  //
  // Variabili per la rilevazione del cambio layout durante la gestione di una risposta
  this.HeaderH = 0;          // Altezza del HeaderBox prima dell'inizio della gestione della risposta
  this.SideMenuBoxW = 0;     // Larghezza del SideMenuBox prima dell'inizio della gestione della risposta
  this.LeftDockW = 0;        // Larghezza form docked
  this.RightDockW = 0;       // Larghezza form docked
  this.TopDockH = 0;         // Altezza form docked
  this.TopDockH = 0;         // Altezza form docked
  //
  // Inizializzo le command zones
  for (var i = 0; i < 48; i++)
    this.CommandZones[i] = 0;
  //
  this.DeferAfterProcess = true; // Usa un timer per attivare il processo di AfterProcess
  this.AfterStart = false;       // Siamo nella fase che segue lo start dell'applicazione?
  this.ScrollMobilePage = false; // Se true su android abilitiamo lo scroll di wep (serve per gestire il resize della tastiera)
  //
  // Stato di visibilita' di alcuni componenti: se lo tengo in memoria risparmio accessi al DOM
  this.CloseAllVisible = true;
  this.WelcomeBoxVisible = true;
}


// **********************************************************************
// Inizializza questo WebEntryPoint leggendo i dati da un nodo <wep> XML
// **********************************************************************
WebEntryPoint.prototype.LoadFromXml = function (node)
{
  // Se il nodo non mi corrisponde, esco
  if (node.nodeName != this.Identifier)
    return;
  //
  // Inizializzo le proprieta' locali
  this.LoadProperties(node);
  //
  var objlist = node.childNodes;
  var n = objlist.length;
  //
  // Ciclo su tutti i nodi che rappresentano oggetti figli
  for (var i = 0; i < n; i++)
  {
    var objnode = objlist.item(i);
    var nome = objnode.nodeName;
    //
    // In base al tipo di oggetto, invio il messaggio di caricamento
    switch (nome)
    {
      case "vsl":
        this.VSList.LoadFromXml(objnode);
        break;

      case "cmh":
        this.CmdObj.LoadFromXml(objnode);
        break;

      case "inh":
        this.IndObj.LoadFromXml(objnode);
        break;

      case "tmh":
        this.TimerObj.LoadFromXml(objnode);
        break;

      case "voice":
        this.VoiceObj.LoadFromXml(objnode);
        break;

      case "par":
        RD3_ServerParams.LoadFromXml(objnode);
        break;

      case "frm":
        {
          // Definisce una form aperta
          var newform = new WebForm();
          newform.LoadFromXml(objnode);
          this.StackForm.push(newform);
        }
        break;

      case "ccm":
        {
          // Definisce un custom command
          var newcmd = new Command();
          newcmd.LoadFromXml(objnode);
          this.CustomCommands.push(newcmd);
        }
        break;

      case "scz":
        {
          if (!this.UseZones())
            break;
          //
          var newZone = new ScreenZone();
          newZone.LoadFromXml(objnode);
          this.ScreenZones.push(newZone);
        }
        break;
    }
  }
}


// **********************************************************************
// Esegue un evento di change che riguarda le proprieta' di questo oggetto
// **********************************************************************
WebEntryPoint.prototype.ChangeProperties = function (node)
{
  // Semplicemente setto le proprieta' a partire dal nodo
  this.LoadProperties(node);
}


// **************************************************************
// Inizializza le proprieta' di questo oggetto leggendole dal
// nodo xml arrivato.
// **************************************************************
WebEntryPoint.prototype.LoadProperties = function (node)
{
  // Ciclo su tutti gli attributi del nodo
  var attrlist = node.attributes;
  var n = attrlist.length;
  for (var i = 0; i < n; i++)
  {
    var attrnode = attrlist.item(i);
    var nome = attrnode.nodeName;
    var valore = attrnode.nodeValue;
    //
    if (nome.substr(0, 2) == "cz")
    {
      this.CommandZones[parseInt(nome.substr(2))] = parseInt(valore);
    }
    //
    switch (nome)
    {
      case "cap":
        this.SetMainCaption(valore);
        break;
      case "img":
        this.SetMainImage(valore);
        break;
      case "hlp":
        this.SetHelpFile(valore);
        break;
      case "dbi":
        this.SetDebugType(parseInt(valore));
        break;
      case "cmd":
        this.SetCmdPrompt(valore);
        break;
      case "met":
        this.SetMenuType(valore);
        break;
        //
      case "rfi":
        this.SetRefreshInterval(parseInt(valore));
        break;
      case "rlo":
        this.SetRefreshLocation(parseInt(valore));
        break;
        //
      case "wid":
        this.SetWidgetMode(valore == "1");
        break;
      case "dec":
        this.SetUseDecimalDot(valore == "1");
        break;
      case "act":
        this.ActivateForm(valore);
        break;
      case "wel":
        this.SetWelcomePage(valore);
        break;
      case "prg":
        this.SetProgressFile(valore);
        break;
      case "ent":
        this.SetEntryPoint(valore);
        break;
      case "vfl":
        this.SetVisualFlags(parseInt(valore));
        break;
        //
      case "sid":
        this.SessionID = valore;
        break;
      case "lan":
        this.SetLanguage(valore);
        break;
        //
      case "clo":
        this.CloseAllEventDef = parseInt(attrnode.nodeValue);
        break;
      case "cla":
        this.CloseAppEventDef = parseInt(attrnode.nodeValue);
        break;
      case "shl":
        this.SetShowLogoff(valore == "1");
        break;
      case "smw":
        this.SetSideMenuWidth(parseInt(valore));
        break;
      case "acc":
        this.SetAccentColor(valore);
        break;
        //
      case "hks":
        this.SetHandledKeys(parseInt(valore));
        break;
        //
      case "ena":
        RD3_ClientParams.EnableGFX = (valore == "1");
        break;
      case "sta":
        RD3_ClientParams.GFXDef["start"] = valore;
        break;
      case "eca":
        RD3_ClientParams.GFXDef["menu"] = valore;
        break;
      case "pma":
        RD3_ClientParams.GFXDef["popup"] = valore;
        break;
      case "sfa":
        RD3_ClientParams.GFXDef["form"] = valore;
        break;
      case "oma":
        RD3_ClientParams.GFXDef["modal"] = valore;
        break;
      case "msa":
        RD3_ClientParams.GFXDef["message"] = valore;
        break;
      case "lma":
        RD3_ClientParams.GFXDef["lastmessage"] = valore;
        break;
      case "cfa":
        RD3_ClientParams.GFXDef["frame"] = valore;
        break;
      case "eta":
        RD3_ClientParams.GFXDef["tree"] = valore;
        break;
      case "cya":
        RD3_ClientParams.GFXDef["list"] = valore;
        break;
      case "qta":
        RD3_ClientParams.GFXDef["qbetip"] = valore;
        break;
      case "cta":
        RD3_ClientParams.GFXDef["tab"] = valore;
        break;
      case "sga":
        RD3_ClientParams.GFXDef["graph"] = valore;
        break;
      case "cba":
        RD3_ClientParams.GFXDef["book"] = valore;
        break;
      case "rda":
        RD3_ClientParams.GFXDef["redirect"] = valore;
        break;
      case "pra":
        RD3_ClientParams.GFXDef["preview"] = valore;
        break;
      case "dka":
        RD3_ClientParams.GFXDef["docked"] = valore;
        break;
      case "ppr":
        RD3_ClientParams.GFXDef["popupres"] = valore;
        break;
      case "ttp":
        RD3_ClientParams.GFXDef["tooltip"] = valore;
        break;
      case "tsk":
        RD3_ClientParams.GFXDef["taskbar"] = valore;
        break;
      case "cmb":
        RD3_ClientParams.GFXDef["combo"] = valore;
        break;
        //
      case "tme":
        RD3_TooltipManager.SetEnabled(valore == "1");
        break;
        //
      case "cns":
        RD3_ClientParams.ComboNameSeparator = valore;
        break;
        //
      case "mth":
        this.SetMotionThreshold(valore);
        break;
        //
      case "par": 
        if (node.childNodes && node.childNodes.length >= 1)
        {
          var objlist = node.childNodes;
          var n1 = objlist.length;
          for (var i1=0; i1<n1; i1++) 
          {
            var objnode = objlist.item(i1);
            if (objnode.nodeName == "par")
              RD3_ServerParams.LoadFromXml(objnode);
          }
        }
        break;
      //
      case "id":
        this.Identifier = valore;
        RD3_DesktopManager.ObjectMap.add(valore, this);
        break;
    }
  }
}


// ******************************************************
// Setter del refresh interval dell'applicazione
// ******************************************************
WebEntryPoint.prototype.SetRefreshInterval = function (value)
{
  if (value != undefined)
    this.RefreshInterval = value;
  //
  // Se la videata e' gia' stata realizzata, aggiusto il timer
  if (this.Realized)
  {
    if (this.TimerId != 0) {
      window.clearInterval(this.TimerId);
      this.TimerId = 0;
    }
    //
    if (this.RefreshInterval > 0)
    {
      // Imposto l'intervallo
      this.TimerId = window.setInterval('RD3_DesktopManager.WebEntryPoint.RefreshIntervalTick()', this.RefreshInterval);
    }
  }
}


// ********************************************************
// Funzione chiamata per registrare il cambiamento
// di posizione del browser
// ********************************************************
function RD3_LocationChange(p)
{
  var s = p.coords.latitude + "|";
  s += p.coords.longitude + "|"
  s += p.coords.accuracy + "|";
  s += p.coords.altitude + "|";
  s += p.coords.altitudeAccuracy + "|";
  s += p.coords.heading + "|";
  s += p.coords.speed + "|";
  if (RD3_Glb.IsChrome() && p.timestamp.getTime)
    s += p.timestamp.getTime() + "|";
  else
    s += p.timestamp + "|";
  var ev = new IDEvent("chgloc", "wep", undefined, RD3_Glb.EVENT_ACTIVE, "loc", s, null, null, null, (RD3_DesktopManager.WebEntryPoint.LastWatchTime == 0) ? 1 : RD3_DesktopManager.WebEntryPoint.RefreshLocation);
  RD3_DesktopManager.WebEntryPoint.LastWatchTime = new Date();
}


// ********************************************************
// Funzione chiamata per registrare il cambiamento di orientazione
// viene chiamata tipicamente ogni 50 ms, qui registro solo i valori ottenuti
// ********************************************************
function RD3_DeviceOrientation(ev)
{
  RD3_DesktopManager.WebEntryPoint.LastOrientationEvent = ev;
}


// ********************************************************
// Funzione chiamata per registrare il movimento
// ********************************************************
function RD3_DeviceMotion(ev)
{
  if (ev)
  {
    var t = RD3_DesktopManager.WebEntryPoint.MotionThreshold;
    var ot = Math.abs(ev.acceleration.x) >= t.accel;
    ot = ot || Math.abs(ev.acceleration.y) >= t.accel;
    ot = ot || Math.abs(ev.acceleration.z) >= t.accel;
    ot = ot || Math.abs(ev.rotationRate.alpha) >= t.rot;
    ot = ot || Math.abs(ev.rotationRate.beta) >= t.rot;
    ot = ot || Math.abs(ev.rotationRate.gamma) >= t.rot;
    //
    // Se sono sopra soglia, genero l'evento solo se non c'e' n'e' un altro in coda uguale al mio ancora non lanciato
    if (ot && !RD3_DesktopManager.MessagePump.GetEvent("wep", "chgmot"))
    {
      var s = ev.acceleration.x + "|";
      s += ev.acceleration.y + "|"
      s += ev.acceleration.z + "|";
      s += ev.accelerationIncludingGravity.x + "|";
      s += ev.accelerationIncludingGravity.y + "|";
      s += ev.accelerationIncludingGravity.z + "|";
      s += ev.rotationRate.alpha + "|";
      s += ev.rotationRate.beta + "|";
      s += ev.rotationRate.gamma + "|";
      var or = RD3_DesktopManager.WebEntryPoint.LastOrientationEvent;
      if (or)
      {
        s += or.alpha + "|";
        s += or.beta + "|";
        s += or.gamma + "|";
        if (or.webkitCompassHeading)
        {
          s += or.webkitCompassHeading + "|";
          s += or.webkitCompassAccuracy + "|";
        }
        else
        {
          s += "||";
        }
      }
      else
      {
        s += "|||||";
      }
      s += (new Date()).getTime() + "|";
      var ev = new IDEvent("chgmot", "wep", undefined, RD3_Glb.EVENT_ACTIVE, "mot", s, null, null, null, 100);
    }
  }
  else
  {
    // simulator
    var s = Math.random() * 10 + "|";
    s += Math.random() * 10 + "|"
    s += Math.random() * 10 + "|";
    s += Math.random() * 20 + "|";
    s += Math.random() * 20 + "|"
    s += Math.random() * 20 + "|";
    //
    s += Math.random() * 30 + "|";
    s += Math.random() * 30 + "|";
    s += Math.random() * 30 + "|";
    //
    s += Math.random() * 90 + "|";
    s += Math.random() * 90 + "|";
    s += Math.random() * 90 + "|";
    //
    s += Math.random() * 360 + "|";
    s += Math.random() * 30 + "|";
    //
    s += (new Date()).getTime() + "|";
    var ev = new IDEvent("chgmot", "wep", undefined, RD3_Glb.EVENT_ACTIVE, "mot", s, null, null, null, 100);
  }
}


// ********************************************************
// Funzione chiamata per registrare un errore
// dovuto alla location
// ********************************************************
function RD3_LocationError(err)
{
  var ev = new IDEvent("chgloc", "wep", undefined, RD3_Glb.EVENT_ACTIVE, "err", err.code);
}


// ******************************************************
// Setter del refresh interval dell'applicazione
// ******************************************************
WebEntryPoint.prototype.SetRefreshLocation = function (value)
{
  if (value != undefined)
    this.RefreshLocation = value;
  //
  // Se la videata e' gia' stata realizzata, aggiusto il timer
  if (this.Realized)
  {
    if (this.RefreshLocation > 0 && this.WatchLocationId == 0 && navigator.geolocation) {
      this.WatchLocationId = navigator.geolocation.watchPosition(RD3_LocationChange, RD3_LocationError, {enableHighAccuracy: true, refreshLocation: this.RefreshLocation});
    }
    if (this.RefreshLocation <= 0 && this.WatchLocationId != 0)
    {
      navigator.geolocation.clearWatch(this.WatchLocationId);
      this.WatchLocationId = 0;
    }
  }
}


// ******************************************************
// Setter del refresh interval dell'applicazione
// ******************************************************
WebEntryPoint.prototype.SetMotionThreshold = function (value)
{
  if (value != undefined)
  {
    var x = value.split("|");
    this.MotionThreshold.accel = parseFloat(x[0]);
    this.MotionThreshold.rot = parseFloat(x[1]);
  }
  //
  // Se la videata e' gia' stata realizzata, aggiusto il timer
  if (this.Realized)
  {
    if (this.MotionThreshold.accel != 999 || this.MotionThreshold.rot != 999)
    {
      if (RD3_Glb.IsTouch())
      {
        window.addEventListener("devicemotion", RD3_DeviceMotion);
        window.addEventListener("deviceorientation", RD3_DeviceOrientation);
      }
      else
      {
        this.MotionTimerID = window.setInterval(RD3_DeviceMotion, 1000 * this.MotionThreshold.accel);
      }
    }
    else
    {
      if (window.removeEventListener)
      {
        window.removeEventListener("devicemotion", RD3_DeviceMotion);
        window.removeEventListener("deviceorientation", RD3_DeviceOrientation);
      }
      window.clearInterval(this.MotionTimerID);
      RD3_DesktopManager.WebEntryPoint.LastOrientationEvent = null;
    }
  }
}


// ******************************************************
// Setter del widgetmode dell'applicazione
// ******************************************************
WebEntryPoint.prototype.SetWidgetMode = function (value)
{
  var oldwm = this.WidgetMode;
  //
  if (value != undefined)
    this.WidgetMode = value;
  //
  if (this.Realized && (this.WidgetMode != oldwm || value == undefined))
  {
    // Tolgo il menu' e nascondo Header
    this.HeaderBox.style.display = !this.HasCaption() ? "none" : "";
    this.AdjustMenuVisibility();
  }
}


// ******************************************************
// Setter dei visual flags dell'applicazione
// ******************************************************
WebEntryPoint.prototype.SetVisualFlags = function (value)
{
   var old = this.VisualFlags;
   var oht = this.HasToolbar();
   var ohs = this.HasStatusBar();
   if (value!=undefined)
   this.VisualFlags = value;
   //
   if (this.Realized && (this.VisualFlags != old || value==undefined) && !RD3_Glb.IsMobile())
   {
     // Sistemo in funzione di quanto chiesto
     this.DebugImageBox.style.display = (this.HasDebugButton() ? "" : "none");
     this.CommandInput.style.display = (this.HasCommandBox() ? "" : "none");
     this.HelpFileBox.style.display = (this.HasHelpButton() ? "" : "none");
     this.CloseAppBox.style.display = (this.HasCloseButton() ? "" : "none");
     this.MainCaptionBox.style.display = (this.HasTitle() ? "" : "none");
     this.HeaderBox.style.display = (this.HasCaption() ? "" : "none");
     this.MainImageBox.style.display = (this.HasIcon() ? "" : "none");
     this.MainImageBox.style.cursor = (this.IsIconActive() ? "pointer" : "");
   }
}

// ********************************************************************************
// Funzioni per il reperimento degli stili visuali
// ********************************************************************************
WebEntryPoint.prototype.HasCaption = function ()
{
  return (this.VisualFlags & 0x1) && !this.WidgetMode;
}

WebEntryPoint.prototype.HasMenuButton = function ()
{
  return (this.VisualFlags & 0x2);
}

WebEntryPoint.prototype.HasIcon = function ()
{
  return (this.VisualFlags & 0x4) && this.MainImage != "";
}

WebEntryPoint.prototype.HasTitle = function ()
{
  return (this.VisualFlags & 0x8);
}

WebEntryPoint.prototype.HasCommandBox = function ()
{
  return (this.VisualFlags & 0x10) && this.CmdPrompt != "";
}

WebEntryPoint.prototype.HasCloseButton = function ()
{
  return (this.VisualFlags & 0x20) && this.ShowLogoff;
}

WebEntryPoint.prototype.HasHelpButton = function ()
{
  return (this.VisualFlags & 0x40) && this.HelpFile != "";
}

WebEntryPoint.prototype.HasDebugButton = function ()
{
  return (this.VisualFlags & 0x80) && this.DebugType != 0;
}

WebEntryPoint.prototype.IsIconActive = function ()
{
  return (this.VisualFlags & 0x100);
}

WebEntryPoint.prototype.HasAjaxInd = function ()
{
  return (this.VisualFlags & 0x200);
}

WebEntryPoint.prototype.HasToolbar = function ()
{
  return (this.VisualFlags & 0x400) && !this.WidgetMode;
}

WebEntryPoint.prototype.HasStatusBar = function ()
{
  return (this.VisualFlags & 0x800) && !this.WidgetMode;
}


// ******************************************************
// Modifica il colore di accento del tema (dal precedente che vale #0081C2
// ******************************************************
WebEntryPoint.prototype.SetAccentColor = function (value)
{
  
}


// ******************************************************
// Setter della visibilita' dell'immagine di logoff
// ******************************************************
WebEntryPoint.prototype.SetShowLogoff = function (value)
{
  if (value != undefined)
    this.ShowLogoff = value;
  //
  if (this.Realized)
    this.CloseAppBox.style.display = this.HasCloseButton() ? "" : "none";
}


// ******************************************************
// Setter della larghezza del menu laterale
// ******************************************************
WebEntryPoint.prototype.SetSideMenuWidth = function (value)
{
  this.SideMenuWidth = value;
}


// ******************************************************
// Setter dei tasti intercettati dall'applicazione
// ******************************************************
WebEntryPoint.prototype.SetHandledKeys = function (value)
{
  if (value != undefined)
    this.HandledKeys = value;
  //
  // Questa proprieta' non puo' variare dopo essere stata inviata, e viene gestita
  // nel KBManager
}

// ******************************************************
// Setter del decimal dot dell'applicazione
// ******************************************************
WebEntryPoint.prototype.SetUseDecimalDot = function (value)
{
  if (value != undefined)
    this.UseDecimalDot = value;
  //
  // Imposto le variabili globali della maskedinput
  if (this.UseDecimalDot)
  {
    glbDecSep = ".";
    glbThoSep = ",";
  }
  else
  {
    glbDecSep = ",";
    glbThoSep = ".";
  }
}


// *********************************************************
// Timer globale
// *********************************************************
WebEntryPoint.prototype.Tick = function ()
{
  var n = this.StackForm.length;
  for (var i = 0; i < n; i++)
  {
    var f = this.StackForm[i];
    if (f && f.Realized && f.Visible)
      f.Tick();
  }
}


// *********************************************************
// Setter della Caption del Web Entry Point
// *********************************************************
WebEntryPoint.prototype.SetMainCaption = function (value) 
{
  if (value != undefined)
    this.MainCaption = value;
  //
  // Se la videata e' gia' stata realizzata, aggiusto le proprieta' visuali
  if (this.Realized && this.MainCaptionBox) 
  {
    this.MainCaptionBox.innerHTML = RD3_Glb.HandleIconString(this.MainCaption);
    document.title = RD3_Glb.RemoveTags(RD3_Glb.HandleIconString(this.MainCaption, true));
  }
}


// *********************************************************
// Setter della main image del Web Entry Point
// *********************************************************
WebEntryPoint.prototype.SetMainImage = function (value)
{
  var old = this.MainImage;
  if (value != undefined)
    this.MainImage = value;
  //
  // Se la videata e' gia' stata realizzata, aggiusto le proprieta' visuali
  if (this.Realized && this.MainImageBox)
  {
    if (this.MainImage == "")
      this.MainImageBox.style.display = "none";
    else {
      this.MainImageBox.src = RD3_Glb.GetImgSrc("images/" + this.MainImage);
      this.MainImageBox.style.display = "";
    }
  }
}

// *********************************************************
// Setter del file di help del Web Entry Point
// *********************************************************
WebEntryPoint.prototype.SetHelpFile = function (value)
{
  if (value != undefined)
    this.HelpFile = value;
  //
  // Se la videata e' gia' stata realizzata, aggiusto le proprieta' visuali
  if (this.Realized && this.HelpFileBox)
  {
    if (!this.HasHelpButton())
    {
      this.HelpFileBox.style.display = "none";
    }
    else
    {
      this.HelpFileBox.style.display = "";
      this.HelpFileBox.src = RD3_Glb.GetImgSrc("images/help.gif");
    }
  }
}

// *********************************************************
// Setter del tipo di debug del Web Entry Point
// *********************************************************
WebEntryPoint.prototype.SetDebugType = function (value)
{
  if (value != undefined)
    this.DebugType = value;
  //
  // Se la videata e' gia' stata realizzata, aggiusto le proprieta' visuali
  if (this.Realized && this.DebugImageBox)
  {
    var disp = this.HasDebugButton() ? "" : "none";
    var im = "";
    switch (this.DebugType)
    {
      case 0 :
        disp = "none";
        break;               // No Debug - No Help
      case 1 :
        im = "images/bug.gif";
        break;       // Debug
      case 2 :
        im = "images/dtthelp.gif";
        break;    // Help
    }
    //
    this.DebugImageBox.style.display = disp;
    if (im != "" && !RD3_Glb.IsMobile())
      this.DebugImageBox.src = RD3_Glb.GetImgSrc(im);
    if (RD3_Glb.IsMobile() && this.DebugType == 1 && window.top != window)
    {
      var deb = window.top.document.getElementById("debug");
      if (deb)
        deb.style.display = "";
    }
  }
}

// *********************************************************
// Setter dell'immagine di debug del Web Entry Point
// *********************************************************
WebEntryPoint.prototype.SetCmdPrompt = function (value)
{
  if (value != undefined)
    this.CmdPrompt = value;
  //
  // Se la videata e' gia' stata realizzata, aggiusto le proprieta' visuali
  if (this.Realized && this.CommandBox)
  {
    if (!this.HasCommandBox())
    {
      this.CommandBox.style.display = "none";
    }
    else
    {
      this.CommandBox.style.display = "";
      this.CommandCaption.nodeValue = this.CmdPrompt;
    }
  }
}

// *********************************************************
// Imposta il tipo di menu' (non cambia dopo inizializzazione)
// *********************************************************
WebEntryPoint.prototype.SetMenuType = function (value)
{
  if (value != undefined)
  {
    if (value == RD3_Glb.MENUTYPE_GROUPED)
    {
      value = RD3_Glb.MENUTYPE_LEFTSB;
      this.CmdObj.ShowGroups = true;
    }
    //
    this.MenuType = value;
  }
}

// *********************************************************
// Setter della welcoma page del Web Entry Point
// *********************************************************
WebEntryPoint.prototype.SetWelcomePage = function (value)
{
  if (value != undefined)
    this.WelcomePage = value;
  //
  // Se la videata e' gia' stata realizzata, aggiusto le proprieta' visuali
  if (this.Realized && this.WelcomePage != "")
  {
    if (this.WelcomeForm)
      this.WelcomeForm.SetPage(this.WelcomePage, this.MainCaption);
    if (this.WelcomeBox)
      this.WelcomeBox.src = this.WelcomePage;
  }
}

// *********************************************************
// Setter del file da verificare per operazioni interrompibili
// *********************************************************
WebEntryPoint.prototype.SetProgressFile = function (value)
{
  if (value != undefined)
    this.ProgressBarFile = value;
  //
  // Questa proprieta' non puo' cambiare a runtime
}

// *********************************************************
// Setter dell'entry point dell'applicazione
// *********************************************************
WebEntryPoint.prototype.SetEntryPoint = function (value)
{
  if (value != undefined)
    this.EntryPoint = value;
  //
  // Questa proprieta' non puo' cambiare a runtime
}


// *********************************************************
// Setter della lingua dell'applicazione
// *********************************************************
WebEntryPoint.prototype.SetLanguage = function (value)
{
  if (value != undefined)
    this.Language = value;
  //
  // Se i messaggi non sono stati tradotti in questa lingua passo all'inglese
  if (!ClientMessagesSet[this.Language])
    this.Language = "ENG";
  //
  this.ApplyLanguage();
}

// *********************************************************
// Applica le modifiche legate alla lingua
// *********************************************************
WebEntryPoint.prototype.ApplyLanguage = function ()
{
  ClientMessages = ClientMessagesSet[this.Language];
  //
  MonthNames = ClientMessages.WEP_CAL_MonthNames;
  DayNames = ClientMessages.WEP_CAL_DayNames;
  CloseCaption = ClientMessages.WEP_CAL_CloseButtonCaption;
}

// ***************************************************************
// Crea gli oggetti DOM utili a questo oggetto
// L'oggetto parent indica all'oggetto dove devono essere contenuti
// i suoi oggetti figli nel DOM
// ***************************************************************
WebEntryPoint.prototype.Realize = function (parent)
{
  // Creo i miei oggetti visuali
  this.RealizeWep(parent);
  this.RealizeHeader(this.HeaderBox);
  //
  // A questo punto devo cominciare a fare realizzare i miei figli
  this.VSList.Realize();
  this.CmdObj.Realize(this.HasSideMenu() ? this.MenuBox : this.SideMenuBox, this.RightHeaderCommandsList);
  this.TimerObj.Realize();
  this.VoiceObj.Realize();
  //
  var ns = this.ScreenZones.length;
  for (var t = 0; t < ns; t++)
    this.ScreenZones[t].Realize();
  //
  // Anche le form aperte devono essere realizzate
  var n = this.StackForm.length;
  for (var i = 0; i < n; i++)
  {
    if (!this.StackForm[i].Modal)
      this.StackForm[i].Realize();
  }
  //
  // Eseguo l'impostazione iniziale delle mie proprieta'
  this.Realized = true;
  this.SetMainCaption();
  this.SetMainImage();
  this.SetHelpFile();
  this.SetDebugType();
  this.SetCmdPrompt();
  this.SetRefreshInterval();
  this.SetWidgetMode();
  this.SetWelcomePage();
  this.SetShowLogoff();
  this.SetSideMenuWidth();
  this.SetRefreshLocation();
  this.SetMotionThreshold();
  this.SetVisualFlags();
  //
  // Al termine della realizzazione aggiungo tutto al DOM
  parent.appendChild(this.WepBox);
  //
  // Solo ora realizzo le form modali
  var n = this.StackForm.length;
  for (var i = 0; i < n; i++)
  {
    if (this.StackForm[i].Modal)
      this.StackForm[i].Realize();
  }
  //
  if (this.ActiveForm == null)
    this.ActiveForm = this.GetLastForm();
  //
  this.ActivateForm(this.ActiveForm);
  this.RecalcLayout = true;
}


// ********************************************************************************
// Questa funzione crea gli oggetti DOM per il WEP
// ********************************************************************************
WebEntryPoint.prototype.RealizeWep = function (parent)
{
  var _this = this;
  //
  // Creo il div globale del wep
  // Questo e' il div grande come tutta la pagina
  this.WepBox = document.createElement("div");
  this.WepBox.setAttribute("id", this.Identifier);
  this.WepBox.className = RD3_ClientParams.BodyContainerType;
  this.WepBox.style.visibility = "hidden";
  //
  // Creo il div destinato a contenere l'header
  // L'header e' posizionato in alto
  this.HeaderBox = document.createElement("nav");
  this.HeaderBox.setAttribute("id", "header-container");
  this.HeaderBox.className = "navbar navbar-default";
  //
  // Creo il div destinato a contenere il menu
  if (this.HasSideMenu()) {
    this.SideMenuBox = document.createElement("div");
    this.SideMenuBox.setAttribute("id", "sidebar");
    this.SideMenuBox.className = "sidebar-offcanvas";
    if (this.SideMenuWidth > 0)
      this.SideMenuBox.style.width = this.SideMenuWidth + "px";
    //
    // Creo il contenitore del menu
    this.MenuBox = document.createElement("div");
    this.MenuBox.setAttribute("id", "menu-accordion");
    this.MenuBox.className = "panel-group";
    this.SideMenuBox.appendChild(this.MenuBox);
  }
  else {
    this.SideCollapseContainer = document.createElement("div");
    this.SideCollapseContainer.className = "collapse navbar-collapse";
    this.SideCollapseContainer.setAttribute("id", "navbar-collapse-menu");
    this.SideMenuBox = document.createElement("ul");
    this.SideMenuBox.className = "nav navbar-nav navbar-left";
    this.SideCollapseContainer.appendChild(this.SideMenuBox);
  }
  //
  // Creo la grigliatura con la struttura della pagina
  var mainRow = document.createElement("div");
  mainRow.setAttribute("id", "wep-main-row");
  mainRow.className = "row" + (this.HasSideMenu() ? " row-offcanvas" : "");
  //
  this.MainGridContainer = document.createElement("div");
  this.MainGridContainer.setAttribute("id", "wep-main-container");
  if (this.HasSideMenu()) {
    this.MainGridContainer.className = "col-sm-12 main-side-" + (this.MenuType == RD3_Glb.MENUTYPE_LEFTSB ? "left" : "right");
    if (this.SideMenuWidth > 0) {
      if (this.MenuType == RD3_Glb.MENUTYPE_LEFTSB)
        this.MainGridContainer.style.paddingLeft = (this.SideMenuWidth + 30) + "px";
      else
        this.MainGridContainer.style.paddingRight = (this.SideMenuWidth + 30) + "px";
    }
  }
  else
    this.MainGridContainer.className = "col-xs-12 col-sm-12 col-md-12";
  mainRow.appendChild(this.MainGridContainer);
  //
  if (this.HasSideMenu()) {
    mainRow.className += " row-offcanvas-" + (this.MenuType == RD3_Glb.MENUTYPE_LEFTSB ? "left" : "right");
    if (this.MenuType == RD3_Glb.MENUTYPE_RIGHTSB)
      mainRow.appendChild(this.SideMenuBox);
    else
      mainRow.insertBefore(this.SideMenuBox, this.MainGridContainer);
  }
  //
  // Creata la struttura principale per il menu e la main area creo la grigliatura per le docked e la MainForm
  // Griglia base : 3 righe
  var gridRow1 = document.createElement("div");
  gridRow1.className = "row";
  this.MainGridRow2 = document.createElement("div");
  this.MainGridRow2.className = "row main-grid-row";
  var gridRow3 = document.createElement("div");
  gridRow3.className = "row";
  this.MainGridContainer.appendChild(gridRow1);
  this.MainGridContainer.appendChild(this.MainGridRow2);
  this.MainGridContainer.appendChild(gridRow3);
  //
  // Nella prima riga va creato il container delle docked superiori
  this.TopDockedBox = document.createElement("div");
  this.TopDockedBox.className = "col-xs-12 col-sm-12 col-md-12";
  this.TopDockedBox.setAttribute("id", "wep-top-docked-container");
  gridRow1.appendChild(this.TopDockedBox);
  //
  // Nella seconda riga vanno inseriti i container delle videate docked destra, sinistra e centrale
  // In AdaptLayout vengono modificate le classi e nascoste in base al contenuto
  this.LeftDockedBox = document.createElement("div");
  this.LeftDockedBox.className = "col-xs-12 col-sm-3 col-md-3";
  this.LeftDockedBox.setAttribute("id", "wep-left-docked-container");
  this.RightDockedBox = document.createElement("div");
  this.RightDockedBox.className = "col-xs-12 col-sm-3 col-md-3";
  this.RightDockedBox.setAttribute("id", "wep-right-docked-container");
  this.FormsBox = document.createElement("div");
  this.FormsBox.className = "col-xs-12 col-sm-4 col-md-4"
  this.FormsBox.setAttribute("id", "wep-main-forms-container");
  this.MainGridRow2.appendChild(this.LeftDockedBox);
  this.MainGridRow2.appendChild(this.FormsBox);
  this.MainGridRow2.appendChild(this.RightDockedBox);
  //
  // Nella terza riga mettiamo il contenitore delle docked inferiori
  this.BottomDockedBox = document.createElement("div");
  this.BottomDockedBox.className = "col-xs-12 col-sm-12 col-md-12";
  this.BottomDockedBox.setAttribute("id", "wep-bottom-docked-container");
  gridRow3.appendChild(this.BottomDockedBox);
  //
  // Creo l'iframe per la welcome page
  // Protezione per applicazioni embeddate dentro Inde.. noi forziamo la compatibilita' ad IE7 in quel caso, eppure le righe di codice sotto
  // non funzionano (non si sa perche')... quindi se sono fallite non facciamo uscire l'eccezione e proviamo nell'altro modo
  var done = false;
  try
  {
    if (RD3_Glb.IsIE(10, false))
    {
      this.WelcomeBox = document.createElement("<iframe id='welcome-container' frameBorder='0' onload='RD3_DesktopManager.WebEntryPoint.OnWelcomeBoxLoad();'></iframe>");
      done = true;
    }
  }
  catch (ex) {
  }
  //
  if (!done)
  {
    this.WelcomeBox = document.createElement("iframe");
    this.WelcomeBox.setAttribute("id", "welcome-container");
    this.WelcomeBox.frameBorder = 0;
    this.WelcomeBox.onload = new Function("RD3_DesktopManager.WebEntryPoint.OnWelcomeBoxLoad();");
  }
  //
  // Aggiungo i div al dom nell'ordine corretto
  document.body.appendChild(this.HeaderBox);
  this.WepBox.appendChild(mainRow);
  //
  if (this.WelcomeBox)
    this.FormsBox.appendChild(this.WelcomeBox); // Deve essere il primo cosi' appare solo se non ci sono form aperte
  //
  // Gestione del resize
  var rf = function (ev) {
    _this.OnResize(ev);
  };
  if (window.addEventListener)
    window.addEventListener("resize", rf, false);
  else
    window.onresize = rf;
  //
  // Nel mobile il resize non e' sempre lanciato, ma l'orientation change si'
  if (RD3_Glb.IsTouch()) {
    window.addEventListener("orientationchange", rf, false);
    //
    if (RD3_Glb.IsAndroid()) {
      window.addEventListener("orientationchange", function() {
        // Ho cambiato l'orientamento, potrebbe dipendere da:
        // - apertura tastiera
        // - rotazione
        // nel secondo caso devo aggiornare la dimensione della width e della height a quelle correnti
        // così quando aprirò la tastiera (focus) posso fissarle.
        // mi memorizzo le dimensioni se non sono fissate nello style del body (in quel caso significa che c'è la tastiera aperta, perchè
        // nel blur le tolgo..)
        if (document.body.style.width == "" && document.body.style.height == "")
          window.setTimeout(function() {
            window.currentHeight = window.innerHeight;
            window.currentWidth = window.innerWidth;
          }, 250);
      }, false);
      //
      // Impostazione valori iniziali
      window.currentHeight = window.innerHeight;
      window.currentWidth = window.innerWidth;
    }
  }
  //
  var cf = function (ev) {
    _this.OnClick(ev);
  };
  this.WepBox.onclick = cf;
  //
  var uf = function (ev) {
    _this.OnBeforeUnload(ev);
  };
  if (document.body.addEventListener)
    document.body.addEventListener("beforeunload", uf, false);
  else
    document.body.onbeforeunload = uf;  
  //
  // Adesso il frame per il caricamento dei blob
  // Protezione per applicazioni embeddate dentro Inde.. noi forziamo la compatibilita' ad IE7 in quel caso, eppure le righe di codice sotto
  // non funzionano (non si sa perche')... quindi se sono fallite non facciamo uscire l'eccezione e proviamo nell'altro modo
  var done = false;
  try
  {
    if (RD3_Glb.IsIE(10, false))
    {
      this.BlobFrame = document.createElement("<iframe name='blobframe' onload='RD3_DesktopManager.WebEntryPoint.OnBlobUpload();'></iframe>");
      done = true;
    }
  }
  catch (ex) {
  }
  //
  if (!done)
  {
    this.BlobFrame = document.createElement("iframe");
    this.BlobFrame.setAttribute("id", "blobframe");
    this.BlobFrame.setAttribute("name", "blobframe");
    this.BlobFrame.onload = new Function("RD3_DesktopManager.WebEntryPoint.OnBlobUpload();");
  }
  //
  this.BlobFrame.style.display = "none";
  this.BlobFrame.src = "about:blank";
  document.body.appendChild(this.BlobFrame);
  //
  // Gestisco la FormList
  if (this.HasSideMenu())
  {
    // Creo la FormList
    this.FormListBox = document.createElement("div");
    this.FormListBox.setAttribute("id","form-list-box-container");
    this.FormListBox.className = "panel panel-default";
    //
    // Creo il div per la testata
    this.FormListHeader = document.createElement("div");    
    this.FormListHeader.className = "panel-heading has-pointer";
    this.FormListHeader.setAttribute("data-toggle", "collapse");
    //this.FormListHeader.setAttribute("data-parent", "#menu-accordion");
    this.FormListHeader.setAttribute("data-target", "#form-list-entry-container");
    this.FormListBox.appendChild(this.FormListHeader);
    //
    var formListTitle = document.createElement("h4");
    formListTitle.className = "panel-title";
    this.FormListHeader.appendChild(formListTitle);
    //
    this.FormListLink = document.createElement("div");
    this.FormListLink.textContent = RD3_ServerParams.VideateAperte;
    formListTitle.appendChild(this.FormListLink);
    //
    // Creo il box che conterra' l'elenco
    this.FormsListCollapsePanel = document.createElement("div");
    this.FormsListCollapsePanel.setAttribute("id", "form-list-entry-container");
    this.FormsListCollapsePanel.className = "panel-collapse collapse in";
    this.FormListBox.appendChild(this.FormsListCollapsePanel);
    //
    this.FormsListCollapsePanelBody = document.createElement("div");
    this.FormsListCollapsePanelBody.className = "panel-body";
    this.FormsListCollapsePanel.appendChild(this.FormsListCollapsePanelBody);
    //
    var ptable = document.createElement("table");
    ptable.className = "table";
    this.FormsListCollapsePanelBody.appendChild(ptable);
    //
    this.FormsListBodyBox = document.createElement("tbody");
    ptable.appendChild(this.FormsListBodyBox);
    //
    this.RealizeCloseAll(this.FormsListBodyBox);
    //
    // E l'aggiungo al side menu box
    this.SideMenuBox.appendChild(this.FormListBox);
  }
}

// ********************************************************************************
// Caricata la welcome-page
// ********************************************************************************
WebEntryPoint.prototype.OnWelcomeBoxLoad = function ()
{
  // Se non posso accedere alla window (cross-scripting) non mi lamento... non posso fare di meglio
  try
  {
    if (!this.WelcomeBox.contentWindow || !this.WelcomeBox.contentWindow.document)
      return;
  }
  catch (ex)
  {
    return;
  }
  //
  // Attacco un po' di eventi per permettere al D&D Manager di funzionare se non ci sono videate aperte
  var mm = new Function("ev", "return RD3_DDManager.OnMouseMove(ev)");
  var mu = new Function("ev", "return RD3_DDManager.OnMouseUp(ev)");
  //
  if (RD3_Glb.IsTouch() && !RD3_Glb.IsIE(10, true))
  {
    this.WelcomeBox.contentWindow.document.addEventListener("touchmove", mm, false);
    this.WelcomeBox.contentWindow.document.addEventListener("touchend", mu, false);
  }
  else if (this.WelcomeBox.contentWindow.document.addEventListener)
  {
    this.WelcomeBox.contentWindow.document.addEventListener("mousemove", mm, true);
    this.WelcomeBox.contentWindow.document.addEventListener("mouseup", mu, true);
  }
  else
  {
    this.WelcomeBox.contentWindow.document.attachEvent("onmousemove", mm);
    this.WelcomeBox.contentWindow.document.attachEvent("onmouseup", mu);
  }
}

// ********************************************************************************
// Questa funzione crea gli oggetti DOM per l'header
// ********************************************************************************
WebEntryPoint.prototype.RealizeHeader = function (parent)
{
  var _this = this;
  //
  var fluidContainer = document.createElement("div");
  fluidContainer.className = RD3_ClientParams.HeaderContainerType;
  parent.appendChild(fluidContainer);
  //
  var leftNavbarHeader = document.createElement("div");
  leftNavbarHeader.className = "navbar-header pull-left";
  fluidContainer.appendChild(leftNavbarHeader);
  //
  // Creiamo il box per l'immagine dell'applicazione (LOGO)
  this.MainImageBox = document.createElement("img");
  this.MainImageBox.className = "main-app-image";
  this.MainImageBox.onclick = new Function("ev","return RD3_DesktopManager.CallEventHandler('"+this.Identifier+"', 'OnIconClick', ev)");
  leftNavbarHeader.appendChild(this.MainImageBox);
  //
  // Creiamo titolo dell'applicazione
  this.MainCaptionBox = document.createElement("a");
  this.MainCaptionBox.className = "navbar-brand";
  leftNavbarHeader.appendChild(this.MainCaptionBox);
  //
  var rightNavbarHeader = document.createElement("div");
  rightNavbarHeader.className = "navbar-header pull-right";
  fluidContainer.appendChild(rightNavbarHeader);
  //
  this.RightHeaderCommandsList = document.createElement("ul");
  this.RightHeaderCommandsList.className = "nav";
  rightNavbarHeader.appendChild(this.RightHeaderCommandsList);
  //
  var li = document.createElement("li");
  li.className = "pull-right";
  this.RightHeaderCommandsList.appendChild(li);
  //
  // Creiamo il pulsante di espansione del menu
  this.toggleButton = document.createElement("button");
  this.toggleButton.className = "navbar-toggle";
  this.toggleButton.setAttribute("data-toggle", (this.MenuType == RD3_Glb.MENUTYPE_RIGHTSB || this.MenuType == RD3_Glb.MENUTYPE_LEFTSB ? "offcanvas" : "collapse"));
  if (this.MenuType == RD3_Glb.MENUTYPE_TASKBAR || this.MenuType == RD3_Glb.MENUTYPE_MENUBAR)
    this.toggleButton.setAttribute("data-target", "#navbar-collapse-menu");
  else {
    this.toggleButton.onclick = function (ev) {
      $('.row-offcanvas').toggleClass('active');
    };
  }
  this.toggleButton.setAttribute("aria-expanded", "false");
  li.appendChild(this.toggleButton);
  //
  var span = document.createElement("span");
  span.className = "icon-bar";
  this.toggleButton.appendChild(span);
  span = document.createElement("span");
  span.className = "icon-bar";
  this.toggleButton.appendChild(span);
  span = document.createElement("span");
  span.className = "icon-bar";
  this.toggleButton.appendChild(span);
  //
  // Pulsante Logoff
  li = document.createElement("li");
  li.className = "pull-right";
  this.RightHeaderCommandsList.appendChild(li);
  this.CloseAppBox = document.createElement("button");
  this.CloseAppBox.className = "btn btn-default navbar-btn";
  this.CloseAppBox.onclick = function (ev) {
    _this.OnCloseApp(ev);
  };
  var closeImage = document.createElement("i");
  closeImage.className = "fa " + RD3_ClientParams.FA_ICON_LOGOFF;
  this.CloseAppBox.appendChild(closeImage);
  li.appendChild(this.CloseAppBox);
  //
  // DEBUG BUTTON
  li = document.createElement("li");
  li.className = "pull-right";
  this.RightHeaderCommandsList.appendChild(li);
  this.DebugImageBox = document.createElement("button");
  this.DebugImageBox.className = "btn btn-default navbar-btn";
  this.DebugImageBox.onclick = function (ev) {
    _this.OnDebug(ev);
  };
  var debugImage = document.createElement("i");
  debugImage.className = "fa " + RD3_ClientParams.FA_ICON_DEBUG;
  this.DebugImageBox.appendChild(debugImage);
  li.appendChild(this.DebugImageBox);
  //
  // HELP BUTTON
  li = document.createElement("li");
  li.className = "pull-right";
  this.RightHeaderCommandsList.appendChild(li);
  this.HelpFileBox = document.createElement("button");
  this.HelpFileBox.className = "btn btn-default navbar-btn";
  this.HelpFileBox.onclick = function (ev) {
    _this.OnHelpButton(ev);
  };
  var helpImage = document.createElement("i");
  helpImage.className = "fa " + RD3_ClientParams.FA_ICON_HELP;
  this.HelpFileBox.appendChild(helpImage);
  li.appendChild(this.HelpFileBox);
  //
  // CMD Box
  li = document.createElement("li");
  li.className = "pull-right";
  this.RightHeaderCommandsList.appendChild(li);
  this.CommandInput = document.createElement("input");
  RD3_Glb.AutocompleteOff(this.CommandInput);
  this.CommandInput.className = "form-control navbar-btn wep-cmd-box";
  this.CommandInput.placeholder = "Search";
  this.CommandInput.onchange = function (ev) {
    _this.OnCommand(ev);
  };
  //
  // Su IE non scatta l'onchange quando premo invio.. in questo caso faccio il blur in modo che scatti 
  if (RD3_Glb.IsIE()) {
    this.CommandInput.onkeypress = function (ev) {
      var eve = (window.event)?window.event:ev;
      var code = (eve.charCode)?eve.charCode:eve.keyCode;
      //
      if (code == 13)
        this.blur();
    };
  }
  li.appendChild(this.CommandInput);
  //
  // Nel caso di menu superiore/inferiore creo il contenitore dei comandi
  if (this.MenuType == RD3_Glb.MENUTYPE_TASKBAR || this.MenuType == RD3_Glb.MENUTYPE_MENUBAR)
    fluidContainer.appendChild(this.SideCollapseContainer);
  //
  if (this.MenuType == RD3_Glb.MENUTYPE_TASKBAR)
    this.HeaderBox.className += " navbar-fixed-bottom";
  else
    this.HeaderBox.className += " navbar-fixed-top";
  //
  //
  // Chiamata per customizzazione header
  this.CustomizeHeader(parent);
}

// ********************************************************************************
// Dummy function per customizzazione header... puo' essere ridefinita in CUSTOM3.JS
// ********************************************************************************
WebEntryPoint.prototype.CustomizeHeader = function (parent)
{
}


// ********************************************************************************
// Questa funzione crea gli oggetti DOM per il pulsante chiudi tutto
// ********************************************************************************
WebEntryPoint.prototype.RealizeCloseAll = function (parent)
{
  this.CloseAllRow = document.createElement("tr");
  parent.appendChild(this.CloseAllRow);
  //
  var td = document.createElement("td");
  this.CloseAllRow.appendChild(td);
  //
  this.CloseAllButton = document.createElement("button");
  this.CloseAllButton.className = "btn btn-primary pull-right";
  td.appendChild(this.CloseAllButton);
  //
  this.CloseAllImg = RD3_Glb.createFAImage(RD3_ClientParams.FA_ICON_CLOSEALL);
  this.CloseAllTxt = document.createTextNode(" " + RD3_ServerParams.ChiudiTutto);
  this.CloseAllButton.appendChild(this.CloseAllImg);
  this.CloseAllButton.appendChild(this.CloseAllTxt);
  //
  this.CloseAllButton.onclick = function (ev) {
    this.OnCloseAll(ev);
  }.bind(this);
  //
  this.HandleCloseAllVisibility();
}


// ********************************************************************************
// Imposta le dimensioni del FormListBox perche'
// 1: Non ci sia dello spazio vuoto dopo
// 2: Mantenga comunque una dimensione sufficiente per tutto l'elenco delle form aperte
// ********************************************************************************
WebEntryPoint.prototype.AdaptFormListBox = function ()
{
  
}


// ********************************************************************************
// Calcola le dimensioni dei div in base alla dimensione del contenuto
// ********************************************************************************
WebEntryPoint.prototype.AdaptLayout = function ()
{
  // Calcolo le dimensioni: per prima cosa l'altezza del WepBox (100% - altezza header)
  var wepHeight = "calc(100% - " + (this.HeaderBox.offsetHeight + 10) + "px)";
  this.WepBox.style.height = wepHeight;
  if (this.MenuType != RD3_Glb.MENUTYPE_TASKBAR)
    this.WepBox.style.marginTop = (this.HeaderBox.offsetHeight + 10) + "px";
  //
  // Adesso calcolo le dimensioni della grigliatura principale, se ci sono docked
  // la dimensione dipende dal contenuto, altrimenti è 0, quindi per prima cosa devo chiedere
  // alle form docked di dimensionarsi
  var n = this.StackForm.length;
  for (var i = 0; i < n; i++) {
    var f = this.StackForm[i];
    f.AdaptDocked();
  }
  //
  if (!this.HasTopDocked()) {
    this.TopDockedBox.style.height = "0px";
    this.TopDockedBox.style.padding = "0px";
  }
  else
    this.TopDockedBox.style.padding = "";
  //
  if (!this.HasBottomDocked()) {
    this.BottomDockedBox.style.height = "0px";
    this.BottomDockedBox.style.padding = "0px";
  }
  else
    this.BottomDockedBox.style.padding = "";
  //
  var centerRowHeight = "calc(100% - " + (this.TopDockedBox.offsetHeight + this.BottomDockedBox.offsetHeight + 15) + "px)";
  this.MainGridRow2.style.height = centerRowHeight;
  if (this.HasSideMenu())
    this.SideMenuBox.style.height = centerRowHeight;
  //
  // Ora calcolo la dimensione orizzontale della riga centrale della mia griglia
  var mainWidth = 12;
  if (this.HasRightDocked()) {
    mainWidth -= this.GetRightDockedColumns();
    this.RightDockedBox.style.display = "";
    this.RightDockedBox.className = "col-xs-12 col-sm-" + this.GetRightDockedColumns() + " col-md-" + this.GetRightDockedColumns();
  }
  else
    this.RightDockedBox.style.display = "none";
  if (this.HasLeftDocked()) {
    mainWidth -= this.GetLeftDockedColumns();
    this.LeftDockedBox.style.display = "";
    this.LeftDockedBox.className = "col-xs-12 col-sm-" + this.GetLeftDockedColumns() + " col-md-" + this.GetLeftDockedColumns();
  }
  else
    this.LeftDockedBox.style.display = "none";
  //
  this.FormsBox.className = "col-xs-12 col-sm-" + mainWidth + " col-md-" + mainWidth;
  if ((this.WidgetMode || this.CmdObj.SuppressMenu || !this.HasSideMenu()) && !this.HasLeftDocked())
    this.FormsBox.className += " is-widget-mode";
  //
  // Aggiorno le dimensioni della MDI area sul server
  this.SendResize();
  //
  // Ora tocca alle form aperte
  var n = this.StackForm.length;
  for (var i = 0; i < n; i++) {
    var f = this.StackForm[i];
    f.AdaptLayout();
  }
  //
  this.VoiceObj.AdaptLayout();
}


// ********************************************************************************
// Adatta l'Header
// ********************************************************************************
WebEntryPoint.prototype.AdaptHeader = function ()
{
  
}


// ********************************************************************************
// Toglie gli elementi visuali dal DOM perche' questo oggetto sta per essere
// distrutto
// ********************************************************************************
WebEntryPoint.prototype.Unrealize = function ()
{
  // Tolgo gli eventi motion
  if (window.removeEventListener)
  {
    window.removeEventListener("devicemotion", RD3_DeviceMotion);
    window.removeEventListener("deviceorientation", RD3_DeviceOrientation);
  }
  //
  // Tolgo WepBox dal DOM
  this.WepBox.parentNode.removeChild(this.WepBox);
  //
  // Mi tolgo dalla mappa degli oggetti
  RD3_DesktopManager.ObjectMap[this.Identifier] = null;
  //
  // Passo il comando a tutti i miei figli
  this.IndObj.Unrealize();
  this.CmdObj.Unrealize();
  this.TimerObj.Unrealize();
  this.VSList.Unrealize();
  this.VoiceObj.Unrealize();
  //
  var n = this.StackForm.length;
  for (var i = 0; i < n; i++)
  {
    this.StackForm[i].Unrealize();
  }
  //
  // Distruggo la DelayDialog
  this.DelayDialog.Unrealize();
  //
  // Rimuovo i collegamenti al DOM
  this.WepBox = null;
  this.HeaderBox = null;
  this.SideMenuBox = null;
  this.FormsBox = null;
  this.MenuBox = null;
  this.FormListBox = null;
  this.FormListHeader = null;
  this.FormListTitle = null;
  this.CloseAllBox = null;
  this.CloseAllButton = null;
  //
  this.Realized = false;
}


// ********************************************************************************
// Devo segnarmi che sto iniziando la gestione di una riposta del server
// ********************************************************************************
WebEntryPoint.prototype.BeforeProcessResponse = function ()
{
  this.InResponse = true;
}


// ********************************************************************************
// Devo gestire le variazioni avvenute
// ********************************************************************************
WebEntryPoint.prototype.AfterProcessResponse = function (force)
{
  if (this.DeferAfterProcess || RD3_GFXManager.Animating())
  {
    this.DeferAfterProcess = false;
    //
    window.setTimeout("RD3_DesktopManager.WebEntryPoint.AfterProcessResponse(" + force + ")", 20);
    return;
  }
  //
  if (this.InResponse || force)
  {
    var recalc = this.RecalcLayout;
    //
    // Mi e' stato chiesto di rinfrescare le toolbar?
    if (this.RefreshCommands)
    {
      this.CmdObj.ActiveFormChanged();
      this.IndObj.ActiveFormChanged();
      //
      this.RefreshCommands = false;
    }
    //
    this.RecalcLayout = false;
    if (recalc || force) {
      this.AdaptLayout();
    }
    //
    // Se necessario faccio rimettere a posto anche la toolbar della welcome mobile
    if (RD3_Glb.IsMobile() && this.WelcomeForm && this.WelcomeForm.Realized && this.WelcomeForm.AdaptToolbar)
      this.WelcomeForm.AdaptToolbarLayout();
    //
    // Giro il messaggio alle form aperte
    var n = this.StackForm.length;
    for (var i = 0; i < n; i++)
      this.StackForm[i].AfterProcessResponse();
    //
    // Giro il messaggio al KBManager
    RD3_KBManager.AfterProcessResponse();
    if (this.UseZones())
    {
      for (var szi = 0; szi < this.ScreenZones.length; szi++)
        this.ScreenZones[szi].AfterProcessResponse();
    }
    //
    // Rendo visibile il wep se era ancora nascosto
    if (this.WepBox.style.visibility == "hidden")
    {
      this.WepBox.style.visibility = "";
      //
      // Blocco il timer di animazione caricamento se presente
      if (typeof (WaitTimer) != "undefined" && WaitTimer)
      {
        window.clearTimeout(WaitTimer);
        WaitTimer = null;
      }
      //
      // Elimino il wait box se presente
      var wb = document.getElementById("wait-box");
      if (wb)
      {
        if (RD3_Glb.IsMobile())
        {
          this.WepBox.style.opacity = 1;
          wb.style.opacity = 0;
          window.setTimeout("document.getElementById('wait-box').parentNode.removeChild(document.getElementById('wait-box'))", 500);
        }
        else
        {
          wb.parentNode.removeChild(wb);
        }
      }
      //
      // Inizio effetto iniziale fading
      if (!RD3_Glb.IsMobile())
      {
        var fx = new GFX("start", true, document.body, false);
        RD3_GFXManager.AddEffect(fx);
      }
      this.SoundAction("login", "play");
      //
      // Finche' e' invisibile safari non aggiusta bene l'header
      if (RD3_Glb.IsSafari() || RD3_Glb.IsChrome() || RD3_Glb.IsFirefox(4) || RD3_Glb.IsIE(8))
        this.OnResize();
      //
      // Dico che e' ora di dare il fuoco a qualcuno
      RD3_KBManager.CheckFocus = true;
    }
    //
    // Se ho dei comandi ritardati li eseguo ora
    if (this.ExecuteRitardati.length > 0)
    {
      // Ciclo sui comandi e li eseguo, poi svuoto l'array
      var nex = this.ExecuteRitardati.length;
      for (var iex = 0; iex < nex; iex++)
      {
        eval(this.ExecuteRitardati[iex]);
      }
      //
      this.ExecuteRitardati.splice(0, nex);
    }
    //
    RD3_KBManager.ActiveButton = null;
    //
    this.InResponse = false;
  }
}


// ********************************************************************************
// Il browser e' stato ridimensionato
// ********************************************************************************
WebEntryPoint.prototype.OnResize = function (evento)
{
  // Devo gestire in modo particolare la tastiera
  if (RD3_Glb.IsAndroid()) {
    // Se ho le dimensioni impostate significa che la tastiera si e' aperta, in questo caso se il resize
    // porta ad una videata più alta dell'ultimo resize signifca che si è chiusa la tastiera:
    // in questo caso togliamo le dimensioni impostate
    if (document.body.style.width != "" && document.body.style.height != "" && window.innerHeight > window.lastHeight) {
      document.body.style.width = "";
      document.body.style.height = "";
      document.body.style.transform = "";
      // 
      // Togliamo il fuoco all'elemento attivo
      if (document.activeElement && document.activeElement.blur)
        document.activeElement.blur();  
    }
    //
    // Ho le dimensioni impostate, un oggetto con il fuoco e l'altezza si è ridotta :
    // si è aperta la tastiera! In questo caso ho tutte le informzioni sufficienti per portare a video il campo con il fuoco
    if (document.activeElement && document.body.style.width != "" && document.body.style.height != "" && window.innerHeight < window.lastHeight) {
      var top = document.activeElement.getBoundingClientRect().top;
      if (window.currentHeight - top < window.innerHeight)
        top = top - (window.currentHeight - window.innerHeight);
      document.body.style.transform = "translate3d(0px, -"+top+"px, 0px)";
    }
    //
    // Current mi servono per fissare le dimensioni della pagina all'apertura della tastiera,
    // last mi serve per capire se la tastiera si e' aperta o chiusa
    if (document.body.style.width == "" && document.body.style.height == "") {
      window.currentHeight = window.innerHeight;
      window.currentWidth = window.innerWidth;
    }
    //
    window.lastHeight = window.innerHeight;
    window.lastWidth = window.innerWidth;
  }
  //
  // Resize, chiudo anche i popup aperti che potrebbero andare fuori posto
  this.CmdObj.ClosePopup();
  //
  // Eseguo il resize ritardato
  if (this.ResizeTimerId > 0)
    window.clearTimeout(this.ResizeTimerId);
  this.ResizeTimerId = window.setTimeout("RD3_DesktopManager.WebEntryPoint.TickResize()", 200);
}


// ********************************************************************************
// Il browser e' stato ridimensionato
// ********************************************************************************
WebEntryPoint.prototype.TickResize = function (delayed)
{
  if (delayed == undefined)
    delayed = false;
  //
  if (this.ResizeTimerId)
    window.clearTimeout(this.ResizeTimerId);
  this.ResizeTimerId = 0;
  if (!this.InResponse)
  {
    if (RD3_Glb.IsAndroid() && !delayed)
    {
      this.ResizeTimerId = window.setTimeout("RD3_DesktopManager.WebEntryPoint.TickResize(true)", 250);
      return;
    }
    //
    // Se faccio un resize perdo l'eventuale OnChanges: allora lo forzo dando il fuoco al wep, memorizzandomi l'elemento che
    // aveva il fuoco
    var oldfoc = RD3_KBManager.ActiveElement;
    this.WepBox.focus();
    //
    this.InResize = true;
    this.AdaptLayout();
    //
    var n = this.StackForm.length;
    for (var i = 0; i < n; i++)
      this.StackForm[i].OnViewportResize();
    this.InResize = false;
    //
    // Se c'e' una combo aperta, chiudo anche quella
    if (RD3_DDManager.OpenCombo)
    {
      // Se sono su mobile e la combo non e' popover)
      if (RD3_Glb.IsMobile() && !RD3_DDManager.OpenCombo.UsePopover)
        RD3_DDManager.OpenCombo.AdaptMobileCombo();
      else
        RD3_DDManager.OpenCombo.Close();
    }
    RD3_DDManager.ClosePopup();
    //
    // Ripristino l'elemento che aveva il fuoco in precedenza
    RD3_KBManager.ActiveElement = oldfoc;
  }
  else
  {
    this.ResizeTimerId = window.setTimeout("RD3_DesktopManager.WebEntryPoint.TickResize()", 50);
  }
}


// ********************************************************************************
// Manda al server un evento di resize di Wep
// ********************************************************************************
WebEntryPoint.prototype.SendResize = function ()
{
  // Per dare al server le dimensioni corrette devo conoscere i padding..
  var padr = 0;
  var padl = 0;
  var padt = 0;
  var padb = 0;
  try
  {
    padr = parseInt(RD3_Glb.GetStyleProp(this.FormsBox, "paddingRight"));
    padl = parseInt(RD3_Glb.GetStyleProp(this.FormsBox, "paddingLeft"));
    padt = parseInt(RD3_Glb.GetStyleProp(this.FormsBox, "paddingTop"));
    padb = parseInt(RD3_Glb.GetStyleProp(this.FormsBox, "paddingBottom"));
  }
  catch (ex)
  {
  }
  //
  var ev = new IDEvent("resize", this.Identifier, null, RD3_Glb.EVENT_SERVERSIDE, null, this.FormsBox.clientWidth - padr, this.FormsBox.clientHeight - padb, RD3_Glb.GetScreenLeft(this.FormsBox) + padl, RD3_Glb.GetScreenTop(this.FormsBox) + padt, null, false, this.WepBox.clientWidth, this.WepBox.clientHeight);
}


// ********************************************************************************
// Una form deve essere aperta
// ********************************************************************************
WebEntryPoint.prototype.OpenForm = function (objnode, openas)
{
  // Definisce una form aperta
  var newform = new WebForm();
  //
  // Verifico se ho previsto la chiusura della form che devo aprire.. qui
  // posso gestire solo le Form MDI, le form modali, docked o popup fanno partire
  // subito la loro animazione di chiusura
  if (this.CloseFormId != "" && this.CloseFormId == objnode.getAttribute("id"))
  {
    // Faccio subito l'unrealize della form da chiudere, altrimenti
    // sovrascriverei gli oggetti nella mappa con la mia loadfromxml
    var ftc = RD3_DesktopManager.ObjectMap[this.CloseFormId];
    if (ftc)
      ftc.Unrealize();
    //
    // La tolgo dallo stack delle form
    var n = this.StackForm.length;
    for (var i = 0; i < n; i++)
    {
      var f = this.StackForm[i];
      if (f.Identifier == this.CloseFormId)
      {
        this.StackForm.splice(i, 1);
        break;
      }
    }
    //
    // Elimino gli eventuali puntatori alla form che ho chiuso
    if (this.ActiveForm && this.ActiveForm.Identifier == this.CloseFormId)
      this.ActiveForm = null;
    if (this.VisibleForm && this.VisibleForm.Identifier == this.CloseFormId)
      this.VisibleForm = null;
    //
    // Annullo la chiusura della form (l'ho chiusa io senza animazione..)
    this.CloseFormId = "";
  }
  //
  newform.LoadFromXml(objnode);
  this.StackForm.push(newform); // Anche le form modali vanno qui
  //
  if (openas)
    newform.Modal = openas;
  //
  newform.Realize();
  //
  // Attivo la form appena aperta portandola in primo piano (solo se e' visibile!)
  if (newform.Visible)
  {
    if (this.UseZones() && newform.Docked)
    {
      this.ActivateForm(newform, true);
    }
    else
    {
      var lastForm = (newform.Owner && newform.HasBackButton() ? RD3_DesktopManager.ObjectMap[newform.Owner] : null);
      if (lastForm && lastForm.Modal == 0 && newform.Modal == 0)
      {
        newform.AdaptLayout();
        RD3_DesktopManager.WebEntryPoint.AnimateForms(lastForm.Identifier, newform.Identifier, true);
      }
      else
      {
        this.ActivateForm(newform, true);
      }
    }
  }
}


// ********************************************************************************
// Una form deve essere chiusa
// ********************************************************************************
WebEntryPoint.prototype.CloseForm = function (ident)
{
  // Se sto chiudendo la form attualmente visibile la chiudo con animazione, altrimenti la chiudo normalmente
  var n = this.StackForm.length;
  for (var i = 0; i < n; i++)
  {
    var f = this.StackForm[i];
    if (f.Identifier == ident)
    {
      if (this.UseZones() && f.Docked)
      {
        this.CloseForm2(f);
        break;
      }
      else
      {
        // Uso l'animazione di ritorno (Mobile, per form aperta da altra form) solo se la form a cui devo ritornare non
        // e' gia' stata chiusa: in quel caso mi chiudo e basta..
        var lastForm = (f.Owner && f.HasBackButton() ? RD3_DesktopManager.ObjectMap[f.Owner] : null);
        if (lastForm && lastForm.Realized && lastForm.Modal == 0 && f.Modal == 0 && this.ActiveForm && f.Identifier == this.ActiveForm.Identifier)
          this.AnimateForms(lastForm.Identifier, f.Identifier, false);
        else
          this.CloseForm2(f);
        break;
      }
    }
  }
}


// ********************************************************************************
// Una form deve essere chiusa
// ********************************************************************************
WebEntryPoint.prototype.CloseForm2 = function (f)
{
  var _this = this;
  //
  // Ritrovo l'indice della form
  var i = 0;
  for (i = 0; i < this.StackForm.length; i++)
  {
    if (this.StackForm[i] == f)
      break;
  }
  //
  // Trovata la form da chiudere: la chiudo con animazione se non e' dockata ne' modale ed e' visibile,
  // altrimenti la chiudo
  if ((!f.Docked && !f.Modal) && f.FormBox.style.display != "none")
  {
    // Nel tema mobile a causa dell'animazione di BackButton puo' succedere che venga chiamata la CloseForm2 di piu' Form
    // e siano entrambe visibili.. (in seattle c'e' sempre una ed una sola form principale visibile.. tranne durante l'animazione che viene fatta alla fine di tutto - activate,
    // in Mobile l'animazione parte subito se c'e' il backbutton.. questo implica che mentre stiamo elaborando la risposta del server ci sono due form visibili..)
    // Allora se mi sono segnato di chiudere una Form all'activate devo chiuderla qui.
    if (RD3_Glb.IsMobile() && this.CloseFormId != "" && this.CloseFormId != f.Identifier && (!this.ActiveForm || this.ActiveForm.Identifier != this.CloseFormId))
    {
      var j = 0;
      var closingF;
      for (j = 0; j < this.StackForm.length; j++)
      {
        if (this.StackForm[j].Identifier == this.CloseFormId)
        {
          closingF = this.StackForm[j];
          break;
        }
      }
      //
      if (closingF)
      {
        // Gestisco la chiusura di timer di form
        this.TimerObj.FormClosed(closingF);
        //
        // La distruggo
        closingF.Unrealize();
        //
        // Tolgo la form dallo stackform
        this.StackForm.splice(j, 1);
      }
    }
    //
    // Mi memorizzo l'id della form da chiudere
    this.CloseFormId = f.Identifier;
  }
  else
  {
    // Se e' modale la chiudo con l'animazione corretta
    if (f.Modal)
    {
      // Tolgo la form dallo stackform
      this.StackForm.splice(i, 1);
      //
      // Se sono la form attiva elimino il riferimento
      if (this.ActiveForm && this.ActiveForm.Identifier == f.Identifier)
      {
        this.ActiveForm = null;
        //
        this.CmdObj.ActiveFormChanged();
        this.IndObj.ActiveFormChanged();
      }
      //
      if (f.PopupFrame && f.PopupFrame.Opened) {
        
        f.PopupFrame.OnCloseCallBack = function() {
          _this.CompleteCloseFormAnimation(f);
        }
        //
        // Chiudo la modale con la sua animazione
        f.PopupFrame.Close();
      }
      else
        this.CompleteCloseFormAnimation(f);
    }
    //
    if (f.Docked)
    {
      if (!this.UseZones())
      {
        // Gestisco la chiusura di timer di form
        this.TimerObj.FormClosed(f);
        //
        // Se sono la form attiva elimino il riferimento
        if (this.ActiveForm && this.ActiveForm.Identifier == f.Identifier)
          this.ActiveForm = null;
        //
        f.Unrealize();
        //
        // Tolgo la form dallo stackform
        var n = this.StackForm.length;
        for (var i=0; i<n; i++) {
          var fs = this.StackForm[i];
          if (fs.Identifier == f.Identifier) {
            this.StackForm.splice(i, 1);
            break;
          } 
        }
        //
        // Gestisco i Command Set, gli indicatori ed i Timer
        this.CmdObj.ActiveFormChanged();
        this.IndObj.ActiveFormChanged();
        this.TimerObj.ActiveFormChanged();
      }
      else
      {
        var scz = this.GetScreenZone(f.DockType);
        scz.CloseForm(f);
      }
    }
    //
    if (!f.Docked && !f.Modal)
    {
      // Gestisco la chiusura di timer di form
      this.TimerObj.FormClosed(f);
      //
      // Se sono la form attiva elimino il riferimento
      if (this.ActiveForm && this.ActiveForm.Identifier == f.Identifier)
        this.ActiveForm = null;
      //
      // La distruggo
      f.Unrealize();
      //
      // Tolgo la form dallo stackform
      this.StackForm.splice(i, 1);
    }
  }
}

// ********************************************************************************
// Una form deve essere attivata
// ********************************************************************************
WebEntryPoint.prototype.ActivateForm = function (ident, isopening)
{
  if (!isopening)
    isopening = false;
  //
  if (this.Realized)
  {
    // nf e' la nuova form da attivare
    var nf = (ident && ident.Identifier) ? ident : RD3_DesktopManager.ObjectMap[ident];
    //
    // Se sto attivando una form non docked, chiudo l'eventuale popover aperto
    if (RD3_Glb.IsMobile() && nf && !nf.Docked)
      RD3_DDManager.ClosePopup();
    //
    // Era gia' attiva... non faccio nulla
    if (nf == this.ActiveForm && this.ActiveForm)
      return;
    //
    var m = nf ? nf.Modal != 0 : false;
    var d = nf ? nf.Docked : false;
    //
    // Attivazione di una MODALE
    if (m)
    {
      this.ActiveForm = nf;
      //
      var fx = new GFX("modal", true, nf, !isopening, null, nf.ModalAnimDef);
      RD3_GFXManager.AddEffect(fx);
      //
      return;
    }
    //
    // Attivazione di una DOCKED
    if (d)
    {
      this.ActiveForm = nf;
      //
      // Se non c'e' l'animazione iniziale faccio l'animazione di apertura docked
      if (this.ActiveForm && this.WepBox.style.visibility != "hidden")
      {
        // Apertura delle docked : non uso animazione ma vado subito nello stato corretto
        nf.AdaptDocked();
        //
        this.CmdObj.ActiveFormChanged();
        this.IndObj.ActiveFormChanged();
        this.TimerObj.ActiveFormChanged();
        //
        // Se il pulsante chiudi tutto e' nascosto lo devo mostrare
        this.HandleCloseAllVisibility();
        //
        if (this.ActiveForm)
          RD3_DesktopManager.HandleFocus2(this.ActiveForm.Identifier,0);
        //
        this.AdaptLayout(); 
      }
      else  // Ci sara' l'animazione iniziale: la docked deve comparire subito..
      {
        nf.SetActive(true, true);
        //
        this.CmdObj.ActiveFormChanged();
        this.IndObj.ActiveFormChanged();
        this.TimerObj.ActiveFormChanged();
        //
        // Se il pulsante chiudi tutto e' nascosto lo devo mostrare
        this.HandleCloseAllVisibility();
        //
        if (this.ActiveForm)
          RD3_DesktopManager.HandleFocus2(this.ActiveForm.Identifier, 0);
      }
      //
      return;
    }
    //
    // Attivazione di una FORM PRINCIPALE
    if (!m && !d)
    {
      var finidx = 1000;   // Indice della form da aprire nello stackform
      var foutidx = 0;     // indice della form da fare uscire nello stackform
      //
      this.ActiveForm = nf;
      var fin = nf;
      var fout = this.VisibleForm;
      //
      // Se non devo animare nessuna form in ingresso animo il welcome box se e' nescosto
      if (!fin && this.WelcomeBox && this.WelcomeBox.style.display == "none")
      {
        fin = this.WelcomeBox;
        finidx = -1000;
      }
      if (!fin && this.WelcomeForm) {
        fin = this.WelcomeForm;
        finidx = -1000;
      }
      //
      // Se non ho trovato nessun candidato valido allora animo in uscita il WelcomeBox se e' visibile
      if (!fout && this.WelcomeBox && this.WelcomeBox.style.display != "none")
        fout = this.WelcomeBox;
      //
      // Trovo gli indici delle form
      var n = this.StackForm.length;
      for (var i = 0; i < n; i++)
      {
        var f = this.StackForm[i];
        //
        if (f == fin)
          finidx = i;
        //
        if (f == fout)
          foutidx = i;
      }
      //
      // Per eseguire l'animazione ho bisogno di due oggetti differenti, se non si verifica questa condizione salto l'animazione
      var skip = false;
      if ((fin == fout) || !fout || !fin)
        skip = true;
      //
      // Se non sto facendo l'animazione di fade iniziale
      if (fin && this.WepBox.style.visibility != "hidden")
      {
        var dir = true; // Direzione dello scroll verticale: true da up a dn, false il contrario
        if (fout)
          dir = (finidx < foutidx); // Imposto la direzione dello scroll se ci sono due form
        //
        var fx = new GFX("form", dir, fin, skip, (fout ? fout : null), fin.ShowAnimDef);
        //
        // Se devo chiudere una form lo comunico all'animazione
        if (this.CloseFormId != "")
          fx.CloseFormAnimation = true;
        //
        RD3_GFXManager.AddEffect(fx);
      }
      else    // Sto facendo l'animazione di fade iniziale: non faccio animazioni
      {
        // In questo caso se ho una form da attivare la faccio comparire..
        if (nf)
        {
          // Forzo le altre form a nascondersi
          var n = this.StackForm.length;
          for (var i = 0; i < n; i++) {
            var fo = this.StackForm[i];
            //
            // Nascondo solo le form principali
            if (!fo.Modal && !fo.Docked)
              fo.SetActive(false, true);
          }
          //
          // Nascondo la welcome
          this.WelcomeBox.style.display = "none";
          //
          nf.SetActive(true, true);
          if (!m && !d)
            this.VisibleForm = nf;
          this.ActiveForm = nf;
          //
          // Imposto i flag di animazione, in modo da bloccare l'animazione dei messaggi alla comparsa..
          this.ActiveForm.Animating = true;
          var nf = this.StackForm.length;
          for (var t = 0; t < nf; t++) {
            if (this.StackForm[t].Docked)
              this.StackForm[t].Animating = true;
          }
          //
          this.CmdObj.ActiveFormChanged();
          this.IndObj.ActiveFormChanged();
          this.TimerObj.ActiveFormChanged();
          //
          // Se il pulsante chiudi tutto e' nascosto lo devo mostrare
          this.HandleCloseAllVisibility();
          //
          RD3_DesktopManager.HandleFocus2(this.ActiveForm.Identifier, 0);
        }
      }
      //
      // Se il menu e' superiore e sto mostrando il togglebutton e il menu e' visibile lo faccio collassare cliccando sul pulsante
      if (this.MenuType == RD3_Glb.MENUTYPE_TASKBAR || this.MenuType == RD3_Glb.MENUTYPE_MENUBAR) {
        if (RD3_Glb.GetStyleProp(this.toggleButton, "display") != "none" && this.toggleButton.getAttribute("aria-expanded") == "true")
          this.toggleButton.click();
      }
    }
    //
    // Apertura di una Docked o di una form principale: se siamo su smartphone con menu laterale lo nascondiamo
    if (!m && this.HasSideMenu())
      window.setTimeout(function () {
        $('.row-offcanvas').removeClass('active')
      }, 300);
  }
  else
  {
    // Memorizzo l'identificativo della form attiva
    this.ActiveForm = ident;
  }
}


// ********************************************************************************
// Gestore dell click sul pulsante chiudi tutto
// ********************************************************************************
WebEntryPoint.prototype.OnCloseAll = function (evento)
{
  var ev = new IDEvent("claclk", this.Identifier, evento, this.CloseAllEventDef);
}


// ********************************************************************************
// Gestore dell click sul pulsante chiudi tutto
// ********************************************************************************
WebEntryPoint.prototype.OnCloseApp = function (evento)
{
  var ev = new IDEvent("clk", "cloapp", evento, this.CloseAppEventDef);
}


// ********************************************************************************
// Apre il file di help in una nuova finestra
// ********************************************************************************
WebEntryPoint.prototype.OnHelpButton = function (evento)
{
  var ok = RD3_ClientEvents.OnHelp(this);
  //
  if (ok)
    RD3_DesktopManager.OpenDocument(this.HelpFile, "help", "");
}


// ********************************************************************************
// Metodo che gestisce la visibilita' del pulsante chiudi tutto:
// se ci sono form nella lista form e' visibile, altrimenti no
// ********************************************************************************
WebEntryPoint.prototype.HandleCloseAllVisibility = function ()
{
  var mdi = this.GetLastForm();
  var popup = this.GetLastForm(true);
  //
  // Decido se devono essere visibili o meno
  var clallvis = popup ? true : false;
  var welvis = mdi ? false : true;
  //
  // Confronto con lo stato attuale: lo cambio solo devo
  if (clallvis != this.CloseAllVisible)
  {
    if (this.FormsListCollapsePanelBody)
      this.FormsListCollapsePanelBody.style.display = clallvis ? "" : "none";
    this.CloseAllVisible = clallvis;
  }
}


// ********************************************************************************
// Ritorna l'ultima form aperta che non sia modale o docked e che sia visibile
// se flPopup = TRUE allora considera anche i popup
// ********************************************************************************
WebEntryPoint.prototype.GetLastForm = function (flPopup)
{
  var n = this.StackForm.length;
  for (var i = n - 1; i >= 0; i--)
  {
    var f = this.StackForm[i];
    //
    if (f.Modal != 0 && !flPopup)
      continue;
    //
    if (!f.IsModalPopup() && !f.Docked && f.Visible)
      return f;
  }
  //
  return null;
}


// ********************************************************************************
// Questa funzione cerca di convincere l'utente che non deve andare via dalla
// pagina prima di premere il pulsante "CHIUDI"
// ********************************************************************************
WebEntryPoint.prototype.OnBeforeUnload = function (ev)
{
  if (!this.Redirecting) 
  {
    if (ev && RD3_ServerParams.UnloadMessage) 
    {
      ev.preventDefault();
      ev.returnValue = RD3_ServerParams.UnloadMessage;
    }
    return RD3_ServerParams.UnloadMessage;
  }
}


// ********************************************************************************
// L'utente ha premuto il tasto DEBUG o TRACE
// pagina prima di premere il pulsante "CHIUDI"
// ********************************************************************************
WebEntryPoint.prototype.OnDebug = function (ev)
{
  var ok = RD3_ClientEvents.OnDebug(this);
  //
  if (ok)
  {
    if (ev.ctrlKey)
    {
      RD3_Debugger.Show();
    }
    else
    {
      switch (this.DebugType)
      {
        case 1 : // Debug
        {
          if (window.RD4_Enabled)
            var ev = new IDEvent("IWDTT", "", ev, RD3_Glb.EVENT_ACTIVE);
          else
            RD3_DesktopManager.OpenDocument("?WCI=IWDTT&WCE=", "debug", "");
          break;
        }
        case 2 : // Trace
          {
            var ev = new IDEvent("cmd", this.Identifier, ev, RD3_Glb.EVENT_ACTIVE, "DTTHELP");
          }
          break;
      }
    }
  }
}


// ********************************************************************************
// Ho cliccato da qualche parte...
// ********************************************************************************
WebEntryPoint.prototype.OnClick = function (ev)
{
  if (this.DisableOnClick > 0)
  {
    this.DisableOnClick--;
    return;
  }
  //
  // Non considero il click sulla menubox di task bar
  var srcobj = (window.event) ? window.event.srcElement : ev.explicitOriginalTarget;
  if (srcobj == this.TaskbarMenuBox)
    return;
  //
  // Chiudo tutti i popup
  this.CmdObj.ClosePopup();
}

// ********************************************************************************
// Ultimato caricamento del blob?
// ********************************************************************************
WebEntryPoint.prototype.OnBlobUpload = function (response)
{
  var s = "";
  if (response)
    s = response;
  else
  {
    try
    {
      var bod = RD3_DesktopManager.WebEntryPoint.BlobFrame.contentWindow.document.body;
      //
      if (bod.innerText)
        s = bod.innerText;
      if (bod.textContent)
        s = bod.textContent;
    }
    catch (ex)
    {
    }
  }
  //
  if (s && s != "")
  {
    // tolgo i 256 "p" messi per ingannare il browser
    s = s.substr(256, s.length - 256);
    //
    this.DelayDialog.Close();
    //
    // Carico una richiesta con questo testo dentro...
    RD3_DesktopManager.ProcessXmlText(s);
  }
}


// ********************************************************************************
// L'utente ha scritto un valore nell'input dei comandi
// ********************************************************************************
WebEntryPoint.prototype.OnCommand = function (ev)
{
  // Verifico se il pulsante premuto e' un INVIO, se lo e' mando il comando al server e svuoto il campo
  var code = (ev.charCode)? ev.charCode: ev.keyCode;
  if (code == 13)
  {
  // Attivo su richiesta il debugger javascript
  var s = this.CommandInput.value.toUpperCase();
  if (s=="JSDEB")
  IEDebug();
  if (s=="ANI-" || s=="ANIOFF")
  RD3_ClientParams.EnableGFX = false;
  if (s=="AWK-" || s=="AWKOFF")
  RD3_ClientParams.UseWebKitGFX = false;
  if (s=="ANI+" || s=="ANION")
  RD3_ClientParams.EnableGFX = true;
  if (s=="AWK+" || s=="AWKON")
  RD3_ClientParams.UseWebKitGFX = true;
  if (s=="SOUND-" || s=="SND-" || s=="SNDOFF")
  RD3_ClientParams.EnableSound = false;
  if (s=="SOUND+" || s=="SND+" || s=="SNDON")
  RD3_ClientParams.EnableSound = true;
  //
  // Mando il comando al server
  var ev = new IDEvent("cmd", this.Identifier, ev, RD3_Glb.EVENT_ACTIVE, s);
  //
  // Svuoto l'input
  this.CommandInput.value = "";
  }
  // Mando il comando al server
  var s = this.CommandInput.value.toUpperCase();
  var ev = new IDEvent("cmd", this.Identifier, ev, RD3_Glb.EVENT_ACTIVE, s);
  //
  // Svuoto l'input
  this.CommandInput.value = "";
}

// **********************************************************************
// Gestisco la pressione dei tasti FK a partire dalla form
// **********************************************************************
WebEntryPoint.prototype.HandleFunctionKeys = function (eve)
{
  var code = (eve.charCode) ? eve.charCode : eve.keyCode;
  var ok = false;
  //
  // Verifico i comandi generali
  ok = RD3_DesktopManager.WebEntryPoint.CmdObj.HandleFunctionKeys(eve, -1, -1);
  //
  return ok;
}


// **********************************************************************
// Il WEP deve prendere il fuoco, lo do alla form oppure al cmd
// **********************************************************************
WebEntryPoint.prototype.Focus = function ()
{
  // Provo a darlo alla form attiva (se c'e')
  if (this.ActiveForm && this.ActiveForm.Focus && this.ActiveForm.Focus())
    return true;
  //
  return false;
}


// **********************************************************************
// Imposta il timer che nascondera' il focus box
// **********************************************************************
WebEntryPoint.prototype.SetHideFocusBox = function ()
{
  this.FocusBoxTimerId = window.setTimeout("RD3_DesktopManager.WebEntryPoint.HideFocusBox();", 200);
}

// **********************************************************************
// Imposta il timer che nascondera' il focus box
// **********************************************************************
WebEntryPoint.prototype.ResetHideFocusBox = function ()
{
  window.clearTimeout(this.FocusBoxTimerId);
}

// **********************************************************************
// Toglie il focus box dal campo
// **********************************************************************
WebEntryPoint.prototype.HideFocusBox = function ()
{
  if (this.FocusBox.parentNode)
    this.FocusBox.parentNode.removeChild(this.FocusBox);
}


// **********************************************************************
// Il mouse e' stato mosso su una delle icone di scroll del menu laterale
// **********************************************************************
WebEntryPoint.prototype.OnScrollMenuOver = function (evento)
{
  
}


// **********************************************************************
// Il mouse e' uscito da una delle icone di scroll del menu laterale
// **********************************************************************
WebEntryPoint.prototype.OnScrollMenuOut = function (evento)
{
  
}


// ********************************************************************************
// Inizio lo scrolling
// ********************************************************************************
WebEntryPoint.prototype.ScrollMenu = function (dir)
{
  
}


// ********************************************************************************
// Effettuo scrolling della toolbar
// ********************************************************************************
WebEntryPoint.prototype.ScrollingMenu = function ()
{
  
}


// ********************************************************************************
// Posiziona e mostra lo scrollbox se necessario
// ********************************************************************************
WebEntryPoint.prototype.AdaptScrollBox = function ()
{
  
}


// *******************************************************
// Metodo che gestisce il cambio dello stato dell'immagine
// *******************************************************
WebEntryPoint.prototype.OnReadyStateChange = function ()
{
  
}


// ********************************************************************************
// Gestore del mouse over su un oggetto di wep
// ********************************************************************************
WebEntryPoint.prototype.OnMouseOverObj = function (ev, obj)
{
  
}


// ********************************************************************************
// Gestore del mouse out su un oggetto di wep
// ********************************************************************************
WebEntryPoint.prototype.OnMouseOutObj = function (ev, obj)
{
  
}


// ********************************************************************************
// Gestore del mouse down su un oggetto di wep
// ********************************************************************************
WebEntryPoint.prototype.OnMouseDownObj = function (ev, obj)
{
  
}


// ********************************************************************************
// Evento di scrolling del menu
// ********************************************************************************
WebEntryPoint.prototype.OnMouseWheel = function (ev)
{
  var box = (this.MenuType == RD3_Glb.MENUTYPE_TASKBAR) ? this.TaskbarMenuBox : this.SideMenuBox;
  //
  // Vediamo se lo scroll e' necessario
  if (!box)
    return false;
  //
  // Se il menu' ci sta tutto non scrollo
  if (this.SuppressMenu || box.scrollHeight - 1 <= box.clientHeight)
    return false;
  //
  var eve = window.event ? window.event : ev;
  var srcElement = eve.srcElement ? eve.srcElement : eve.target;
  var delta = window.event ? eve.wheelDelta : (eve.detail != 0 ? -eve.detail * 40 : eve.wheelDelta);
  //
  // Eseguo scrolling
  box.scrollTop -= delta / 2;
  //
  // Aggiorno lo stato delle immaginette di scrolling
  this.AdaptScrollBox();
  //
  RD3_Glb.StopEvent(ev);
  return false;
}



// ********************************************************************************
// Mostra o nasconde le form dello stack in base alla form attiva
// ********************************************************************************
WebEntryPoint.prototype.HandleFormStack = function ()
{
  // Se ci sono Form
  if (this.StackForm.length > 0)
  {
    var m = this.ActiveForm ? this.ActiveForm.IsModalPopup() : false;
    var d = this.ActiveForm ? this.ActiveForm.Docked : false;
    var mdi = this.ActiveForm ? this.ActiveForm.Modal == 0 : true;
    //
    var n = this.StackForm.length;
    for (var i = 0; i < n; i++)
    {
      var f = this.StackForm[i];
      //
      // Se la form e' da attivare la attivo, altrimenti la nascondo
      if (f == this.ActiveForm)
      {
        f.SetActive(true, true);
        this.ActiveForm.AlreadyVisible = true;
        //
        // Se la form attiva e' MDI la memorizzo come form di background
        if (f.Modal == 0 && !d)
          this.VisibleForm = f;
      }
      else
      {
        // Non la faccio sparire se quella che ho aperto e' popup
        f.SetActive(false, mdi);
      }
    }
  }
  //
  this.HandleCloseAllVisibility();
  this.AdaptScrollBox();
}


// ********************************************************************************
// Ritorna TRUE se il menu' e' laterale
// ********************************************************************************
WebEntryPoint.prototype.HasSideMenu = function ()
{
  return this.MenuType == RD3_Glb.MENUTYPE_RIGHTSB || this.MenuType == RD3_Glb.MENUTYPE_LEFTSB;
}


// ********************************************************************************
// Devo far apparire il menu' della taskbar
// ********************************************************************************
WebEntryPoint.prototype.OnStartClick = function (eve)
{
  
}


// *********************************************************
// Imposta il tooltip
// *********************************************************
WebEntryPoint.prototype.GetTooltip = function (tip, obj)
{
  if (obj == this.CloseAppBox)
  {
    tip.SetTitle(ClientMessages.TIP_TITLE_ChiudiAppl);
    tip.SetText(RD3_ServerParams.ChiudiAppl);
    tip.SetAnchor(RD3_Glb.GetScreenLeft(obj), RD3_Glb.GetScreenTop(obj) + (obj.offsetHeight / 2));
    tip.SetPosition(3);
    return true;
  }
  //
  if (obj == this.SuppressMenuBox)
  {
    if (this.CmdObj.SuppressMenu)
    {
      tip.SetTitle(ClientMessages.TIP_TITLE_MostraMenu);
      tip.SetText(RD3_ServerParams.MostraMenu);
    }
    else
    {
      tip.SetTitle(ClientMessages.TIP_TITLE_NascondiMenu);
      tip.SetText(RD3_ServerParams.NascondiMenu);
    }
    //
    tip.SetAnchor(RD3_Glb.GetScreenLeft(obj) + obj.offsetWidth, RD3_Glb.GetScreenTop(obj) + (obj.offsetHeight / 2));
    //
    if (this.MenuType == RD3_Glb.MENUTYPE_RIGHTSB)
      tip.SetPosition(3);
    else
      tip.SetPosition(1);
    //
    return true;
  }
  else if (obj == this.DebugImageBox)
  {
    // Attualmente non hanno un title
  }
  else if (obj == this.HelpFileBox)
  {
    // Attualmente non hanno un title
  }
  //
  return false;
}


// **********************************************************************
// Esegue un azione su un suono
// **********************************************************************
WebEntryPoint.prototype.SoundAction = function (soundname, action, volume, t1, t2, notify, progress, videodiv)
{
  
}


// **********************************************************************
// Esegue un azione su un suono usando HTML5
// **********************************************************************
WebEntryPoint.prototype.SoundAction5 = function (soundurl, action, volume, t1, t2, notify, progress, videodiv)
{
  
}


// **********************************************************************
// Ritorna l'oggetto audio per il file cercato
// **********************************************************************
function RD3_AudioPlay(ev)
{
  
}


// **********************************************************************
// Ritorna l'oggetto audio per il file cercato
// **********************************************************************
WebEntryPoint.prototype.GetSoundObject5 = function (soundurl, flcreate)
{
  if (!this.SoundObjects)
    this.SoundObjects = new HashTable(true);
  //
  var obj = this.SoundObjects[soundurl];
  if (!obj && flcreate)
  {
    obj = new Audio(soundurl);
    //obj.setAttribute("controls","true");
    obj.style.display = "none";
    obj.style.position = "absolute";
    obj.style.left = "0px";
    obj.style.top = "0px";
    obj.addEventListener("click", RD3_AudioPlay, true);
    document.body.appendChild(obj);
    this.SoundObjects[soundurl] = obj;
  }
  return obj;
}


// *********************************************************
// Ritorna l'insieme degli oggetti da tirare
// *********************************************************
WebEntryPoint.prototype.GetDropList = function (dragobj)
{
  var a = new Array();
  var dl = new Array();
  //
  // Vediamo di quali form devo calcolare gli oggetti da droppare
  if (this.ActiveForm && this.ActiveForm.IsModalPopup())
  {
    // Form popup modale... solo lei
    a.push(this.ActiveForm);
  }
  else
  {
    // Tutte le form visibili
    var n = this.StackForm.length;
    for (var i = 0; i < n; i++)
    {
      var f = this.StackForm[i];
      if (f && f.Realized && f.FormBox.style.display != "none")
      {
        a.push(f);
      }
    }
  }
  //
  // Tutti i frame di tutte le form dichiarate
  var n = a.length;
  for (var i = 0; i < n; i++)
  {
    a[i].ComputeDropList(dl, dragobj);
  }
  //
  this.CmdObj.ComputeDropList(dl, dragobj);
  //
  return dl;
}


// *********************************************************
// Gestione generic drop
// *********************************************************
WebEntryPoint.prototype.OnDrop = function (dropobj, dragobj, evento)
{
  if (!dropobj || !dragobj)
    return;
  //
  // Comincio ad identificare gli oggetti
  var iddropobj = dropobj.GetDOMObj().id;
  var iddragobj = dragobj.GetDOMObj().id;
  //
  // Identifico bottone
  var b = evento.button;
  if (RD3_Glb.IsIE())
  {
    if (b & 0x01)
      b = 0;
    else if (b & 0x02)
      b = 2;
    else if (b & 0x04)
      b = 1;
  }
  //
  // Coordinate browser
  var xb = evento.clientX;
  var yb = evento.clientY;
  var x = xb;
  var y = yb;
  //
  // In caso di pannello o tree, x e y sono rispetto al frame (+ eventuali scrollbar)
  if (dropobj.GetParentFrame)
  {
    var dropfr = dropobj.GetParentFrame();
    x = xb - RD3_Glb.GetScreenLeft(dropfr.FrameBox);
    y = yb - RD3_Glb.GetScreenTop(dropfr.FrameBox);
  }
  //
  // Convertiamo le coordinate rispetto alla box nella UM del book
  if (dropobj instanceof BookBox)
  {
    x = xb - RD3_Glb.GetScreenLeft(dropobj.BoxBox);
    y = yb - RD3_Glb.GetScreenTop(dropobj.BoxBox);
    var r = new Rect(x, y, 0, 0);
    dropobj.AdaptCoords(r);
    x = r.x;
    y = r.y;
  }
  //
  // Ho fatto il drag di una cella in lista: devo dire al pannello di cambiare riga e posizionarsi su quella draggata, in questo modo 
  // lato server posso prendere i dati giusti dalla riga sottesa
  if (dragobj instanceof PCell && dragobj.InList) {
    var pval = dragobj.PValue;
    var pan = dragobj.ParentField.ParentPanel;
    //
    // IN bootstrap i pannelli hanno sempre tutte righe (righe dinamiche, come nel mobile) quindi basta impostare il change sull'index -1 del PValue
    if (pval && pan && pan.PanelMode==RD3_Glb.PANEL_LIST)
      pan.ChangeActualRow(pval.Index - 1, evento);
  }
  //
  // Lancio evento
  var ev = new IDEvent("gdd", iddropobj, evento, RD3_Glb.EVENT_ACTIVE, iddragobj, b, Math.floor(xb), Math.floor(yb), Math.floor(x), null, false, Math.floor(y));
  return true;
}


// *********************************************************************************************
// Completa la chiusura di una form alla fine dell'animazione del GFX
// lastform : false se l'animazione fa comparire una nuova form, true se e' la chiusura dell'ultima form
// *********************************************************************************************
WebEntryPoint.prototype.CompleteCloseFormAnimation = function (webform, lastform)
{
  // Gestisco la chiusura di timer di form
  this.TimerObj.FormClosed(webform);
  //
  // Tolgo la form dallo stackform
  for (var i = 0; i < this.StackForm.length; i++)
  {
    if (this.StackForm[i] == webform)
    {
      this.StackForm.splice(i, 1);
      break;
    }
  }
  //
  // La distruggo
  webform.Unrealize();
  //
  // Se non e' entrata nessuna form devo annullare i puntatori alla form visibile
  if (lastform)
  {
    this.VisibleForm = null;
    this.ActiveForm = null;
  }
  //
  this.CloseFormId = "";
}

// *********************************************************************************************
// Tocchi finali della fine di un animazione riguardante una form MDI, modale o docked
// *********************************************************************************************
WebEntryPoint.prototype.HandleFinishFormAnimation = function (mustadapt)
{
  this.HandleFormStack();
  //
  // Gestisco il cambio di form attiva..
  this.CmdObj.ActiveFormChanged();
  this.IndObj.ActiveFormChanged();
  this.TimerObj.ActiveFormChanged();
  //
  var acf = this.ActiveForm;
  if (acf)
  {
    // Faccio l'adapt alla fine se sono arrivato qui saltando lo start e l'oggetto da animare non era gia' a video..
    if (mustadapt) {
      acf.RecalcLayout = true;
      acf.StartingFormAnimation();
    }
    //
    // Se il fuoco non e' gia' stato gestito in questa richiesta allora lo gestisco adesso
    if (!RD3_KBManager.DontCheckFocus && !RD3_Glb.IsMobile())
      RD3_DesktopManager.HandleFocus2(acf.Identifier, 0);
  }
}

WebEntryPoint.prototype.IsTransformable = function (id)
{
  if ((id == "forms-container" || id.indexOf("dock-container") != -1) && RD3_ServerParams.EnableFrameResize)
    return true;
  else
    return false;
}

WebEntryPoint.prototype.DragObj = function (id, obj, x, y)
{
  
}


// **************************************************
// L'oggetto da dimensionare e' il menu laterale
// **************************************************
WebEntryPoint.prototype.DropElement = function ()
{
  return this.SideMenuBox;
}


// **************************************************
// Applico il cursore del resize all'oggetto giusto
// **************************************************
WebEntryPoint.prototype.ApplyCursor = function (cn)
{
  
}

// ***************************************
// Metodi per gestire il tipo di resize
// ***************************************
WebEntryPoint.prototype.CanResizeW = function ()
{
  return true;
}
WebEntryPoint.prototype.CanResizeH = function ()
{
  return false;
}
WebEntryPoint.prototype.CanResizeL = function ()
{
  return true;
}
WebEntryPoint.prototype.CanResizeR = function ()
{
  return true;
}
WebEntryPoint.prototype.CanResizeT = function ()
{
  return false;
}
WebEntryPoint.prototype.CanResizeD = function ()
{
  return false;
}

// **************************************************
// Ridimensiono il menu laterale
// **************************************************
WebEntryPoint.prototype.OnTransform = function (x, y, w, h, evento)
{
  if (w < RD3_ClientParams.SideMenuMinWidth)
    w = RD3_ClientParams.SideMenuMinWidth;
  //
  // Per prima cosa accodo il messaggio di resize del menu: infatti l'impostazione della dimensione fa scattare i resize,
  // quindi dopo e' troppo tardi per inviare l'evento (finirebbe dopo i resize, mentre va gestito prima, cosi' negli eventi il server ha gia'
  // il valore corretto)
  var ev = new IDEvent("menures", this.Identifier, evento, RD3_Glb.EVENT_SERVERSIDE, w);
  this.SetSideMenuWidth(w);
}


// **************************************************
// Gestione del Refresh Interval
// **************************************************
WebEntryPoint.prototype.RefreshIntervalTick = function ()
{
  var ev = new IDEvent("rfi", this.Identifier, null, RD3_Glb.EVENT_ACTIVE);
}


// **************************************************
// Ritorna la prima docked del tipo considerato
// se flvisible=true, torna solo le dock visibili
// **************************************************
WebEntryPoint.prototype.GetDockedForm = function (docktype, flvisible)
{
  for (var i = 0; i < this.StackForm.length; i++)
  {
    var f = this.StackForm[i];
    if (f.Docked && f.DockType == docktype)
    {
      if (!flvisible || f.Visible)
        return f;
    }
  }
}


// **************************************************
// Esegue animazione forms
// **************************************************
WebEntryPoint.prototype.AnimateForms = function (frm1ident, frm2ident, fldx)
{
  var frm1 = RD3_DesktopManager.ObjectMap[frm1ident];
  var frm2 = RD3_DesktopManager.ObjectMap[frm2ident];
  //
  frm1.EnableIDScroll(false);
  frm2.EnableIDScroll(false);
  //
  frm1.FormBox.style.display = "";
  frm2.FormBox.style.display = "";
  RD3_Glb.SetTransitionProperty(frm1.FormBox, "-webkit-transform");
  RD3_Glb.SetTransitionProperty(frm2.FormBox, "-webkit-transform");
  RD3_Glb.SetTransitionDuration(frm1.FormBox, "0ms");
  RD3_Glb.SetTransitionDuration(frm2.FormBox, "0ms");
  //
  var pini = 0;
  var pfin = 0;
  var cini = 0;
  var cfin = 0;
  //
  if (fldx)
  {
    // frm1 a video, frm2 entra da destra
    f1ini = 0;
    f1fin = -this.FormsBox.offsetWidth;
    f2ini = this.FormsBox.offsetWidth;
    f2fin = 0;
  }
  else
  {
    // frm1 a video, frm2 entra da sinistra
    f1ini = -this.FormsBox.offsetWidth;
    f1fin = 0;
    f2ini = 0;
    f2fin = this.FormsBox.offsetWidth;
  }
  //
  // Posiziono gli elementi usando il 3d
  RD3_Glb.SetTransform(frm1.FormBox, "translate3d(" + f1ini + "px,0px,0px)");
  RD3_Glb.SetTransform(frm2.FormBox, "translate3d(" + f2ini + "px,0px,0px)");
  //
  // Eseguo l'animazione
  var sc = "RD3_Glb.SetTransitionDuration(document.getElementById('" + frm1.FormBox.id + "'), '250ms');";
  sc += "RD3_Glb.SetTransitionDuration(document.getElementById('" + frm2.FormBox.id + "'), '250ms');";
  sc += "RD3_Glb.SetTransform(document.getElementById('" + frm1.FormBox.id + "'), 'translate3d(" + f1fin + "px,0px,0px)');";
  sc += "RD3_Glb.SetTransform(document.getElementById('" + frm2.FormBox.id + "'), 'translate3d(" + f2fin + "px,0px,0px)');";
  //
  if (!this.ea)
    this.ea = new Function("ev", "return RD3_DesktopManager.CallEventHandler('" + this.Identifier + "', 'OnEndAnimation', ev)");
  RD3_Glb.AddEndTransaction(frm1.FormBox, this.ea, false);
  window.setTimeout(sc, 30);
  //
  this.AniForm1 = frm1;
  this.AniForm2 = frm2;
  this.AniMode = fldx;
}


// *******************************************************************
// Gestisce animazione combo mobile
// *******************************************************************
WebEntryPoint.prototype.OnEndAnimation = function (ev)
{
  if (RD3_Glb.GetTransitionDuration(this.AniForm1.FormBox) == "0ms")
    return;
  //
  RD3_Glb.RemoveEndTransaction(this.AniForm1.FormBox, this.ea, false);
  RD3_Glb.SetTransitionProperty(this.AniForm1.FormBox, "");
  RD3_Glb.SetTransitionProperty(this.AniForm2.FormBox, "");
  RD3_Glb.SetTransitionDuration(this.AniForm1.FormBox, "");
  RD3_Glb.SetTransitionDuration(this.AniForm2.FormBox, "");
  RD3_Glb.SetTransform(this.AniForm1.FormBox, "");
  RD3_Glb.SetTransform(this.AniForm2.FormBox, "");
  this.AniForm1.EnableIDScroll(true);
  this.AniForm2.EnableIDScroll(true);
  //
  if (this.AniMode)
  {
    // E' entrata la form 2 da destra, quindi form 1 deve essere deattivata
    this.ActivateForm(this.AniForm2, true);
  }
  else
  {
    // E' entrata la form 1 da sinistra, quindi form 2 deve essere chiusa
    this.CloseForm2(this.AniForm2);
    //
    // Siccome sono tornato indietro, i pannelli della form1 perdono hiliterow
    for (var i = 0; i < this.AniForm1.Frames.length; i++)
    {
      if (this.AniForm1.Frames[i] instanceof IDPanel)
        this.AniForm1.Frames[i].HiliteRow(0);
    }
  }
  //
  this.AniForm1 = undefined;
  this.AniForm2 = undefined;
  this.AniMode = undefined;
  //
  if (this.FormToActivate)
  {
    var f = this.FormToActivate;
    this.FormToActivate = null;
    this.ActivateForm(f);
  }
}

//*******************************************************************
// Gestione del bottone back per Android
//*******************************************************************
WebEntryPoint.prototype.OnBackButton = function ()
{
  
}

//********************************************************************
// Restituisce la ScreenZone corrispondente alla posizione richiesta
//********************************************************************
WebEntryPoint.prototype.GetScreenZone = function (zonePos)
{
  for (var i = 0; i < this.ScreenZones.length; i++)
  {
    var scz = this.ScreenZones[i];
    if (scz && scz.Position == zonePos)
      return scz;
  }
  //
  return null;
}


// ********************************************************************************
// Il browser e' stato ridimensionato
// ********************************************************************************
WebEntryPoint.prototype.OnIconClick = function (evento)
{
  if (this.IsIconActive())
    RD3_SendCommand("IconClick");
}

WebEntryPoint.prototype.OnTouchDown = function (ev)
{
  
}

WebEntryPoint.prototype.OnTouchMove = function (ev)
{
  
}

WebEntryPoint.prototype.OnTouchUp = function (ev)
{
  
}

/**
 * Le ScreenZone non sono supportate nel tema bootstrap
 */
WebEntryPoint.prototype.UseZones = function ()
{
  return false;
}

/**
 * Restituisce true se e' presente una docked destra
 */
WebEntryPoint.prototype.HasRightDocked = function ()
{
  var n = this.StackForm.length;
  for (var i = 0; i < n; i++)
  {
    var f = this.StackForm[i];
    if (f.Docked && f.DockType == RD3_Glb.FORMDOCK_RIGHT && f.Visible)
      return true;
  }
  //
  return false;
}

/**
 * Restituisce true se e' presente una docked destra
 */
WebEntryPoint.prototype.HasLeftDocked = function ()
{
  var n = this.StackForm.length;
  for (var i = 0; i < n; i++)
  {
    var f = this.StackForm[i];
    if (f.Docked && f.DockType == RD3_Glb.FORMDOCK_LEFT && f.Visible)
      return true;
  }
  //
  return false;
}

/**
 * Restituisce true se e' presente una docked sopra
 */
WebEntryPoint.prototype.HasTopDocked = function ()
{
  var n = this.StackForm.length;
  for (var i = 0; i < n; i++)
  {
    var f = this.StackForm[i];
    if (f.Docked && f.DockType == RD3_Glb.FORMDOCK_TOP && f.Visible)
      return true;
  }
  //
  return false;
}

/**
 * Restituisce true se e' presente una docked sopra
 */
WebEntryPoint.prototype.HasBottomDocked = function ()
{
  var n = this.StackForm.length;
  for (var i = 0; i < n; i++)
  {
    var f = this.StackForm[i];
    if (f.Docked && f.DockType == RD3_Glb.FORMDOCK_BOTTOM && f.Visible)
      return true;
  }
  //
  return false;
}

/**
 * Restituisce la larghezza in colonne della docked destra
 */
WebEntryPoint.prototype.GetRightDockedColumns = function ()
{
  var n = this.StackForm.length;
  for (var i = 0; i < n; i++)
  {
    var f = this.StackForm[i];
    if (f.Docked && f.DockType == RD3_Glb.FORMDOCK_RIGHT && f.Visible)
      return f.GetDockedColumns();
  }
  //
  return 0;
}

/**
 * Restituisce la larghezza in colonne della docked sinistra
 */
WebEntryPoint.prototype.GetLeftDockedColumns = function ()
{
  var n = this.StackForm.length;
  for (var i = 0; i < n; i++)
  {
    var f = this.StackForm[i];
    if (f.Docked && f.DockType == RD3_Glb.FORMDOCK_LEFT && f.Visible)
      return f.GetDockedColumns();
  }
  //
  return 0;
}

/**
 * Gestisce la visibilita' e la struttura dell'applicazione in base a WidgetMode/SuppressMenu/Menu Visibili
 */
WebEntryPoint.prototype.AdjustMenuVisibility = function () 
{
  if (this.HasSideMenu()) {
    // Gestisco il ToggleButton per il mobile
    this.toggleButton.style.display = (this.WidgetMode || this.CmdObj.SuppressMenu) ? "none" : "";
    this.toggleButton.style.minHeight = (this.WidgetMode || this.CmdObj.SuppressMenu) ? "5px" : "";
    this.toggleButton.style.minWidth = (this.WidgetMode || this.CmdObj.SuppressMenu) ? "5px" : "";
    //
    // Qui non devo considerare HasMenu perche' in laterale c'e' anche la lista delle videate, quindi anche se non ho menu' 
    // devo poter mostare lo stesso la parte laterale (quindi considero solo widgetMode e suppressmenu)
    this.SideMenuBox.style.display = (this.WidgetMode || this.CmdObj.SuppressMenu) ? "none" : "";
    if (this.WidgetMode || this.CmdObj.SuppressMenu)
      this.MainGridContainer.className = "col-xs-12 col-sm-12 col-md-12";
    else {
      this.MainGridContainer.className = "col-sm-12  main-side-" + (this.MenuType == RD3_Glb.MENUTYPE_LEFTSB ? "left" : "right");
      if (this.SideMenuWidth > 0) {
        if (this.MenuType == RD3_Glb.MENUTYPE_LEFTSB)
          this.MainGridContainer.style.paddingLeft = (this.SideMenuWidth + 30) + "px";
        else
          this.MainGridContainer.style.paddingRight = (this.SideMenuWidth + 30) + "px";
      }
    }
  }
  else {
    // Gestisco il ToggleButton per il mobile (in questo caso devo nasconderlo anche se non ci sono menu')
    this.toggleButton.style.display = (this.WidgetMode || this.CmdObj.SuppressMenu || !this.CmdObj.HasMenu()) ? "none" : "";
    this.toggleButton.style.minHeight = (this.WidgetMode || this.CmdObj.SuppressMenu || !this.CmdObj.HasMenu()) ? "5px" : "";
    this.toggleButton.style.minWidth = (this.WidgetMode || this.CmdObj.SuppressMenu || !this.CmdObj.HasMenu()) ? "5px" : "";  
  }
  //
  this.RecalcLayout = true;
}