/*
 * Instant Developer Cloud
 * Copyright Pro Gamma Spa 2000-2021
 * All rights reserved
 */
/* global moment */

var Client = Client || {};
/**
 * @class An html control (input, radio group, checkbox, html editor, etc.)
 * @param {Object} widget
 * @param {View|Element} parent - the parent element
 * @param {View} view
 */
Client.IdfControl = function (widget, parent, view)
{
  Client.Widget.call(this, widget, parent, view);
};


// Make Client.IdfControl extend Client.Widget
Client.IdfControl.prototype = new Client.Widget();


Client.IdfControl.stretches = {
  AUTO: 1,
  NONE: 2,
  FILL: 3,
  ENLARGE: 4,
  CROP: 5
};


Client.IdfControl.blobCommands = {
  UPLOAD: "upload",
  DELETE: "delete",
  VIEW: "view",
  OPEN: "open"
};


Client.IdfControl.nameSeparator = "; ";
Client.IdfControl.valueSeparator = ";";
Client.IdfControl.rValueSeparator = "@#@";


/**
 * Update element properties
 * @param {Object} props
 */
Client.IdfControl.prototype.updateElement = function (props)
{
  props = props || {};
  //
  Client.Widget.prototype.updateElement.call(this, props);
  //
  let update = {};
  //
  if (props.container !== undefined)
    this.container = props.container;
  //
  if (props.customElement !== undefined)
    this.customElement = props.customElement;
  //
  // This property is sent by IDC when one or more elements are added as field children
  if (props.customChildrenConf !== undefined) {
    this.customChildrenConf = props.customChildrenConf;
    update.customChildren = true;
  }
  //
  if (props.type !== undefined)
    this.type = props.type;
  //
  if (props.isPassword !== undefined)
    this.isPassword = props.isPassword;
  //
  if (props.heightResize !== undefined)
    this.heightResize = props.heightResize;
  //
  if (props.visualStyle !== undefined) {
    this.visualStyle = props.visualStyle;
    this.type = this.type || Client.IdfVisualStyle.getByIndex(this.visualStyle).getControlType();
    this.isPassword = this.isPassword || Client.IdfVisualStyle.getByIndex(this.visualStyle).getPasswordFlag();
    //
    // Mask can be set by visual style, so check if I have to update it
    if (this.getMask())
      update.mask = true;
  }
  //
  if (props.maxLength !== undefined) {
    this.maxLength = props.maxLength;
    update.maxLength = true;
    //
    // maxLength can modify mask, so if I there is a mask, update it
    if (this.getMask())
      update.mask = true;
  }
  //
  if (props.scale !== undefined) {
    this.scale = props.maxLength;
    //
    // scale can modify mask, so if I there is a mask, update it
    if (this.getMask())
      update.mask = true;
  }
  //
  if (props.optional !== undefined) {
    this.optional = props.optional;
    update.valueList = true;
  }
  //
  if (props.valueList !== undefined) {
    this.valueList = JSON.parse(JSON.stringify(props.valueList));
    update.valueList = true;
    //
    // If value list type is 0 it means server tell me to empty my value list
    if (this.valueList.type === 0) {
      update.value = true;
      update.skipOpenCombo = true;
    }
    //
    // When I receive a new value list, I have to update my value if I have a value source or if I'm a smart lookup
    if (this.hasValueSource || this.smartLookup)
      update.value = true;
  }
  //
  if (props.hasValueSource !== undefined)
    this.hasValueSource = props.hasValueSource;
  //
  if (props.smartLookup !== undefined)
    this.smartLookup = props.smartLookup;
  //
  if (props.autoLookup !== undefined)
    this.autoLookup = props.autoLookup;
  //
  if (props.editorType !== undefined)
    this.editorType = props.editorType;
  //
  if (props.enabled !== undefined) {
    this.enabled = props.enabled;
    update.enabled = true;
  }
  //
  if (props.isLookup !== undefined)
    this.isLookup = props.isLookup;
  //
  if (props.activatorImage !== undefined) {
    this.activatorImage = props.activatorImage;
    update.activatorImage = true;
  }
  //
  if (props.activatorWidth !== undefined) {
    this.activatorWidth = props.activatorWidth;
    update.activatorWidth = true;
  }
  //
  if (props.isClickable !== undefined) {
    this.isClickable = props.isClickable;
    update.isClickable = true;
  }
  //
  if (props.canActivate !== undefined) {
    this.canActivate = props.canActivate;
    update.activatorImage = true;
    update.isClickable = true;
  }
  //
  if (props.activableDisabled !== undefined) {
    this.activableDisabled = props.activableDisabled;
    update.activatorImage = true;
  }
  //
  if (props.superActive !== undefined)
    this.superActive = props.superActive;
  //
  if (props.isRowQbe !== undefined)
    this.isRowQbe = props.isRowQbe;
  //
  if (props.canSort !== undefined)
    this.canSort = props.canSort;
  //
  if (props.alignment !== undefined) {
    this.alignment = props.alignment;
    update.alignment = true;
  }
  //
  if (props.backColor !== undefined) {
    this.backColor = props.backColor;
    update.backColor = true;
    update.activatorBackColor = true;
  }
  //
  if (props.color !== undefined) {
    this.color = props.color;
    update.color = true;
  }
  //
  if (props.mask !== undefined) {
    this.mask = props.mask;
    update.mask = true;
  }
  //
  if (props.fontModifiers !== undefined) {
    this.fontModifiers = props.fontModifiers;
    update.fontModifiers = true;
  }
  //
  if (props.badge !== undefined) {
    this.badge = props.badge;
    update.badge = true;
  }
  //
  if (props.blobMime !== undefined)
    this.blobMime = props.blobMime;
  //
  if (props.htmlBlobMime !== undefined)
    this.htmlBlobMime = props.htmlBlobMime;
  //
  if (props.blobUrl !== undefined)
    this.blobUrl = props.blobUrl;
  //
  if (props.uploadBlobEnabled !== undefined) {
    this.uploadBlobEnabled = props.uploadBlobEnabled;
    update.blob = true;
  }
  //
  if (props.deleteBlobEnabled !== undefined) {
    this.deleteBlobEnabled = props.deleteBlobEnabled;
    update.blob = true;
  }
  //
  if (props.viewBlobEnabled !== undefined) {
    this.viewBlobEnabled = props.viewBlobEnabled;
    update.blob = true;
  }
  //
  if (props.image !== undefined) {
    this.image = props.image;
    update.image = true;
  }
  //
  if (props.imageResizeMode !== undefined) {
    this.imageResizeMode = props.imageResizeMode;
    update.imageResizeMode = true;
  }
  //
  if (props.multiUpload !== undefined)
    this.multiUpload = props.multiUpload;
  //
  if (props.uploadExtensions !== undefined)
    this.uploadExtensions = props.uploadExtensions;
  //
  if (props.value !== undefined) {
    this.value = props.value || "";
    update.value = true;
    //
    // If my parent field has a value source I always have to reset value list when value changes
    if (this.hasValueSource)
      this.valueList = undefined;
    //
    // If I'm a smart lookup:
    // 1) If server sends me "" or "LKENULL", it means I have to empty combo value and remove value list too
    // 2) If server sends me another value:
    //    - I could not have a value list yet, because server sends it on user event;
    //    - I could have a value list that didn't arrive from server
    //    In both cases create a value list having just one item containing current value
    if (this.smartLookup || this.hasValueSource) {
      if (this.value === "" || this.value === "LKENULL") {
        this.valueList = {items: [], clientSide: true};
        update.valueList = true;
        update.skipOpenCombo = true;
      }
      else if (!this.valueList || !this.valueList.items.length || this.valueList.clientSide) {
        this.valueList = {items: [], clientSide: true};
        //
        // In case of multiple value source, split its values and push them as value list items
        if (this.hasValueSource && this.isMultipleCombo()) {
          let newValues = this.value.replaceAll(this.getComboNameSeparator(), Client.IdfControl.valueSeparator);
          newValues = this.value.split(Client.IdfControl.valueSeparator);
          for (let i = 0; i < newValues.length; i++)
            this.valueList.items.push({name: newValues[i], value: newValues[i]});
        }
        else
          this.valueList.items.push({name: this.value, value: this.value});
        //
        update.valueList = true;
        update.skipOpenCombo = true;
      }
    }
  }
  //
  if (props.showHtmlEditorToolbar !== undefined) {
    this.showHtmlEditorToolbar = props.showHtmlEditorToolbar;
    update.showHtmlEditorToolbar = true;
  }
  //
  if (props.className !== undefined) {
    this.oldClassName = this.className;
    this.className = props.className;
    update.className = true;
  }
  //
  if (props.subFrameConf !== undefined) {
    // If my subFrame is changed, remove old subFrame
    if (this.subFrame && this.subFrame.id !== props.subFrameConf?.id) {
      this.subFrame.mainObjects[0].getRootObject().remove();
      delete this.subFrame;
    }
    //
    this.subFrameConf = props.subFrameConf;
    update.subFrame = true;
  }
  //
  if (props.placeholder !== undefined) {
    this.placeholder = props.placeholder;
    update.placeholder = true;
  }
  //
  if (props.numRows !== undefined)
    this.numRows = props.numRows;
  //
  if (props.filter !== undefined) {
    this.filter = props.filter;
    update.filter = true;
  }
  //
  if (props.isInQbe !== undefined) {
    this.isInQbe = props.isInQbe;
    update.qbeStatus = true;
    update.valueList = true;
  }
  //
  if (props.comboMultiSel !== undefined) {
    this.comboMultiSel = props.comboMultiSel;
    //
    // Update QBE status just if I'm in QBE
    if (this.isInQbe)
      update.qbeStatus = true;
  }
  //
  if (props.comboSeparator !== undefined) {
    this.comboSeparator = props.comboSeparator;
    //
    // Update QBE status just if I'm in QBE
    if (this.isInQbe)
      update.qbeStatus = true;
  }
  //
  if (props.openCombo !== undefined) {
    if (this.getType() === Client.IdfField.controlTypes.COMBO)
      this.activator?.getRootObject().click();
    return;
  }
  //
  // If there isn't a container yet, do nothing
  if (!this.container)
    return;
  //
  // Create control
  this.createControl();
  //
  if (this.customElement) {
    let customProps = {};
    for (let p in props) {
      if (this.customElement.customProps[p])
        customProps[p] = props[p];
    }
    this.control.updateElement(customProps);
    this.invokeCustomMethods();
  }
  //
  // Create activator
  this.createActivator();
  //
  // Create badge
  this.createBadge();
  //
  // Update control, activator and badge
  this.updateObjects(update);
  //
  // When these properties change I have to destroy and recreate control, so save their old values in order to compare new values with the old ones
  this.oldType = this.getType();
  this.oldAlignment = this.getAlignment();
  this.oldBlobMime = this.blobMime;
};


/**
 * Handle an event
 * @param {Object} event
 */
Client.IdfControl.prototype.onEvent = function (event)
{
  let events = [];
  //
  // Skip errorText event
  if (event.content?.name === "errorText")
    return events;
  //
  // If this event fired on activator, remember it
  event.activator = (event.obj === this.activator?.id);
  //
  let routeToParent;
  //
  if (this.customChildrenConf)
    routeToParent = true;
  else {
    switch (event.id) {
      case "onKey":
        routeToParent = this.handleKey(event);
        events.push(...this.handleKeyMovement(event));
        break;

      case "chgProp":
        routeToParent = this.handleChange(event);
        break;

      case "onClick":
        events.push(...this.handleClick(event));
        break;

      case "onDblclick":
        // If I have an activator, handle activator click locally
        if (this.activator)
          this.handleActivatorClick(event);
        break;

      case "onDragover":
        // I need to prevent dragover default behaviour otherwise ondrop doesn't fire
        if (event.obj === this.control.id)
          event.content.srcEvent.preventDefault();
        break;

      case "onDrop":
        if (event.obj === this.control.id) {
          // Use blob upload channel when user drop files on a multiupload control
          if (this.multiUpload) {
            let content = {command: Client.IdfControl.blobCommands.UPLOAD, files: event.content.srcEvent.dataTransfer.files};
            events.push(...this.parentWidget.onEvent({id: "onBlobCommand", obj: this.id, content}));
          }
          else if (this.isTextEdit())
            this.handlePaste(event);
        }
        break;

      case "onPaste":
        this.handlePaste(event);
        break;

      case "onFocusin":
      case "onFocusout":
        this.handleFocus(event);
        break;

      default:
        if (this.customElement?.events.includes(event.id))
          routeToParent = true;
        break;
    }
  }
  //
  if (!routeToParent)
    return events;
  //
  // Change event target so that parent can understand event comes from control
  event.obj = this.id;
  //
  // Route event to parent
  events.push(...this.parentWidget.onEvent(event));
  //
  return events;
};


/**
 * Handle onKey event
 * @param {Object} event
 */
Client.IdfControl.prototype.handleKey = function (event)
{
  let isEnterKey = event.content.keyCode === 13;
  //
  // pageUp, pageDown, end, home, left, up, right, down, del
  let isMovementKey = [33, 34, 35, 36, 37, 38, 39, 40, 45].includes(event.content.keyCode);
  //
  if (!this.enabled) {
    // In I'm not enabled, allow just enter, movement and tab keys
    if (!isEnterKey && !isMovementKey && event.content.keyCode !== 9)
      event.content.srcEvent.preventDefault();
    //
    return;
  }
  //
  if (event.content.type === "keydown") {
    // F2 =>
    if (event.content.keyCode === Client.mainFrame.wep?.FKActField + 111)
      return true;
    //
    // I need to handle "enter" key and maxLength manually just in case of EDIT having text as datatype
    if (!this.isTextEdit())
      return;
    //
    // ctrl/meta key + A or C or V or X or Z
    let isShortcutKey = (event.content.ctrlKey || event.content.metaKey) && [65, 67, 86, 88, 90].includes(event.content.keyCode);
    //
    // [backspace, del] + [pageUp, pageDown, end, home, left, up, right, down]
    let isSpecialKey = [8, 46].includes(event.content.keyCode) || isMovementKey;
    //
    // Not shortcut, not special and not enter
    let isNormalKey = !isShortcutKey && !isSpecialKey && !isEnterKey;
    //
    // In case of "enter" key, check if I have to prevent default
    if (isEnterKey) {
      if (this.numRows === 1 && !this.heightResize)
        event.content.srcEvent.preventDefault();
    }
    else if (isNormalKey) {
      // Since control for text datatype is an editable span, I have to handle max length manually
      if (this.maxLength > 0 && event.content.srcEvent.target.innerText.length >= this.maxLength)
        event.content.srcEvent.preventDefault();
    }
  }
  else if (event.content.type === "keyup") {
    // In case of "enter" key up on an HTML editor or on a control having numRows > 1, stop propagation and never route event to parent
    if (isEnterKey && (this.getType() === Client.IdfField.controlTypes.HTMLEDITOR || this.numRows > 1)) {
      event.content.srcEvent.stopPropagation();
      return;
    }
    //
    // Route key up event just in case of printable charachters into a text edit
    if (this.isTextEdit() && (event.content.srcEvent.key.length === 1 || [8, 32, 46].includes(event.content.keyCode))) {
      event.content.name = "value";
      return true;
    }
  }
};


/**
 * Handle key movement event
 * @@param {Object} event
 */
Client.IdfControl.prototype.handleKeyMovement = function (event)
{
  let events = [];
  if (event.content.type !== "keydown")
    return events;
  //
  let domObj = event.content.srcEvent.srcElement;
  let column = 0;
  let row = 0;
  switch (event.content.keyCode) {
    case 9: // Tab
      column = event.content.shiftKey ? -1 : 1;
      break;

    case 13: // Enter
      if (!Client.mainFrame.wep?.tabWithEnter)
        return events;
      //
      column = 1;
      break;

    case 37: // Left arrow
      if (event.content.shiftKey || event.content.ctrlKey || domObj.selectionStart !== 0)
        return events;
      //
      column = -1;
      break;

    case 39: // Right arrow
      if (event.content.shiftKey || event.content.ctrlKey || domObj.selectionEnd !== domObj.value.length)
        return events;
      //
      column = 1;
      break;

    case 38: // Up arrow
    case 40: // Down arrow
      if (event.content.shiftKey || event.content.ctrlKey || this.numRows > 1)
        return events;
      //
      row = event.content.keyCode === 38 ? -1 : 1;
      break;

    default:
      return events;
  }
  //
  event.content.srcEvent.preventDefault();
  events.push(...this.parentWidget.focusNearControl({control: this, column, row}));
  //
  return events;
};


/**
 * Handle chgProp event
 * @param {Object} event
 */
Client.IdfControl.prototype.handleChange = function (event)
{
  let type = this.getType();
  //
  if (event.content.name === "filter" && type === Client.IdfField.controlTypes.COMBO) {
    // Change on smartLookup has to be sent immediately if user clears combo value or if combo is open
    if (this.smartLookup && (event.content.value === "" || this.control.isOpen()))
      event.immediate = true;
    //
    // Check if I have to open combo on filter and avoid routing event to my parent:
    // - smartLookup must never open combo on filter;
    // - valueSource has to open combo if it's not superactive and combo is already open;
    // - valueList has to open combo if filter is not "".
    let isValueList = !this.smartLookup && !this.hasValueSource;
    let openValueSource = this.hasValueSource && !this.superActive && this.control.isOpen();
    //
    if ((isValueList && event.content.value !== "") || openValueSource) {
      this.openCombo(event.content.value);
      return;
    }
    //
    // If filter on a value list is "", close combo
    if (isValueList && event.content.value === "")
      this.control.closeCombo();
  }
  //
  // If click occurred on a checkbox, the chgProp event is fired with {name: "checked", value: true/false} as content.
  // Since my parent expects content of chgProp event to be {name: "value", value: "something"}, change it in this way
  if (type === Client.IdfField.controlTypes.CHECK && event.content.name === "checked") {
    event.content.name = "value";
    //
    // In QBE checkbox has three status: true, indeterminate and false.
    // If old status was "indeterminate", browser sends me "true" as new checked value because "indeterminate" means checkbox was not checked.
    // But since I want "false" after "indeterminate", I replace "true" value with "false"
    if (this.isInQbe && this.oldIndeterminateStatus)
      event.content.value = false;
    //
    // If I have a value list with 2 or more items, get first item value if checkbox is checked, otherwise get second item value
    if (this.valueList && this.valueList.items.length >= 2)
      event.content.value = event.content.value ? this.valueList.items[0].value : this.valueList.items[1].value;
    else
      event.content.value = event.content.value ? "on" : "";
    //
    // If old status was "true", browser sends me "false" as new checked value.
    // But since checkbox in QBE also has "indeterminate" status, I replace "false" value with the "indeterminate" one (i.e. "---").
    // In this way the writeValue method will set checked=null and indeterminate=true instead of checked=false. And so I obtain the third status
    if (this.isInQbe && this.oldCheckStatus)
      event.content.value = "---";
  }
  //
  if (event.content.name === "value") {
    // Combo in QBE sends its value just on combo close, except when value is "".
    if (this.isInQbe && type === Client.IdfField.controlTypes.COMBO) {
      if (event.content.value !== "" && event.content.value !== "LKENULL" && !event.comboClosed)
        return;
    }
    //
    // smartLookup or valueSource or superActive or customElement value changes have to be sent immediately
    event.immediate = (this.smartLookup || this.hasValueSource || this.superActive || this.customElement);
    //
    // If change occurred on blob/multiupload hidden input, route onBlobCommand (UPLOAD) to my parent passing uploaded files
    if ((this.dataType === Client.IdfField.dataTypes.BLOB || this.multiUpload) && this.supportControl && event.obj === this.supportControl.id) {
      event.id = "onBlobCommand";
      event.content = {command: Client.IdfControl.blobCommands.UPLOAD};
      event.content.files = this.supportControl.getRootObject().files;
    }
  }
  //
  return true;
};


/**
 * Handle onClick event
 * @param {Object} event
 */
Client.IdfControl.prototype.handleClick = function (event)
{
  let events = [];
  //
  // Handle activator click
  if (event.activator) {
    this.handleActivatorClick(event);
    return events;
  }
  //
  let blobCommand;
  //
  let uploadBlobClick = event.obj === this.uploadBlobConf?.id;
  let deleteBlobClick = event.obj === this.deleteBlobConf?.id;
  let viewBlobClick = event.obj === this.viewBlobConf?.id;
  let multiUploadClick = event.obj === this.control.id && this.multiUpload;
  //
  // If click occurred on upload blob button or on multi upload control, click on hidden input to open file browser
  if (uploadBlobClick || multiUploadClick) {
    let fileInput = this.supportControl.getRootObject();
    fileInput.click();
    return events;
  }
  else if (deleteBlobClick)
    blobCommand = Client.IdfControl.blobCommands.DELETE;
  else if (viewBlobClick)
    this.handleViewBlobClick(event);
  //
  // If click occurred on link of a SIZE blob, the blob command is OPEN
  let link = this.control?.elements?.[0]?.elements?.[0];
  if (link && event.obj === link.id && this.blobMime === Client.IdfFieldValue.blobMimeTypes.SIZE)
    this.handleOpenBlobClick(event);
  //
  if (blobCommand)
    events.push(...this.parentWidget.onEvent({id: "onBlobCommand", obj: this.id, content: {command: blobCommand}}));
  //
  return events;
};


/**
 * Handle click on view blob button
 * @param {Object} event
 */
Client.IdfControl.prototype.handleViewBlobClick = function (event)
{
  Client.mainFrame.wep.showPreview(Client.mainFrame.wep.SRV_MSG_ShowDoc, this.blobUrl.replace("&amp;", "&"));
};


/**
 * Handle click on open blob button
 * @param {Object} event
 */
Client.IdfControl.prototype.handleOpenBlobClick = function (event)
{
  if (this.blobMime === Client.IdfFieldValue.blobMimeTypes.TEXT)
    open().document.body.innerText = this.value;
  else
    Client.mainFrame.open({href: this.blobUrl.replace("&amp;", "&")});
};


/**
 * Handle onPaste event
 * @param {Object} event
 */
Client.IdfControl.prototype.handlePaste = function (event)
{
  if (!this.enabled)
    return event.content.srcEvent.preventDefault();
  //
  let maxLength = this.maxLength;
  let mask = this.getMask();
  //
  // If there are both maxLength and mask, get their minimum length as maxLength
  if (this.maxLength > 0 && mask)
    maxLength = Math.min(this.maxLength, mask.length);
  else if (mask) // Otherwise, if there is just mask, use its length as maxLength
    maxLength = mask.length;
  //
  // If there is no maxLength, I have to do nothing
  if (maxLength <= 0)
    return;
  //
  // Get source event
  let srcEvent = event.id === "onDrop" ? event.content.srcEvent : event.content;
  //
  // Get pasted text
  let clipboardData = event.id === "onDrop" ? srcEvent.dataTransfer : (srcEvent.clipboardData || window.clipboardData);
  let pastedText = clipboardData.getData("text/plain");
  //
  let rootObject = this.control.getRootObject();
  //
  // Get actual selection
  let selStart = rootObject.selectionStart;
  let selEnd = rootObject.selectionEnd;
  //
  // Remove selected text. It will be replaced by pasted text
  let oldText = rootObject.value;
  oldText = `${oldText.slice(0, selStart)}${oldText.slice(selEnd)}`;
  //
  // Get actual text length
  let textLength = oldText.length;
  //
  // If adding pasted text to actual text causes max length to be exceeded, truncate pasted text
  if (textLength + pastedText.length > maxLength)
    pastedText = pastedText.substr(0, maxLength - textLength);
  //
  // Prevent default paste
  srcEvent.preventDefault();
  //
  // If there is a pastedText, execute paste operation manually and restore old selection
  if (pastedText) {
    rootObject.value = `${oldText.slice(0, selStart)}${pastedText}${oldText.slice(selStart)}`;
    rootObject.selectionStart = selStart;
    rootObject.selectionEnd = selEnd;
  }
};


/**
 * Handle selection change event
 * @param {Object} event
 */
Client.IdfControl.prototype.handleSelectionChange = function (event)
{
  return this.parentWidget.handleSelectionChange(event);
};


/**
 * Create control configuration
 */
Client.IdfControl.prototype.createControlConfig = function ()
{
  let config;
  let supportControlConfig;
  //
  switch (this.getType()) {
    case Client.IdfField.controlTypes.EDIT:
      if (this.isTextEdit()) {
        let htmlClass = this.showHTML() ? " show-html" : "";
        config = {c: "IonText", type: "span", className: "control-span" + htmlClass, contentEditable: true, events: ["onClick", "onDblclick", "onPaste", "onDrop", "onDragover"]};
        //
        if (this.multiUpload)
          supportControlConfig = {c: "Input", type: "file", visible: false, events: ["onClick", "onChange"]};
      }
      else {
        config = {c: "IonInput", labelPosition: "hidden", className: "control-edit", events: ["onClick", "onDblclick", "onChange"]};
        //
        if (this.isPassword) {
          config.type = "password";
          config.events.push("onFocusin");
          config.events.push("onFocusout");
        }
        else {
          config.mask = this.getMask();
          //
          if (Client.IdfField.isDateOrTime(this.dataType)) {
            supportControlConfig = {c: "IonDateTime", usePicker: Client.IonDateTime.SHOW_PICKER_NEVER, displayFormat: this.getMask(true), style: {width: 0, height: "1px"}};
            config.maskType = "D";
          }
          else
            config.maskType = "N";
        }
      }
      break;

    case Client.IdfField.controlTypes.COMBO:
      config = {c: "IonAutoComplete", labelPosition: "hidden", backdrop: "none", showIcon: true, className: "control-edit", comboClass: "control-combo", list: this.getComboList(), events: ["onFilter", "onChange", "onClick", "onDblclick"]};
      break;

    case Client.IdfField.controlTypes.CHECK:
      config = {c: "IonCheckbox", className: "control-checkbox"};
      break;

    case Client.IdfField.controlTypes.OPTION:
      config = {c: "IonList", noLines: true, radioGroup: true, className: "control-radio-group", children: [], events: ["onChange", "onClick", "onDblclick"]};
      if (this.valueList) {
        for (let i = 0; i < this.valueList.items.length; i++) {
          let optionItem = {c: "IonItem", className: "control-radio-item", wrapper: false};
          optionItem.children = [{c: "IonRadio", label: this.valueList.items[i].name, value: this.valueList.items[i].value}];
          config.children.push(optionItem);
        }
      }
      break;

    case Client.IdfField.controlTypes.BUTTON:
      config = {c: "IonButton", className: "control-button", events: ["onClick"]};
      break;

    case Client.IdfField.controlTypes.HTMLEDITOR:
      config = {c: "HtmlEditor", className: "control-htmleditor", events: ["onChange"]};
      break;

    case Client.IdfField.controlTypes.CUSTOM:
      if (this.customElement)
        config = this.customElement.createConfig();
      break;

    case Client.IdfField.controlTypes.BLOB:
      config = this.createBlobConfig();
      supportControlConfig = {c: "Input", type: "file", visible: false, events: ["onClick", "onChange"]};
      break;

    case Client.IdfField.controlTypes.CHECKLIST:
      config = {c: "IonList", noLines: true, className: "control-radio-group", children: []};
      if (this.valueList) {
        for (i = 0; i < this.valueList.items.length; i++) {
          let optionItem = {c: "IonItem", className: "control-radio-item", wrapper: false};
          optionItem.children = [{c: "IonCheckbox", label: this.valueList.items[i].name}];
          config.children.push(optionItem);
        }
      }
      break;

    case Client.IdfField.controlTypes.LISTGROUPHEADER:
      config = {c: "IonText", type: "span", className: "control-span"};
      break;
  }
  //
  return {mainControlConfig: this.createElementConfig(config), supportControlConfig: supportControlConfig ? this.createElementConfig(supportControlConfig) : undefined};
};


/**
 * Create blob configuration
 */
Client.IdfControl.prototype.createBlobConfig = function ()
{
  let config = {c: "Container", className: "control-blob", children: [], events: ["onClick"]};
  //
  let blobContainerConfig;
  //
  switch (this.blobMime) {
    case Client.IdfFieldValue.blobMimeTypes.TEXT:
    case Client.IdfFieldValue.blobMimeTypes.IMAGE:
    case Client.IdfFieldValue.blobMimeTypes.EMPTY:
    case Client.IdfFieldValue.blobMimeTypes.SIZE:
      blobContainerConfig = {c: "Container", className: "control-blob-container", children: [], events: ["onClick"]};
      //
      // In case of SIZE mime type, add a child representing blob link
      if (this.blobMime === Client.IdfFieldValue.blobMimeTypes.SIZE) {
        blobContainerConfig.className += " control-blob-size";
        blobContainerConfig.children.push({c: "Container", events: ["onClick"]});
      }
      break;

    default:
      blobContainerConfig = {c: "Iframe", className: "control-blob-container", events: ["onClick"], frameBorder: "no"};
      break;
  }
  //
  config.children.push(blobContainerConfig);
  //
  return config;
};


/**
 * Write value on my element
 */
Client.IdfControl.prototype.writeValue = function ()
{
  switch (this.getType()) {
    case Client.IdfField.controlTypes.EDIT:
      this.writeEditValue();
      break;

    case Client.IdfField.controlTypes.COMBO:
      this.writeComboValue();
      break;

    case Client.IdfField.controlTypes.CHECK:
      this.writeCheckValue();
      break;

    case Client.IdfField.controlTypes.BUTTON:
      let {caption, icon} = Client.Widget.extractCaptionData(this.value);
      this.control.updateElement({label: caption, icon});
      break;

    case Client.IdfField.controlTypes.BLOB:
      this.writeBlobValue();
      break;

    case Client.IdfField.controlTypes.CHECKLIST:
      if (this.valueList) {
        let values = this.value.split(this.comboSeparator);
        //
        for (let i = 0; i < this.valueList.items.length; i++) {
          let val = values.find(item => item === this.valueList.items[i].value);
          let checkInput = this.control.elements[i].elements[0];
          checkInput.updateElement({checked: !!val});
        }
      }
      break;

    default:
      let props = {};
      if (this.customElement && !this.customElement.subFrameId) {
        // I push also a change of the default property
        let defaultBindingProperty = Client[this.customElement._class]?.defaultBindingProperty;
        if (defaultBindingProperty) {
          let isNumber = false;
          if (defaultBindingProperty in this.control)
            isNumber = typeof this.control[defaultBindingProperty] === "number";
          else
            isNumber = isNaN(this.value);
          //
          props[defaultBindingProperty] = isNumber ? new Number(this.value) : this.value;
        }
      }
      //
      props.value = this.value;
      this.control.updateElement(props);
      break;
  }
};


/**
 * Write value on EDIT
 */
Client.IdfControl.prototype.writeEditValue = function ()
{
  let props = {};
  let writeValue = true;
  //
  if (this.showHTML()) {
    props.value = "";
    props.innerHTML = this.getHTMLIcon(this.value);
  }
  else {
    let value = this.value;
    //
    if (Client.IdfField.isDateOrTime(this.dataType)) {
      if (!value || value === "Invalid date")
        value = "";
      else {
        // Update support control value
        let isoDateTime = this.getISODateTime(value);
        this.supportControl.updateElement({value: isoDateTime});
        //
        // If value is in ISO format I have to covert it to mask format
        if (value.includes("T"))
          value = moment(isoDateTime).format(this.getMask(true));
      }
    }
    else if (this.isPassword) {
      // In case of password, write value just if server sends me "********"
      writeValue = !!value;
      for (let i = 0; i < value.length; i++) {
        if (value[i] !== "*") {
          writeValue = false;
          break;
        }
      }
    }
    //
    if (writeValue)
      props.value = value;
  }
  //
  // Update control
  this.control.updateElement(props);
};


/**
 * Write value on COMBO
 */
Client.IdfControl.prototype.writeComboValue = function ()
{
  let props = {};
  //
  if (this.value === "")
    props.filter = "";
  //
  if (this.smartLookup) {
    // Change combo value just if value is valid
    let isValid = false;
    let newValue = [];
    //
    if (this.value !== undefined) {
      let oldValue = this.value === "*" ? this.control.value : this.value;
      //
      if (this.isMultipleCombo()) {
        oldValue = this.value.replaceAll(this.getComboNameSeparator(), Client.IdfControl.valueSeparator);
        oldValue = this.value.split(Client.IdfControl.valueSeparator);
      }
      //
      for (let i = 0; i < this.valueList.items.length; i++) {
        let item = this.valueList.items[i];
        //
        // In case of multiple combo (i.e. in qbe), when server sends the entire value list the old selected items are sent with a special value: LKE1001, LKE1002 and so on.
        // So retrieve these values into value list and use them as control new value
        if (this.value === "*" && this.isMultipleCombo()) {
          let itemValue = parseInt(item.value.replace("LKE", ""));
          if (itemValue > 1000) {
            newValue.push(item.value);
            isValid = true;
          }
        }
        else if ((item.value && oldValue.includes(item.value)) || (item.name && oldValue.includes(item.name))) {
          newValue.push(item.value);
          isValid = true;
        }
      }
    }
    //
    if (isValid)
      props.value = newValue.join(Client.IdfControl.valueSeparator);
    //
    if (this.value === "")
      props.value = "";
  }
  else
    props.value = this.value;
  //
  // Update control
  this.control.updateElement(props);
};


/**
 * Write value on CHECK
 */
Client.IdfControl.prototype.writeCheckValue = function ()
{
  let props = {};
  //
  if (this.value === "---") {
    props.indeterminate = true;
    props.checked = null;
  }
  else {
    props.indeterminate = false;
    //
    if (this.valueList && this.valueList.items.length >= 2)
      props.checked = this.value === this.valueList.items[0].value;
    else
      props.checked = this.value === "on";
  }
  //
  this.oldCheckStatus = props.checked;
  this.oldIndeterminateStatus = props.indeterminate;
  //
  // Update control
  this.control.updateElement(props);
};


/**
 * Write value on BLOB
 */
Client.IdfControl.prototype.writeBlobValue = function ()
{
  let props = {};
  let control = this.control.elements[0];
  //
  props.style = {backgroundImage: ""};
  props.innerHTML = "";
  props.src = "";
  switch (this.blobMime) {
    case Client.IdfFieldValue.blobMimeTypes.IMAGE:
      props.style = {backgroundImage: "url('" + this.value + "')"};
      break;

    case Client.IdfFieldValue.blobMimeTypes.SIZE:
      control = this.control.elements[0].elements[0];
      props.innerHTML = this.value;
      break;

    case Client.IdfFieldValue.blobMimeTypes.TEXT:
    case Client.IdfFieldValue.blobMimeTypes.EMPTY:
      props.innerHTML = this.value;
      break;

    default:
      props.src = this.value;
      break;
  }
  //
  // Update control
  control.updateElement(props);
};


/**
 * Create control
 */
Client.IdfControl.prototype.createControl = function ()
{
  let type = this.getType();
  //
  // If control already exists and its type or its blobMime are not changed, do nothing
  if (this.control && type === this.oldType && this.blobMime === this.oldBlobMime)
    return;
  //
  // I have to create control, so remove old control, if any
  if (this.control) {
    let index = this.mainObjects.findIndex(obj => obj.id === this.control.id);
    this.mainObjects.splice(index, 1);
    //
    this.container.removeChild(this.control);
    //
    // If there is an old support control, remove it
    if (this.supportControl) {
      index = this.mainObjects.findIndex(obj => obj.id === this.supportControl.id);
      this.mainObjects.splice(index, 1);
      this.container.removeChild(this.supportControl);
      //
      delete this.supportControl;
    }
  }
  //
  // Create new control
  let controlConf = this.createControlConfig();
  this.control = this.container.insertBefore({child: controlConf.mainControlConfig, sib: this.activator && this.getActivatorPosition() === "right" ? this.activator.id : undefined});
  this.elements.push(this.control);
  this.mainObjects.push(this.control);
  //
  this.invokeCustomMethods();
  //
  // If there is a support control, create it
  if (controlConf.supportControlConfig) {
    this.supportControl = this.container.insertBefore({child: controlConf.supportControlConfig});
    this.mainObjects.push(this.supportControl);
    //
    // In case of multi upload, set multiple property on file input
    if (this.multiUpload) {
      let fileInput = this.supportControl.getRootObject();
      fileInput.multiple = true;
    }
  }
  //
  // Enable key events
  this.control.enableKeyEvent({inputs: true, type: "down"});
  this.control.enableKeyEvent({inputs: true, type: "up"});
  //
  // If I'm a combo, override Client.IonAutoComplete.positionCombo in order to customize combo position and dimensions
  if (type === Client.IdfField.controlTypes.COMBO)
    this.adjustCombo();
  //
  // In case of text EDIT, control is a contenteditable span. But I want it to act like an input
  if (this.isTextEdit())
    this.simulateInput();
  //
  let update = {};
  update.enabled = true;
  update.blob = true;
  update.skipOpenCombo = true;
  update.isClickable = true;
  update.value = (this.value !== undefined);
  update.maxLength = (this.maxLength !== undefined);
  update.valueList = (this.valueList !== undefined);
  update.showHtmlEditorToolbar = (this.showHtmlEditorToolbar !== undefined);
  update.alignment = (this.getAlignment() !== undefined);
  update.backColor = (this.backColor !== undefined);
  update.color = (this.color !== undefined);
  update.mask = (this.getMask() !== undefined);
  update.fontModifiers = (this.fontModifiers !== undefined);
  update.image = (this.image !== undefined);
  update.imageResizeMode = (this.imageResizeMode !== undefined);
  update.className = (this.className !== undefined);
  update.filter = (this.filter !== undefined);
  update.qbeStatus = (this.isInQbe !== undefined);
  update.subFrame = (this.subFrame !== undefined);
  update.placeholder = (this.placeholder !== undefined);
  //
  this.updateObjects(update);
};


/**
 * Update blob control
 */
Client.IdfControl.prototype.updateBlobControl = function ()
{
  // Get overlay
  let blobOverlay = this.control.elements[1];
  //
  let overlayVisible = this.uploadBlobEnabled || this.deleteBlobEnabled || this.viewBlobEnabled;
  //
  // If blob has no enabled commands, remove blob overlay
  if (!overlayVisible) {
    // If it exists, remove it
    if (blobOverlay) {
      this.control.removeChild(blobOverlay);
      //
      delete this.uploadBlobConf;
      delete this.deleteBlobConf;
      delete this.viewBlobConf;
    }
    //
    return;
  }
  //
  if (!blobOverlay) {
    // Create overlay configuration
    let overlayConfig = this.createElementConfig({c: "Container", className: "control-blob-overlay", children: [], events: ["onClick"]});
    //
    // Create upload blob icon configuration
    let tooltip = Client.Widget.getHTMLTooltip(Client.IdfResources.t("TIP_TITLE_LoadDoc"), Client.mainFrame.wep ? Client.mainFrame.wep.SRV_MSG_LoadDoc : "");
    this.uploadBlobConf = this.createElementConfig({c: "IonButton", className: "control-blob-upload", icon: "download", tooltip, events: ["onClick"]});
    overlayConfig.children.push(this.uploadBlobConf);
    //
    // Create delete blob icon configuration
    tooltip = Client.Widget.getHTMLTooltip(Client.IdfResources.t("TIP_TITLE_DeleteDoc"), Client.mainFrame.wep ? Client.mainFrame.wep.SRV_MSG_DeleteDoc : "");
    this.deleteBlobConf = this.createElementConfig({c: "IonButton", className: "", icon: "trash", tooltip, events: ["onClick"]});
    overlayConfig.children.push(this.deleteBlobConf);
    //
    // Create view blob icon configuration
    tooltip = Client.Widget.getHTMLTooltip(Client.IdfResources.t("TIP_TITLE_ShowDoc"), Client.mainFrame.wep ? Client.mainFrame.wep.SRV_MSG_ShowDoc : "");
    this.viewBlobConf = this.createElementConfig({c: "IonButton", className: "", icon: "open", tooltip, events: ["onClick"]});
    overlayConfig.children.push(this.viewBlobConf);
    //
    // Create overlay as control child
    blobOverlay = this.control.insertBefore({child: overlayConfig});
  }
  //
  // Update buttons visibility
  let uploadButton = blobOverlay.elements[0];
  uploadButton.updateElement({visible: this.uploadBlobEnabled});
  //
  let deleteButton = blobOverlay.elements[1];
  deleteButton.updateElement({visible: this.deleteBlobEnabled});
  //
  let viewButton = blobOverlay.elements[2];
  viewButton.updateElement({visible: this.viewBlobEnabled});
  //
  // Update accepted extensions
  let fileInput = this.supportControl.getRootObject();
  fileInput.accept = (this.uploadExtensions !== "*.*") ? this.uploadExtensions : "";
};


/**
 * Update control enabled status
 */
Client.IdfControl.prototype.updateEnabled = function ()
{
  switch (this.getType()) {
    case Client.IdfField.controlTypes.EDIT:
    case Client.IdfField.controlTypes.COMBO:
      // Do nothing
      break;

    case Client.IdfField.controlTypes.OPTION:
      for (let i = 0; i < this.control.elements.length; i++) {
        let radioInput = this.control.elements[i].elements[0];
        radioInput.updateElement({disabled: !this.enabled});
      }
      break;

    default:
      this.control.updateElement({disabled: !this.enabled});
      break;
  }
};


/**
 * Create activator
 */
Client.IdfControl.prototype.createActivator = function ()
{
  let image = this.getActivatorImage();
  let needActivator = !!image && (this.activatorWidth === undefined || this.activatorWidth > 0);
  //
  if (!needActivator) {
    // If I had an activator, I don't need it anymore
    if (this.activator) {
      this.container.removeChild(this.activator);
      delete this.activator;
    }
    //
    return;
  }
  //
  // If I already had an activator and alignment is not changed, do nothing
  if (this.activator && this.getAlignment() === this.oldAlignment)
    return;
  //
  // I have to create activator, so remove the old one, if any
  if (this.activator) {
    let index = this.mainObjects.findIndex(obj => obj.id === this.activator.id);
    this.mainObjects.splice(index, 1);
    //
    this.container.removeChild(this.activator);
  }
  //
  let activatorConf = this.createElementConfig({c: "IonButton", className: "control-activator" + (this.isRowQbe ? " qbe" : "") + " " + this.getActivatorPosition(), events: ["onClick"]});
  this.activator = this.container.insertBefore({child: activatorConf, sib: this.getActivatorPosition() === "left" ? this.control.id : (this.badgeObj ? this.badgeObj.id : undefined)});
  this.mainObjects.push(this.activator);
  //
  let update = {
    activatorWidth: (this.activatorWidth !== undefined),
    activatorBackColor: (this.backColor !== undefined),
    activatorImage: true,
    badgeMarginRight: true
  };
  //
  this.updateObjects(update);
};


/**
 * Create badge
 */
Client.IdfControl.prototype.createBadge = function ()
{
  // A row qbe control doesn't need badge
  if (this.isRowQbe)
    return;
  //
  if (!this.badge) {
    // If I had a badge, I don't need it anymore
    if (this.badgeObj) {
      this.container.removeChild(this.badgeObj);
      delete this.badgeObj;
    }
    //
    return;
  }
  //
  // If I already have a badge, do nothing
  if (this.badgeObj)
    return;
  //
  // Create badge
  let badgeConf = this.createElementConfig({c: "IonBadge", className: "generic-badge" + (this.badgeInside ? " inside" : "")});
  this.badgeObj = this.container.insertBefore({child: badgeConf});
  this.mainObjects.push(this.badgeObj);
  //
  let update = {
    badge: true,
    badgeMarginRight: true
  };
  //
  this.updateObjects(update);
};


/**
 * Update control, activator and badge
 * @param {Object} update
 */
Client.IdfControl.prototype.updateObjects = function (update)
{
  let type = this.getType();
  //
  // Update blob control
  if (update.blob && type === Client.IdfField.controlTypes.BLOB)
    this.updateBlobControl();
  //
  // Update html editor toolbar
  if (update.showHtmlEditorToolbar && type === Client.IdfField.controlTypes.HTMLEDITOR)
    this.control.updateElement({className: "control-htmleditor" + (this.showHtmlEditorToolbar ? "" : " hide-buttons")});
  //
  // Update enabled state
  if (update.enabled)
    this.updateEnabled();
  //
  // Update max length
  if (update.maxLength && type === Client.IdfField.controlTypes.EDIT)
    this.control.updateElement({maxLength: this.maxLength});
  //
  // Update value list
  if (update.valueList && type === Client.IdfField.controlTypes.COMBO)
    this.control.updateElement({list: this.getComboList()});
  //
  // Write value, if it's changed
  if (update.value) {
    this.writeValue();
    //
    // If I'm a smart lookup or a value source (not row qbe), I have to open my combo on value list change
    if (update.valueList && (this.smartLookup || (this.hasValueSource && !this.isRowQbe)) && !update.skipOpenCombo)
      this.openCombo();
  }
  //
  // Update alignment
  if (update.alignment) {
    let alignment = type === Client.IdfField.controlTypes.CHECK ? Client.IdfVisualStyle.alignments.CENTER : this.getAlignment();
    let textAlign = Client.IdfVisualStyle.getTextAlign(alignment);
    this.control.updateElement({style: {textAlign: textAlign, justifyContent: textAlign}});
  }
  //
  // Update background color
  if (update.backColor) {
    if (type === Client.IdfField.controlTypes.EDIT || type === Client.IdfField.controlTypes.COMBO)
      this.control.getRootObject().style.backgroundColor = this.backColor;
    else
      this.control.updateElement({style: {backgroundColor: this.backColor}});
  }
  //
  // Update color
  if (update.color) {
    // In case of OPTION, I have to set color on each option
    if (type === Client.IdfField.controlTypes.OPTION) {
      for (let i = 0; i < this.control.elements.length; i++)
        this.control.elements[i].updateElement({style: {color: this.color}});
    }
    else
      this.control.updateElement({style: {color: this.color}});
  }
  //
  // Update mask
  if (update.mask) {
    // Use mask just for EDIT that are not in QBE
    if (type === Client.IdfField.controlTypes.EDIT)
      this.control.updateElement({mask: this.isInQbe ? "" : this.getMask()});
  }
  //
  // Update font modifiers
  if (update.fontModifiers) {
    let font = Client.IdfVisualStyle.getFont(this.fontModifiers, true);
    this.control.updateElement({style: {fontStyle: font.style, fontWeight: font.weight, textDecoration: font.decoration}});
  }
  //
  // Update image
  if (update.image)
    this.updateImage();
  //
  // Update image resize mode
  if (update.imageResizeMode)
    this.updateImageResizeMode();
  //
  // Update className
  if (update.className) {
    Client.Widget.updateElementClassName(this.control, this.oldClassName, true);
    Client.Widget.updateElementClassName(this.control, this.className);
  }
  //
  // Update filter
  if (update.filter)
    this.updateFilter();
  //
  // Update QBE status
  if (update.qbeStatus)
    this.updateQbeStatus();
  //
  // Update subframe
  if (update.subFrame) {
    let controlVisible = true;
    if (this.subFrameConf) {
      controlVisible = false;
      //
      let append = this.subFrame?.id !== this.subFrameConf.id;
      //
      this.subFrame = Client.eleMap[this.subFrameConf.id];
      if (!this.subFrame)
        this.subFrame = this.container.insertBefore({child: this.subFrameConf});
      //
      if (append)
        this.container.appendChildObject(this.subFrame.mainObjects[0], this.subFrame.mainObjects[0].getRootObject());
      //
      delete this.subFrameConf;
    }
    //
    // Hide control if there is a subFrame
    this.control.updateElement({visible: controlVisible});
  }
  //
  // Update custom children
  if (update.customChildren)
    this.updateCustomChildren();
  //
  // Update placeholder
  if (update.placeholder) {
    if ([Client.IdfField.controlTypes.EDIT,
      Client.IdfField.controlTypes.COMBO,
      Client.IdfField.controlTypes.HTMLEDITOR,
      Client.IdfField.controlTypes.CUSTOM].includes(type)) {
      if (this.isTextEdit())
        this.control.getRootObject().setAttribute("placeholder", this.placeholder);
      else
        this.control.updateElement({placeholder: this.placeholder});
    }
  }
  //
  // Add/remove clickable css class
  if (update.isClickable)
    Client.Widget.updateElementClassName(this.control, "control-clickable", !this.isClickable && !this.canActivate);
  //
  // Update activator
  if (this.activator) {
    // Update activator width
    if (update.activatorWidth)
      this.activator.updateElement({style: {width: this.activatorWidth + "px", height: this.activatorWidth + "px"}});
    //
    // Update image
    if (update.activatorImage)
      this.updateActivatorImage();
    //
    // Update background color
    if (update.activatorBackColor)
      this.activator.updateElement({style: {backgroundColor: this.backColor}});
  }
  //
  // Update badge
  if (this.badgeObj) {
    // Update badge
    if (update.badge)
      this.badgeObj.updateElement({innerText: this.badge});
    //
    // Update badge margin right
    if (update.badgeMarginRight || update.activatorWidth)
      this.updateBadgeMarginRight();
  }
};


/**
 * Update image
 */
Client.IdfControl.prototype.updateImage = function ()
{
  let type = this.getType();
  //
  let src = this.image ? (Client.mainFrame.isIDF ? "images/" : "") + this.image : "";
  let url = src ? "url('" + src + "')" : "";
  //
  // In case of EDIT or COMBO I cannot set backgroundImage using updateElement method
  // because this would set it on domObj (input) while I want backgrounImage to be set on outerObj (ion-input)
  if (type === Client.IdfField.controlTypes.EDIT || type === Client.IdfField.controlTypes.COMBO)
    this.control.getRootObject().style.backgroundImage = url;
  else
    this.control.updateElement({style: {backgroundImage: url}});
};


/**
 * Update image resize mode
 */
Client.IdfControl.prototype.updateImageResizeMode = function ()
{
  let type = this.getType();
  //
  let control = (type === Client.IdfField.controlTypes.BLOB) ? this.control.elements[0] : this.control;
  //
  let className = control.getRootObject().className || "";
  //
  // Remove old resize mode class
  className = className.split(" ").filter((c) => !c.startsWith("control-blob-img")).join(" ");
  //
  // Add new resize mode class
  className = className + " " + this.getImageResizeModeClass();
  //
  className = className.trim();
  //
  control.updateElement({className});
};


/**
 * Update activator image
 */
Client.IdfControl.prototype.updateActivatorImage = function ()
{
  let image = this.getActivatorImage();
  //
  Client.Widget.setIconImage(image, this.activator);
};


/**
 * Update badge margin right to avoid activator overlap
 */
Client.IdfControl.prototype.updateBadgeMarginRight = function ()
{
  // If there isn't an activator or it's not on right side or badge is not shown inside control, badge position is correct by default, so do nothing
  if (!this.activator || this.getActivatorPosition() !== "right" || !this.badgeInside)
    return;
  //
  let activatorWidth = this.activatorWidth || this.activator.getRootObject().getBoundingClientRect().width || 23;
  let marginRigth = parseInt(activatorWidth / 2) + 4;
  this.badgeObj.updateElement({style: {marginRight: marginRigth + "px"}});
};


/**
 * Update badge margin right to avoid activator overlap
 */
Client.IdfControl.prototype.updateCustomChildren = function ()
{
  this.customChildren = this.customChildren || [];
  //
  // Detach old custom children
  for (let i = 0; i < this.customChildren.length; i++) {
    this.customChildren[i].getRootObject().remove();
    this.customChildren.splice(i, 1);
    i--;
  }
  //
  if (this.customChildrenConf) {
    for (let i = 0; i < this.customChildrenConf.length; i++) {
      let customChildConf = this.customChildrenConf[i];
      //
      this.customChildren[i] = this.container.insertBefore({child: this.createElementConfig(customChildConf)});
      this.customChildren[i].parentWidget = this;
    }
  }
  //
  // Hide control if there are custom children
  this.control.updateElement({visible: false});
};


/**
 * Override some combo methods to adapt them to what I need
 */
Client.IdfControl.prototype.adjustCombo = function ()
{
  this.control.positionCombo = () => {
    Client.IonAutoComplete.prototype.positionCombo.call(this.control);
    //
    // Get my container's rects
    let containerRects = this.container.getRootObject().getBoundingClientRect();
    //
    // Adjust combo position and width
    this.control.comboObj.style.left = containerRects.left + "px";
    this.control.comboObj.style.width = containerRects.width > 0 ? "auto" : "0px";
    //
    // Check if I have to set combo background color, items color or items font modifier
    if (this.backColor !== undefined || this.color !== undefined || this.fontModifiers !== undefined) {
      if (!this.control.comboObj.children[0] || !this.control.comboObj.children[0].children[0])
        return;
      //
      // Get internal ion-list
      let comboList = this.control.comboObj.children[0].children[0];
      //
      // Set background color and/or color on items
      for (let i = 0; i < comboList.children.length; i++) {
        if (this.backColor !== undefined)
          comboList.children[i].style.backgroundColor = this.backColor;
        if (this.color !== undefined)
          comboList.children[i].style.color = this.color;
        if (this.fontModifiers !== undefined) {
          let font = Client.IdfVisualStyle.getFont(this.fontModifiers, true);
          //
          if (font.style)
            comboList.children[i].style.fontStyle = font.style;
          if (font.weight)
            comboList.children[i].style.fontWeight = font.weight;
          if (font.decoration)
            comboList.children[i].style.textDecoration = font.decoration;
        }
      }
    }
  };
  //
  this.control.setValue = (value, emitChange, ev) => {
    let updateValue = false;
    //
    // In case of multiple combo, if user selects an item and then the empty item I would have
    // "LKE2;LKENULL" as value. So if I found "LKENULL" into value, it becomes simply "LKENULL".
    if (this.isMultipleCombo() && value && value.indexOf("LKENULL") !== -1)
      value = "LKENULL";
    //
    let newValue = [];
    if (this.valueList && this.valueList.items.length && value) {
      let oldValue = value.split(Client.IdfControl.valueSeparator);
      //
      for (let i = 0; i < this.valueList.items.length; i++) {
        let item = this.valueList.items[i];
        if ((item.value && oldValue.includes(item.value)) || (item.name && oldValue.includes(item.name))) {
          newValue.push(item.value);
          updateValue = true;
        }
      }
    }
    else if (value === "")
      updateValue = true;
    //
    newValue = newValue.join(Client.IdfControl.valueSeparator);
    if (updateValue)
      Client.IonAutoComplete.prototype.setValue.call(this.control, newValue, emitChange, ev);
  };
  //
  this.control.isOpen = function () {
    return this.listObj?.style.opacity === "1";
  };
  //
  this.control.closeCombo = () => {
    Client.IonAutoComplete.prototype.closeCombo.call(this.control);
    //
    // A value source has to show its value when combo is closed and its name when combo is opened.
    // So on combo close I have to make control loose its value list (that cause control to show name)
    // in order to force it to recreate a value list containing just the value to show.
    // Since value source loose its value list on value change, update it now (even if its the same)
    if (this.hasValueSource && !this.isInQbe)
      this.updateElement({value: this.value});
    //
    // Combo in QBE sends its value just on combo close
    if (this.isInQbe) {
      Client.mainFrame.sendEvents([{
          id: "chgProp",
          obj: this.control.id,
          comboClosed: true,
          content: {
            name: "value",
            value: this.control.value
          }
        }]);
    }
  };
  //
  this.control.close = function (firstLevel) {
    if (this.comboObj) {
      this.comboObj.remove();
      if (this.combo2)
        this.combo2.remove();
    }
    this.closed = true;
    //
    Client.Element.prototype.close.call(this, firstLevel);
  };
  //
  this.control.onInputKeyUp = (ev) => {
    if (this.enabled)
      Client.IonAutoComplete.prototype.onInputKeyUp.call(this.control, ev);
  };
  //
  this.control.onInputKeyDown = (ev) => {
    if (this.enabled)
      Client.IonAutoComplete.prototype.onInputKeyDown.call(this.control, ev);
  };
};


/**
 * Check if this control shows html
 */
Client.IdfControl.prototype.showHTML = function ()
{
  return (this.isTextEdit() && (this.getMask() === "=" || this.isClickable || Client.Widget.extractCaptionData(this.value).icon));
};


/**
 * Get control type
 */
Client.IdfControl.prototype.getType = function ()
{
  // Use html editor control if parent field editor type is html editor
  if (this.editorType === Client.IdfField.editorTypes.HTMLEDITOR)
    return Client.IdfField.controlTypes.HTMLEDITOR;
  //
  // Use blob control if parent field data type is blob
  if (this.dataType === Client.IdfField.dataTypes.BLOB)
    return Client.IdfField.controlTypes.BLOB;
  //
  if (this.customElement)
    return Client.IdfField.controlTypes.CUSTOM;
  //
  let type = this.type;
  switch (type) {
    case Client.IdfField.controlTypes.AUTO:
      // If there is a value list or I have a value source or I'm a smart lookup, use combo
      if (this.valueList || this.hasValueSource || this.smartLookup)
        type = Client.IdfField.controlTypes.COMBO;
      else // Otherwise use edit
        type = Client.IdfField.controlTypes.EDIT;
      break;

    case Client.IdfField.controlTypes.CHECK:
      // A check without a value list becomes an edit
      if (!this.valueList)
        type = Client.IdfField.controlTypes.EDIT;
      break;
  }
  //
  return type;
};


/**
 * Return activator image
 */
Client.IdfControl.prototype.getActivatorImage = function ()
{
  if (this.isRowQbe) {
    if (this.canSort || this.enabled)
      return this.qbeFilter ? "funnel" : "funnel";
    else
      return "";
  }
  //
  // Just AUTO, EDIT and COMBO control type can have an activator
  let controlsTypes = [
    Client.IdfField.controlTypes.EDIT,
    Client.IdfField.controlTypes.COMBO
  ];
  if (controlsTypes.indexOf(this.getType()) === -1)
    return "";
  //
  // A negative activator width means I don't want the activator image to be shown
  if (this.activatorWidth <= 0)
    return "";
  //
  // If I'm disabled and I don't have to show icon when disabled and I cannot activate or I'm not activable when disabled, don't show activator
  if (!this.enabled && !Client.mainFrame.wep?.showDisabledIcons && (!this.canActivate || !this.activableDisabled))
    return "";
  //
  // If there is a custom activator image, get it
  if (this.activatorImage)
    return this.activatorImage;
  //
  // If there is a value list or I have a value source or I'm a smart lookup or I'm a lookup and I have to show icon, get dropdown ion-icon as activator
  if (this.valueList || this.hasValueSource || this.smartLookup || (this.isLookup && Client.mainFrame.wep?.showSmartLookupIcon))
    return "arrow-dropdown";
  //
  // If parent field is date or time, show specific activator just in case they are not activable
  if (Client.IdfField.isDateOrTime(this.dataType) && !this.canActivate) {
    // In QBE don't show activator for datetime field
    if ((this.isInQbe && this.dataType === Client.IdfField.dataTypes.DATETIME))
      return "";
    else if (Client.IdfField.isDate(this.dataType)) // Get calendar ion-icon for date and datetime field
      return "calendar";
    else if (this.dataType === Client.IdfField.dataTypes.TIME) // Get time ion-icon for time field
      return "time";
  }
  //
  // If can activate, get more ion-icon
  if (this.canActivate)
    return "more";
  //
  // No activator required
  return "";
};


/**
 * Return activator position
 */
Client.IdfControl.prototype.getActivatorPosition = function ()
{
  let position = "right";
  //
  // Check if I have to position activator on right
  if (Client.mainFrame.wep && Client.mainFrame.wep.rightAlignedIcons)
    return position;
  //
  // In case of row QBE, activator position is right
  if (this.isRowQbe)
    return position;
  //
  // Get my alignment
  let alignment = this.getAlignment();
  //
  // If alignment is AUTO, it's RIGHT for numeric fields, left for other types of fields
  if (alignment === Client.IdfVisualStyle.alignments.AUTO)
    alignment = Client.IdfField.isNumeric(this.dataType) ? Client.IdfVisualStyle.alignments.RIGHT : Client.IdfVisualStyle.alignments.LEFT;
  //
  // If alignment is RIGHT, activator position is left
  if (alignment === Client.IdfVisualStyle.alignments.RIGHT)
    position = "left";
  //
  return position;
};


/**
 * Get alignment
 */
Client.IdfControl.prototype.getAlignment = function ()
{
  if (this.alignment !== -1)
    return this.alignment;
  //
  return Client.IdfVisualStyle.getByIndex(this.visualStyle).getAlignment();
};


/**
 * Get image resize mode class
 */
Client.IdfControl.prototype.getImageResizeModeClass = function ()
{
  let className = "";
  //
  switch (this.imageResizeMode) {
    case Client.IdfControl.stretches.FILL:
      className = "control-blob-img-fill";
      break;

    case Client.IdfControl.stretches.ENLARGE:
      className = "control-blob-img-enlarge";
      break;

    case Client.IdfControl.stretches.CROP:
      className = "control-blob-img-crop";
      break;

    default:
      if (this.getType() === Client.IdfField.controlTypes.BLOB)
        className = "control-blob-img";
      break;
  }
  //
  return className;
};


/**
 * Get mask
 * @param {Boolean} upperCase
 */
Client.IdfControl.prototype.getMask = function (upperCase)
{
  let mask = this.mask || Client.IdfVisualStyle.getByIndex(this.visualStyle)?.getMask();
  //
  // Get default mask if needed
  if (!mask) {
    if (Client.IdfField.isDateOrTime(this.dataType))
      mask = this.getDateTimeMask(upperCase);
    else if (Client.IdfField.isNumeric(this.dataType))
      mask = this.getNumericMask();
  }
  //
  // Adapt mask to max length and scale
  return this.adaptMask(mask);
};


/**
 * Adapt given mask to max length and scale
 * @param {String} mask
 */
Client.IdfControl.prototype.adaptMask = function (mask)
{
  if (!mask)
    return mask;
  //
  if (Client.IdfField.isNumeric(this.dataType)) {
    if (this.dataType !== Client.IdfField.dataTypes.FLOAT) {
      // Get decimal separator position
      let sepIndex = mask.lastIndexOf(".");
      //
      // Calculate scale
      // Scale for currency type is mask's decimal digits
      // Scale for integer type is 0 (no decimal digits)
      let scale = this.scale;
      if (this.dataType === Client.IdfField.dataTypes.CURRENCY)
        scale = mask.length - sepIndex - 1;
      if (this.dataType === Client.IdfField.dataTypes.INTEGER)
        scale = 0;
      //
      // Adapt integer part using scale and max length. I eventually have to cut off some integer digits
      let integerDigits = this.maxLength - scale;
      if (integerDigits > 0) {
        if (sepIndex === -1)
          sepIndex = mask.length;
        //
        for (let i = sepIndex - 1; i >= 0; i--) {
          let ch = mask.charAt(i);
          //
          // If current character is not an integer digit placeholder, continue
          if (ch !== "0" && ch !== "#")
            continue;
          //
          integerDigits--;
          //
          // If I reached the maximum allowed integer digits, truncate mask
          if (integerDigits === 0 && i > 0) {
            mask = mask.substr(i);
            break;
          }
        }
      }
      //
      // Adapt decimal part cutting off unnecessary decimal digits
      // Get decimal separator position (mask may has changed)
      sepIndex = mask.lastIndexOf(".");
      //
      if (sepIndex !== -1 && scale >= 0) {
        // If there is "." without decimal digits, cut off decimal part
        if (scale === 0)
          mask = mask.substring(0, sepIndex - 1);
        else {
          for (let i = sepIndex + 1; i < mask.length; i++) {
            let ch = mask.charAt(i);
            //
            // If current character is not a decimal digit placeholder, continue
            if (ch !== "0" && ch !== "#")
              continue;
            //
            scale--;
            //
            // If I reached the allowed decimal digits, truncate mask
            if (scale === 0) {
              mask = mask.substring(0, i + 1);
              break;
            }
          }
        }
      }
    }
    //
    // Replace "0" with "#" in integer part
    for (let i = 0; i < mask.length; i++) {
      let ch = mask.charAt(i);
      if (ch === ".")
        break;
      if (ch === "0")
        mask = mask.substr(0, i) + "#" + mask.substring(i + 1);
    }
    //
    // Change decimal separator if needed
    if (!Client.mainFrame.wep?.decimalDot) {
      let sepIndex = mask.lastIndexOf(".");
      mask = mask.replace(/,/g, ".");
      if (sepIndex !== -1)
        mask = mask.substr(0, sepIndex) + "," + mask.substring(sepIndex + 1);
    }
  }
  else if (Client.IdfField.isText(this.dataType)) { // Truncate max if needed
    if (mask.length > this.maxLength)
      mask = mask.substr(0, this.maxLength);
  }
  //
  return mask;
};


/**
 * Get value
 * @param {String} value
 */
Client.IdfControl.prototype.getValueToSend = function (value)
{
  let type = this.getType();
  //
  // If given value is valid, that is the value to send.
  // Instead, checkbox and date/time have always to ask internal control for value to send
  if (value !== undefined && type !== Client.IdfField.controlTypes.CHECK && !Client.IdfField.isDateOrTime(this.dataType) && !Client.IdfField.isNumeric(this.dataType))
    return value;
  //
  // I don't have a valid value: ask internal control for value to send
  switch (type) {
    case Client.IdfField.controlTypes.COMBO:
      value = this.smartLookup ? this.control.lastChange : this.control.value;
      break;

    case Client.IdfField.controlTypes.CHECK:
      let rootObj = this.control.getRootObject();
      if (rootObj.indeterminate)
        value = this.isRowQbe ? "" : "---";
      else if (rootObj.checked)
        value = this.isRowQbe ? this.valueList.items[0].value : "on";
      else
        value = this.isRowQbe ? this.valueList.items[1].value : "";
      break;

    case Client.IdfField.controlTypes.CHECKLIST:
      value = [];
      for (let i = 0; i < this.control.elements.length; i++) {
        let checkInput = this.control.elements[i].elements[0].getRootObject();
        //
        if (checkInput.checked)
          value.push(this.valueList.items[i].value);
      }
      break;

    case Client.IdfField.controlTypes.EDIT:
      if (Client.IdfField.isDateOrTime(this.dataType)) {
        value = value !== undefined ? value : this.control.value;
        if (!value || value === "Invalid date")
          value = "";
        else if (!this.isInQbe || value.includes("T"))
          value = moment(this.getISODateTime(value)).format(this.getMask(true));
      }
      else if (Client.IdfField.isNumeric(this.dataType))
        value = this.control.domObj.value;
      else if (this.isTextEdit())
        value = this.control.getRootObject().value || "";
      else
        value = this.control.value || "";
      //
      break;

    default:
      value = this.control.value || "";
      break;
  }
  //
  return value;
};


/**
 * Update QBE status
 */
Client.IdfControl.prototype.updateQbeStatus = function ()
{
  let props = {};
  //
  let type = this.getType();
  //
  switch (type) {
    case Client.IdfField.controlTypes.COMBO:
      props.multiple = this.isMultipleCombo();
      props.nameSeparator = this.getComboNameSeparator();
      props.valueSeparator = Client.IdfControl.valueSeparator;
      break;

    case Client.IdfField.controlTypes.EDIT:
      props.mask = this.isInQbe ? "" : this.getMask();
      break;
  }
  //
  this.control.updateElement(props);
};


/**
 * Update filter
 */
Client.IdfControl.prototype.updateFilter = function ()
{
  let type = this.getType();
  //
  switch (type) {
    case Client.IdfField.controlTypes.CHECKLIST:
      for (let i = 0; i < this.control.elements.length; i++) {
        let checkItem = this.control.elements[i];
        let valueListItem = this.valueList.items[i];
        //
        let visible = (!this.filter || valueListItem.name.toLowerCase().indexOf(this.filter) !== -1);
        checkItem.updateElement({visible});
      }
      break;
  }
};


/**
 * Open combo
 * @param {String} filter
 */
Client.IdfControl.prototype.openCombo = function (filter)
{
  filter = filter || "";
  //
  this.control.fullCombo = true;
  this.control.openCombo(filter, true);
};


/**
 * Get list to populate combo
 */
Client.IdfControl.prototype.getComboList = function ()
{
  let list = [];
  if (!this.valueList)
    return list;
  //
  let columnsLength = [];
  let useHtml = false;
  //
  // If I'm optional or I'm in QBE, add an empty item as first item (if does not exist yet)
  if ((this.optional || this.isInQbe) && this.valueList.items && this.valueList.items[0] && this.valueList.items[0].value !== "")
    this.valueList.items.splice(0, 0, {name: "", value: "", enabled: true});
  //
  // If value list has headers, I have to use html items
  if (this.valueList.headers) {
    useHtml = true;
    //
    // Since I want to create a table structure, I have to know which is the width to give to each column.
    // So calculate the max length (in terms of chars) of each column
    //
    // Start with header columns
    let cols = this.valueList.headers.split("|");
    for (let i = 0; i < cols.length; i++)
      columnsLength[i] = cols[i].length;
    //
    // Then go with items columns
    for (let i = 0; i < this.valueList.items.length; i++) {
      let item = this.valueList.items[i];
      //
      cols = item.name.split("|");
      for (let j = 0; j < cols.length; j++) {
        if (cols[j].length > columnsLength[j])
          columnsLength[j] = cols[j].length;
      }
    }
    //
    // Create combo header item
    let headerItem = this.createComboItem({name: this.valueList.headers, value: this.valueList.headers, s: "combo-header", useHtml: true, columnsLength: columnsLength, fixed: true});
    list.push(headerItem);
  }
  //
  let actualGroup;
  for (let i = 0; i < this.valueList.items.length; i++) {
    let item = this.valueList.items[i];
    //
    // If item belongs to a group and I didn't create group item yet, create it now
    if (item.group && item.group !== actualGroup) {
      let groupItem = this.createComboItem({name: item.group, value: item.group, s: "combo-group-header"});
      list.push(groupItem);
    }
    //
    // item object has already the right properties, but I want to add "useHtml" and "columnsLength".
    // So clone item attaching these two properties and use it to create a combo item
    let itemCopy = Object.assign({useHtml: useHtml, columnsLength: columnsLength}, item);
    let listItem = this.createComboItem(itemCopy);
    list.push(listItem);
    //
    actualGroup = item.group;
  }
  //
  return list;
};


/**
 * Create a combo item
 * @param {Object} itemObj
 */
Client.IdfControl.prototype.createComboItem = function (itemObj)
{
  // If name is "" use " " instead, otherwise IonAutoComplete won't create item
  let itemName = itemObj.name || " ";
  let itemHtml;
  let itemSrc;
  //
  if (Client.Widget.extractCaptionData(itemName).icon && !itemObj.useHtml && !itemObj.image) {
    let {caption, icon} = Client.Widget.extractCaptionData(itemName);
    itemName = caption;
    itemSrc = (icon.indexOf("fa ") === 0 ? "fai:" : "ion:") + icon;
  }
  //
  // Get src if I have an image
  if (itemObj.image)
    itemSrc = Client.Widget.isIconImage(itemObj.image) ? (itemObj.image.indexOf("fa ") === 0 ? "fai:" : "ion:") + itemObj.image : (Client.mainFrame.isIDF ? "images/" : "") + itemObj.image;
  //
  // If item content has to be html, create a "div" row and its "div" columns
  if (itemObj.useHtml) {
    let cols = itemName.split("|");
    //
    itemHtml = "<div style='display: flex;'>";
    for (let i = 0; i < cols.length; i++) {
      let width = (i !== cols.length - 1 ? (itemObj.columnsLength[i] * 7) + "px" : "auto");
      itemHtml += "<div style='width:" + width + ";'>" + cols[i] + "</div>";
    }
    itemHtml += "</div>";
    //
    // Use decode column as item name
    itemName = cols[this.valueList.decodeColumn - 1];
  }
  //
  let itemClass = itemObj.s;
  if (!itemClass) {
    if (itemObj.value === "" || itemObj.value === "LKENULL")
      itemClass = "combo-item-empty";
    else
      itemClass = itemObj.enabled ? "combo-item" : "combo-item-disabled";
  }
  //
  let item = {};
  item.n = itemName;
  item.v = itemObj.value;
  item.disabled = (itemObj.enabled === false); // undefined is true
  item.s = itemClass;
  item.src = itemSrc;
  item.html = itemHtml;
  item.fixed = itemObj.fixed;
  //
  return item;
};


/**
 * Get combo name separator
 */
Client.IdfControl.prototype.getComboNameSeparator = function ()
{
  return this.comboSeparator || (Client.mainFrame.isIDF ? Client.mainFrame.wep.comboNameSeparator : Client.IdfControl.nameSeparator);
};


/**
 * Get value from rValue
 * @param {String} rValue
 */
Client.IdfControl.prototype.getComboValueFromRValue = function (rValue)
{
  let value = "";
  //
  if (!this.valueList || !rValue)
    return value;
  //
  let rValuesArray = rValue.split(Client.IdfControl.rValueSeparator);
  //
  for (let i = 0; i < rValuesArray.length; i++) {
    let item = this.valueList.items.find(item => item.rValue === rValuesArray[i]);
    if (item)
      value += (i > 0 ? this.getComboNameSeparator() : "") + item.value;
  }
  //
  return value;
};


/**
 * Get rValue from value
 * @param {String} value
 */
Client.IdfControl.prototype.getComboRValueFromValue = function (value)
{
  let rValue = "";
  //
  if (!this.valueList || !value)
    return rValue;
  //
  let valuesArray = value.split(this.getComboNameSeparator());
  //
  for (let i = 0; i < valuesArray.length; i++) {
    let item = this.valueList.items.find(item => item.value === valuesArray[i]);
    if (item)
      rValue += (i > 0 ? Client.IdfControl.rValueSeparator : "") + item.rValue;
  }
  //
  return rValue;
};


/**
 * Return true if combo is multiple
 */
Client.IdfControl.prototype.isMultipleCombo = function ()
{
  return this.isInQbe && this.comboMultiSel;
};


/**
 * Get date time mask
 * @param {Boolean} upperCase
 */
Client.IdfControl.prototype.getDateTimeMask = function (upperCase)
{
  let mask = "";
  //
  let dateMask = Client.mainFrame.wep?.dateMask || "";
  let timeMask = Client.mainFrame.wep?.timeMask || "";
  //
  if (upperCase) {
    dateMask = dateMask.toUpperCase();
    timeMask = timeMask.toUpperCase();
    timeMask = timeMask.replace("nn", "mm").replace("NN", "mm");
  }
  //
  let dateTimeMask = Client.mainFrame.isIDF ? dateMask + " " + timeMask : "";
  //
  switch (this.dataType) {
    case Client.IdfField.dataTypes.DATE:
      mask = dateMask;
      break;

    case Client.IdfField.dataTypes.TIME:
      mask = timeMask;
      break;

    case Client.IdfField.dataTypes.DATETIME:
      mask = dateTimeMask;
      break;
  }
  //
  return mask;
};


/**
 * Get numeric mask
 */
Client.IdfControl.prototype.getNumericMask = function ()
{
  let mask = "";
  //
  switch (this.dataType) {
    case Client.IdfField.dataTypes.CURRENCY:
    case Client.IdfField.dataTypes.DECIMAL:
      mask = Client.mainFrame.isIDF ? Client.mainFrame.wep.currencyMask : "";
      break;

    case Client.IdfField.dataTypes.FLOAT:
      mask = Client.mainFrame.isIDF ? Client.mainFrame.wep.floatMask : "";
      break;

    case Client.IdfField.dataTypes.INTEGER: // DT_INTEGER
      //
      // Maschera per gli interi: # per maxlength-1 seguiti da uno 0
      for (let t = 0; t < this.maxLength - 1; t++)
        mask += "#";
      mask += "0";
      break;
  }
  //
  return mask;
};


/**
 * Get ISO format for current value
 * @param {String} value
 */
Client.IdfControl.prototype.getISODateTime = function (value)
{
  let mask;
  //
  if (!value.includes("T")) {
    mask = this.getMask(true);
    //
    if (this.dataType === Client.IdfField.dataTypes.TIME) {
      value = "1970-01-01T" + value;
      mask = "YYYY/MM/DDT" + mask;
    }
  }
  //
  return moment(value, mask).toISOString();
};


/**
 * Handle activator click
 * @@param {Object} event
 */
Client.IdfControl.prototype.handleActivatorClick = function (event)
{
  // In case of row qbe or double, always route activator click to parent
  if (this.isRowQbe || (event.id === "onDblclick"))
    return;
  //
  let type = this.getType();
  //
  // If I'm a combo but not smart lookup nor value source, open combo immediately and don't route event to my parent
  if (type === Client.IdfField.controlTypes.COMBO && !this.smartLookup && !this.hasValueSource) {
    this.openCombo();
    return;
  }
  //
  // If my data type is date or time and I'm enabled but I cannot be activated, show date/time picker
  if (Client.IdfField.isDateOrTime(this.dataType) && this.enabled && !this.canActivate) {
    // Some browsers have showPicker method for inputs, some don't (i.e. Safari) and so I show picker by simulating click on input
    if (this.supportControl.textObj.showPicker)
      this.supportControl.textObj.showPicker();
    else
      this.supportControl.textObj.click();
  }
};


/**
 * Check if this is a text EDIT control
 */
Client.IdfControl.prototype.isTextEdit = function ()
{
  return this.getType() === Client.IdfField.controlTypes.EDIT && Client.IdfField.isText(this.dataType) && !this.isPassword;
};


/**
 * Make editable span simulate input behaviour
 */
Client.IdfControl.prototype.simulateInput = function ()
{
  let rootObject = this.control.getRootObject();
  //
  // In order to handle mask I need to define some properties (value, selectionStart, selectionEnd) on span
  // because maskedinp.js works on those properties (it has been implemented on input only)
  // and I don't want to change the whole maskedinp to handle span case
  Object.defineProperty(rootObject, "value", {
    get: function () {
      return this.innerText;
    },
    set: function (newValue) {
      this.innerText = newValue;
    }
  });
  //
  Object.defineProperty(rootObject, "selectionStart", {
    get: function () {
      return this.getSelection(rootObject.childNodes).start;
    }.bind(this),
    set: function (newValue) {
      rootObject.selStart = newValue;
    }.bind(this)
  });
  //
  Object.defineProperty(rootObject, "selectionEnd", {
    get: function () {
      return this.getSelection(rootObject.childNodes).end;
    }.bind(this),
    set: function (newValue) {
      this.setSelection(rootObject.childNodes, rootObject.selStart, newValue);
    }.bind(this)
  });
  //
  // I also need to define these function to make maskedinput work as expected
  rootObject.select = function () {};
  rootObject.onchange = function () {};
};


/**
 * Get selection
 * @param {HTMLElement[]} nodes
 */
Client.IdfControl.prototype.getSelection = function (nodes)
{
  let start = 0, end = 0;
  let selection = document.getSelection();
  //
  // Don't mess with visible cursor
  let range = selection.getRangeAt(0);
  let inSelection = false;
  for (let node of nodes) {
    let length = node.nodeType === Node.TEXT_NODE ? node.length : 1;
    //
    if (!inSelection) {
      if (range.startContainer === node) {
        inSelection = true;
        start += range.startOffset;
      }
      else
        start += length;
    }
    //
    if (inSelection && range.endContainer === node) {
      end += range.endOffset;
      break;
    }
    else
      end += length;
  }
  //
  return {start, end};
};


/**
 * Set selection
 * @param {HTMLElement[]} nodes
 * @param {Integer} start
 * @param {Integer} end
 */
Client.IdfControl.prototype.setSelection = function (nodes, start, end)
{
  if (start === -1 || end === -1 || nodes.length === 0)
    return;
  //
  let range = document.createRange();
  let startContainer, endContainer;
  let startOffset, endOffset;
  let pos = 0;
  for (let node of nodes) {
    let length = node.nodeType === Node.TEXT_NODE ? node.length : 1;
    //
    if (start >= pos && start < pos + length) {
      startContainer = node;
      startOffset = start - pos;
    }
    if (end >= pos && end <= pos + length) {
      endContainer = node;
      endOffset = end - pos;
    }
    pos += length;
    if (end < pos)
      break;
  }
  //
  range.setStart(startContainer, startOffset);
  range.setEnd(endContainer, endOffset);
  //
  let sel = document.getSelection();
  sel.removeAllRanges();
  sel.addRange(range);
};


/**
 * Invoke custom methods
 */
Client.IdfControl.prototype.invokeCustomMethods = function ()
{
  this.customElement?.methodInvocations?.forEach(mi => this.control[mi.method].apply(this.control, mi.args));
};


/**
 * Return true if given event represents a click on my activator
 * @param {Object} event
 */
Client.IdfControl.prototype.isActivatorClick = function (event)
{
  if (this.activator)
    return Client.Utils.isMyParent(event.content.srcEvent.srcElement, this.activator.id);
};


/**
 * Give focus to the element
 * @param {Object} event
 */
Client.IdfControl.prototype.handleFocus = function (event)
{
  if (!this.isPassword)
    return;
  //
  if (event.id === "onFocusin") {
    // On focus in, empty password if value has no other charachters than "*"
    let emptyValue = true;
    for (let i = 0; i < this.value.length; i++) {
      if (this.value[i] !== "*") {
        emptyValue = false;
        break;
      }
    }
    //
    if (emptyValue)
      this.control.updateElement({value: ""});
  }
  else if (event.id === "onFocusout" && !this.control.value.length)
    this.control.updateElement({value: this.value});
};


/**
 * Give focus to the element
 * @param {Object} options
 */
Client.IdfControl.prototype.focus = function (options)
{
  this.control.focus(options);
};


Client.IdfControl.prototype.isDraggable = function (element)
{
  // Accept a generic drop only if i'm a panel control AND the panel has the 'candrop' flag enabled
  return this.parentWidget instanceof Client.IdfFieldValue && this.parentIdfFrame.canDrag && !this.enabled;
};


Client.IdfControl.prototype.acceptsDrop = function (widget, targetDomElement)
{
  // Accept a generic drop only if i'm a panel control AND the panel has the 'candrop' flag enabled
  return this.parentWidget instanceof Client.IdfFieldValue && this.parentIdfFrame.canDrop;
};
