/*
 * Instant Developer Cloud
 * Copyright Pro Gamma Spa 2000-2021
 * All rights reserved
 */

var Client = Client || {};


/**
 * @class A frame containing buttons
 * @param {Object} widget
 * @param {View|Element} parent - the parent element
 * @param {View} view
 */
Client.IdfTree = function (widget, parent, view)
{
  // Set the defaults
  widget = Object.assign({
    vertical: true,
    dragDrop: false,
    compacted: false,
    activateOnExpand: true,
    showMultipleSelection: false,
    activeMultipleSelection: false,
    //
    // Set default events definition
    clickEventDef: Client.IdfMessagesPump?.eventTypes.ACTIVE,
    expandEventDef: Client.IdfMessagesPump?.eventTypes.ACTIVE,
    checkEventDef: Client.IdfMessagesPump?.eventTypes.DEFERRED,
    expandAnimationDef: Client.IdfWebEntryPoint?.getAnimationDefault("tree")
  }, widget);
  //
  Client.IdfFrame.call(this, widget, parent, view);
};


// Make Client.IdfTree extend Client.IdfFrame
Client.IdfTree.prototype = new Client.IdfFrame();

Client.IdfTree.getRequirements = Client.IdfFrame.getRequirements;

Client.IdfTree.transPropMap = Object.assign({}, Client.IdfFrame.transPropMap, {
  mul: "showMultipleSelection",
  aoe: "activateOnExpand",
  cms: "popupMenu",
  sel: "activeNode",
  act: "activeMultipleSelection",
  ded: "dragDrop",
  xpc: "expandEventDef",
  chc: "checkEventDef",
  exa: "expandAnimationDef"
});


/**
 * Convert properties values
 * @param {Object} props
 */
Client.IdfTree.convertPropValues = function (props)
{
  props = props || {};
  //
  Client.IdfFrame.convertPropValues(props);
  //
  for (let p in props) {
    switch (p) {
      case Client.IdfTree.transPropMap.mul:
      case Client.IdfTree.transPropMap.aoe:
      case Client.IdfTree.transPropMap.act:
      case Client.IdfTree.transPropMap.ded:
        props[p] = props[p] === "1";
        break;
    }
  }
};


/**
 * Create elements configuration
 * @@param {Object} widget
 */
Client.IdfTree.prototype.createElementsConfig = function (widget)
{
  Client.IdfFrame.prototype.createElementsConfig.call(this, widget);
  //
  this.mainContainerConf.className += " tree-container";
  this.toolbarConf.className += " tree-toolbar";
  this.contentContainerConf.className += " tree-content";
  //
  // Create content container configuration
  if (Client.mainFrame.idfMobile) {
    this.treeContainerConf = this.createElementConfig({c: "AltContainer", className: "tree-content-main-list", selectedPage: 0});
    this.treeContainerConf.animations = [{trigger: "change", type: "slide", easing: "ease", duration: 350, delay: 0, from: "left"}];
    //
    // Create the first level container
    this.levelContainersConf = [];
    this.levelContainersConf.push(this.createElementConfig({c: "IonList", className: "tree-content-main-list", noLines: false}));
    this.treeContainerConf.children.push(this.levelContainersConf[0]);
  }
  else
    this.treeContainerConf = this.createElementConfig({c: "IonList", className: "tree-content-main-list", noLines: true, events: ["onDragstart", "onDragenter", "onDragover", "onDragleave", "onDragend", "onDrop"]});
  //
  this.contentContainerConf.children.push(this.treeContainerConf);
};


Client.IdfTree.prototype.createToolbarConfig = function ()
{
  Client.IdfFrame.prototype.createToolbarConfig.call(this);
  //
  if (Client.mainFrame.idfMobile) {
    // create the backbutton for the tree
    this.backTreeButtonConf = this.createElementConfig({c: "IonButton", icon: "arrow-back", className: "generic-btn frame-toolbar-btn tree-back-btn", events: ["onClick"], visible: false});
    if (!this.onlyContent)
      this.toolbarConf.children.splice(1, 0, this.backTreeButtonConf);
    else {
      // If it's a content only tree, add the back button to the main view toolbar
      this.parentIdfView.navbarConf.children.splice(1, 0, this.backTreeButtonConf);
      this.backTreeButtonConf.style = { "order": "10" };
      //
      if (Client.eleMap[this.parentIdfView.navbarConf.id]) {
        // If the navbar is already created, i need also to create the button and add it to the toolbar
        let viewNav = Client.eleMap[this.parentIdfView.navbarConf.id];
        let bbBut = this.view.createElement(this.backTreeButtonConf, viewNav, this.view);
        viewNav.domObj.appendChild(bbBut.domObj);
        viewNav.elements.push(bbBut);
      }
    }
  }
};


/**
 * Get root object. Root object is the object where children will be inserted
 * @param {Boolean} el - if true, get the element itself istead of its domObj
 */
Client.IdfTree.prototype.getRootObject = function (el)
{
  if (this.moving)
    return Client.IdfFrame.prototype.getRootObject.call(this, el);
  //
  let rootObject = Client.eleMap[Client.mainFrame.idfMobile ? this.levelContainersConf[0].id : this.treeContainerConf.id];
  return el ? rootObject : rootObject.domObj;
};


Object.defineProperty(Client.IdfTree.prototype, "childrenTreeNodes", {
  get() {
    return Object.getOwnPropertyDescriptor(Client.IdfTreeNode.prototype, "childrenTreeNodes").get.call(this);
  }
});


/**
 * Update inner elements properties
 * @param {Object} props
 */
Client.IdfTree.prototype.updateElement = function (props)
{
  props = props || {};
  //
  Client.IdfFrame.prototype.updateElement.call(this, props);
  //
  let updateDraggable, updateEnabled;
  for (let p in props) {
    let v = props[p];
    switch (p) {
      case "popupMenu":
        this.popupMenu = v;
        break;

      case "showMultipleSelection":
        this.setShowMultipleSelection(v);
        break;

      case "activeMultipleSelection":
        this.setActiveMultipleSelection(v);
        break;

      case "activateOnExpand":
        this.activateOnExpand = v;
        break;

      case "activeNode":
        this.setActiveNode(v);
        break;

      case "dragDrop":
        this.dragDrop = v;
        updateDraggable = true;
        break;

      case "enabled":
        this.enabled = v;
        updateDraggable = true;
        updateEnabled = true;
        break;

      case "compacted":
        this.compacted = v;
        Client.Widget.updateElementClassName(Client.eleMap[this.contentContainerConf.id], "compact-tree", !this.compacted);
        break;
    }
  }
  //
  if (props.updateToolbar)
    this.updateToolbar();
  //
  if (!this.realizing && (updateDraggable || updateEnabled)) {
    this.childrenTreeNodes.forEach(tn => {
      if (updateEnabled)
        tn.updateEnabled(true);
      if (updateDraggable)
        tn.updateDraggable(true);
    });
  }
};


Client.IdfTree.prototype.setShowMultipleSelection = function (value)
{
  this.showMultipleSelection = value;
  let el = Client.eleMap[this.treeContainerConf.id];
  Client.Widget.updateElementClassName(el, "enable-selection", !this.showMultipleSelection);
};


Client.IdfTree.prototype.setActiveMultipleSelection = function (value)
{
  this.activeMultipleSelection = value;
  this.checkEventDef = this.activeMultipleSelection ? Client.IdfMessagesPump.eventTypes.ACTIVE : Client.IdfMessagesPump.eventTypes.DEFERRED;
};

/**
 * Set the active node
 * @param {String} value - selected node id
 */
Client.IdfTree.prototype.setActiveNode = function (value)
{
  if (this.activeNode === value)
    return;
  //
  // Clear the current selection
  Client.eleMap[this.activeNode]?.setActiveNode(false);
  //
  this.activeNode = value;
  //
  Client.eleMap[this.activeNode]?.setActiveNode(true);
};


/**
 * Get container at "lvl" level
 * @param {Integer} lvl
 */
Client.IdfTree.prototype.getLevelContainer = function (lvl)
{
  if (!this.levelContainersConf)
    return;
  //
  let level = lvl - 1;
  if (level < 0)
    level = 0;
  if (level >= this.levelContainersConf.length) {
    // Create the containers if needed
    let treeContainer = Client.eleMap[this.treeContainerConf.id];
    while (level >= this.levelContainersConf.length) {
      this.levelContainersConf.push(this.createElementConfig({c: "Container", className: "tree-content-lvl", visible: false}));
      treeContainer.elements.push(this.view.createElement(this.levelContainersConf[this.levelContainersConf.length - 1], treeContainer, this.view));
    }
  }
  //
  return Client.eleMap[this.levelContainersConf[level].id];
};


/**
 * Move the altcontainer to the selected level
 * @param {Client.IdfTreeNode} node
 */
Client.IdfTree.prototype.handleNodeExpansion = function (node)
{
  if (node.expanded)
    this.expandedNode = node;
  //
  let lvl = node.expanded ? node.level : node.level - 1;
  let el = Client.eleMap[this.treeContainerConf.id];
  el.updateElement({selectedPage: lvl});
  //
  // Show/Hide the backbutton
  let el2 = Client.eleMap[this.backTreeButtonConf.id];
  el2.updateElement({visible: el.selectedPage !== 0});
  //
  if (node.expanded)
    this.updateToolbar();
};


/**
 * Handle an event
 * @param {Object} event
 */
Client.IdfTree.prototype.onEvent = function (event)
{
  let events = Client.IdfFrame.prototype.onEvent.call(this, event);
  //
  switch (event.id) {
    case "onClick":
      if (Client.mainFrame.isIDF && event.obj === this.backTreeButtonConf?.id && this.expandedNode) {
        // Click on the back button, collapse the node and retard the hiding of the child (to not interfere with the animation)
        let node = Client.eleMap[this.expandedNode.id];
        node.updateElement({expanded: false, retarded: true});
        //
        // Send expand event
        events.push({
          id: "trnexp",
          def: this.expandEventDef,
          content: {
            oid: this.expandedNode.id
          }
        });
        //
        if (node.parent instanceof Client.IdfTreeNode)
          this.expandedNode = node.parent;
        else
          delete this.expandedNode;
        //
        this.updateToolbar();
      }
      break;

    case "onDragstart":
      this.handleDragStart(event);
      break;

    case "onDragenter":
      this.handleDragEnter(event);
      break;

    case "onDragover":
      this.handleDragOver(event);
      break;

    case "onDragleave":
      this.handleDragLeave(event);
      break;

    case "onDrop":
      events.push(...this.handleDrop(event));
      break;
  }
  //
  return events;
};


/**
 * Get the droppable node's div referred a target object
 * @param {Object} target
 */
Client.IdfTree.isDroppableNode = function (target)
{
  return target.closest(".treenode-header") !== null;
};


Client.IdfTree.prototype.handleDragStart = function (event)
{
  if (Client.mainFrame.isEditing())
    return;
  //
  // Set the ID of the dragged object
  let srcEvent = event.content.srcEvent;
  let draggedNode = Client.eleMap[srcEvent.target.id].parentWidget;
  srcEvent.dataTransfer.setData("text", draggedNode.id);
  srcEvent.dataTransfer.effectAllowed = "copy";
};


Client.IdfTree.prototype.handleDragEnter = function (event)
{
  if (Client.mainFrame.isEditing())
    return;
  //
  let srcEvent = event.content.srcEvent;
  if (!Client.IdfTree.isDroppableNode(srcEvent.target))
    return;
  //
  // Enable the drag
  srcEvent.preventDefault();
  //
  // Set the class to hilight the target
  srcEvent.target.closest(".treenode-header")?.classList.add("tree-drop-hover");
  //
  // Change the cursor
  srcEvent.dataTransfer.dropEffect = "copy";
};


Client.IdfTree.prototype.handleDragOver = function (event)
{
  if (Client.mainFrame.isEditing())
    return;
  //
  let srcEvent = event.content.srcEvent;
  if (!Client.IdfTree.isDroppableNode(srcEvent.target))
    return;
  //
  // Change the cursor
  srcEvent.preventDefault();
  srcEvent.dataTransfer.dropEffect = "copy";
};


Client.IdfTree.prototype.handleDragLeave = function (event)
{
  if (Client.mainFrame.isEditing())
    return;
  //
  // Clear all the drag effect classes
  // -> this is a drag leave, maybe we are not on a treenode. so we need to protect the closest() with a ? since
  // this element may not be child of the node
  let srcEvent = event.content.srcEvent;
  srcEvent.target.closest(".treenode-header")?.classList.remove("tree-drop-hover");
};


Client.IdfTree.prototype.handleDrop = function (event)
{
  if (Client.mainFrame.isEditing())
    return [];
  //
  let srcEvent = event.content.srcEvent;
  let draggedNode = Client.eleMap[srcEvent.dataTransfer.getData("text")];
  let droppedNode = Client.eleMap[srcEvent.target.closest(".treenode-header").id]?.parentWidget;
  //
  // Clear all the drop effect classes
  srcEvent.target.closest(".treenode-header")?.classList.remove("tree-drop-hover");
  //
  if (!droppedNode)
    return;
  //
  // Now the event data
  let events = [];
  if (Client.mainFrame.isIDF)
    events.push({
      id: "drp",
      def: Client.IdfMessagesPump?.eventTypes.ACTIVE,
      content: {
        oid: droppedNode.id,
        obn: draggedNode.id
      }
    });
  else {
    // IDC handles only generic D&D
  }
  //
  return events;
};


Client.IdfTree.prototype.resetCache = function ()
{
  for (let i = 0; i < this.childrenTreeNodes.length; i++) {
    let chNode = this.childrenTreeNodes[i];
    this.removeChild(chNode);
    i--;
  }
};


/**
 * Update toolbar
 */
Client.IdfTree.prototype.updateToolbar = function ()
{
  Client.IdfFrame.prototype.updateToolbar.call(this);
  //
  if (!Client.mainFrame.idfMobile)
    return;
  //
  // Mobile setting
  let caption = this.caption || "";
  if (this.expandedNode) {
    let node = Client.eleMap[this.expandedNode.id];
    caption = node.expanded ? node.label : caption;
  }
  //
  // Set caption on caption element
  let captionEl = Client.eleMap[this.captionConf.id];
  captionEl.updateElement({innerText: caption});
  //
  // Update the global menu/back button
  this.parentIdfView?.checkMobileButtons();
};


/**
 * Get click detail
 * @param {Object} event
 * @param {Widget} srcWidget
 */
Client.IdfTree.prototype.getClickDetail = function (event, srcWidget)
{
  let detail = Client.IdfFrame.prototype.getClickDetail.call(this, event, srcWidget);
  //
  let node;
  if (srcWidget instanceof Client.IdfTreeNode)
    node = srcWidget.id;
  //
  if (Client.mainFrame.isIDF)
    detail.par4 = node || -1;
  else if (node)
    detail.node = node;
  //
  return detail;
};

/*
 * A tree doesn't accept the generic Drop on himself, only on its nodes..
 */
Client.IdfTree.prototype.acceptsDrop = function (element)
{
  return false;
};


/*
 * On mobile: Collapse all expanded nodes, deep-first
 */
Client.IdfTree.prototype.resetTree = function ()
{
  this.childrenTreeNodes.forEach(tn => tn.collapseBranch());
};
