/*
 * Instant Developer Cloud
 * Copyright Pro Gamma Spa 2000-2021
 * All rights reserved
 */

var Client = Client || {};


/**
 * @class A panel field value
 * @param {Object} widget
 * @param {View|Element} parent - the parent element
 * @param {View} view
 */
Client.IdfFieldValue = function (widget, parent, view)
{
  this.parentField = parent;
  this.parentPanel = parent.parent;
  this.index = widget.index;
  //
  // Set default values
  widget = Object.assign({
    text: "",
    visible: -1,
    enabled: -1,
    visualStyle: -1,
    alignment: -1,
    isRowQbe: false
  }, widget);
  //
  // Add this value to field values array
  this.parentField.addValue(this);
  //
  Client.Widget.call(this, widget, parent, view);
  //
  if (this.customElement?.subFrameId)
    this.subFrameConf = parent.parentIdfView?.getSubFrame(this.customElement.subFrameId);
  //
  if (this.isRowQbe) {
    let controlType = this.getControlType();
    //
    // If this is a row QBE value and its control type is checkbox, set "indeterminate" as initial status
    if (controlType === Client.IdfField.controlTypes.CHECK)
      this.text = "---";
    //
    // In case of COMBO (not smart lookup), ask server for value list if my parent does not have one
    if (controlType === Client.IdfField.controlTypes.COMBO && !this.parentField.smartLookup) {
      Client.mainFrame.sendEvents([{
          id: "qbecombo",
          def: Client.IdfMessagesPump.eventTypes.URGENT,
          content: {
            oid: this.parentField.id
          }
        }]);
    }
  }
};


// Make Client.IdfFieldValue extend Client.Widget
Client.IdfFieldValue.prototype = new Client.Widget();


Client.IdfFieldValue.transPropMap = {
  idx: "index",
  txt: "text",
  vis: "visible",
  ena: "enabled",
  err: "errorText",
  ety: "errorType",
  rse: "rowSelectorType",
  mim: "blobMime",
  mty: "htmlBlobMime",
  url: "blobUrl",
  aln: "alignment",
  bkc: "backColor",
  frc: "color",
  msk: "mask",
  ftm: "fontModifiers"
};


Client.IdfFieldValue.blobMimeTypes = {
  TEXT: "text",
  IMAGE: "image",
  SIZE: "size",
  EMPTY: "empty"
};


Client.IdfFieldValue.rowSelectorTypes = {
  DOC_ERROR: 1,
  DOC_UPDATED: 2,
  INSERTED_DOC_ERROR: 3,
  INSERTED_DOC_UPDATED: 4
};


Client.IdfFieldValue.errorTypes = {
  NONE: 0,
  ERROR: 1,
  CONFIRM_WARNING: 2,
  WARNING: 3
};


Object.defineProperties(Client.IdfFieldValue.prototype, {
  listContainer: {
    get: function () {
      return Client.eleMap[this.listContainerId];
    }
  },
  listControl: {
    get: function () {
      return Client.eleMap[this.listControlId];
    }
  },
  rowSelector: {
    get: function () {
      return Client.eleMap[this.rowSelectorId];
    }
  }
});

/**
 * Convert properties values
 * @param {Object} props
 */
Client.IdfFieldValue.convertPropValues = function (props)
{
  props = props || {};
  //
  for (let p in props) {
    switch (p) {
      case Client.IdfFieldValue.transPropMap.idx:
      case Client.IdfFieldValue.transPropMap.vis:
      case Client.IdfFieldValue.transPropMap.ena:
      case Client.IdfFieldValue.transPropMap.ety:
      case Client.IdfFieldValue.transPropMap.rse:
      case Client.IdfFieldValue.transPropMap.aln:
        props[p] = parseInt(props[p]);
        break;
    }
  }
};


/**
 * Realize widget UI
 * @param {Object} widget
 * @param {View|Element|Widget} parent
 * @param {View} view
 */
Client.IdfFieldValue.prototype.realize = function (widget, parent, view)
{
  // Create widget children (custom elements)
  this.createChildren(widget);
};


/**
 * Update element properties
 * @param {Object} props
 */
Client.IdfFieldValue.prototype.updateElement = function (props)
{
  let applyVisualStyle;
  //
  if (this.skipUpdate)
    return;
  //
  props = props || {};
  let propsToUpdate = {};
  //
  Client.Widget.prototype.updateElement.call(this, props);
  //
  if (props.text !== undefined) {
    this.text = props.text;
    propsToUpdate.text = true;
  }
  //
  if (props.visible !== undefined && props.visible !== this.visible) {
    this.visible = props.visible;
    this.parentPanel.calcLayout();
  }
  //
  if (props.enabled !== undefined) {
    this.enabled = props.enabled;
    propsToUpdate.enabled = true;
    //
    applyVisualStyle = true;
  }
  //
  if (props.errorText !== undefined)
    this.errorText = props.errorText;
  //
  if (props.errorType !== undefined) {
    this.errorType = props.errorType;
    applyVisualStyle = true;
  }
  //
  if (props.blobMime !== undefined)
    this.blobMime = props.blobMime;
  //
  if (props.htmlBlobMime !== undefined)
    this.htmlBlobMime = props.htmlBlobMime;
  //
  if (props.blobUrl !== undefined)
    this.blobUrl = props.blobUrl;
  //
  if (props.alignment !== undefined) {
    this.alignment = props.alignment;
    propsToUpdate.alignment = true;
  }
  //
  if (props.backColor !== undefined) {
    this.backColor = props.backColor;
    propsToUpdate.backColor = true;
  }
  //
  if (props.color !== undefined) {
    this.color = props.color;
    propsToUpdate.color = true;
  }
  //
  if (props.mask !== undefined) {
    this.mask = props.mask;
    propsToUpdate.mask = true;
  }
  //
  if (props.fontModifiers !== undefined) {
    this.fontModifiers = props.fontModifiers;
    propsToUpdate.fontModifiers = true;
  }
  //
  if (props.badge !== undefined)
    propsToUpdate.badge = true;
  //
  if (props.valueList !== undefined) {
    this.valueList = props.valueList;
    propsToUpdate.valueList = true;
  }
  //
  if (props.controlType !== undefined)
    propsToUpdate.controlType = true;
  //
  if (props.rowSelectorType !== undefined) {
    this.rowSelectorType = props.rowSelectorType;
    this.updateRowSelectorIcon();
  }
  //
  // Apply visual style if needed
  if (applyVisualStyle)
    this.applyVisualStyle();
  //
  // Update controls
  this.updateControls(propsToUpdate);
};


/**
 * Handle an event
 * @param {Object} event
 */
Client.IdfFieldValue.prototype.onEvent = function (event)
{
  let events = [];
  //
  if (event.content instanceof Array && this.customElement) {
    events.push(...this.customElement.onEvent(event));
    events.forEach(e => {
      e.content.oid = this.id;
      e.content.ace = this.parentField.id;
    });
  }
  else {
    events.push(...Client.Widget.prototype.onEvent.call(this, event));
    //
    switch (event.id) {
      case "chgProp":
        if (this.customElement) {
          events.push(...this.customElement.onEvent(event));
          //
          // If the default property has changed I also push a change of the value
          if (event.content.name === Client[this.customElement._class].defaultBindingProperty) {
            event = JSON.parse(JSON.stringify(event));
            event.content.name = "value";
            event.content.value += "";
          }
        }
        //
        if (event.content.name === "value" || event.content.name === "filter")
          events.push(...this.handleChange(event));
        else if (event.content.name === "checked" && event.obj === this.multiSelCheckbox?.id)
          events.push(...this.parentPanel.handleDataRowSelection(event.content.value, this.index));
        break;

      case "onKey":
        events.push(...this.handleChange(event));
        //
        // F2 => activate
        if (event.content.keyCode === Client.mainFrame.wep?.FKActField + 111)
          events.push(...this.handleActivate(event));
        break;

      case "onClick":
      case "onDblclick":
      case "onContextmenu":
        // Handle changeRow/rowSelectorClick event
        events.push(...this.parentPanel.handleDataRowClick(event, this));
        //
        if (event.obj !== this.rowSelector?.id && (event.id !== "onContextmenu" || this.parentPanel.activateOnRightClick))
          events.push(...this.handleActivate(event));
        break;

      case "onBlobCommand":
        events.push(...this.handleBlobCommand(event));
        break;

      case "onFocusin":
      case "onFocusout":
        if (event.id === "onFocusout" && !Client.mainFrame.isIDF && !this.parentField.isStatic()) {
          let control = this.getSourceControl(event);
          let changeEvent = {obj: control.id, content: {name: "value", value: control.getValueToSend()}, immediate: true};
          events.push(...this.handleChange(changeEvent));
        }
        //
        events.push(...this.handleFocus(event));
        break;
    }
  }
  //
  if (this.customChildrenConf) {
    let srcElement = Client.Utils.findElementFromDomObj(event.content.srcEvent?.target);
    if (!srcElement?.events?.includes(event.id) && event.id !== "chgProp")
      return events;
    //
    event.content.srcEvent.stopPropagation();
    //
    let ev = {
      id: "fireEvent",
      obj: this.parentField.id,
      content: {srcId: event.id, row: this.index, srcObjId: event.id === "chgProp" ? this.id : srcElement.id}
    };
    events.push(ev);
  }
  //
  return events;
};


/**
 * Create a column configuration that represents row selector and will be part of panel grid
 */
Client.IdfFieldValue.prototype.createRowSelectorConfig = function ()
{
  // Create row selector column configuration
  let offsetCol = this.parentPanel.getListRowOffset() ? " offset-col" : "";
  let rowSelectorConf = this.createElementConfig({c: "IonCol", className: "panel-list-col row-selector-col" + offsetCol, xs: "auto", events: ["onClick"], visible: this.parentPanel.showRowSelector});
  //
  // Create row selector button configuration
  let rowSelectorIconConf = this.createElementConfig({c: "IonButton", className: "row-selector", icon: this.isRowQbe ? "remove-circle" : "arrow-round-forward"});
  rowSelectorConf.children.push(rowSelectorIconConf);
  //
  // Save element id
  this.rowSelectorId = rowSelectorConf.id;
  //
  return rowSelectorConf;
};


/**
 * Create a column configuration that will be part of panel grid
 */
Client.IdfFieldValue.prototype.createListConfig = function ()
{
  let controlConfig = this.parentField.createControlConfig(false, this.index);
  //
  // Create column configuration
  let listContainerConf = this.createElementConfig(controlConfig.container);
  //
  // Create list control configuration
  let listControlConf = this.createElementConfig(controlConfig.control);
  listContainerConf.children.push(listControlConf);
  //
  // Save elements ids
  this.listContainerId = listContainerConf.id;
  this.listControlId = listControlConf.id;
  //
  return listContainerConf;
};


/**
 * Get visual style
 */
Client.IdfFieldValue.prototype.getVisualStyle = function ()
{
  return this.visualStyle !== -1 ? this.visualStyle : this.parentField.visualStyle;
};


/**
 * Get value control type
 */
Client.IdfFieldValue.prototype.getControlType = function ()
{
  let controlType;
  //
  if (this.listControl)
    controlType = this.listControl.getType();
  else if (this.outListControl)
    controlType = this.outListControl.getType();
  else if (this.formControl)
    controlType = this.formControl.getType();
  //
  return controlType;
};


/**
 * Get value list
 */
Client.IdfFieldValue.prototype.getValueList = function ()
{
  return this.valueList || this.parentField.valueList;
};


/**
 * Get alignment
 */
Client.IdfFieldValue.prototype.getAlignment = function ()
{
  // Get value, field and visual style alignment
  let alignment = this.alignment !== -1 ? this.alignment : undefined;
  let fieldAlignment = this.parentField.alignment !== -1 ? this.parentField.alignment : undefined;
  let vis = Client.IdfVisualStyle.getByIndex(this.getVisualStyle());
  //
  alignment = alignment || fieldAlignment || vis?.getAlignment();
  //
  // If there is no alignment or it's AUTO, assign RIGHT or LEFT
  if (!alignment || alignment === Client.IdfVisualStyle.alignments.AUTO)
    alignment = this.parentField.isRightAligned() ? Client.IdfVisualStyle.alignments.RIGHT : Client.IdfVisualStyle.alignments.LEFT;
  //
  return alignment;
};


/**
 * Check if this value is clickable
 */
Client.IdfFieldValue.prototype.isClickable = function ()
{
  return this.parentField.isHyperLink || Client.IdfVisualStyle.getByIndex(this.getVisualStyle())?.getHyperLinkFlag();
};


/**
 * Update row selector icon
 */
Client.IdfFieldValue.prototype.updateRowSelectorIcon = function ()
{
  if (!this.rowSelector)
    return;
  //
  let icon = "arrow-round-forward";
  let cssClass = "";
  //
  switch (this.rowSelectorType) {
    case Client.IdfFieldValue.rowSelectorTypes.DOC_ERROR:
    case Client.IdfFieldValue.rowSelectorTypes.INSERTED_DOC_ERROR:
      icon = "close-circle";
      cssClass = "error";
      break;

    case Client.IdfFieldValue.rowSelectorTypes.DOC_UPDATED:
    case Client.IdfFieldValue.rowSelectorTypes.INSERTED_DOC_UPDATED:
      icon = "medical";
      cssClass = "updated";
      break;
  }
  //
  let rowSelectorButton = this.rowSelector.elements[0];
  //
  // Set image
  Client.Widget.setIconImage(icon, rowSelectorButton);
  //
  // Set css class
  let classList = rowSelectorButton.getRootObject().classList;
  classList.remove("updated");
  classList.remove("error");
  //
  rowSelectorButton.updateElement({className: classList.value + " " + cssClass});
};


/**
 * Show or hide checkbox for multiple selection
 * @param {Boolean} show
 */
Client.IdfFieldValue.prototype.updateMultiSelVisibility = function (show)
{
  if (!this.rowSelectorId)
    return;
  //
  // Get row selector button
  let rowSelectorButton = this.rowSelector.elements[0];
  //
  if (show) {
    // Hide row selector button
    rowSelectorButton.updateElement({visible: false});
    //
    let disabled = this.parentPanel.isNewRow(this.index) || this.isInQbe();
    //
    // Create checkbox
    if (!this.multiSelCheckbox) {
      let checkboxConf = this.createElementConfig({c: "IonCheckbox", className: "control-checkbox"});
      this.multiSelCheckbox = this.rowSelector.insertBefore({child: checkboxConf});
    }
    //
    this.multiSelCheckbox.updateElement({disabled});
  }
  else {
    // Show row selector button
    rowSelectorButton.updateElement({visible: true});
    //
    // Remove checkbox
    if (this.multiSelCheckbox) {
      this.rowSelector.removeChild(this.multiSelCheckbox);
      delete this.multiSelCheckbox;
    }
  }
};


/**
 * Select or unselect row
 * @param {Boolean} value
 */
Client.IdfFieldValue.prototype.selectRow = function (value)
{
  // If this value does not handle row selector column, do nothing
  if (!this.rowSelectorId)
    return;
  //
  this.multiSelCheckbox?.updateElement({checked: value});
};


/**
 * Set layout for container of list value
 * @param {Object} layout - {style, xs}
 */
Client.IdfFieldValue.prototype.setListLayout = function (layout)
{
  if (!this.listContainer)
    return;
  //
  let fixedLeft = this.parentPanel.isFixedField(this.parentField) ? this.parentPanel.getFixedFieldLeft(this.parentField) + "px" : "";
  layout.style.left = fixedLeft;
  //
  // Apply layout to list container
  this.listContainer.updateElement(layout);
  Client.Widget.updateElementClassName(this.listContainer, "fixed-col", !fixedLeft);
  //
  // If I handle the row selector, set it the same height as mine
  if (this.rowSelector) {
    this.rowSelector.updateElement({style: {height: layout.style.height, left: this.parentPanel.fixedColumns ? 0 : ""}});
    Client.Widget.updateElementClassName(this.rowSelector, "fixed-col", !this.parentPanel.fixedColumns);
  }
};


/**
 * Clear out list and form controls assigned to me
 */
Client.IdfFieldValue.prototype.clearControls = function ()
{
  if (this.outListContainer) {
    this.elements.splice(this.elements.findIndex(e => e === this.outListContainer));
    this.outListContainer.parent = this.parentField;
    this.outListContainer.parentWidget = this.parentField;
  }
  //
  delete this.outListContainer;
  delete this.outListControl;
  //
  if (this.formContainer) {
    this.elements.splice(this.elements.findIndex(e => e === this.formContainer));
    this.formContainer.parent = this.parentField;
    this.formContainer.parentWidget = this.parentField;
  }
  //
  delete this.formContainer;
  delete this.formControl;
};


/**
 * Assign out list and form container to me
 * @param {Object} params
 *                      - outListContainer
 *                      - outListControl
 *                      - formContainer
 *                      - formControl
 */
Client.IdfFieldValue.prototype.assignControls = function (params)
{
  // If I have an out list container, it means I'm the value of an out of list field
  if (params.outListContainer) {
    // Now I'm the temporary owner of shared out list container and control
    this.outListContainer = params.outListContainer;
    this.outListControl = params.outListControl;
    this.elements.push(this.outListContainer);
    this.outListContainer.parent = this;
    this.outListContainer.parentWidget = this;
  }
  //
  if (params.formContainer) {
    // Now I'm the temporary owner of shared form container and control
    this.formContainer = params.formContainer;
    this.formControl = params.formControl;
    this.elements.push(this.formContainer);
    this.formContainer.parent = this;
    this.formContainer.parentWidget = this;
  }
  //
  this.updateControls({all: true, skipInList: true});
  //
  // Assign sub frame (if any) to proper control
  this.assignSubFrame();
  //
  this.assignCustomChildren();
};


/**
 * Apply visual style
 */
Client.IdfFieldValue.prototype.applyVisualStyle = function ()
{
  let readOnly = !this.parentField.isEnabled(this.index);
  //
  // Don't apply QBE style to static field and to simple lookup field
  let qbe = this.isInQbe() && !this.parentField.isStatic() && (!this.parentField.isLookup() || this.parentField.autoLookup || this.parentField.smartLookup);
  //
  let error = this.errorType === Client.IdfFieldValue.errorTypes.ERROR;
  let warning = this.errorType === Client.IdfFieldValue.errorTypes.WARNING || this.errorType === Client.IdfFieldValue.errorTypes.CONFIRM_WARNING;
  //
  // Prepare visual style options
  let visOptions = {objType: "field", list: false, readOnly, qbe, error, warning};
  //
  // Set visual style on form field value
  if (this.formContainer)
    this.addVisualStyleClasses(this.formContainer, visOptions);
  //
  // If I have an out list container, it means I'm the value of an out of list field
  if (this.outListContainer)
    this.addVisualStyleClasses(this.outListContainer, visOptions);
  else if (this.listContainer) { // Otherwise I'm the value of a list field
    visOptions.list = true;
    visOptions.alternate = (this.index % 2 === 0);
    visOptions.activeRow = (this.index === this.parentPanel.getActiveRowIndex());
    //
    this.addVisualStyleClasses(this.listContainer, visOptions);
    //
    // If this value has to handle the row selector, set visual style on it
    // Use first visual style for row selector
    if (this.rowSelector)
      this.addVisualStyleClasses(this.rowSelector, visOptions);
  }
  //
  this.updateControls({visualStyle: true});
};


/**
 * Update internal controls
 * @param {Object} propsToUpdate - example {visualStyle: true, editorType: true, ...}
 */
Client.IdfFieldValue.prototype.updateControls = function (propsToUpdate)
{
  propsToUpdate = propsToUpdate || {};
  //
  let updateAll = propsToUpdate.all;
  //
  let skippedProps = ["parent", "subFrameConf", "customChildrenConf"];
  let fieldProps = Object.keys(this.parentField).concat(Object.keys(this));
  //
  //I add the custom properties
  if (this.customElement)
    fieldProps = fieldProps.concat(Object.keys(this.customElement.customProps));
  //
  let controlProps = {};
  for (let i = 0; i < fieldProps.length; i++) {
    let fieldProp = fieldProps[i];
    if (skippedProps.includes(fieldProp))
      continue;
    //
    // Always update enabled status
    if (fieldProp === "enabled") {
      if ((this.parentField.controlType || Client.IdfVisualStyle.getByIndex(this.getVisualStyle())?.getControlType()) === Client.IdfField.controlTypes.BUTTON)
        controlProps.enabled = this.parentField.isEnabled(this.index) || this.parentField.activableDisabled;
      else
        controlProps.enabled = this.parentField.isEnabled(this.index) && !this.parentField.isStatic();
    }
    else if (propsToUpdate[fieldProp] || updateAll) { // Otherwise check if current property has to be sent to control
      if (fieldProp === "text") {
        controlProps.value = this.text || "";
        controlProps.blobMime = this.blobMime;
        controlProps.htmlBlobMime = this.htmlBlobMime;
        controlProps.blobUrl = this.blobUrl;
      }
      else if (fieldProp === "visualStyle")
        controlProps.visualStyle = this.getVisualStyle();
      else if (fieldProp === "type")
        controlProps.isLookup = this.parentField.isLookup();
      else if (fieldProp === "alignment") {
        // If I have an alignment, update control alignment
        let alignment = this.getAlignment();
        if (alignment)
          controlProps.alignment = alignment;
      }
      else if (fieldProp === "image")
        controlProps.image = this.parentPanel.showFieldImageInValue ? this.parentField.image : undefined;
      else if (fieldProp === "canSort")
        controlProps.canSort = this.parentField.isSortable();
      else if (fieldProp === "isHyperLink")
        controlProps.isClickable = this.isClickable();
      else if (fieldProp === "controlType")
        controlProps.type = this.parentField.controlType;
      else if (fieldProp === "customElement") {
        if (this.customElement && !this.customElement.subFrameId)
          controlProps.customElement = this.customElement;
      }
      else if (this.customElement && this.customElement.customProps[fieldProp])
        controlProps[fieldProp] = this.customElement[fieldProp];
      else
        controlProps[fieldProp] = this[fieldProp] ?? this.parentField[fieldProp];
    }
  }
  //
  // Since updateBlobCommands is not a field nor a fieldValue property, I have to handle it here
  if (propsToUpdate.updateBlobCommands || updateAll) {
    let blobCommands = this.getBlobCommands();
    controlProps.uploadBlobEnabled = blobCommands.upload;
    controlProps.deleteBlobEnabled = blobCommands.delete;
    controlProps.viewBlobEnabled = blobCommands.view;
  }
  //
  // Since isInQbe is not a field nor a fieldValue property, I have to handle it here
  if (propsToUpdate.isInQbe || updateAll)
    controlProps.isInQbe = this.isInQbe();
  //
  let listControlProps = Object.assign({}, controlProps);
  let formControlProps = Object.assign({}, controlProps);
  //
  if (controlProps.container || updateAll) {
    listControlProps.container = this.parentField.isInList() ? this.listContainer : this.outListContainer;
    formControlProps.container = this.formContainer;
  }
  //
  // I have two "numRows" properties: listNumRows and formNumRows.
  // Since I want control to have a single "numRows" property,
  // I have to assign listNumRows as numRows for listControl and outListControl and formNumRows for formControl
  listControlProps.numRows = listControlProps.listNumRows;
  delete listControlProps.listNumRows;
  delete listControlProps.formNumRows;
  //
  formControlProps.numRows = formControlProps.formNumRows;
  delete formControlProps.listNumRows;
  delete formControlProps.formNumRows;
  //
  if (!propsToUpdate.skipInList)
    this.listControl?.updateElement(listControlProps);
  //
  this.outListControl?.updateElement(listControlProps);
  this.formControl?.updateElement(formControlProps);
  //
  return true;
};


/**
 * Update visibility
 * @param {Boolean} form
 */
Client.IdfFieldValue.prototype.updateVisibility = function (form)
{
  let visible = this.isVisible(form);
  //
  if (!form) {
    if (this.listContainer) {
      // If field is visible use "visibility" style property because value has to take up space also when it's not visible
      if (this.parentField.isVisible())
        this.listContainer.updateElement({style: {display: "flex", visibility: visible ? "visible" : "hidden"}});
      else
        this.listContainer.updateElement({style: {visibility: "visible", display: visible ? "flex" : "none"}});
    }
    else if (this.outListContainer) {
      // Get field parent column
      let fieldColumn = this.parentPanel.getListFieldColumn(this.parentField.id);
      fieldColumn.visible = visible;
      //
      let el = Client.eleMap[this.parentField.listParentColConf.id];
      el.updateElement({style: {display: visible ? "flex" : "none"}});
    }
  }
  //
  if (form && this.formContainer) {
    // Get field parent column
    let fieldColumn = this.parentPanel.getFormFieldColumn(this.parentField.id);
    fieldColumn.visible = visible;
    //
    let el = Client.eleMap[this.parentField.formParentColConf.id];
    el.updateElement({style: {display: visible ? "flex" : "none"}});
  }
};


/**
 * Return true if I'm visible in given layout
 * @param {Boolean} form
 */
Client.IdfFieldValue.prototype.isVisible = function (form)
{
  // If parent field is not visible, I'm not visible
  if (!this.parentField.isVisible(form))
    return false;
  //
  return !!this.visible;
};


/**
 * Return true if I'm enabled
 */
Client.IdfFieldValue.prototype.isEnabled = function ()
{
  let enabled = !!this.enabled;
  if (this.enabled === -1)
    enabled = this.parentField.enabled;
  //
  return enabled;
};


/**
 * Return true if I'm in QBE
 */
Client.IdfFieldValue.prototype.isInQbe = function ()
{
  return (this.parentPanel.status === Client.IdfPanel.statuses.qbe && this.index === 1 || this.isRowQbe);
};


/**
 * Handle value change
 * @param {Object} event
 */
Client.IdfFieldValue.prototype.handleChange = function (event)
{
  let events = [];
  //
  // If change occurred on a blob, do nothing
  if (this.parentField.dataType === Client.IdfField.dataTypes.BLOB)
    return events;
  //
  // Check if change event occurred on one of my controls
  let control;
  if (event.obj === this.listControl?.id)
    control = this.listControl;
  else if (event.obj === this.outListControl?.id)
    control = this.outListControl;
  else if (event.obj === this.formControl?.id)
    control = this.formControl;
  //
  if (!control)
    return events;
  //
  if (!Client.mainFrame.isIDF && control.isTextEdit() && !this.parentField.superActive && !event.immediate)
    return events;
  //
  // Check if this change is related to a smart lookup asking for list
  let requireSmartLookupList = this.parentField.smartLookup && event.content.value === "*";
  //
  // If it's not a real value change (unless it's related to a smart lookup asking for list), do nothing
  if (event.content.name === "value" && event.content.value === this.text && !requireSmartLookupList)
    return events;
  //
  // Get control value to send
  let valueToSend = requireSmartLookupList ? "*" : control.getValueToSend(event.content.value);
  //
  // Update my text and my control value unless it's related to a smart lookup asking for list
  if (!requireSmartLookupList)
    this.updateElement({text: event.content.value});
  //
  if (Client.mainFrame.isIDF) {
    let flag = 0;
    let delay;
    //
    if (control.getType() === Client.IdfField.controlTypes.EDIT && !this.parentField.superActive)
      delay = Client.IdfMessagesPump.defaultDelay;
    //
    // If  parent field is super active, send change immediately
    if (this.parentField.superActive || event.immediate) {
      flag = Client.IdfMessagesPump.eventTypes.IMMEDIATE;
      delay = Client.IdfMessagesPump.superActiveDelay;
    }
    //
    // If value to send ends with a space, don't send it immediately. In fact, server will remove that space.
    // So wait a bit in order to give time to user to write another charachter
    if (this.parentField.superActive && valueToSend.trim() !== valueToSend)
      delay = Client.IdfMessagesPump.whiteSpaceDelay;
    //
    if (this.isRowQbe) {
      if (event.content.name === "value" || event.content.name === "filter") {
        let eventId = "qbeset";
        let eventContent = {oid: this.parentField.id, par1: valueToSend};
        //
        // In case of filter on a smart lookup I have to send a "qbecombo" event as soon as possible
        if (event.content.name === "filter" && this.parentField.smartLookup)
          eventId = "qbecombo";
        else if (event.content.name === "value") { // In case of value, send a "qbeset" event having "qbefilter" as object name
          eventContent.obn = "qbefilter";
          //
          // In case of rowQbe smart lookup, I have to send to server rValues instead of values
          if (this.parentField.smartLookup) {
            let newQbeFilter = this.listControl.getComboRValueFromValue(valueToSend);
            eventContent.par1 = newQbeFilter;
            //
            // Update parent field qbe filter
            this.parentField.updateElement({qbeFilter: newQbeFilter});
          }
        }
        //
        events.push({
          id: eventId,
          def: Client.IdfMessagesPump.eventTypes.URGENT,
          delay,
          content: eventContent
        });
      }
    }
    else {
      events.push({
        id: "chg",
        content: {
          par1: valueToSend
        }
      });
      //
      if (this.customElement)
        events.push(...this.customElement.onEvent(event));
      //
      events.forEach(e => {
        e.def = this.parentField.changeEventDef | flag;
        e.delay = delay;
        e.updateStartTime = !!this.parentField.superActive;
        e.content.oid = this.id;
      });
      //
      // A superactive value source has to open combo on each pressed key, so handle its activation in order to make server send value list
      //if (this.parentField.hasValueSource && this.parentField.superActive)
      //  events.push(...this.handleActivate(event));
    }
  }
  else {
    if (this.isRowQbe) {
      // TODO
    }
    else
      events.push({obj: this.parentField.id, id: "chgProp", content: {name: "text", value: valueToSend, index: this.index, clid: Client.id}});
  }
  //
  return events;
};


/**
 * Handle value activation
 * @param {Object} event
 */
Client.IdfFieldValue.prototype.handleActivate = function (event)
{
  let events = [];
  //
  event.content = event.content || {};
  //
  // If parent field is disabled and cannot be activate when disabled, return no events
  if (!this.parentField.isEnabled() && !this.parentField.activableDisabled)
    return events;
  //
  let controlType = this.getControlType();
  //
  // Get source control of given event
  let control = this.getSourceControl(event);
  //
  // Check if given event represents a click on control activator
  let isActivatorClick = control.isActivatorClick(event);
  //
  // If this is field value for row QBE, open QBE popup on activator click
  if (this.isRowQbe) {
    if (isActivatorClick) {
      // If control type is CHECK I don't need QBE popup because there aren't special filter on checkbox
      if (controlType === Client.IdfField.controlTypes.CHECK)
        return events;
      //
      // Open QBE popup
      new Client.IdfFilterPopup({id: "filter-popup", field: this.parentField}, this.view, this.view);
    }
    //
    return events;
  }
  //
  let isActivable = controlType === Client.IdfField.controlTypes.BUTTON || this.parentField.canActivate || this.isClickable();
  //
  // If parent field cannot activate
  if (!isActivable && !this.parentField.hasValueSource) {
    // If user clicks on smart lookup activator, I have to send filter to server in order to receive the value list and thus open combo
    if (this.parentField.smartLookup && isActivatorClick) {
      event.immediate = true;
      event.content.value = "*";
      event.obj = control.id;
      events.push(...this.handleChange(event));
    }
    //
    return events;
  }
  //
  // If the field has an associated command, I notify the click on the command
  if (this.parentField.command) {
    events.push(...Client.eleMap[this.parentField.command].handleClick(event));
    return events;
  }
  //
  if (Client.mainFrame.isIDF) {
    // If parent field is static, click on value means click on header
    let par1 = this.parentField.isStatic() ? "cap" : undefined;
    //
    // If parent field is static, send parent field object id instead of value id
    let oid = this.parentField.isStatic() ? this.parentField.id : this.id;
    //
    events.push({
      id: "clk",
      def: this.parentField.clickEventDef,
      content: {
        oid,
        xck: event.content.offsetX,
        yck: event.content.offsetY,
        par1
      }
    });
  }
  //
  return events;
};


/**
 * Get blob commands
 */
Client.IdfFieldValue.prototype.getBlobCommands = function ()
{
  let blobCommands = {
    upload: this.parentPanel.isCommandEnabled(Client.IdfPanel.commands.CZ_BLOBEDIT),
    delete: this.parentPanel.isCommandEnabled(Client.IdfPanel.commands.CZ_BLOBDELETE),
    view: this.parentPanel.isCommandEnabled(Client.IdfPanel.commands.CZ_BLOBSAVEAS)
  };
  //
  let valueMimeType = this.blobMime || Client.IdfFieldValue.blobMimeTypes.EMPTY;
  //
  // If I'm not enabled or panel is in QBE status or it's on new row no upload and delete
  if (!this.parentField.isEnabled(this.index) || this.parentPanel.status === Client.IdfPanel.statuses.qbe || this.parentPanel.isNewRow()) {
    blobCommands.upload = false;
    blobCommands.delete = false;
  }
  //
  // With blob empty no view and delete
  if (valueMimeType === Client.IdfFieldValue.blobMimeTypes.EMPTY) {
    blobCommands.view = false;
    blobCommands.delete = false;
  }
  //
  return blobCommands;
};


/**
 * Get blob commands
 * @param {Object} event
 */
Client.IdfFieldValue.prototype.handleBlobCommand = function (event)
{
  let events = [];
  switch (event.content.command) {
    case Client.IdfControl.blobCommands.UPLOAD:
      this.handleBlobUpload(event);
      break;

    case Client.IdfControl.blobCommands.DELETE:
      // TODO: this.doHighlightDelete(true);
      //
      let options = {
        type: Client.Widget.msgTypes.CONFIRM,
        text: Client.IdfResources.t("PAN_MSG_ConfirmDeleteBLOB", [this.parentField.header])
      };
      //
      Client.Widget.showMessageBox(options, result => {
        if (result === "Y") {
          events.push(...this.handleBlobDelete(event));
          Client.mainFrame.sendEvents(events);
        }
      });
      //
      break;
  }
  //
  return events;
};


/**
 * Handle blob upload
 * @param {Object} event
 */
Client.IdfFieldValue.prototype.handleBlobUpload = function (event)
{
  let errorMsg = "";
  let formData = new FormData();
  for (let i = 0; i < event.content.files.length; i++) {
    let file = event.content.files[i];
    //
    // Check if file extensions is valid
    let canUpload = false;
    if (this.parentField.uploadExtensions === "*.*")
      canUpload = true;
    else {
      // Get allowed mime types
      let allowedMimes = this.parentField.uploadExtensions.split(',');
      for (let j = 0; j < allowedMimes.length; j++) {
        if (file.name.substring(file.name.length - allowedMimes[j].length) === allowedMimes[j]) {
          canUpload = true;
          break;
        }
      }
    }
    //
    if (!canUpload) {
      errorMsg += errorMsg ? Client.IdfResources.t("SWF_ER_FILENOTSEND") : "";
      errorMsg += "<br>" + file.name + " : " + Client.IdfResources.t("SWF_ER_VALIDATIONFAILED");
    }
    //
    // Check if file size is valid
    if (!file.size) {
      canUpload = false;
      //
      errorMsg += errorMsg ? Client.IdfResources.t("SWF_ER_FILENOTSEND") : "";
      errorMsg += "<br>" + file.name + " : " + Client.IdfResources.t("SWF_ER_VALIDATIONFAILED");
    }
    else if (file.size > this.parentField.maxUploadSize) {
      canUpload = false;
      //
      // Calculate max size string
      let maxSizeString = this.parentField.maxUploadSize;
      let units = ["B", "KB", "MB", "GB"];
      for (let i = 0; i < units.length; i++) {
        if (this.parentField.maxUploadSize < Math.pow(1024, i + 1)) {
          maxSizeString = Math.round(this.parentField.maxUploadSize / Math.pow(1024, i), 0) + " " + units[i];
          break;
        }
      }
      //
      errorMsg += errorMsg ? Client.IdfResources.t("SWF_ER_FILENOTSEND") : "";
      errorMsg += "<br>" + file.name + " : " + Client.IdfResources.t("SWF_ER_FILESIZEEXCEEDED") + " (max " + maxSizeString + ")";
    }
    //
    // If file can be uploaded, add it to form data
    if (canUpload)
      formData.append("file" + i, file);
  }
  //
  // If there is an error message, show it
  if (errorMsg)
    return Client.Widget.showMessageBox({type: Client.Widget.msgTypes.ALERT, text: errorMsg});
  //
  let wci = (this.parentField.multiUpload ? "IWFiles" : "IWUpload");
  //
  if (Client.idfOffline || (!Client.mainFrame.isIDF && !this.parentField.multiUpload)) {
    for (let i = 0; i < event.content.files.length; i++) {
      let file = event.content.files[i];
      let reader = new FileReader();
      reader.onload = ev => {
        let events = [];
        if (Client.mainFrame.isIDF)
          events.push({
            id: wci,
            def: Client.IdfMessagesPump.eventTypes.ACTIVE,
            content: {
              par1: this.parentField.id,
              par2: file.name,
              par3: file.type,
              par4: ev.target.result,
              par5: file.size
            }
          });
        else
          events.push({
            obj: this.parentField.id,
            id: "chgProp",
            content: {
              name: "text",
              value: ev.target.result,
              index: this.index,
              clid: Client.id
            }
          });
        //
        Client.mainFrame.sendEvents(events);
      };
      //
      reader.readAsDataURL(file);
    }
  }
  else {
    // Calculate query string
    let qstring, req;
    if (Client.mainFrame.isIDF) {
      qstring = `?WCI=${wci}&WCE=${this.parentField.id}`;
      req = Client.mainFrame.messagesPump.createRequest();
    }
    else {
      qstring = Client.Utils.getRESTQueryString({msgType: "input-upload", objId: this.id});
      req = new XMLHttpRequest();
    }
    //
    // Send post request
    req.open("POST", qstring, true);
    //
    if (Client.mainFrame.isIDF)
      req.addEventListener("load", function () {
        if (this.status === 200) {
          let text = this.responseText || "";
          text = text.substr(256, text.length - 256);
          //
          let parser = new DOMParser();
          let xmlDoc = parser.parseFromString(text, "text/xml");
          //
          Client.mainFrame.handleIDFResponse(req.reqCode, xmlDoc);
        }
      }, false);
    //
    req.send(formData);
  }
};


/**
 * Handle blob delete
 * @param {Object} event
 */
Client.IdfFieldValue.prototype.handleBlobDelete = function (event)
{
  let events = [];
  if (Client.mainFrame.isIDF)
    events.push({
      id: "pantb",
      def: this.parentPanel.toolbarEventDef,
      content: {
        oid: this.parentPanel.id,
        obn: "delblob" + this.parentField.index
      }
    });
  else
    events.push({
      obj: this.parentField.id,
      id: "chgProp",
      content: {
        name: "text",
        index: this.index,
        clid: Client.id
      }
    });
  //
  return events;
};


/**
 * Handle focus event
 * @param {Object} event
 */
Client.IdfFieldValue.prototype.handleFocus = function (event)
{
  let events = [];
  switch (event.id) {
    case "onFocusout":
      // If the field is active when it loses focus I have to send the change event (which is queued)
      // However, if the onFocus event is also active,
      // I wait to send so as to merge the events into the same request
      if (Client.IdfMessagesPump.isActiveEvent(this.parentField.changeEventDef)
              && !Client.IdfMessagesPump.isActiveEvent(this.parentPanel.focusEventDef))
        Client.mainFrame.messagesPump.sendEvents(true);
      break;
  }
  //
  if (Client.mainFrame.isIDF) {
    events.push({
      id: "fev",
      def: this.parentPanel.focusEventDef,
      delay: 250,
      content: {
        oid: this.parentPanel.id,
        obn: this.parentField.index,
        par1: event.id === "onFocusin" ? "1" : "0"
      }
    });
  }
  else {
    // TODO
  }
  return events;
};


/**
 * Handle selection change event
 * @param {Object} event
 */
Client.IdfFieldValue.prototype.handleSelectionChange = function (event)
{
  return this.parentField.handleSelectionChange(event);
};


/**
 * Assign sub frame to proper control
 */
Client.IdfFieldValue.prototype.assignSubFrame = function ()
{
  // A subframe is a unique object shared among list and form field's interfaces.
  // This means that on selected row change or on panel mode change it has to be removed from its actual container and appended to its new one
  let subFrameConf = this.subFrameConf || this.parentField.subFrameConf;
  //
  // If no subFrameConf, do nothing
  if (!subFrameConf)
    return;
  //
  // Get list control
  let listControl = this.parentField.isInList() ? this.listControl : this.outListControl;
  //
  // If actual parent panel mode is LIST, assign sub frame to list control. Otherwise give it to form control
  if (this.parentPanel.layout === Client.IdfPanel.layouts.list) {
    this.formControl?.updateElement({subFrameConf: null});
    listControl?.updateElement({subFrameConf});
  }
  else {
    listControl?.updateElement({subFrameConf: null});
    this.formControl?.updateElement({subFrameConf});
  }
};


/**
 * Assign custom children to proper control
 * @param {Integer} layout
 */
Client.IdfFieldValue.prototype.assignCustomChildren = function (layout)
{
  // If no customChildrenConf, do nothing
  if (!this.customChildrenConf)
    return;
  //
  // Get list control
  let listControl = this.parentField.isInList() ? this.listControl : this.outListControl;
  //
  layout = layout ?? this.parentPanel.layout;
  //
  // If I have to assign custom children to form layout, remove them from list control and add them to form control
  if (layout === Client.IdfPanel.layouts.form) {
    listControl?.updateElement({customChildrenConf: null});
    this.formControl?.updateElement({customChildrenConf: this.customChildrenConf});
  }
  else {
    this.formControl?.updateElement({customChildrenConf: null});
    listControl?.updateElement({customChildrenConf: this.customChildrenConf});
  }
};


/**
 * Update qbe filter
 */
Client.IdfFieldValue.prototype.updateQbeFilter = function ()
{
  if (!this.isRowQbe)
    return;
  //
  let text = this.parentField.qbeFilter;
  //
  // In case of CHECK control, if there isn't a qbe filter I have to set "indeterminate" state
  if (this.getControlType() === Client.IdfField.controlTypes.CHECK && !this.parentField.qbeFilter)
    text = "---";
  //
  if (this.parentField.smartLookup) {
    // If qbe filter comes from server, empty value list
    if (this.valueList)
      delete this.valueList;
    //
    // If qbe filter is an array it means that it has be set by IdfFilterPopup. So calculate string to show into combo
    if (this.parentField.qbeFilter instanceof Array) {
      text = "";
      //
      // Get combo name separator
      let sep = this.listControl.getComboNameSeparator();
      //
      for (let i = 0; i < this.parentField.qbeFilter.length; i++)
        text += (text.length > 0 ? sep : "") + this.parentField.qbeFilter[i].name;
    }
    else if (this.parentField.qbeFilter.indexOf("fld:") !== -1) // Otherwise it comes from server as string
      text = this.listControl.getComboValueFromRValue(this.parentField.qbeFilter);
  }
  //
  this.updateElement({text});
};


/**
 * Get source control
 * @param {Object} event
 */
Client.IdfFieldValue.prototype.getSourceControl = function (event)
{
  let control;
  switch (event.obj) {
    case this.listContainer?.id:
    case this.listControl?.id:
      control = this.listControl;
      break;

    case this.outListContainer?.id:
    case this.outListControl?.id:
      control = this.outListControl;
      break;

    case this.formContainer?.id:
    case this.formControl?.id:
      control = this.formControl;
      break;
  }
  //
  return control;
};


/**
 * Give focus to the element
 * @param {Object} options
 */
Client.IdfFieldValue.prototype.focus = function (options)
{
  let control;
  if (this.parentPanel.layout === Client.IdfPanel.layouts.list)
    control = this.listControl || this.outListControl;
  else
    control = this.formControl;
  //
  if (options.selectAll) {
    options.selectionStart = 0;
    options.selectionEnd = this.text.length;
  }
  //
  control?.focus(options);
};


/**
 * Return the target to use for opening a popup on this widget
 * @returns {DomNode}
 */
Client.IdfFieldValue.prototype.getPopupTarget = function ()
{
  if (this.parentPanel.layout === Client.IdfPanel.layouts.list)
    return this.listContainer || this.outListContainer;
  else
    return this.formContainer;
};


/**
 * Handle selection change event
 * @param {Object} options
 */
Client.IdfFieldValue.prototype.focusNearControl = function (options)
{
  options = Object.assign({fieldValue: this}, options);
  return this.parentPanel.focusNearControl(options);
};


/**
 * Open combo
 */
Client.IdfFieldValue.prototype.openCombo = function ()
{
  let control;
  //
  // If panel mode is LIST get list control or out list control
  if (this.parentPanel.layout === Client.IdfPanel.layouts.list)
    control = this.parentField.isInList() ? this.listControl : this.outListControl;
  else // Otherwise get formControl
    control = this.formControl;
  //
  control?.updateElement({openCombo: true});
};