/*
 * Instant Developer Cloud
 * Copyright Pro Gamma Spa 2000-2021
 * All rights reserved
 */

var Client = Client || {};


/**
 * @class A panel page
 * @param {Object} widget
 * @param {View|Element} parent - the parent element
 * @param {View} view
 */
Client.IdfPanelPage = function (widget, parent, view)
{
  // In case of IDC I have to calculate index
  if (!Client.mainFrame.isIDF) {
    Client.IdfPanelPage.count = Client.IdfPanelPage.count || 0;
    widget.index = Client.IdfPanelPage.count;
    Client.IdfPanelPage.count++;
  }
  //
  // Add this page to panel pages array
  parent.addPage(this);
  //
  // Get page's fields from parent panel
  this.fields = [];
  for (let i = 0; i < parent.fields.length; i++) {
    let field = parent.fields[i];
    if (field.pageIndex === widget.index) {
      field.page = this;
      this.fields.push(field);
    }
  }
  //
  // Get page's groups from parent panel
  this.groups = [];
  for (let i = 0; i < parent.groups.length; i++) {
    let group = parent.groups[i];
    if (group.pageIndex === widget.index) {
      group.page = this;
      this.groups.push(group);
    }
  }
  //
  // Set default values
  widget = Object.assign({
    enabled: true,
    visible: true
  }, widget);
  //
  Client.Widget.call(this, widget, parent, view);
};


// Make Client.IdfPanelPage extend Client.Widget
Client.IdfPanelPage.prototype = new Client.Widget();


Client.IdfPanelPage.transPropMap = {
  flg: "flags",
  img: "image",
  pst: "pageStyle",
  ind: "index"
};


/**
 * Convert properties values
 * @param {Object} props
 */
Client.IdfPanelPage.convertPropValues = function (props)
{
  props = props || {};
  //
  for (let p in props) {
    switch (p) {
      case Client.IdfPanelPage.transPropMap.flg:
      case Client.IdfPanelPage.transPropMap.pst:
      case Client.IdfPanelPage.transPropMap.ind:
        props[p] = parseInt(props[p]);
        break;
    }
  }
};


/**
 * Realize widget UI
 * @param {Object} widget
 * @param {View|Element|Widget} parent
 * @param {View} view
 */
Client.IdfPanelPage.prototype.realize = function (widget, parent, view)
{
  let containerConf = this.createElementConfig({c: "Container", className: "panel-page", events: ["onClick"], children: [
      {c: "IonText", type: "span", className: "panel-page-text"}
    ]});
  //
  let parentContainer = Client.eleMap[this.parent.pagesContainerConf.id];
  this.mainObjects.push(view.createElement(containerConf, parentContainer, view));
  //
  this.textObj = this.getRootObject(true).elements[0];
};



/**
 * Update element properties
 * @param {Object} props
 */
Client.IdfPanelPage.prototype.updateElement = function (props)
{
  props = props || {};
  //
  Client.Widget.prototype.updateElement.call(this, props);
  //
  if (props.flags !== undefined) {
    let enabled = !!(props.flags & 0x01);
    if (enabled !== this.enabled)
      props.enabled = enabled;
    //
    let visible = !!(props.flags & 0x02);
    if (visible !== this.visible)
      props.visible = visible;
  }
  //
  let updateHeader = false;
  for (let p in props) {
    let v = props[p];
    switch (p) {
      case "enabled":
        this.enabled = v;
        break;

      case "visible":
        this.setVisible(v);
        break;

      case "tooltip":
      case "caption":
        updateHeader = true;
        break;

      case "image":
        this.setImage(v);
        break;

      case "pageStyle":
        this.setPageStyle(v);
        break;

      case "badge":
        this.setBadge(v);
        break;

      case "isActive":
        this.setActive(v);
        break;
    }
  }
  //
  // Update header
  if (updateHeader)
    this.updateHeader();
  //
  // If I have to calculate layout, do it now
  if (props.visible !== undefined && !this.realizing)
    this.parent.calcLayout();
  //
  if (props.enabled !== undefined) {
    // Update fields controls and apply visual style
    for (let i = 0; i < this.fields.length; i++) {
      this.fields[i].updateControls();
      this.fields[i].applyVisualStyle();
    }
  }
};


Client.IdfPanelPage.prototype.setVisible = function (value)
{
  this.visible = value;
  this.getRootObject(true).updateElement({visible: this.visible});
};


Client.IdfPanelPage.prototype.setImage = function (value)
{
  this.image = value;
  let rootObject = this.getRootObject(true);
  if (this.image) {
    if (!this.imageObj) {
      let imageConf = this.createElementConfig({c: "IonButton", className: "generic-btn small panel-page-image"});
      this.imageObj = rootObject.insertBefore({child: imageConf, sib: this.textObj.id});
    }
    //
    // Set image
    Client.Widget.setIconImage(this.image, this.imageObj);
    //
    // If image is an url, add "image" class
    Client.Widget.updateElementClassName(this.imageObj, "image", Client.Widget.isIconImage(this.image));
  }
  else {
    if (this.imageObj) {
      rootObject.removeChild(this.imageObj);
      delete this.imageObj;
    }
  }
};


Client.IdfPanelPage.prototype.setPageStyle = function (value)
{
  // Remove old page style and add the new one
  let rootObject = this.getRootObject(true);
  Client.Widget.updateElementClassName(rootObject, "panel-page-" + this.pageStyle, true);
  this.pageStyle = value;
  Client.Widget.updateElementClassName(rootObject, "panel-page-" + this.pageStyle);
};


Client.IdfPanelPage.prototype.setBadge = function (value)
{
  let rootObject = this.getRootObject(true);
  if (this.badge) {
    if (!this.badgeObj) {
      let badgeConf = this.createElementConfig({c: "IonBadge", className: "generic-badge internal", innerText: this.badge});
      this.badgeObj = rootObject.insertBefore({child: badgeConf});
    }
    else
      this.badgeObj.updateElement({innerText: this.badge});
  }
  else {
    if (this.badgeObj) {
      rootObject.removeChild(this.badgeObj);
      delete this.badgeObj;
    }
  }
};


Client.IdfPanelPage.prototype.setActive = function (value)
{
  this.isActive = value;
  //
  // Remove active class and add it if needed
  Client.Widget.updateElementClassName(this.getRootObject(true), "active", !this.isActive);
};


/**
 * Handle an event
 * @param {Object} event
 */
Client.IdfPanelPage.prototype.onEvent = function (event)
{
  let events = Client.Widget.prototype.onEvent.call(this, event);
  //
  switch (event.id) {
    case "onClick":
      event.page = {id: this.id, index: this.index};
      events.push(...this.parent.handlePageClick(event));
      break;
  }
  //
  return events;
};


/**
 * Update my header
 */
Client.IdfPanelPage.prototype.updateHeader = function ()
{
  let tooltip = Client.Widget.getHTMLTooltip(this.caption, this.tooltip);
  //
  // Update tooltip
  let rootObject = this.getRootObject(true);
  rootObject.updateElement({tooltip});
  //
  // Update header text
  this.textObj.updateElement({innerHTML: Client.Widget.getHTMLForCaption(this.caption)});
};


/**
 * Return true if I'm visible
 */
Client.IdfPanelPage.prototype.isVisible = function ()
{
  return this.visible && this.isActive;
};


/**
 * Return true if I'm enabled
 */
Client.IdfPanelPage.prototype.isEnabled = function ()
{
  return this.enabled;
};
