/*
 * Instant Developer Cloud
 * Copyright Pro Gamma Spa 2000-2021
 * All rights reserved
 */

var Client = Client || {};


/**
 * @class A popup to filter idfField's values
 * @param {Object} widget
 * @param {View|Element} parent - the parent element
 * @param {View} view
 */
Client.IdfFilterPopup = function (widget, parent, view)
{
  this.field = widget.field;
  this.fieldValue = this.field.getValueByIndex(this.field.parent.getActiveRowIndex());
  this.valueList = this.field.valueList;
  //
  this.filterTypeControls = [];
  this.filterValueControls = [];
  this.filterItems = [];
  this.filterItemCloseButtons = [];
  //
  // Init combo selection
  if (this.fieldValue.getControlType() === Client.IdfField.controlTypes.COMBO) {
    let qbeFilter = this.field.qbeFilter || "";
    let sep = this.field.smartLookup && qbeFilter.indexOf(Client.IdfControl.rValueSeparator) !== -1 ? Client.IdfControl.rValueSeparator : this.field.comboSeparator;
    this.comboSelection = qbeFilter ? qbeFilter.split(sep) : [];
  }
  //
  Client.Widget.call(this, widget, parent, view);
};


// Make Client.IdfFilterPopup extend Client.Widget
Client.IdfFilterPopup.prototype = new Client.Widget();


Client.IdfFilterPopup.filterTypes = {VALUE: "1", STARTS: "2", ENDS: "3", CONTAINS: "4", DIFFERENT: "5", MAJOR: "6", MINOR: "7", BETWEEN: "8", EMPTY: "9", NOTEMPTY: "10"};


/**
 * Create elements configuration
 */
Client.IdfFilterPopup.prototype.createElementsConfig = function ()
{
  // Get proper reference element id for the popup
  let fieldContainerId;
  if (this.field.parent.layout === Client.IdfPanel.layouts.list)
    fieldContainerId = this.field.isInList() ? this.field.listContainerId : this.field.outListContainerId;
  else
    fieldContainerId = this.field.formContainerId;
  //
  let refId = Client.eleMap[fieldContainerId].getRootObject().id;
  let options = {mode: "popup", autoclose: true, ref: {id: refId, offset: 1, position: "bottom-left"}, width: "400px", height: "376px", extcls: "filter-popup-modal"};
  this.viewConf = this.createElementConfig({type: "view", options, elements: []});
  //
  this.pageConf = this.createElementConfig({c: "IonPage", className: "filter-popup-page"});
  this.viewConf.elements.push(this.pageConf);
  //
  // Create header configuration
  this.createHeaderConfig();
  //
  // Create footer configuration
  this.createFooterConfig();
  //
  // Create content configuration
  this.createContentConfig();
};


/**
 * Create header configuration
 */
Client.IdfFilterPopup.prototype.createHeaderConfig = function ()
{
  let headerConf = this.createElementConfig({c: "IonHeader", className: "filter-popup-header"});
  this.pageConf.children.push(headerConf);
  //
  // Create navbar configuration
  let navbarConf = this.createElementConfig({c: "IonNavBar", className: "filter-popup-navbar"});
  headerConf.children.push(navbarConf);
  //
  // Create title configuration
  let innerHTML = Client.Widget.getHTMLForCaption(this.field.header);
  this.titleConf = this.createElementConfig({c: "IonTitle", className: "filter-popup-title", innerHTML});
  navbarConf.children.push(this.titleConf);
  //
  // Create buttons configuration
  let buttonsConf = this.createElementConfig({c: "IonButtons", className: "filter-popup-navbar-buttons"});
  navbarConf.children.push(buttonsConf);
  //
  // Create close button configuration
  this.closeButtonConf = this.createElementConfig({c: "IonButton", icon: "close", className: "filter-popup-close-button", events: ["onClick"]});
  buttonsConf.children.push(this.closeButtonConf);
};


/**
 * Create footer configuration
 */
Client.IdfFilterPopup.prototype.createFooterConfig = function ()
{
  let footerConf = this.createElementConfig({c: "IonFooter", className: "filter-popup-footer"});
  this.pageConf.children.push(footerConf);
  //
  // Create apply button configuration
  this.applyButtonConf = this.createElementConfig({c: "IonButton", label: Client.IdfResources.t("FIL_DOFILTER"), className: "filter-popup-footer-button main-action", events: ["onClick"]});
  footerConf.children.push(this.applyButtonConf);
  //
  // Create clear button configuration
  this.clearButtonConf = this.createElementConfig({c: "IonButton", label: Client.IdfResources.t("FIL_CLEARFILTER"), className: "filter-popup-footer-button", events: ["onClick"]});
  footerConf.children.push(this.clearButtonConf);
};


/**
 * Create content configuration
 */
Client.IdfFilterPopup.prototype.createContentConfig = function ()
{
  this.contentConf = this.createElementConfig({c: "IonContent", className: "filter-popup-content"});
  this.pageConf.children.push(this.contentConf);
  //
  if (this.field.parent.searchMode === Client.IdfPanel.searchModes.header) {
    // If field is sortable, create sort container configuration
    if (this.field.isSortable())
      this.createSortConfig();
    //
    // If panel can group, create grouping container configuration
    if (this.field.parent.canGroup && this.field.parent.showGroups)
      this.createSortConfig(true);
  }
  //
  this.filtersAreaConf = this.createElementConfig({c: "Container", className: "filter-popup-filters-area"});
  this.contentConf.children.push(this.filtersAreaConf);
};


/**
 * Create sort configuration
 * @param {Boolean} grouping
 */
Client.IdfFilterPopup.prototype.createSortConfig = function (grouping)
{
  let activateDesc;
  if (this.field.sortMode === Client.IdfField.sortModes.DESC) {
    if (grouping && this.field.groupingMode === Client.IdfField.groupingModes.DESC)
      activateDesc = true;
    else if (!grouping && this.field.groupingMode === Client.IdfField.groupingModes.NONE)
      activateDesc = true;
  }
  //
  let activateAsc = this.field.sortMode === Client.IdfField.sortModes.ASC;
  if (this.field.sortMode === Client.IdfField.sortModes.ASC) {
    if (grouping && this.field.groupingMode === Client.IdfField.groupingModes.ASC)
      activateAsc = true;
    else if (!grouping && this.field.groupingMode === Client.IdfField.groupingModes.NONE)
      activateAsc = true;
  }
  //
  let containerConf = this.createElementConfig({c: "Container", className: "filter-popup-item sort"});
  this.contentConf.children.push(containerConf);
  //
  // Create text configuration
  let text = grouping ? Client.IdfResources.t("FIL_GROUP_CAPTION") : Client.IdfResources.t("FIL_SORT_CAPTION");
  let textConf = this.createElementConfig({c: "IonText", type: "span", innerText: text, className: "filter-popup-sort-text"});
  containerConf.children.push(textConf);
  //
  // Create buttons configuration
  let buttonsConf = this.createElementConfig({c: "IonButtons", className: "filter-popup-sort-buttons"});
  containerConf.children.push(buttonsConf);
  //
  // Create DESC button configuration
  let descButtonConf = this.createElementConfig({c: "IonButton", icon: "arrow-dropdown", className: "filter-popup-button" + (activateAsc ? " active" : ""), events: ["onClick"]});
  buttonsConf.children.push(descButtonConf);
  if (grouping)
    this.groupingDescButtonId = descButtonConf.id;
  else
    this.sortDescButtonId = descButtonConf.id;
  //
  // Create ASC button configuration
  let ascButtonConf = this.createElementConfig({c: "IonButton", icon: "arrow-dropup", className: "filter-popup-button" + (activateDesc ? " active" : ""), events: ["onClick"]});
  buttonsConf.children.push(ascButtonConf);
  if (grouping)
    this.groupingAscButtonId = ascButtonConf.id;
  else
    this.sortAscButtonId = ascButtonConf.id;
  //
  // Create clear button configuration
  let sortActive = this.field.sortMode !== Client.IdfField.sortModes.NONE;
  let groupingActive = this.field.groupingMode !== Client.IdfField.groupingModes.NONE;
  if ((grouping && groupingActive) || (!grouping && sortActive)) {
    let label = grouping ? Client.IdfResources.t("FIL_DEGROUP_LABEL") : Client.IdfResources.t("FIL_SORT_CLEAR");
    let clearButtonConf = this.createElementConfig({c: "IonButton", label, className: "filter-popup-button", events: ["onClick"]});
    buttonsConf.children.push(clearButtonConf);
    if (grouping)
      this.clearGroupingButtonId = clearButtonConf.id;
    else
      this.clearSortButtonId = clearButtonConf.id;
  }
};


/**
 * Realize widget UI
 * @param {Object} widget
 * @param {View|Element|Widget} parent
 * @param {View} view
 */
Client.IdfFilterPopup.prototype.realize = function (widget, parent, view)
{
  // Create elements configuration
  this.createElementsConfig(widget);
  //
  Client.mainFrame.openView(this.viewConf);
  //
  this.createFilters();
};


/**
 * Update inner elements properties
 * @param {Object} props
 */
Client.IdfFilterPopup.prototype.updateElement = function (props)
{
  Client.Widget.prototype.updateElement.call(this, props);
};


/**
 * Handle an event
 * @param {Object} event
 */
Client.IdfFilterPopup.prototype.onEvent = function (event)
{
  let events = Client.Widget.prototype.onEvent.call(this, event);
  //
  switch (event.id) {
    case "onClick":
      if (event.obj === this.closeButtonConf.id)
        this.close(true);
      else if (event.obj === this.applyButtonConf.id)
        events.push(...this.handleApplyFilter());
      else if (event.obj === this.clearButtonConf.id)
        events.push(...this.handleClearFilter());
      else if (event.obj === this.sortAscButtonId)
        events.push(...this.handleSort(Client.IdfField.sortModes.ASC));
      else if (event.obj === this.sortDescButtonId)
        events.push(...this.handleSort(Client.IdfField.sortModes.DESC));
      else if (event.obj === this.clearSortButtonId)
        events.push(...this.handleSort(Client.IdfField.sortModes.NONE));
      else if (event.obj === this.groupingAscButtonId)
        events.push(...this.handleGrouping(Client.IdfField.groupingModes.ASC));
      else if (event.obj === this.groupingDescButtonId)
        events.push(...this.handleGrouping(Client.IdfField.groupingModes.DESC));
      else if (event.obj === this.clearGroupingButtonId)
        events.push(...this.handleGrouping(Client.IdfField.groupingModes.NONE));
      else {
        let closeButton = this.filterItemCloseButtons.find(button => button.id === event.obj);
        let closeButtonIdx = this.filterItemCloseButtons.findIndex(button => button.id === event.obj);
        if (closeButton) {
          this.filterItems[closeButtonIdx].close(true);
          //
          this.filterItems.splice(closeButtonIdx, 1);
          this.filterItemCloseButtons.splice(closeButtonIdx, 1);
          this.filterTypeControls.splice(closeButtonIdx, 1);
          this.filterValueControls.splice(closeButtonIdx, 1);
        }
      }
      break;

    case "chgProp":
      if (event.content.name === "value")
        events.push(...this.handleChange(event));
      else if (event.content.name === "checked" && this.field.smartLookup) {
        // Push rValue of selected items into smart lookup selection
        if (this.valueList) {
          let selectedItems = this.filterValueControls[0].getValueToSend();
          for (let i = 0; i < this.valueList.items.length; i++) {
            let item = this.valueList.items[i];
            //
            let idx = this.comboSelection.indexOf(item.rValue);
            let selected = selectedItems.find(sel => sel === item.value);
            //
            if (selected && idx === -1)
              this.comboSelection.push(item.rValue);
            else if (!selected && idx !== -1)
              this.comboSelection.splice(idx, 1);
          }
        }
      }
      break;

    case "onKey":
      events.push(...this.handleChange(event));
      break;
  }
  //
  return events;
};


/**
 * Create filter item
 * @param {Object} filter
 */
Client.IdfFilterPopup.prototype.createFilterItem = function (filter)
{
  filter = filter || {};
  //
  let controlType = this.fieldValue.getControlType();
  //
  let filtersArea = Client.eleMap[this.filtersAreaConf.id];
  //
  // Create filter item
  let filterItemConf = this.createElementConfig({c: "Container", className: "filter-popup-item"});
  let filterItem = this.view.createElement(filterItemConf, filtersArea, this.view);
  filtersArea.elements.push(filterItem);
  //
  this.filterItems.push(filterItem);
  //
  // Create filter type control
  this.createFilterTypeControl(filter.type, filterItem);
  //
  // Create filter value control
  this.createFilterValueControl(filter, filterItem);
  //
  if (controlType === Client.IdfField.controlTypes.EDIT) {
    // Create filter item close button
    let props = {
      c: "IonButton",
      icon: "close-circle",
      style: {visibility: filter.value || Client.IdfFilterPopup.isEmptyOrNotEmpty(filter.type) ? "visible" : "hidden"},
      className: "generic-btn filter-popup-remove-item-button",
      events: ["onClick"]
    };
    //
    let filterItemCloseButtonConf = this.createElementConfig(props);
    let filterItemCloseButton = this.view.createElement(filterItemCloseButtonConf, filterItem, this.view);
    filterItem.elements.push(filterItemCloseButton);
    //
    this.filterItemCloseButtons.push(filterItemCloseButton);
  }
};


/**
 * Get filter types based on field data type
 */
Client.IdfFilterPopup.prototype.getFilterTypes = function ()
{
  let types = [];
  //
  if (Client.IdfField.isNumeric(this.field.dataType) || Client.IdfField.isDateOrTime(this.field.dataType)) {
    types.push({value: Client.IdfFilterPopup.filterTypes.VALUE, name: Client.IdfResources.t("FIL_VALUE")});
    types.push({value: Client.IdfFilterPopup.filterTypes.MAJOR, name: Client.IdfResources.t("FIL_MAJOR")});
    types.push({value: Client.IdfFilterPopup.filterTypes.MINOR, name: Client.IdfResources.t("FIL_MINOR")});
    //
    // If data type is time or date time, don't add BETWEEN filter
    if (Client.IdfField.isTime(this.field.dataType))
      types.push({value: Client.IdfFilterPopup.filterTypes.BEETWEEN, name: Client.IdfResources.t("FIL_BETWEEN")});
    //
    types.push({value: Client.IdfFilterPopup.filterTypes.EMPTY, name: Client.IdfResources.t("FIL_EMPTY")});
    types.push({value: Client.IdfFilterPopup.filterTypes.NOTEMPTY, name: Client.IdfResources.t("FIL_NOTEMPTY")});
  }
  else {
    if (this.field.comboSeparator === ";") {
      types.push({value: Client.IdfFilterPopup.filterTypes.STARTS, name: Client.IdfResources.t("FIL_STARTS")});
      types.push({value: Client.IdfFilterPopup.filterTypes.ENDS, name: Client.IdfResources.t("FIL_ENDS")});
      types.push({value: Client.IdfFilterPopup.filterTypes.CONTAINS, name: Client.IdfResources.t("FIL_CONTAINS")});
    }
    types.push({value: Client.IdfFilterPopup.filterTypes.VALUE, name: Client.IdfResources.t("FIL_VALUE")});
    types.push({value: Client.IdfFilterPopup.filterTypes.DIFFERENT, name: Client.IdfResources.t("FIL_DIFFERENT")});
    types.push({value: Client.IdfFilterPopup.filterTypes.EMPTY, name: Client.IdfResources.t("FIL_EMPTY")});
    types.push({value: Client.IdfFilterPopup.filterTypes.NOTEMPTY, name: Client.IdfResources.t("FIL_NOTEMPTY")});
  }
  //
  return types;
};


/**
 * Create control to choose filter type
 * @param {String} filterTypeValue
 * @param {Element} container
 */
Client.IdfFilterPopup.prototype.createFilterTypeControl = function (filterTypeValue, container)
{
  // If control type is EDIT I have to create a COMBO control having filter types as items.
  // Otherwise (CHECK, OPTION, COMBO) I have to create an EDIT control to allow user to search values to filter by
  let isEdit = this.fieldValue.getControlType() === Client.IdfField.controlTypes.EDIT;
  //
  // Get filter types
  let filterTypes = this.getFilterTypes();
  //
  let props = {
    c: "IdfControl",
    enabled: !isEdit,
    canActivate: isEdit,
    activableDisabled: isEdit,
    visualStyle: this.fieldValue.getVisualStyle(),
    type: isEdit ? Client.IdfField.controlTypes.COMBO : Client.IdfField.controlTypes.EDIT,
    placeholder: isEdit ? "" : Client.IdfResources.t("FIL_SEARCH_PLACE"),
    valueList: isEdit ? {items: filterTypes} : undefined,
    value: isEdit ? filterTypeValue || filterTypes[0].value : undefined,
    container,
    className: "filter-popup-type-control"
  };
  let filterTypeControlConf = this.createElementConfig(props);
  //
  let filterTypeControl = this.view.createElement(filterTypeControlConf, container, this.view);
  this.elements.push(filterTypeControl);
  //
  this.filterTypeControls.push(filterTypeControl);
};


/**
 * Create control to insert filter value
 * @param {Object} filter
 * @param {Element} container
 */
Client.IdfFilterPopup.prototype.createFilterValueControl = function (filter, container)
{
  // Get field value control type
  let controlType = this.fieldValue.getControlType();
  //
  let filterValueType = Client.IdfField.controlTypes.CHECKLIST;
  let className = "filter-popup-checklist";
  let controlContainer = Client.eleMap[this.filtersAreaConf.id];
  //
  // If control type is EDIT I have to create an EDIT control to allow use to filter.
  // If control type is CHECK, I have to create an OPTION control having "true", "false", "empty" and "not empty" as items.
  // Otherwise (CHECK, OPTION, COMBO) I have to create a CHECKLIST control to allow user to filter by one or more values
  if (controlType === Client.IdfField.controlTypes.EDIT) {
    filterValueType = Client.IdfField.controlTypes.EDIT;
    className = Client.IdfFilterPopup.isEmptyOrNotEmpty(filter.type) ? "" : "filter-popup-value-control";
    controlContainer = container;
  }
  else if (controlType === Client.IdfField.controlTypes.CHECK) {
    filterValueType = Client.IdfField.controlTypes.OPTION;
    this.valueList = {items: [{value: "on", name: "true"}, {value: "", name: "false"}, {value: "!", name: "empty"}, {value: ".", name: "notempty"}]};
  }
  //
  let props = {
    c: "IdfControl",
    dataType: this.field.dataType,
    enabled: !Client.IdfFilterPopup.isEmptyOrNotEmpty(filter.type),
    canActivate: false,
    visualStyle: this.fieldValue.getVisualStyle(),
    type: filterValueType,
    comboSeparator: this.field.comboSeparator,
    value: filter.value,
    valueList: this.valueList,
    className,
    container: controlContainer,
    heightResize: false,
    numRows: 1
  };
  let filterValueControlConf = this.createElementConfig(props);
  //
  let filterValueControl = this.view.createElement(filterValueControlConf, controlContainer, this.view);
  this.elements.push(filterValueControl);
  //
  this.filterValueControls.push(filterValueControl);
};


/**
 * Create filters
 */
Client.IdfFilterPopup.prototype.createFilters = function ()
{
  let qbeFilter = this.field.qbeFilter || "";
  let filters = this.parseFilters(qbeFilter);
  filters = filters || [];
  //
  let controlType = this.fieldValue.getControlType();
  if (controlType === Client.IdfField.controlTypes.EDIT) {
    for (let i = 0; i < filters.length; i++)
      this.createFilterItem(filters[i]);
    //
    // Create an empty item in order to add more filters
    this.createFilterItem();
  }
  else {
    this.createFilterItem(filters[0]);
    //
    if (controlType === Client.IdfField.controlTypes.COMBO) {
      Client.mainFrame.sendEvents([{
          id: "qbecombo",
          def: Client.IdfMessagesPump.eventTypes.URGENT,
          content: {
            oid: this.field.id,
            obn: this.id,
            par1: this.field.smartLookup ? "*" : ""
          }
        }]);
    }
  }
};


/**
 * Handle change on my controls
 * @param {Object} event
 */
Client.IdfFilterPopup.prototype.handleChange = function (event)
{
  let events = [];
  //
  // Get field value control type
  let controlType = this.fieldValue.getControlType();
  //
  let filterTypeControl = this.filterTypeControls.find(control => control.id === event.obj);
  let filterValueControl = this.filterValueControls.find(control => control.id === event.obj);
  //
  let typeControlIdx = this.filterTypeControls.findIndex(control => control.id === event.obj);
  //
  // If change occurred on a filter type control
  if (filterTypeControl) {
    if (controlType !== Client.IdfField.controlTypes.EDIT) {
      let events = [];
      //
      if (event.content.value === undefined)
        return events;
      //
      // In case of smart lookup, ask server for new value list filtered by event.content.value
      if (this.field.smartLookup) {
        events.push({
          id: "qbecombo",
          def: Client.IdfMessagesPump.eventTypes.ACTIVE,
          content: {
            oid: this.field.id,
            obn: this.id,
            par1: event.content.value ? event.content.value : "*"
          }
        });
      }
      else // Otherwise filter value list locally
        this.filterValueControls[typeControlIdx].updateElement({filter: event.content.value.toLowerCase()});
      //
      return events;
    }
    //
    // If new type is "empty" or "not empty" I have to remove value (it not make sense with that filter types) and disabled filter value control
    let filterType = filterTypeControl.getValueToSend();
    let emptyOrNotEmptyType = Client.IdfFilterPopup.isEmptyOrNotEmpty(filterType);
    //
    let props = {};
    props.value = emptyOrNotEmptyType ? "" : undefined;
    props.enabled = !emptyOrNotEmptyType;
    props.className = emptyOrNotEmptyType ? "" : "filter-popup-value-control";
    //
    // Update filter value control associated with filter type control
    this.filterValueControls[typeControlIdx].updateElement(props);
  }
  else if (filterValueControl) // Otherwise if it occurred on a filter value control, update its value
    filterValueControl.updateElement({value: event.content.value, style: {visibility: "visible"}});
  //
  // Get last filter type and value
  let lastFilterTypeControl = this.filterTypeControls[this.filterTypeControls.length - 1];
  let lastFilterValueControl = this.filterValueControls[this.filterValueControls.length - 1];
  let lastFilterType = lastFilterTypeControl.getValueToSend();
  let lastFilterValue = lastFilterValueControl.getValueToSend();
  //
  // Check if last filter type is "empty" or "not empty"
  let emptyOrNotEmptyType = Client.IdfFilterPopup.isEmptyOrNotEmpty(lastFilterType);
  //
  // I have to create new filter item in two cases:
  // 1) change occurred on a filter type control and last filter type is "empty" or "not empty";
  // 2) change occurred on a filter value control and last filter value is not empty
  if ((filterTypeControl && emptyOrNotEmptyType) || (filterValueControl && lastFilterValue)) {
    // Show close button on every filter item
    for (let i = 0; i < this.filterItemCloseButtons.length; i++)
      this.filterItemCloseButtons[i].updateElement({style: {visibility: "visible"}});
    //
    this.createFilterItem();
  }
  //
  return events;
};


/**
 * Apply filter
 */
Client.IdfFilterPopup.prototype.handleApplyFilter = function ()
{
  let qbeFilter = "";
  let controlType = this.fieldValue.getControlType();
  //
  for (let i = 0; i < this.filterValueControls.length; i++) {
    let typeControl = this.filterTypeControls[i];
    let valueControl = this.filterValueControls[i];
    //
    let type = typeControl.getValueToSend();
    let value = valueControl.getValueToSend();
    //
    if (!value && !Client.IdfFilterPopup.isEmptyOrNotEmpty(type))
      continue;
    //
    switch (controlType) {
      case Client.IdfField.controlTypes.EDIT:
        qbeFilter += (i > 0 ? this.field.comboSeparator : "") + this.stringifyFilter({type, value});
        //
        // If qbeFilter starts with "=", add ";" at the beginning in order to handle multiple filters of that type
        if (qbeFilter.startsWith("=") && this.filterValueControls.length > 1)
          qbeFilter = ";" + qbeFilter;
        break;

      case Client.IdfField.controlTypes.CHECK:
        qbeFilter = value;
        break;

      default:
        // Smart lookup has to send rValue instead of value
        value = this.field.smartLookup ? this.comboSelection : value;
        qbeFilter += this.stringifyFilter({type: Client.IdfFilterPopup.filterTypes.VALUE, value});
        break;
    }
  }
  //
  // In case of smart lookup I have to set filter client side
  if (this.field.smartLookup) {
    this.field.updateElement({qbeFilter});
    //
    // Set filter on lookups linked to me
    let parentPanel = this.field.parent;
    for (let i = 0; i < parentPanel.fields.length; i++) {
      let field = parentPanel.fields[i];
      if (field.type === this.field.type)
        field.updateElement({qbeFilter});
    }
  }
  //
  // Close popup
  this.close(true);
  //
  // Send filter to server
  return [{id: "qbeset", def: Client.IdfMessagesPump.eventTypes.URGENT, content: {oid: this.field.id, obn: "qbefilter", par1: qbeFilter}}];
};


/**
 * Clear filter
 */
Client.IdfFilterPopup.prototype.handleClearFilter = function ()
{
  // In case of smart lookup I have to clear filter client side
  if (this.field.smartLookup) {
    this.field.updateElement({qbeFilter: ""});
    //
    // Clear also filter of lookups linked to me
    let parentPanel = this.field.parent;
    for (let i = 0; i < parentPanel.fields.length; i++) {
      let field = parentPanel.fields[i];
      if (field.type === this.field.type)
        field.updateElement({qbeFilter: ""});
    }
  }
  //
  // Close popup
  this.close(true);
  //
  // Clear filter
  return [{id: "qbeset", def: Client.IdfMessagesPump.eventTypes.URGENT, content: {oid: this.field.id, obn: "clear", par1: ""}}];
};


/**
 * Handle sort
 * @param {Integer} sortMode
 */
Client.IdfFilterPopup.prototype.handleSort = function (sortMode)
{
  // Close popup
  this.close(true);
  //
  return this.field.handleSort(sortMode);
};


/**
 * Handle grouping
 * @param {Integer} groupingMode
 */
Client.IdfFilterPopup.prototype.handleGrouping = function (groupingMode)
{
  // Close popup
  this.close(true);
  //
  return this.field.handleGrouping(groupingMode);
};


/**
 * Return true if given filterType is EMPTY or NOTEMPTY
 * @param {String} filterType
 */
Client.IdfFilterPopup.isEmptyOrNotEmpty = function (filterType)
{
  return filterType === Client.IdfFilterPopup.filterTypes.EMPTY || filterType === Client.IdfFilterPopup.filterTypes.NOTEMPTY;
};


/**
 * Parse filters
 * @param {String} filters
 */
Client.IdfFilterPopup.prototype.parseFilters = function (filters)
{
  if (!filters)
    return [];
  //
  let sep = this.field.comboSeparator;
  if (this.field.smartLookup && filters.indexOf(Client.IdfControl.rValueSeparator) !== -1)
    sep = Client.IdfControl.rValueSeparator;
  //
  let qbeFilters = filters.split(sep);
  //
  filters = [];
  for (let i = 0; i < qbeFilters.length; i++) {
    let qbeFilter = qbeFilters[i];
    if (!qbeFilter)
      continue;
    //
    let firstPart = qbeFilter.substring(0, qbeFilter.length - 1);
    let middlePart = qbeFilter.substring(1, qbeFilter.length - 1);
    let lastPart = qbeFilter.substring(1);
    //
    if (this.field.smartLookup) {
      // In case of smart lookup use item value as filter
      if (this.valueList) {
        let item = this.valueList.items.find(item => item.rValue === qbeFilter);
        if (item) {
          if (!filters[0])
            filters.push({type: Client.IdfFilterPopup.filterTypes.VALUE, value: ""});
          //
          filters[0].value += (i > 0 ? this.field.comboSeparator : "") + item.value;
        }
      }
    }
    else if (qbeFilter.startsWith("*") && qbeFilter.endsWith("*"))
      filters.push({type: Client.IdfFilterPopup.filterTypes.CONTAINS, value: middlePart});
    else if (qbeFilter.startsWith("*"))
      filters.push({type: Client.IdfFilterPopup.filterTypes.ENDS, value: lastPart});
    else if (qbeFilter.endsWith("*"))
      filters.push({type: Client.IdfFilterPopup.filterTypes.STARTS, value: firstPart});
    else if (qbeFilter.startsWith("#"))
      filters.push({type: Client.IdfFilterPopup.filterTypes.DIFFERENT, value: lastPart});
    else if (qbeFilter.startsWith(">"))
      filters.push({type: Client.IdfFilterPopup.filterTypes.MAJOR, value: lastPart});
    else if (qbeFilter.startsWith("<"))
      filters.push({type: Client.IdfFilterPopup.filterTypes.MINOR, value: lastPart});
    else if (qbeFilter.indexOf(":") !== -1 && !Client.IdfField.isTime(this.field.dataType)) {
      let idx = qbeFilter.indexOf(":");
      filters.push({type: Client.IdfFilterPopup.filterTypes.BETWEEN, value: qbeFilter.substring(0, idx), toValue: qbeFilter.substring(idx + 1)});
    }
    else if (qbeFilter === "!")
      filters.push({type: Client.IdfFilterPopup.filterTypes.EMPTY});
    else if (qbeFilter === ".")
      filters.push({type: Client.IdfFilterPopup.filterTypes.NOTEMPTY});
    else if (qbeFilter.startsWith("="))
      filters.push({type: Client.IdfFilterPopup.filterTypes.VALUE, value: lastPart});
    else { // Filter by value using app configuration
      let filterType = Client.IdfFilterPopup.filterTypes.VALUE;
      //
      let controlType = this.fieldValue.getControlType();
      if (controlType === Client.IdfField.controlTypes.EDIT) {
        if (!Client.IdfField.isDateOrTime(this.field.dataType) && !Client.IdfField.isNumeric(this.field.dataType) && Client.mainFrame.wep.panelLikeSearch)
          filterType = (Client.mainFrame.wep.panelLikeMode === 1 ? Client.IdfFilterPopup.filterTypes.STARTS : Client.IdfFilterPopup.filterTypes.CONTAINS);
        //
        filters.push({type: filterType, value: qbeFilter});
      }
      else {
        if (!filters[0])
          filters.push({type: filterType, value: ""});
        //
        filters[0].value += (i > 0 ? this.field.comboSeparator : "") + qbeFilter;
      }
    }
  }
  //
  return filters;
};


/**
 * Stringify filter
 * @param {Object} filter
 */
Client.IdfFilterPopup.prototype.stringifyFilter = function (filter)
{
  let controlType = this.fieldValue.getControlType();
  let filterString = "";
  //
  switch (filter.type) {
    case Client.IdfFilterPopup.filterTypes.VALUE:
      if (Client.IdfField.isDateOrTime(this.field.dataType))
        filterString = filter.value;
      else {
        if (controlType === Client.IdfField.controlTypes.EDIT)
          filterString = "=" + filter.value;
        else {
          let sep = this.field.smartLookup ? Client.IdfControl.rValueSeparator : this.field.comboSeparator;
          //
          for (let j = 0; j < filter.value.length; j++)
            filterString += (j > 0 ? sep : "") + filter.value[j];
        }
      }
      break;

    case Client.IdfFilterPopup.filterTypes.STARTS:
      filterString = filter.value + "*";
      break;

    case Client.IdfFilterPopup.filterTypes.ENDS:
      filterString = "*" + filter.value;
      break;

    case Client.IdfFilterPopup.filterTypes.CONTAINS:
      filterString = "*" + filter.value + "*";
      break;

    case Client.IdfFilterPopup.filterTypes.DIFFERENT:
      filterString = "#" + filter.value;
      break;

    case Client.IdfFilterPopup.filterTypes.MAJOR:
      filterString = ">" + filter.value;
      break;

    case Client.IdfFilterPopup.filterTypes.MINOR:
      filterString = "<" + filter.value;
      break;

    case Client.IdfFilterPopup.filterTypes.BETWEEN:
      filterString = filter.value + ":" + filter.toValue;
      break;

    case Client.IdfFilterPopup.filterTypes.EMPTY:
      filterString = "!";
      break;

    case Client.IdfFilterPopup.filterTypes.NOTEMPTY:
      filterString = ".";
      break;
  }
  //
  return filterString;
};


/**
 * Remove the element and its children from the element map
 * @param {boolean} firstLevel - if true remove the dom of the element too
 * @param {boolean} triggerAnimation - if true and on firstLevel trigger the animation of 'removing'
 */
Client.IdfFilterPopup.prototype.close = function (firstLevel, triggerAnimation)
{
  Client.eleMap[this.viewConf.id].close();
  //
  Client.Widget.prototype.close.call(this, firstLevel, triggerAnimation);
};


/**
 * Called by IdfField when its valueList changes
 * @param {Object} valueList
 */
Client.IdfFilterPopup.prototype.updateValueList = function (valueList)
{
  this.valueList = valueList;
  //
  let filterString = this.stringifyFilter({type: Client.IdfFilterPopup.filterTypes.VALUE, value: this.comboSelection});
  let oldFilter = this.parseFilters(filterString)[0] || {};
  //
  // Close old value control
  this.filterValueControls[0].close(true);
  this.filterValueControls.splice(0, 1);
  //
  // Create new value control
  this.createFilterValueControl(oldFilter);
};