/*
 * Instant Developer Cloud
 * Copyright Pro Gamma Spa 2000 - 2021
 * All rights reserved
 */

var Client = Client || {};


/**
 * @class A frame containing buttons
 * @param {Object} widget
 * @param {View|Element} parent - the parent element
 * @param {View} view
 */
Client.IdfTabbedView = function (widget, parent, view)
{
  // Set default events definition
  widget = Object.assign({
    clickEventDef: Client.IdfMessagesPump?.eventTypes.ACTIVE,
    changePageAnimationDef: Client.IdfWebEntryPoint.getAnimationDefault("tab"),
    onlyContent: true
  }, widget);
  //
  Client.IdfFrame.call(this, widget, parent, view);
};


// Make Client.IdfTabbedView extend Client.IdfFrame
Client.IdfTabbedView.prototype = new Client.IdfFrame();

Client.IdfTabbedView.getRequirements = Client.IdfFrame.getRequirements;

Client.IdfTabbedView.transPropMap = Object.assign({}, Client.IdfFrame.transPropMap, {
  sel: "selectedPage",
  pla: "placement",
  hid: "hiddenTabs",
  cpa: "changePageAnimationDef"
});

Client.IdfTabbedView.place = {
  TOP: 1,
  BOTTOM: 3
};


/**
 * Convert properties values
 * @param {Object} props
 */
Client.IdfTabbedView.convertPropValues = function (props)
{
  props = props || {};
  //
  Client.IdfFrame.convertPropValues(props);
  //
  for (let p in props) {
    switch (p) {
      case Client.IdfTabbedView.transPropMap.hid:
        props[p] = props[p] === "1";
        break;

      case Client.IdfTabbedView.transPropMap.sel:
        // IDF is 1 based and IDC 0 based
        props[p] = parseInt(props[p], 10) - 1;
        break;

      case Client.IdfTabbedView.transPropMap.pla:
        props[p] = parseInt(props[p], 10);
        break;
    }
  }
};


/**
 * Create elements configuration
 * @param {Object} widget
 */
Client.IdfTabbedView.prototype.createElementsConfig = function (widget)
{
  Client.IdfFrame.prototype.createElementsConfig.call(this, widget);
  //
  // The Tabbed view has a different structure
  this.tabsContainerConf = this.createElementConfig({c: "IonTabs", className: "tabbed-view-box", ignoreTabbar: true});
  this.contentContainerConf.children.push(this.tabsContainerConf);
  //
  // Set animation (if required)
  let a = Client.IdfWebEntryPoint.getAnimationByDef(this.changePageAnimationDef);
  if (a)
    this.tabsContainerConf.animations = [{trigger: "change", ...a}];
};


/**
 * Append a child DOM Object to root object DOM
 * @param {Element} child - child element that requested the insertion
 * @param {HTMLElement} domObj - child DOM object to add
 */
Client.IdfTabbedView.prototype.appendChildObject = function (child, domObj)
{
  let el = Client.eleMap[this.tabsContainerConf.id];
  el.appendChildObject(child, domObj);
  el.elements.push(child);
  child.parent = el;
};


/**
 * Update inner elements properties
 * @param {Object} props
 */
Client.IdfTabbedView.prototype.updateElement = function (props)
{
  props = props || {};
  let el = Client.eleMap[this.tabsContainerConf.id];
  //
  Client.IdfFrame.prototype.updateElement.call(this, props);
  //
  if (props.selectedPage !== undefined) {
    this.selectedPage = props.selectedPage;
    el.updateElement({selectedPage: props.selectedPage});
  }
  //
  if (props.placement !== undefined) {
    this.placement = props.placement;
    el.updateElement({placement: (props.placement === Client.IdfTabbedView.place.BOTTOM ? "bottom" : "top")});
  }
  //
  // hiddenTabs from IDF, onlycontent from IDC
  if (props.hiddenTabs !== undefined) {
    this.hiddenTabs = props.hiddenTabs;
    Client.Widget.updateElementClassName(el, "only-content", !this.hiddenTabs);
  }
};


/**
 * Remove a single page
 * @param {Client.IdfTab} page
 */
Client.IdfTabbedView.prototype.deletePage = function (page)
{
  page.close();
  //
  // Remove from my element list
  let elements = this.elements || [];
  for (let i = 0; i < elements.length; i++)
    if (elements[i] === page) {
      elements.splice(i, 1);
      break;
    }
  //
  // Remove also from the IonTabs list
  let el = Client.eleMap[this.tabsContainerConf.id];
  if (el) {
    elements = el.elements || [];
    for (let i = 0; i < elements.length; i++)
      if (elements[i] === page) {
        elements.splice(i, 1);
        break;
      }
  }
};


/**
 * Handle an event
 * @param {Object} event
 */
Client.IdfTabbedView.prototype.onEvent = function (event)
{
  let events = Client.IdfFrame.prototype.onEvent.call(this, event);
  //
  switch (event.id) {
    case "chgProp":
      // This event has been generated by a widget child element, but server side it has to be notified to widget (valid for IDC)
      if (event.content.name === "selectedPage") {
        this.selectedPage = event.content.value;
        //
        if (this.elements[this.selectedPage]?.elements[0]?.delayResize)
          Client.mainFrame.sendEvents(this.elements[this.selectedPage].elements[0].handleResize());
        //
        // Give event the IDF format
        if (Client.mainFrame.isIDF)
          events.push({
            id: "tab",
            def: this.clickEventDef,
            content: {
              oid: this.elements[this.selectedPage].id,
              obn: this.id
            }
          });
      }
      //
      if (!Client.mainFrame.isIDF) {
        event.obj = this.id;
        events.push(event);
      }
      break;

    case "onChangePage":
      // This event has been generated by a widget child element, but server side it has to be notified to widget (valid for IDC)
      if (!Client.mainFrame.isIDF) {
        event.obj = this.id;
        events.push(event);
      }
      break;
  }
  //
  return events;
};


/**
 * Get root object. Root object is the object where children will be inserted
 * @param {Boolean} el - if true, get the element itself istead of its domObj
 */
Client.IdfTabbedView.prototype.getRootObject = function (el)
{
  let rootObject = this.mainObjects[0];
  return el ? rootObject : rootObject.domObj;
};


/**
 * Get click detail
 * @param {Object} event
 * @param {Widget} srcWidget
 */
Client.IdfTabbedView.prototype.getClickDetail = function (event, srcWidget)
{
  let detail = Client.IdfFrame.prototype.getClickDetail.call(this, event, srcWidget);
  //
  // TODO: identify the click on the page tab
  if (Client.mainFrame.isIDF)
    detail.par4 = srcWidget?.id;
  else {
    if (srcWidget instanceof Client.IdfTab)
      detail.tab = srcWidget.index;
  }
  //
  return detail;
};
