// ************************************************
// Instant Developer RD3 Framework for Bootstrap
// (c) 1999-2016 Pro Gamma Srl - All rights reserved
//
// Classe TreeNode: Rappresenta un nodo in un
// frame di tipo Tree
// ************************************************

function TreeNode(ptree,pnode)
{
  // Proprieta' di questo oggetto di modello
  this.Identifier = null;				// Identificatore del nodo (univoco anche rispetto alla form/tree)
  this.Caption = "";						// Nome del nodo
  this.Tooltip = "";						// Tooltip del nodo
  this.Image = "";							// Icona del nodo
  this.Selected = false;        // Nodo multiselezionato?
  this.CanBeChecked = true;     // Il nodo puo' essere multiselezionato?
  this.Expanded = false;     		// Il nodo e' attualmente espanso?
  this.AlreadyExpanded = false; // Il nodo e' mai stato espanso?
  this.Badge = "";              // Testo del Badge da assegnare al nodo
  this.ClassName;               // Classe aggiuntiva da applicare al nodo (this.CaptionBox)
  //
  // Oggetti figli di questo nodo
  this.Nodes = new Array();     // Elenco dei nodi figli di questo
  //
  // Altre variabili di modello...
  this.ParentTree = ptree;      // L'oggetto albero a cui appartengo
  this.ParentNode = pnode;      // L'oggetto nodo a cui appartengo
  this.WaitForChildren = false; // Sono in attesa dei miei figli (prima espansione..)
  this.ExecuteGFX = true;       // Flag che permette di disabilitare l'animazione di fold in alcuni casi
  //
  // Variabili di collegamento con il DOM
  this.Realized = false; // Se vero, gli oggetti del DOM sono gia' stati creati
  //
  // Oggetti visuali relativi al nodo
  this.NodeBox = null;     // Il DIV complessivo del nodo
  this.CaptionBox = null;  // Il DIV della sola barra del titolo del nodo
  this.ExpandImg = null;   // Immagine da mostrare se il nodo ha dei figli e che indica l'espansione
  this.CheckBox = null;    // CheckBox del Nodo
  this.Filler = null;      // Filler per fare le righine verticali dei nodi padri
  this.NodeContent = null; // Span che contiene l'icona e il testo del nodo
  this.NodeImg = null;     // Immagine associata al nodo
  this.NodeText = null;    // Span contente il testo della caption del nodo
  this.ChildrenBox = null; // Il DIV che contiene i figli del nodo
  this.BadgeObj = null;    // Il DIV che mostra il Badge
  //
  //this.Ea = null;        // Funzione di End Animation mobile
}


// *******************************************************************
// Inizializza questo TreeNode leggendo i dati da un nodo <trn> XML
// *******************************************************************
TreeNode.prototype.LoadFromXml = function(node) 
{
  // Inizializzo le proprieta' locali
  this.LoadProperties(node);
  //
  var objlist = node.childNodes;
  var n = objlist.length;
  //
  // Ciclo su tutti i nodi che rappresentano oggetti figli
  for (var i=0; i<n; i++) 
  {
    var objnode = objlist.item(i);
    var nome = objnode.nodeName;
    //
    // In base al tipo di oggetto, invio il messaggio di caricamento
    switch (nome)
    {
      case "trn":
      {
        var newnode = new TreeNode(this.ParentTree,this);
        newnode.LoadFromXml(objnode);
        //
        this.Nodes.push(newnode);
      }
      break;
    }
  }    
}


// **********************************************************************
// Esegue un evento di change che riguarda le proprieta' di questo oggetto
// **********************************************************************
TreeNode.prototype.ChangeProperties = function(node)
{
  // Vediamo se nel nodo di cambiamento sono indicati anche nodi figli...
  var objlist = node.childNodes;
  //
  // In IE il primo nodo e' gia' l'elemento, negli altri il primo nodo e' un "\n"
  var trn = RD3_Glb.HasNode(node, "trn");
  //
  if (objlist.length>0 && trn)
  {
    // In questo caso elimino i figli miei e poi carico gli altri
    this.RemoveChildren();
    this.LoadFromXml(node);
    //
    if (this.Realized && !RD3_Glb.IsMobile())
    {
      var n=this.Nodes.length;
      for (var i=0; i<n; i++)
      {
        this.Nodes[i].Realize(this.ChildrenBox);
      }
      //
      // Tento di riselezionare il nodo corretto nell'albero
      if (!this.ParentTree.SelectedNode && this.ParentTree.RefreshSelected != "")
        this.ParentTree.SetSelectedNode(this.ParentTree.RefreshSelected);
    }
    //
    // Se ero in attesa di miei figli mi allora mi memorizzo che sono arrivati ed eseguo l'animazione di folding,
    // se non ero in attesa di figli e me ne sono arrivati salto l'animazione e vado subito in stato finale..
    if (this.WaitForChildren)
      this.WaitForChildren = false;
    else
      this.ExecuteGFX = false;
    //
    // Ricalcolo il tipo di immagine da mostrare
    this.HandleExpandable();
    this.ExecuteGFX = true;
    if (RD3_Glb.IsMobile() && this.Expanded)
    	this.LoadNestedNode(true);
  }
  else
  {
    // Normale cambio di proprieta'
    this.LoadProperties(node);
    //
    // Se stavo aspettando dei figli devo eseguire l'handle dell'espansione anche se non ci sono figli, in modo da togliere
    // l'immagine di attesa figli
    if (this.WaitForChildren) {
      this.WaitForChildren = false;
      this.HandleExpandable();
    }
  }
}


// **************************************************************
// Inizializza le proprieta' di questo oggetto leggendole dal
// nodo xml arrivato.
// **************************************************************
TreeNode.prototype.LoadProperties = function(node)
{
  // Ciclo su tutti gli attributi del nodo
  var attrlist = node.attributes;
  var n = attrlist.length;
  for (var i=0; i<n; i++)
  {
    var attrnode = attrlist.item(i);
    var nome = attrnode.nodeName;
    var valore = attrnode.nodeValue;
    //
    switch(nome)
    {
      case "aex": this.SetAlreadyExpanded(attrnode.nodeValue=="1"); break;
      case "exp": this.SetExpanded(attrnode.nodeValue=="1"); break;
      case "cap": this.SetCaption(attrnode.nodeValue); break;
      case "tip": this.SetTooltip(attrnode.nodeValue); break;
      case "img": this.SetImage(attrnode.nodeValue); break;
      case "cch": this.SetCanCheck(attrnode.nodeValue=="1"); break;
      case "sel": this.SetSelected(attrnode.nodeValue=="1"); break;
      case "bdg": this.SetBadge(attrnode.nodeValue); break;
      case "cln": this.SetClassName(attrnode.nodeValue); break;
                  
      case "id": 
        this.Identifier = valore;
        RD3_DesktopManager.ObjectMap.add(valore, this);
      break;
    }
  }
}


// *******************************************************************
// Setter delle proprieta'
// *******************************************************************
TreeNode.prototype.SetAlreadyExpanded= function(value) 
{
  var old = this.AlreadyExpanded;
  this.AlreadyExpanded = value;
  //
  if (this.Realized && old != this.AlreadyExpanded)
    this.HandleExpandable();
}

TreeNode.prototype.SetExpanded= function(value, immediate, force) 
{
  var old = this.Expanded;
  this.Expanded = value;
  //
  if (this.Realized && (old != this.Expanded || force)) {
    this.HandleExpandable();
    if (this.ExpandSkipped)
      this.ExpandSkipped = false;
  }
  else if (this.Expanded) {
    // Mi memorizzo che il nodo va espanso..
    this.ExpandSkipped = true;
  }
}

TreeNode.prototype.SetCaption= function(value) 
{
  this.Caption = value;
  //
  if (this.Realized)
  {
    if (this.NodeText)
    	this.NodeText.innerHTML = RD3_Glb.HandleIconString(this.Caption);
    else
    	this.CaptionBox.innerHTML = RD3_Glb.HandleIconString(this.Caption);
  }
}

TreeNode.prototype.SetTooltip= function(value) 
{
  this.Tooltip = value;
  //
  if (this.Realized) {
  	RD3_TooltipManager.SetObjTitle(this.CaptionBox, this.Tooltip);
  }
}

TreeNode.prototype.SetImage= function(value) 
{
  this.Image = value;
  //
  if (this.Realized) {
    if (this.Image != "") {
      this.NodeImg.style.display = "";
      this.NodeImg.src = RD3_Glb.GetImgSrc("images/" + this.Image);
    }
    else
      this.NodeImg.style.display = "none";
  }
}

TreeNode.prototype.SetCanCheck = function (value) {
  this.CanBeChecked = value;
  //
  if (this.Realized) {
    // I check vanno mostrati se io li posso mostrare e il mio albero li puo' mostrare
    if (this.CanBeChecked && this.ParentTree.MultipleSelection)
      this.CheckBox.style.display = "";
    else
      this.CheckBox.style.display = "none";
  }
}

TreeNode.prototype.SetSelected = function (value) {
  this.Selected = value;
  //
  if (this.Realized) {
    if (this.Selected)
      this.CheckBox.checked = true;
    else
      this.CheckBox.checked = false;
  }
}

TreeNode.prototype.SetBadge= function(value) 
{
  this.Badge = value;
  //
  if (this.Realized)
  {
    if (this.Badge == "" && this.BadgeObj != null)
    {
      this.BadgeObj.parentElement.removeChild(this.BadgeObj);
      this.BadgeObj = null;
    }
    else if (this.Badge != "")
    {
      if (this.BadgeObj == null)
      {
	      this.BadgeObj = document.createElement("span");
	      this.BadgeObj.setAttribute("id", this.Identifier+":bdg");
	      this.BadgeObj.className = "badge";
	      this.BadgeObj.style.marginLeft = "10px";
	      //
	      this.CaptionBox.appendChild(this.BadgeObj);
	    }
	    //
	    this.BadgeObj.innerHTML = this.Badge;
    }
  }
}

// *******************************************************************
// Applico la classe
// *******************************************************************
TreeNode.prototype.SetClassName = function(cls)
{
  var old = this.ClassName;
  if (cls != undefined)
    this.ClassName = cls;
  //
  if (this.Realized && (old != this.ClassName || !cls)) 
  {
    // Rimuovo la classe precedente
    if (old != "") 
      RD3_Glb.RemoveClass2(this.CaptionBox, old);
    //
    // Applico la nuova classe alla fine della lista
    if (this.ClassName && this.ClassName != "")
      RD3_Glb.AddClass(this.CaptionBox, this.ClassName);
  }
}

// ***************************************************************
// Crea gli oggetti DOM utili a questo oggetto
// L'oggetto parent indica all'oggetto dove devono essere contenuti
// i suoi oggetti figli nel DOM
// ***************************************************************
TreeNode.prototype.Realize = function(parent)
{
  // realizzo i miei oggetti visuali
  // Creo il mio contenitore globale
  this.NodeBox = document.createElement("div");
  this.NodeBox.setAttribute("id", this.Identifier);
  this.NodeBox.className = "tree-node-container";
  //
  this.CaptionBox = document.createElement("div");
  this.CaptionBox.setAttribute("id", this.Identifier+":cap");
  this.CaptionBox.className = "tree-node-caption";
  //
  this.ExpandImg = document.createElement("button");
  this.ExpandImg.className = "btn btn-link btn-sm tree-exp-button";
  this.ExpandImg.appendChild(RD3_Glb.createFAImage(RD3_ClientParams.FA_ICON_TREEEXP));
  //
  this.CheckBox = document.createElement("input");
  this.CheckBox.type = "checkbox";
  this.CheckBox.className = "tree-node-check";
  //
  this.NodeContent = document.createElement("span");
  //
  this.NodeImg = document.createElement("img");
  this.NodeImg.className = "tree-node-img";
  //
  this.NodeText = document.createElement("span");
  this.NodeText.setAttribute("id", this.Identifier+":txt");
  this.NodeText.className = "tree-node-text";
  //
  this.NodeContent.appendChild(this.NodeImg);
  this.NodeContent.appendChild(this.NodeText);
  //
  this.CaptionBox.appendChild(this.ExpandImg);
  this.CaptionBox.appendChild(this.CheckBox);
  this.CaptionBox.appendChild(this.NodeContent);
  //
  this.ChildrenBox = document.createElement("div");
  this.ChildrenBox.setAttribute("id", this.Identifier+":chi");
  this.ChildrenBox.className = "tree-node-children";
  //
  this.NodeBox.appendChild(this.CaptionBox);
  this.NodeBox.appendChild(this.ChildrenBox);
  //
  // Gestisco il click e l'expand
  var _this = this;
  this.CaptionBox.onclick = function(ev) { _this.OnClickNode(ev); };
  this.ExpandImg.onclick = function(ev) { _this.OnExpandNode(ev); };
  this.CheckBox.onclick = function(ev) { _this.OnCheckNode(ev); };
  this.CheckBox.onkeypress = function(ev) { _this.OnKeyPress(ev); };
  this.CaptionBox.oncontextmenu = function(ev) { _this.OnRightClick(ev); };
  //
  parent.appendChild(this.NodeBox);
  //
  // Poi chiedo ai miei figli di realizzarsi
  var n = this.Nodes.length;
  for(var i=0; i<n; i++)
    this.Nodes[i].Realize(this.ChildrenBox);
  //
  // Eseguo l'impostazione iniziale delle mie proprieta' (quelle che cambiano l'aspetto visuale)
  this.Realized = true;
  this.HandleExpandable();
  this.SetCaption(this.Caption);
  this.SetTooltip(this.Tooltip);
  this.SetExpanded(this.Expanded);
  this.SetAlreadyExpanded(this.AlreadyExpanded);
  this.SetImage(this.Image);
  this.SetExpanded(this.Expanded);
  this.SetCanCheck(this.CanBeChecked);
  this.SetSelected(this.Selected);
  this.SetEnabled(this.ParentTree.Enabled);
  this.SetBadge(this.Badge);
  this.SetClassName();
}


// ********************************************************************************
// Gestore evento di click su nodo?
// ********************************************************************************
TreeNode.prototype.OnClickNode= function(evento)
{ 
  if (!this.ParentTree.Enabled)
    return;
  //
  if (window.event && evento==undefined)
    evento = window.event;
  //
  var ev = new IDEvent("clk", this.Identifier, evento, this.ParentTree.ClickEventDef);
  //
  if (ev.ClientSide)
    this.ParentTree.SetSelectedNode(this.Identifier);
}



// ********************************************************************************
// Gestore evento di click destro su caption
// ********************************************************************************
TreeNode.prototype.OnRightClick= function(evento)
{ 
  if (!this.ParentTree.Enabled)
    return false;
  //
  if (RD3_Glb.IsTouch())
  {
    var x = RD3_Glb.GetScreenLeft(this.NodeText);
    var w = this.NodeText.offsetWidth;
    if (evento.clientX < x || evento.clientX>x+w)
      return false;
  }
  //  
  if (this.ParentTree.PopupMenu)
  {
    this.ParentTree.TouchMoved = true;
    this.ParentTree.TouchStartX = -1;
    //
    var ev = new IDEvent("rclk", this.Identifier, evento, this.ParentTree.ClickEventDef, "", this.ParentTree.PopupMenu.Identifier);
    //
    // Disabilito click 
    evento.preventDefault();
    evento.stopPropagation();
    return true;
  }
  //
  return false;
}

// ********************************************************************************
// Gestore evento di mouse over
// ********************************************************************************
TreeNode.prototype.OnMouseOverObj= function(evento, obj)
{ 
  
}


// ********************************************************************************
// Gestore evento di mouse out
// ********************************************************************************
TreeNode.prototype.OnMouseOutObj= function(evento, obj)
{ 
  
}

// ********************************************************************************
// Gestore evento di click su caption
// ********************************************************************************
TreeNode.prototype.OnExpandNode= function(evento)
{ 
  if (!this.ParentTree.Enabled)
    return;
  //
  if (window.event && evento==undefined)
    evento = window.event;
  //
  //
  if (!this.AlreadyExpanded || this.Nodes.length == 0)
  {
    var ev = new IDEvent("trnexp", this.Identifier, evento, this.ParentTree.FirstExpandDef);
    this.WaitForChildren = true;
    if (this.ExpandImg)
      RD3_Glb.updateFAImage(this.ExpandImg, RD3_ClientParams.FA_ICON_TREELOAD);
  }
  else
  {
    var ev = new IDEvent("trnexp", this.Identifier, evento, this.ParentTree.ExpandEventDef);
    if (ev.ClientSide)
      this.SetExpanded(!this.Expanded);
  }
  //
  // Dato che fermero' il bubbling dell'evento faccio scattare io l'onclick di webentrypoint
  // perche' altrimenti verrebbe saltato
  // RD3_DesktopManager.WebEntryPoint.OnClick(evento);
  //
  // Devo fermare il bubbling dell'evento, in modo da non fare scattare il click sul nodo
  RD3_Glb.StopEvent(evento);
  return false;
}


// ********************************************************************************
// Gestore evento di check del nodo
// ********************************************************************************
TreeNode.prototype.OnCheckNode= function(evento)
{ 
  if (!this.ParentTree.Enabled)
    return;
  //
  if (window.event && evento==undefined)
    evento = window.event;
  //
  var value;
  var def = this.ParentTree.CheckEventDef;
  if (RD3_Glb.IsMobile()) {
  	value = this.Selected ? "" : "on";
  	def = RD3_Glb.EVENT_ACTIVE;
  }
  else {
  	value = this.CheckBox.checked ? "on" : "";
  }
  var ev = new IDEvent("chg", this.Identifier, evento, def, "check", value);
  //
  // Devo fermare il bubbling dell'evento, in modo da non fare scattare il click sul nodo
  if (evento.stopPropagation)
    evento.stopPropagation();
  else
    evento.cancelBubble = true;
  //
  return true;
}


// ********************************************************************************
// Gestore evento di check del nodo
// ********************************************************************************
TreeNode.prototype.OnKeyPress= function(evento)
{ 
  
}


// **********************************************************************
// Rimuove i figli di questo nodo
// **********************************************************************
TreeNode.prototype.RemoveChildren = function()
{
  var n=this.Nodes.length;
  for (var i=0; i<n; i++)
    this.Nodes[i].Unrealize();
}


// **********************************************************************
// Rimuove questo nodo
// **********************************************************************
TreeNode.prototype.Unrealize = function()
{
  // Tolgo l'oggetto dalla mappa comune
  RD3_DesktopManager.ObjectMap.remove(this.Identifier);
  //
  // Passo il messaggio ai figli
  var n=this.Nodes.length;
  for (var i=0; i<n; i++)
    this.Nodes[i].Unrealize();
  //
  // Elimino gli oggetti visuali
  if (RD3_Glb.IsMobile() && this.CaptionBox && this.CaptionBox.parentNode)
    this.CaptionBox.parentNode.removeChild(this.CaptionBox);
  if (this.NodeBox && this.NodeBox.parentNode)
    this.NodeBox.parentNode.removeChild(this.NodeBox);
  if (this.BackArrowImg && this.BackArrowImg.parentNode)
    this.BackArrowImg.parentNode.removeChild(this.BackArrowImg);
  if (this.BackImg && this.BackImg.parentNode)
    this.BackImg.parentNode.removeChild(this.BackImg);
  //
  // Annullo i riferimenti
  this.NodeBox = null;
  this.CaptionBox = null; 
  this.ExpandImg = null;  
  this.CheckBox = null; 
  this.NodeImg = null;
  this.NodeText = null; 
  this.ChildrenBox = null;
  this.BadgeObj = null;
  //
  this.Realized = false;
}


// **********************************************************************
// Deve tornare vero se l'oggetto e' draggabile
// **********************************************************************
TreeNode.prototype.IsDraggable = function(id)
{
  // i nodi si prendono solo dal testo
  var exit = (id.substr(id.length-4) != ":txt" ? true : false);
  //
  if (RD3_Glb.IsMobile() && id.substr(id.length-4) == ":cap")
    exit = false;
  //
  if (exit)
    return false;
  //
  // La draggabilita' di un nodo dipende dall'albero: se l'albero e' draggabile il nodo e' draggabile
  return (this.ParentTree.DragDrop || this.ParentTree.CanDrag) && this.ParentTree.Enabled;
}


// **********************************************************************
// Drop effettuato sull'oggetto
// **********************************************************************
TreeNode.prototype.OnDrop = function(obj, evento)
{
  // Non lancio evento se:
  // 1) E' stata attivata la nuova gestione drop
  // 2) L'albero non e' attivo
  // 3) Non c'e' la vecchia vesione drop
  // 4) Non e' stato tirato un treenode
  if (this.CanDrop || !this.ParentTree.Enabled || !this.ParentTree.DragDrop || !(obj instanceof TreeNode))
    return false;
  //
  // Invio semplicemente l'evento di drop
  var ev = new IDEvent("drp", this.Identifier, evento, RD3_Glb.EVENT_ACTIVE, obj.Identifier);
  return true;
}


// **********************************************************************
// Metodo che valuta in base ai figli e allo stato di espansione se
// mostrare o meno l'immagine di espansione o l'immagine del nodo 
// senza figli
// **********************************************************************
TreeNode.prototype.HandleExpandable= function(flup) 
{
  if (this.Realized)
  {
    var exp = true;
    var n = this.Nodes.length;
    if (!this.AlreadyExpanded || (!this.Expanded && n > 0))
    {
      // Sono collassato, mostro l'immagine di espansione
      this.ExpandImg.style.visibility = "";
      exp = false;
    }
    else if (this.AlreadyExpanded && this.Expanded && n > 0)
    {
      // Sono espanso, mostro l'immagine di collassamento
      this.ExpandImg.style.visibility = "";
      exp = true;
    }
    else
    {
      // Altrimenti nascondo l'immagine di espansione e mostro quella del figlio singolo
      this.ExpandImg.style.visibility = "hidden";
      exp = false;
    }
    //
    // In base allo stato (espanso o meno) mostro i miei figli
    if (!this.WaitForChildren)
    {
      // Se non ho figli non eseguo nessuna animazione ma vado subito nello stato finale
      var fx = new GFX("tree", exp, this.ChildrenBox, (n<=0)||(!this.ExecuteGFX), null, this.ParentTree.ExpandAnimDef);
      RD3_GFXManager.AddEffect(fx);
    }
    //
    // In base allo stato scelgo l'immagine di espansione giusta
    if (!this.WaitForChildren)
      RD3_Glb.updateFAImage(this.ExpandImg, exp ? RD3_ClientParams.FA_ICON_TREECOLL : RD3_ClientParams.FA_ICON_TREEEXP);
  }
}


// **********************************************************************
// Lo stato del flag mosta check box dell'albero e' cambiato:
// questo metodo aggiorna la visualizzazione di questo nodo
// **********************************************************************
TreeNode.prototype.MultipleSelectionChanged = function()
{
  // Mostro o meno i miei checkbox
  this.SetCanCheck(this.CanBeChecked);
  //
  // Passo il messaggio ai miei figli
  var n = this.Nodes.length;
  for (var i=0; i<n; i++)
  {
    this.Nodes[i].MultipleSelectionChanged();
  }
  //
  // Se sono passato da multi-selezione a non multi-selezione
  // perdo la selezione corrente
  if (!this.ParentTree.MultipleSelection)
  	this.SetNodeClass(false, true);
}


// **********************************************************************
// Metodo che svuota la cache del nodo
// **********************************************************************
TreeNode.prototype.ResetCache = function()
{
	// Svuoto la cache dei miei figli
  this.RemoveChildren();
  //
  this.Nodes.splice(0, this.Nodes.length);
}


// **********************************************************************
// Restituisce l'oggetto Dom a cui associare un Popup Menu
// **********************************************************************
TreeNode.prototype.GetDOMObj = function()
{
  return this.NodeText?this.NodeText:this.CaptionBox;
}


// *****************************************************************************
// Restituisce l'oggetto visuale su cui deve venire applicata l'HL per il drag
// *****************************************************************************
TreeNode.prototype.DropElement = function()
{
  return this.NodeContent?this.NodeContent:this.CaptionBox;
}


// ********************************************************************************
// Compone la lista di drop della box
// ********************************************************************************
TreeNode.prototype.ComputeDropList = function(list, dragobj)
{
  if (!this.Realized)
    return;
  //
  // Questa box vuole essere droppata da dragobj?
  // Si da per scontato che l'albero abbia DragDrop attivo (lo controlla lui)
  if (this!=dragobj)
  {
    list.push(this);
    //
    var drObj = (this.NodeContent ? this.NodeContent : this.CaptionBox);
    //
    // Calcolo le coordinate assolute...
    this.AbsLeft = RD3_Glb.GetScreenLeft(drObj, true);
    this.AbsTop = RD3_Glb.GetScreenTop(drObj, true);
    //
    this.AbsRight = this.AbsLeft + drObj.offsetWidth - 1;
    this.AbsBottom = this.AbsTop + drObj.offsetHeight - 1;
  }
  //
  // ora i sottonodi
  if (this.Expanded && this.Nodes)
  {
    var n = this.Nodes.length;
    for (var i = 0; i<n; i++)
      this.Nodes[i].ComputeDropList(list, dragobj);
  }
}


// **********************************************************************
// Ritorna il frame che contiene il nodo
// **********************************************************************
TreeNode.prototype.GetParentFrame = function()
{
  return this.ParentTree;
}


// **********************************************************************
// Abilita o disabilita il nodo
// **********************************************************************
TreeNode.prototype.SetEnabled = function(value)
{
  if (this.CheckBox)
    this.CheckBox.disabled = !value;
  //
  if (this.CaptionBox)
    RD3_Glb.ApplyCursor(this.CaptionBox, value?"pointer":"default");
  //
  // Aggiorno tutti i sottonodi
  var n = this.Nodes.length;
  for (var i = 0; i<n; i++)
    this.Nodes[i].SetEnabled(value);
}

// **********************************************************************
// Ritorna TRUE se il nodo e' l'ultimo nella collection del padre
// **********************************************************************
TreeNode.prototype.IsLastChild = function()
{
  if (this.ParentNode) {
    if (this.ParentNode.Nodes && this==this.ParentNode.Nodes[this.ParentNode.Nodes.length-1])
      return true;
  }
  else if (this.ParentTree.RootNodes && this==this.ParentTree.RootNodes[this.ParentTree.RootNodes.length-1])
    return true;
  //
  return false;
}


// ****************************************************************************
// Usato nel D&D: restituisce l'offset left da usare durante il posizionamento
// tenendo conto delle scrollbar e del browser
// ****************************************************************************
TreeNode.prototype.AccountOverFlowX = function()
{
  // Se IE non devo tenere conto delle scrollbar, sugli altri browser si
  return RD3_Glb.IsIE(10, false) ? 0 : this.ParentTree.ContentBox.scrollLeft;
}


// ****************************************************************************
// Usato nel D&D: restituisce l'offset top da usare durante il posizionamento
// tenendo conto delle scrollbar e del browser
// ****************************************************************************
TreeNode.prototype.AccountOverFlowY = function()
{
  // Se il sistema usa getBoundingClientRect allora alle scrollbar ci pensa lei, io non devo fare nulla
  if (this.ParentTree.ContentBox && this.ParentTree.ContentBox.getBoundingClientRect)
    return 0;
  //
  // Se IE non devo tenere conto delle scrollbar, sugli altri browser si
  return RD3_Glb.IsIE(10, false) ? 0 : this.ParentTree.ContentBox.scrollTop;
}


// *********************************************************
// Imposta il tooltip
// *********************************************************
TreeNode.prototype.GetTooltip = function(tip, obj)
{
  if (this.Tooltip == "" || RD3_Glb.IsMobile())
    return false;
  //
  tip.SetObj(this.NodeBox);
  tip.SetTitle(this.Caption);
  tip.SetText(this.Tooltip);
  tip.SetAutoAnchor(true);
  tip.SetPosition(2);
  return true;
}


// ********************************************************************************
// Il comando e' stato toccato dall'utente
// ********************************************************************************
TreeNode.prototype.OnTouchDown= function(evento)
{ 
  
}

// ********************************************************************************
// Il comando e' stato smesso di toccare dall'utente
// ********************************************************************************
TreeNode.prototype.OnTouchUp= function(evento, click)
{ 
  
}


// ********************************************************************************
// Elimina la classe hover da tutto il menu', escluso l'oggetto indicato
// ********************************************************************************
TreeNode.prototype.SetNodeClass= function(mycmd, onlyNode)
{ 
	var x = null;
	//
	// Se devo cambiare solo il nodo corrente (io) prendo solo me stesso
	if (onlyNode)
	  x = new Array(this);
	else if (mycmd)
		x = this.Nodes;
	else if (this.ParentNode)
		x = this.ParentNode.Nodes;
	else
		x = this.ParentTree.RootNodes;
	//
	var n = x.length;
	for (var i=0; i<n; i++)
	{
		var mb = x[i].CaptionBox;
		if (mb)
		{
			if (this.CanBeChecked && this.ParentTree.MultipleSelection)
			{
				// Se ci sono i check, non posso impostare la classe qui
			}
			else
			{
				if (x[i]==this.ParentTree.ActiveNode)
	    		RD3_Glb.AddClass(mb, "tree-node-hover");
				else
	    		RD3_Glb.RemoveClass(mb, "tree-node-hover");
	    }
    	//
    	x[i].HandleExpandable();
    }
	}
}


// ********************************************************************************
// Torna true se questo elemento e' quello attivo nell'albero
// ********************************************************************************
TreeNode.prototype.IsActiveNode= function()
{ 
	return this.ParentTree.ActiveNode == this;
}


// *******************************************************************
// Chiamato quando cambia il colore di accento
// *******************************************************************
TreeNode.prototype.AccentColorChanged = function(reg, newc) 
{
  
}

// **************************************************************************************************
// Collassa l'intero ramo dell'albero
// **************************************************************************************************
TreeNode.prototype.CollapseBranch= function()
{ 
  if (this.Expanded)
  {
    // Per prima cosa collasso tutti i miei figli: in questo modo il collassamento avviene deep-first
    var n = this.Nodes.length;
    for (var i=0; i<n; i++)
	    this.Nodes[i].CollapseBranch();
	  //
	  // Mi collasso in maniera immediata
	  this.SetExpanded(false, true, false, true);
	}
}
