/*
 * Instant Developer Cloud
 * Copyright Pro Gamma Spa 2000-2021
 * All rights reserved
 */

var Client = Client || {};


/**
 * @class A group of panel fields
 * @param {Object} widget
 * @param {View|Element} parent - the parent element
 * @param {View} view
 */
Client.IdfGroup = function (widget, parent, view)
{
  // Add this group to panel groups array
  parent.addGroup(this);
  //
  this.listContainersConf = [];
  let inList = true;
  //
  // Get group's fields from parent panel
  this.fields = [];
  for (let i = 0; i < parent.fields.length; i++) {
    let field = parent.fields[i];
    if (field.groupId === widget.id) {
      field.group = this;
      this.fields.push(field);
      //
      inList = inList && field.isInList();
    }
  }
  //
  // Set default values
  widget = Object.assign({
    listHeaderPosition: Client.IdfGroup.headerPositions.INNER,
    formHeaderPosition: Client.IdfGroup.headerPositions.INNER,
    inList,
    enabled: true,
    visible: true,
    collapsible: false,
    collapsed: false,
    pageIndex: 0,
    collapseAnimationDef: Client.IdfWebEntryPoint.getAnimationDefault("group")
  }, widget);
  //
  // Set original dimensions and position
  this.orgListWidth = widget.listWidth;
  this.orgListHeight = widget.listHeight;
  this.orgListLeft = widget.listLeft;
  this.orgListTop = widget.listTop;
  this.orgFormWidth = widget.formWidth;
  this.orgFormHeight = widget.formHeight;
  this.orgFormLeft = widget.formLeft;
  this.orgFormTop = widget.formTop;
  //
  Client.Widget.call(this, widget, parent, view);
};


// Make Client.IdfGroup extend Client.Widget
Client.IdfGroup.prototype = new Client.Widget();


Client.IdfGroup.transPropMap = {
  flg: "flags",
  img: "image",
  lle: "listLeft",
  lto: "listTop",
  lwi: "listWidth",
  lhe: "listHeight",
  fle: "formLeft",
  fto: "formTop",
  fwi: "formWidth",
  fhe: "formHeight",
  pag: "pageIndex",
  lhp: "listHeaderPosition",
  fhp: "formHeaderPosition",
  hhe: "headerHeight",
  hwi: "headerWidth",
  inl: "inList",
  clp: "collapsible",
  col: "collapsed",
  mfl: "listMovedFields",
  mff: "formMovedFields",
  cla: "collapseAnimationDef"
};


Client.IdfGroup.headerPositions = {
  NONE: 1,
  BORDER: 2,
  OUTER: 3,
  INNER: 4
};


/**
 * Convert properties values
 * @param {Object} props
 */
Client.IdfGroup.convertPropValues = function (props)
{
  props = props || {};
  //
  for (let p in props) {
    switch (p) {
      case Client.IdfGroup.transPropMap.flg:
      case Client.IdfGroup.transPropMap.lwi:
      case Client.IdfGroup.transPropMap.lhe:
      case Client.IdfGroup.transPropMap.lle:
      case Client.IdfGroup.transPropMap.lto:
      case Client.IdfGroup.transPropMap.fwi:
      case Client.IdfGroup.transPropMap.fhe:
      case Client.IdfGroup.transPropMap.fle:
      case Client.IdfGroup.transPropMap.fto:
      case Client.IdfGroup.transPropMap.pag:
      case Client.IdfGroup.transPropMap.lhp:
      case Client.IdfGroup.transPropMap.fhp:
      case Client.IdfGroup.transPropMap.hhe:
      case Client.IdfGroup.transPropMap.hwi:
        props[p] = parseInt(props[p]);
        break;

      case Client.IdfGroup.transPropMap.inl:
      case Client.IdfGroup.transPropMap.clp:
      case Client.IdfGroup.transPropMap.col:
      case Client.IdfGroup.transPropMap.mfl:
      case Client.IdfGroup.transPropMap.mff:
        props[p] = props[p] === "1";
        break;
    }
  }
};


/**
 * Create elements configuration
 */
Client.IdfGroup.prototype.createElementsConfig = function ()
{
  // Each field has a "list" and a "form" configuration
  let config = {};
  //
  // 1) Create list container configuration
  if (this.isShown()) {
    config.list = this.createContainerConfig();
    config.aggregate = this.createAggregateContainerConfig();
  }
  //
  // 2) Create form container configuration
  if (this.isShown(true))
    config.form = this.createContainerConfig(true);
  //
  return config;
};


/**
 * Create elements configuration for list/form mode
 * @param {Boolean} form
 */
Client.IdfGroup.prototype.createContainerConfig = function (form)
{
  let containerConf, headerConf, headerTextConf;
  //
  if (!form && this.isInList()) {
    // Create list container
    let offsetCol = this.parent.getHeaderOffset() ? " offset-col" : "";
    this.listContainerConf = this.createElementConfig({c: "IonCol", className: "panel-list-col panel-list-group" + offsetCol});
    //
    // Create list header row configuration
    let listHeaderConf = this.createElementConfig({c: "IonRow", className: "panel-list-group-header"});
    this.listContainerConf.children.push(listHeaderConf);
    //
    // Create list header column configuration
    let listHeaderColConf = this.createElementConfig({c: "IonCol", className: "panel-list-col"});
    listHeaderConf.children.push(listHeaderColConf);
    //
    // Create list header text configuration
    headerTextConf = this.createElementConfig({c: "IonText", type: "span"});
    listHeaderColConf.children.push(headerTextConf);
    //
    // Create list content row
    this.listContentConf = this.createElementConfig({c: "IonRow", className: "panel-list-group-content"});
    this.listContainerConf.children.push(this.listContentConf);
    //
    // Save elements ids
    this.listHeaderId = listHeaderConf.id;
    this.listHeaderTextId = headerTextConf.id;
    this.listContentId = this.listContentConf.id;
    //
    containerConf = this.listContainerConf;
  }
  else {
    // Create container configuration with a collapse button and a text for header
    headerConf = this.createElementConfig({c: "Container", className: "panel-group-header"});
    let collapseButtonConf = this.createElementConfig({c: "IonButton", className: "treenode-exp-icon", events: ["onClick"]});
    headerTextConf = this.createElementConfig({c: "IonText", type: "span"});
    headerConf.children.push(collapseButtonConf);
    headerConf.children.push(headerTextConf);
    //
    containerConf = this.createElementConfig({c: "Container", className: "panel-group-content collapsible-container"});
    containerConf.animations = [{trigger: "animation", prop: "collapseElement", duration: (this.collapseAnimationDef.indexOf("none") === 0 ? 0 : 250)},
      {trigger: "animation", prop: "expandElement", duration: (this.collapseAnimationDef.indexOf("none") === 0 ? 0 : 250)}];
    //
    // Save elements ids
    if (form) {
      this.formHeaderId = headerConf.id;
      this.formCollapseId = collapseButtonConf.id;
      this.formHeaderTextId = headerTextConf.id;
      this.formContainerId = containerConf.id;
    }
    else {
      this.outListHeaderId = headerConf.id;
      this.outListCollapseId = collapseButtonConf.id;
      this.outListHeaderTextId = headerTextConf.id;
      this.outListContainerId = containerConf.id;
    }
  }
  //
  return {headerConf, containerConf};
};


/**
 * Create aggregate container configuration
 */
Client.IdfGroup.prototype.createAggregateContainerConfig = function ()
{
  if (!this.isInList())
    return;
  //
  this.aggregateContainerConf = this.createElementConfig({c: "IonCol", className: "panel-list-col"});
  //
  // Create aggregate container
  this.aggregateRowConf = this.createElementConfig({c: "IonRow", className: "panel-list-group-content"});
  this.aggregateContainerConf.children.push(this.aggregateRowConf);
  //
  this.aggregateRowId = this.aggregateRowConf.id;
  //
  return this.aggregateContainerConf;
};


/**
 * Create a column configuration that will be part of panel grid
 * @param {Integer} index
 */
Client.IdfGroup.prototype.createListConfig = function (index)
{
  // Create column configuration
  let offsetCol = this.parent.getListRowOffset() ? " offset-col" : "";
  let columnConf = this.createElementConfig({c: "IonCol", className: "panel-list-col" + offsetCol});
  //
  // Create row configuration
  let rowConf = this.createElementConfig({c: "IonRow", className: "panel-list-group-content"});
  columnConf.children.push(rowConf);
  //
  // Save element configuration
  this.listContainersConf[index] = columnConf;
  //
  return this.listContainersConf[index];
};


/**
 * Realize widget UI
 * @param {Object} widget
 * @param {View|Element|Widget} parent
 * @param {View} view
 */
Client.IdfGroup.prototype.realize = function (widget, parent, view)
{
  // Create elements configuration
  let config = this.createElementsConfig();
  //
  let headerEl;
  let contentEl;
  //
  // Create list version of this group
  if (config.list) {
    // When a group is in list, it has a main container (an IonCol) having two children: header row and container row.
    // Instead, an out of list group doesn't own an IonCol (the main grid does) and so I have to explicitly create header and container
    let headerConf = config.list.headerConf;
    //
    // Create group elements and append them to parent (panel rootObject). Then they will be appended in the proper object
    if (headerConf) {
      headerEl = view.createElement(headerConf, parent, view);
      this.mainObjects.push(headerEl);
    }
    //
    contentEl = view.createElement(config.list.containerConf, parent, view);
    this.mainObjects.push(contentEl);
    //
    // Create aggregate container and append it to parent (panel rootObject). Then it will be appended in the proper object
    if (config.aggregate)
      this.mainObjects.push(view.createElement(config.aggregate, parent, view));
  }
  //
  // Create form version of this group
  if (config.form) {
    // Create group elements and append them to parent (panel rootObject). Then they will be appended in the proper object
    headerEl = view.createElement(config.form.headerConf, parent, view);
    this.mainObjects.push(headerEl);
    //
    contentEl = view.createElement(config.form.containerConf, parent, view);
    this.mainObjects.push(contentEl);
  }
};



/**
 * Update element properties
 * @param {Object} props
 */
Client.IdfGroup.prototype.updateElement = function (props)
{
  props = props || {};
  //
  let calcLayout;
  let updateStructure;
  let updateHeader, updateClassName, updateImage, updateEnabled;
  //
  Client.Widget.prototype.updateElement.call(this, props);
  //
  if (props.flags !== undefined) {
    let enabled = !!(props.flags & 0x01);
    if (enabled !== this.enabled)
      props.enabled = enabled;
    //
    let visible = !!(props.flags & 0x02);
    if (visible !== this.visible)
      props.visible = visible;
  }
  //
  if (props.enabled !== undefined) {
    this.enabled = props.enabled;
    updateEnabled = true;
  }
  //
  if (props.visible !== undefined) {
    this.visible = props.visible;
    calcLayout = true;
  }
  //
  if (props.tooltip !== undefined)
    updateHeader = true;
  //
  if (props.collapsible !== undefined) {
    this.collapsible = props.collapsible;
    this.updateCollapsible();
  }
  //
  if (props.collapsed !== undefined) {
    this.collapsed = props.collapsed;
    //
    // Collapse or expand list container
    this.handleCollapse();
    //
    // Collapse or expand form container
    this.handleCollapse(true);
  }
  //
  if (props.inList !== undefined)
    this.inList = props.inList;
  //
  if (props.caption !== undefined)
    updateHeader = true;
  //
  if (props.listHeaderPosition !== undefined) {
    this.listHeaderPosition = props.listHeaderPosition;
    updateHeader = true;
  }
  //
  if (props.formHeaderPosition !== undefined) {
    this.formHeaderPosition = props.formHeaderPosition;
    updateHeader = true;
  }
  //
  if (props.headerWidth !== undefined)
    this.headerWidth = isNaN(props.headerWidth) ? undefined : props.headerWidth;
  //
  if (props.headerHeight !== undefined)
    this.headerHeight = isNaN(props.headerHeight) ? undefined : props.headerHeight;
  //
  if (props.listWidth !== undefined) {
    this.listWidth = isNaN(props.listWidth) ? undefined : props.listWidth;
    this.orgListWidth = this.listWidth;
    updateStructure = true;
  }
  //
  if (props.listHeight !== undefined) {
    this.listHeight = isNaN(props.listHeight) ? undefined : props.listHeight;
    this.orgListHeight = this.listHeight;
    updateStructure = true;
  }
  //
  if (props.listLeft !== undefined) {
    this.listLeft = isNaN(props.listLeft) ? undefined : props.listLeft;
    this.orgListLeft = this.listLeft;
    updateStructure = true;
  }
  //
  if (props.listTop !== undefined) {
    this.listTop = isNaN(props.listTop) ? undefined : props.listTop;
    this.orgListTop = this.listTop;
    updateStructure = true;
  }
  //
  if (props.formWidth !== undefined) {
    this.formWidth = isNaN(props.formWidth) ? undefined : props.formWidth;
    this.orgFormWidth = this.formWidth;
    updateStructure = true;
  }
  //
  if (props.formHeight !== undefined) {
    this.formHeight = isNaN(props.formHeight) ? undefined : props.formHeight;
    this.orgFormHeight = this.formHeight;
    updateStructure = true;
  }
  //
  if (props.formLeft !== undefined) {
    this.formLeft = isNaN(props.formLeft) ? undefined : props.formLeft;
    this.orgFormLeft = this.formLeft;
    updateStructure = true;
  }
  //
  if (props.formTop !== undefined) {
    this.formTop = isNaN(props.formTop) ? undefined : props.formTop;
    this.orgFormTop = this.formTop;
    updateStructure = true;
  }
  //
  if (props.className !== undefined) {
    this.oldClassName = this.className;
    this.className = props.className;
    //
    // The className can have a responsive grid, in that case we must extract it
    let cls = Client.Widget.extractGridClasses(this.className);
    this.className = cls.className;
    this.gridClass = cls.gridClass;
    //
    updateClassName = true;
  }
  //
  if (props.image !== undefined) {
    this.image = props.image;
    updateImage = true;
  }
  //
  // If I have to update parent panel structure and layout, do it now
  if (!this.realizing)
    this.parent.updateObjects({structure: updateStructure, layout: calcLayout || updateStructure});
  //
  if (updateEnabled) {
    // Update fields controls
    for (let i = 0; i < this.fields.length; i++) {
      this.fields[i].updateControls();
      this.fields[i].applyVisualStyle();
    }
  }
  //
  // Update header for both list and form
  if (updateHeader) {
    this.updateHeader();
    this.updateHeader(true);
  }
  //
  // Update className for both list and form
  if (updateClassName) {
    this.updateClassName();
    this.updateClassName(true);
  }
  //
  // Update image for both list and form
  if (updateImage) {
    this.updateImage();
    this.updateImage(true);
  }
};


/**
 * Handle an event
 * @param {Object} event
 */
Client.IdfGroup.prototype.onEvent = function (event)
{
  let events = Client.Widget.prototype.onEvent.call(this, event);
  //
  switch (event.id) {
    case "onClick":
      if ([this.outListCollapseId, this.formCollapseId].includes(event.obj)) {
        this.updateElement({collapsed: !this.collapsed});
        //
        // Send collapse event
        if (Client.mainFrame.isIDF)
          events.push({
            id: "grpcol",
            def: Client.IdfMessagesPump.eventTypes.ACTIVE,
            content: {
              oid: this.id,
              obn: this.collapsed ? "col" : "exp",
              xck: event.content.offsetX,
              yck: event.content.offsetY
            }
          });
        else // On IDC send onCollapse event
          events.push({
            id: "onCollapse",
            obj: this.id,
            content: this.collapsed
          });
      }
      //
      break;
  }
  //
  return events;
};


/**
 * Return true if this group contains just in list fields
 */
Client.IdfGroup.prototype.isInList = function ()
{
  return this.inList;
};


/**
 * Calculate layout
 */
Client.IdfGroup.prototype.calcLayout = function ()
{
  // Calculate layout for list mode
  if (this.isShown() && this.parent.getListFieldColumn(this.id)) {
    if (!this.isInList())
      this.parent.setEdgeColumns(null, this);
    //
    this.calcListFormLayout();
  }
  //
  // Calculate layout for form mode
  if (this.isShown(true) && this.parent.getFormFieldColumn(this.id)) {
    this.parent.setEdgeColumns(true, this);
    this.calcListFormLayout(true);
  }
};


/**
 * Calculate layout for list or form mode
 * @param {Boolean} form
 */
Client.IdfGroup.prototype.calcListFormLayout = function (form)
{
  let groupColumn;
  let groupColStyle = {};
  //
  let xs = this.canAdaptWidth(form) ? "" : "auto";
  //
  let width = form ? this.formWidth : this.listWidth;
  let height = form ? this.formHeight : this.listHeight;
  let left = form ? this.formLeft : this.listLeft;
  let top = form ? this.formTop : this.listTop;
  //
  // Use flex to handle width resize
  if (!form && this.isInList()) {
    let adaptableFieldsCount = this.fields.filter(f => f.canAdaptWidth()).length;
    groupColStyle.flexGrow = adaptableFieldsCount;
    groupColStyle.flexShrink = adaptableFieldsCount;
    //
    let groupWidth = this.getWidth();
    groupColStyle.flexBasis = groupWidth.total + "px";
    groupColStyle.minWidth = groupWidth.fixed + "px";
    //
    // Assign fixed height to group
    groupColStyle.height = this.parent.getHeaderHeight() + "px";
  }
  else {
    groupColStyle.padding = "0px";
    //
    // Get field parent column
    groupColumn = form ? this.parent.getFormFieldColumn(this.id) : this.parent.getListFieldColumn(this.id);
    //
    // Calculate margin left
    let fieldColumnLeft = groupColumn.rect.left || 0;
    let deltaLeft = left - fieldColumnLeft;
    groupColStyle.marginLeft = groupColumn.isMostLeft ? left + "px" : deltaLeft + "px";
    //
    // Calculate margin right
    let deltaRight = groupColumn.isMostRight ? (this.parent.width - width - left) : groupColumn.rect.deltaRight;
    groupColStyle.marginRight = deltaRight + "px";
    //
    // Calculate margin top
    let fieldColumnTop = groupColumn.rect.top || 0;
    let deltaTop = top - fieldColumnTop;
    groupColStyle.marginTop = groupColumn.isMostTop ? top + "px" : deltaTop + "px";
    //
    // Get panel toolbar height
    let toolbarHeight = this.parent.getToolbarHeight();
    //
    // Calculate margin bottom
    let deltaBottom = groupColumn.isMostBottom ? (this.parent.height - height - top - toolbarHeight) : groupColumn.rect.deltaBottom;
    groupColStyle.marginBottom = deltaBottom + "px";
  }
  //
  // Update in list group
  if (!form && this.isInList()) {
    // Update field column element
    // List containers are: header row container + data rows containers + aggregate row container
    let listContainers = [this.listContainerConf].concat(this.listContainersConf.concat([this.aggregateContainerConf]));
    for (let i = 0; i < listContainers.length; i++) {
      let el = Client.eleMap[listContainers[i]?.id];
      //
      if (!el)
        continue;
      //
      // Just the first list container (i.e. header) has a specific height
      if (i !== 0)
        groupColStyle.height = "";
      //
      el.updateElement({style: groupColStyle, xs: xs});
    }
  }
  else { // Otherwise update out list group
    let el = Client.eleMap[groupColumn.conf.id];
    if (el) {
      el.updateElement({style: groupColStyle});
      //
      let domObj = el.getRootObject();
      if (domObj.className.indexOf("panel-group") === -1)
        domObj.className += " panel-group";
    }
  }
};


/**
 * Append field dom objects to their own column
 */
Client.IdfGroup.prototype.place = function ()
{
  if (this.isShown() && this.parent.getListFieldColumn(this.id)) {
    this.placeListForm();
    //
    // If group is in list also place its aggregate container
    if (this.isInList())
      this.placeListForm({aggregate: true});
  }
  //
  if (this.isShown(true) && this.parent.getFormFieldColumn(this.id))
    this.placeListForm({form: true});
};


/**
 * Append list/form field dom object to its own column
 * @param {Object} options
 *                        - form
 *                        - aggregate: true if I have to place aggregate container
 */
Client.IdfGroup.prototype.placeListForm = function (options)
{
  options = options || {};
  let form = options.form;
  let aggregate = options.aggregate;
  //
  let parentColumn;
  if (!form && this.isInList()) {
    // Look for the last in list field before my first field. That is the position inside panel grid header where to place group container
    let myFirstField = this.fields[0];
    //
    let found;
    let nextInListDomObj;
    //
    for (let i = 0; i < this.parent.fields.length; i++) {
      let field = this.parent.fields[i];
      //
      if (found && field.isInList() && !field.group) {
        let el = Client.eleMap[aggregate ? field.aggregateContainerId : field.listContainerId];
        nextInListDomObj = el.getRootObject();
        break;
      }
      //
      if (field.id === myFirstField.id)
        found = true;
    }
    //
    // Get panel grid header
    let gridHeader = Client.eleMap[this.parent.getListFieldColumn(this.id, aggregate).conf.id];
    let gridHeaderDomObj = gridHeader.getRootObject();
    //
    // Get group container
    let groupEl = Client.eleMap[aggregate ? this.aggregateContainerConf.id : this.listContainerConf.id];
    let groupDomObj = groupEl.getRootObject();
    //
    // If I found the next in list field, append group container after its container. Otherwise append group container as last row child
    gridHeaderDomObj.insertBefore(groupDomObj, nextInListDomObj);
    //
    let groupIndex = Array.prototype.findIndex.call(gridHeaderDomObj.childNodes, child => child.id === groupEl.id);
    groupIndex = groupIndex === -1 ? gridHeader.elements.length : groupIndex;
    //
    gridHeader.elements.splice(groupIndex, 0, groupEl);
    groupEl.parent = gridHeader;
  }
  else {
    // Get column where to put group header and container
    if (form) {
      parentColumn = this.parent.getFormFieldColumn(this.id);
      this.formParentColConf = parentColumn.conf;
    }
    else {
      parentColumn = this.parent.getListFieldColumn(this.id);
      this.listParentColConf = parentColumn.conf;
    }
    //
    let headerId = form ? this.formHeaderId : this.outListHeaderId;
    let contentId = form ? this.formContainerId : this.outListContainerId;
    //
    let groupCol = Client.eleMap[parentColumn.conf.id];
    let headerEl = Client.eleMap[headerId];
    let contentEl = Client.eleMap[contentId];
    //
    // A field belonging to a group appends its elements to group column.
    // But I want fields to be appended to group content, that is a group column child.
    // Thus I remove fields elements from group column and append them to group content
    for (let i = 0; i < groupCol.elements.length; i++) {
      // Remove field row from group column
      let row = groupCol.elements.splice(i--, 1)[0];
      //
      // Append it to group content
      contentEl.getRootObject().appendChild(row.getRootObject());
      contentEl.elements.push(row);
      row.parent = contentEl;
    }
    //
    let groupRootObject = groupCol.getRootObject();
    //
    // Append header to group column
    groupRootObject.appendChild(headerEl.getRootObject());
    groupCol.elements.push(headerEl);
    headerEl.parent = groupCol;
    //
    // Append content to group column
    groupRootObject.appendChild(contentEl.getRootObject());
    groupCol.elements.push(contentEl);
    contentEl.parent = groupCol;
  }
};


/**
 * Remove field dom objects from their parent column
 */
Client.IdfGroup.prototype.unplace = function ()
{
  if (this.isShown()) {
    this.unplaceListForm();
    //
    // If group is in list also unplace its aggregate container
    if (this.isInList())
      this.unplaceListForm({aggregate: true});
  }
  //
  if (this.isShown(true))
    this.unplaceListForm({form: true});
};


/**
 * Remove field dom object from its parent column
 * @param {Object} options
 *                        - form
 *                        - aggregate: true if I have to unplace aggregate container
 */
Client.IdfGroup.prototype.unplaceListForm = function (options)
{
  options = options || {};
  let form = options.form;
  let aggregate = options.aggregate;
  //
  let headerId, contentId;
  if (!form && this.isInList())
    contentId = aggregate ? this.aggregateContainerConf.id : this.listContainerConf.id;
  else {
    headerId = form ? this.formHeaderId : this.outListHeaderId;
    contentId = form ? this.formContainerId : this.outListContainerId;
  }
  //
  // Detach header (in case of in list fields, the header is the content, so an header object does not exists)
  let headerEl = Client.eleMap[headerId];
  if (headerEl) {
    let headerRootObject = headerEl.getRootObject();
    //
    // Remove header from dom
    headerRootObject.parentNode.removeChild(headerRootObject);
    //
    // Remove header from parent elements
    let index = headerEl.parent.elements.findIndex(el => el.id === headerEl.id);
    headerEl.parent.elements.splice(index, 1);
  }
  //
  // Detach content
  let contentEl = Client.eleMap[contentId];
  let contentRootObject = contentEl.getRootObject();
  //
  // Remove content from dom
  contentRootObject.parentNode.removeChild(contentRootObject);
  //
  // Remove content from parent elements
  index = contentEl.parent.elements.findIndex(el => el.id === contentEl.id);
  contentEl.parent.elements.splice(index, 1);
};


/**
 * Check if this group can apdapt its list/form width
 * @param {Boolean} form
 */
Client.IdfGroup.prototype.canAdaptWidth = function (form)
{
  let canAdapt = false;
  for (let i = 0; i < this.fields.length; i++)
    canAdapt = canAdapt || this.fields[i].canAdaptWidth(form);
  //
  return canAdapt;
};


/**
 * Calculate group width as fields width sum
 * @param {Boolean} form
 */
Client.IdfGroup.prototype.getWidth = function (form)
{
  let width = 0;
  let total = 0, fixed = 0;
  //
  for (let i = 0; i < this.fields.length; i++) {
    let field = this.fields[i];
    //
    let width = field.getRects({form, checkVisibility: true}).width;
    total += width;
    fixed += field.canAdaptWidth() ? Client.IdfField.minWidth : width;
  }
  //
  return {total, fixed};
};


/**
 * Apply visual style
 */
Client.IdfGroup.prototype.applyVisualStyle = function ()
{
  // I don't need to apply visual style while group is realizing. When parent panel will be realized,
  // it will apply visual style to itself and to its children (i.e. fields and groups)
  if (this.realizing)
    return;
  //
  // Get group and panel visual styles
  let vis = Client.IdfVisualStyle.getByIndex(this.getVisualStyle());
  let panelVis = Client.IdfVisualStyle.getByIndex(this.parent.getVisualStyle());
  //
  // Get group and panel background colors from visual styles
  let backColor = vis.getPropertyValue(Client.IdfVisualStyle.transPropMap.col12);
  let panelBackColor = panelVis.getPropertyValue(Client.IdfVisualStyle.transPropMap.col6);
  //
  if (this.isShown()) {
    // Set visual style on list or out of list group container
    let groupColumn = this.parent.getListFieldColumn(this.id);
    let el = this.isInList() ? Client.eleMap[this.listContainerConf.id] : Client.eleMap[groupColumn.conf.id];
    this.addVisualStyleClasses(el, {objType: "group", list: this.isInList()});
    //
    // If list header position is BORDER and group background color is trasparent,
    // apply panel background color to avoid group top border to appear on header text
    if (this.listHeaderPosition === Client.IdfGroup.headerPositions.BORDER && backColor === "transparent" && !this.isInList()) {
      el = Client.eleMap[this.outListHeaderId];
      el.updateElement({style: {backgroundColor: panelBackColor}});
    }
  }
  //
  if (this.isShown(true)) {
    // Set visual style on form group
    let groupColumn = this.parent.getFormFieldColumn(this.id);
    let el = Client.eleMap[groupColumn.conf.id];
    this.addVisualStyleClasses(el, {objType: "group", list: false});
    //
    // If form header position is BORDER and group background color is trasparent,
    // apply panel background color to avoid group top border to appear on header text
    if (this.formHeaderPosition === Client.IdfGroup.headerPositions.BORDER && backColor === "transparent") {
      el = Client.eleMap[this.formHeaderId];
      el.updateElement({style: {backgroundColor: panelBackColor}});
    }
  }
};


/**
 * Get visual style
 */
Client.IdfGroup.prototype.getVisualStyle = function ()
{
  return this.visualStyle !== -1 ? this.visualStyle : this.parent.visualStyle;
};


/**
 * Update my header
 * @param {Boolean} form
 */
Client.IdfGroup.prototype.updateHeader = function (form)
{
  // If I have to update header for a form/list group and parent panel has no form/list, do nothing
  if (!this.isShown(form))
    return;
  //
  let tooltip = Client.Widget.getHTMLTooltip(this.caption, this.tooltip);
  //
  // Update header text and tooltip
  let headerTextContainer;
  if (!form)
    headerTextContainer = this.isInList() ? Client.eleMap[this.listHeaderTextId] : Client.eleMap[this.outListHeaderTextId];
  else
    headerTextContainer = Client.eleMap[this.formHeaderTextId];
  //
  headerTextContainer.updateElement({innerHTML: Client.Widget.getHTMLForCaption(this.caption), tooltip});
  //
  // Update header position
  let headerPosition = form ? this.formHeaderPosition : this.listHeaderPosition;
  let headerContainer;
  if (!form)
    headerContainer = this.isInList() ? Client.eleMap[this.listHeaderId] : Client.eleMap[this.outListHeaderId];
  else
    headerContainer = Client.eleMap[this.formHeaderId];
  //
  // Get header height
  let headerHeight = headerContainer.getRootObject().offsetHeight;
  //
  let style = {display: "", marginTop: ""};
  switch (headerPosition) {
    case Client.IdfGroup.headerPositions.NONE:
      style.display = "none";
      break;

    case Client.IdfGroup.headerPositions.BORDER:
      style.marginTop = "-" + ((headerHeight / 2) + 1) + "px";
      break;

    case Client.IdfGroup.headerPositions.OUTER:
      style.marginTop = "-" + headerHeight + "px";
      break;
  }
  //
  // The only header position allowed for in list group header is NONE. So cancel marginTop
  if (!form && this.isInList())
    style.marginTop = "";
  //
  headerContainer.updateElement({style});
};


/**
 * Update collapse button visibility
 */
Client.IdfGroup.prototype.updateCollapsible = function ()
{
  let collapseButton;
  //
  if (this.isShown() && !this.isInList()) {
    collapseButton = Client.eleMap[this.outListCollapseId];
    collapseButton.updateElement({visible: this.collapsible});
  }
  //
  if (this.isShown(true)) {
    collapseButton = Client.eleMap[this.formCollapseId];
    collapseButton.updateElement({visible: this.collapsible});
  }
};


/**
 * Handle collapsed status
 * @param {Boolean} form
 */
Client.IdfGroup.prototype.handleCollapse = function (form)
{
  // If I have to handle collapse for a form/list group and parent panel has no form/list, do nothing.
  // Also do nothing if this is an in list group. In fact, in list groups cannot be collapsed
  if (!this.isShown(form) || (!form && this.isInList()))
    return;
  //
  // Update collapse button icon
  let collapseButtonId = form ? this.formCollapseId : this.outListCollapseId;
  let collapseButton = Client.eleMap[collapseButtonId];
  collapseButton.updateElement({icon: this.collapsed ? "add" : "remove"});
  //
  // Update container collapsed status
  let containerId = form ? this.formContainerId : this.outListContainerId;
  let container = Client.eleMap[containerId];
  //
  Client.Widget.updateElementClassName(container, "collapsed", !this.collapsed);
  Client.Widget.updateElementClassName(container, "expanded", this.collapsed);
};


/**
 * Update visibility
 * @param {Integer} index
 */
Client.IdfGroup.prototype.updateVisibility = function (index)
{
  let modes = ["list", "form"];
  //
  for (let i = 0; i < modes.length; i++) {
    let form = modes[i] === "form";
    let parentColumn = form ? this.parent.getFormFieldColumn(this.id) : this.parent.getListFieldColumn(this.id);
    //
    // If I have to update visibility for a form/list group and parent panel has no form/list, do nothing
    if (!this.isShown(form) || !parentColumn)
      continue;
    //
    let visible = this.isVisible(form);
    let containerId;
    if (!form && this.isInList())
      containerId = this.listContainerConf.id;
    else {
      let column = form ? this.parent.getFormFieldColumn(this.id) : this.parent.getListFieldColumn(this.id);
      column.visible = visible;
      //
      containerId = column.conf.id;
    }
    //
    // Update physical column visibility
    let el = Client.eleMap[containerId];
    el.updateElement({style: {display: visible ? "flex" : "none"}});
  }
  //
  // Tell my fields to update their visibility
  for (let i = 0; i < this.fields.length; i++)
    this.fields[i].updateVisibility(index);
};


/**
 * Update image
 * @param {Boolean} form
 */
Client.IdfGroup.prototype.updateImage = function (form)
{
  // If I have to update image for a form/list group and parent panel has no form/list, do nothing
  if (!this.isShown(form))
    return;
  //
  let headerContainer;
  //
  if (form)
    headerContainer = Client.eleMap[this.formContainerId];
  else
    headerContainer = this.isInList() ? Client.eleMap[this.listContainerConf.id] : Client.eleMap[this.outListContainerId];
  //
  let src = (Client.mainFrame.isIDF ? "images/" : "") + this.image;
  headerContainer.updateElement({style: {backgroundImage: "url('" + src + "')"}});
};


/**
 * Update group className
 * @param {Boolean} form
 */
Client.IdfGroup.prototype.updateClassName = function (form)
{
  // If I have to update className for a form/list group and parent panel has no form/list, do nothing
  if (!this.isShown(form))
    return;
  //
  // Get container to apply className to
  let container;
  if (form)
    container = Client.eleMap[this.formContainerId];
  else
    container = this.isInList() ? Client.eleMap[this.listHeaderId] : Client.eleMap[this.outListContainerId];
  //
  // Remove old className and add the new one
  let classList = container.getRootObject().classList;
  if (this.oldClassName)
    classList.remove(this.oldClassName);
  if (this.className)
    classList.add(this.className);
  //
  container.updateElement({className: classList.value});
};


/**
 * Return true if I'm enabled
 */
Client.IdfGroup.prototype.isEnabled = function ()
{
  if (!this.enabled)
    return false;
  //
  // If I belong to a page that is not enabled, I'm not enabled
  if (this.page && !this.page.isEnabled())
    return false;
  //
  return true;
};


/**
 * Return true if I'm visible in given layout
 * @param {Boolean} form
 */
Client.IdfGroup.prototype.isVisible = function (form)
{
  if (!this.isShown(form) || !this.visible)
    return false;
  //
  // If I belong to a page that is not visible, I'm not visible
  if (this.page && !this.page.isVisible())
    return false;
  //
  return true;
};


/**
 * Return true if at least one of group fields is shown in given layout
 * @param {Boolean} form
 */
Client.IdfGroup.prototype.isShown = function (form)
{
  if (form && !this.parent.hasForm)
    return false;
  //
  if (!form && !this.parent.hasList)
    return false;
  //
  let isShown = false;
  for (let i = 0; i < this.fields.length; i++) {
    isShown = this.fields[i].isShown(form);
    if (isShown)
      break;
  }
  //
  return isShown;
};


/**
 * Handle reset cache command
 * @param {Object} options
 */
Client.IdfGroup.prototype.resetCache = function (options)
{
  let from = options.from || 1;
  let to = options.to || this.fields[0].values.length;
  //
  for (let i = from; i <= to; i++) {
    // If I have a data block coming after reset cache, I don't have to clear list container configurations belonging to that block. It's better to reuse them
    if (i >= options.dataBlockStart && i <= options.dataBlockEnd)
      continue;
    //
    if (!this.listContainersConf[i])
      continue;
    //
    delete this.listContainersConf[i];
  }
};